/*
 * Paint - an image editor for OS/2 using JLib by John Fairhurst
 *
 * Palette.hpp - the palette dialog which holds the past 16 colours used.
 *
 * Version history: 2/2/97  First Version
 *                  3/2/97  Added menu, invented palette files
 *                  9/3/97  Turned into a frame
 *
 */

#include <string.h>

#include <JPM.hpp>
#include <JColour.hpp>
#include <JMenu.hpp>
#include <JCmdH.hpp>
#include <JProfile.hpp>
#include <JResID.hpp>
#include <JSaveDlg.hpp>
#include <JFile.hpp>
#include <JPParam.hpp>

#include "Palette.hpp"
#include "PaintMgr.hpp"
#include "Painting.hpp"
#include "Settings.hpp"
#include "Colours.hpp"
#include "ResID.h"

#define DEFAULT_PALETTE_FILE  "Default.Pal"
#define CLEAN_PALETTE_FILE    "Minimal.Pal"

// Mouseclick handler ---------------------------------------------------------
PaletteHandler::PaletteHandler( Palette *p) : JCUAMouseHandler(), palette( p)
{}

BOOL PaletteHandler::contextMenu( const JMouseEvent &e)
{
   return palette->menu();
}


// Value set extension to manage colours --------------------------------------
PVSet::PVSet( Palette *pal, Painting *p) : JValueset( pal, JPoint(),
                                                      JSize(), 0,
                                                      JSize( 4, 4),
                                                      JValueset::colour),
                                           pic( p)
{
   // turn the background white
   JWindow::setColour( JPP::background, JColour::white);
}

BOOL PVSet::selected( const JPoint &cel)
{
   pic->ps()->setColour( asColour( cel));
   return true;
}

BOOL PVSet::opened( const JPoint &cel)
{
   try {
      setColour( cel, colourMgr.pick( this, asColour( cel)));
   }
   catch( Cancelled)
   {}

   return true;
}

// palette frame window; owned by a painting window ---------------------------
Palette::Palette( Painting *p)
        : JFrame( "Palette", JFrame::normal | JFrame::noByteAlign),
          pic( p), count( 0), h( this),
          vSet( this, p), cels( 16),
          pmenu( JResID( idm_palette), mgr.father()),
          fname( "Palette")
{
   // set up the window physically
   setOwner( p);
   setClient( &vSet);
   setIcon( JResID( id_ico_palette));
   loadPos( *mgr.prf(), "Pal");

   // load the default palette.
   readPalette( DEFAULT_PALETTE_FILE);

   // add mouse handler to us and to the vset
   attach( &h);
   adopt( &vSet);
   vSet.attach( &h);

   if( mgr.settings.floatPalette()) {
      pmenu.toggleItem( idm_pal_float);
      setOwner( pic);
   }
}

Palette::~Palette()
{
   vSet.detach( &h);
   detach( &h);
   hide();       // Do this so the ini never saves us as open

   if( mgr.settings.autosavePalette())
      writePalette( DEFAULT_PALETTE_FILE);
}

// A few functions to play around with Paint Palette files -------------------
// set the palette from a file
// !! This ought to run asynchronously I s'pose...
void Palette::readPalette( char *filename)
{
   try {
      JRFile file( filename);
      ulong  fsize = file.size();

      JBuffer buffer( fsize);
      file.read( buffer);

      readPalette( &buffer);
   }
   catch( JException *)
   {}
   if( JPM::current()->exn) jlib_catch();
}

// set the palette from a buffer
void Palette::readPalette( JBuffer *b)
{
   ulong *filedata = (ulong *) b->pvAddr();

   cels = filedata[ 0]; count = 0;

   // we want to redimension the valueset.
   vSet.setSize( JSize( 4, cels / 4 + ( (cels % 4) ? 1 : 0 )));

   for( int i = 1; i < cels + 1; i++)
      addColour( JColour( filedata[ i]));

   pic->ps()->setColour( vSet.asColour( vSet.chosen()));
}

// fill a buffer with the palette data, ready for spooling
JBuffer &Palette::fillPalette( JBuffer &b)
{
   ulong *pal = (ulong *) b.pvAddr();

   pal[ 0] = cels;

   for( int i = 0; i < cels; i++) {
      // dependency on palette being 4 entries wide
      JColour col = vSet.asColour( JPoint( (i % 4) + 1, (i / 4) + 1));
      pal[ i + 1] = col.asULong();
   }
   return b;
}

// write the current palette to a file
void Palette::writePalette( char *filename)
{
   JBuffer bpal( 4 * (cels + 1));
   fillPalette( bpal);
   try {
      JWFile file( filename);
      file.write( bpal);
   }
   catch( JException)
   {}
   if( JPM::current()->exn) jlib_catch();
}

// Accessor functions --------------------------------------------------------
// interface to add a new colour. Will not increase the maximum size, colour
// insertion point 'wraps round' when we reach the end.
Palette &Palette::addColour( const JColour &c)
{
   JPoint cel;
   if( count >= cels) count = 0;
   cel.x = (count % 4) + 1;
   cel.y = (count / 4) + 1;
   count++;
   vSet.setColour( cel, c);
   vSet.hilight( cel);
   return *this;
}

// request the palette's presence
// We try to sit to the left of the painting window which owns us
Palette &Palette::request()
{
   JPoint isabel;
   JSize  szPal = size();
   JPoint posPic = pic->origin();
   JSize  szPic = pic->size();

   if( posPic.x - szPal.x < 0) isabel.x = 0;
   else isabel.x = posPic.x - szPal.x;

   isabel.y = posPic.y + szPic.y - szPal.y;

   moveTo( isabel);
   show();
   activate();

   return *this;
}

// Dialog overrides ----------------------------------------------------------
// set the focus back to the picture who nominally owns us and disappear.
BOOL Palette::closing()
{
   pic->activate();
   hide();
   return true;
}

// process our menu choices
BOOL Palette::command( const JCmdEvent &e)
{
   switch( e.value()) {
      case idm_pal_float:
      {
         // toggle the item & set the owner accordingly
         if( pmenu.toggleItem( idm_pal_float))
            setOwner( pic);
         else
            setOwner( mgr.father());
         return true;
      }
      case idm_pal_save:
      {
         // create a Paint Palette file in memory...
         JBuffer *ppal = new JBuffer( 4 * (cels + 1));
         fillPalette( *ppal);

         // ...and let JLib take the strain ;-)
         JSaveDlg *sd = new JSaveDlg( this, ppal, fname,
                                      JResID( id_ico_palette));
         sd->runModally( this);

         return true;
      }
      case idm_pal_load:
      {
         // I hate this open-file dialog. It stinks :^(
         JOpenDlg *dlg = new JOpenDlg( this);

         dlg->run( this);
         return true;
      }
      case idm_pal_clean:
         // clean up the palette
         readPalette( CLEAN_PALETTE_FILE);
         return true;
      case idm_pal_default:
         // Make the current palette the default.
         writePalette( DEFAULT_PALETTE_FILE);
         return true;
      case idm_pal_settings:
         // open settings notebook at the palette page
         mgr.showSettings()->turnToPalettePage();
         return true;
   }
   return false;
}

BOOL Palette::menu()
{
   pmenu.popUp( *this);
   return true;
}

// Palette save|open dlg client methods --------------------------------------
BOOL Palette::savingImminent( JVBuffer *b, const JStr &nm, ulong id)
{ return false; } // let JLib do it

void Palette::savingComplete( JVBuffer *b, ulong id, BOOL cancelled)
{
   delete b;
}

void Palette::openedFiles( JSequence<JOpenDlgFile> &list, ulong id)
{
   JSequence<JOpenDlgFile>::cursor c( list);

   c.toFirst();
   readPalette( &c.current()->data);

   // update the palette filename
   fname = c.current()->fname;
}

void Palette::openingError( ulong id)
{}
