/*
 * Paint - an image editor for OS/2 using JLib by John Fairhurst
 *
 * Settings.hpp - settings notebook for the application
 *                settings data buffer class
 *
 * Version history:  8/2/97  First Version
 *                   3/3/97  Updated settingsdata; defcmod is a string now
 *
 */

#include <string.h>

#include <JLib.h>
#include <JPM.hpp>
#include <JCmdH.hpp>
#include <JProfile.hpp>
#include <JEntFld.hpp>
#include <JSet.hpp>
#include <JPtr.hpp>

#include "Settings.hpp"
#include "Colours.hpp"
#include "PaintMgr.hpp"
#include "ResID.h"

// Private data class with dialogs and pages in -------------------------------
struct SPage
{
   ulong           id;
   JPage          *pg;
   SettingsDialog *dlg;

   SPage( ulong i, JPage *p = 0, SettingsDialog *d = 0 ) : id( i), pg( p),
                                                           dlg( d) {}
  ~SPage() { delete pg; delete dlg; }
   int operator == ( const SPage &op2) const { return id == op2.id; }
};

ulong key( const SPage &spage) { return spage.id; }

typedef JKeySet<JPtr<SPage>,ulong> PageSet;

struct SData
{
   PageSet pages;
   SData() {}
};

// Ctor, dtor, utilities ------------------------------------------------------
Settings::Settings()
         : JDialog( mgr.father(), JResID( dlg_settings)),
           notebook( this, id_notebook),
           data( new SData)
{
   backup = mgr.settings;
   insert( dlg_pal_settings, new PaletteSettings( this, &notebook),
           "~Palette", "Page 1 of 1");
}

Settings::~Settings()
{
   delete data;
}

// utility, insert a page via the list
void Settings::insert( ulong i, SettingsDialog *d, char *t, char *a)
{
   JPage *page = new JPage( notebook.addSection( d, t, a));
   data->pages.add( new SPage( i, page, d));
}

// display the window, updating what it thinks is going on
Settings &Settings::invoke()
{
   updateFromData( mgr.settings);
   show();
   activate();
   return *this;
}

// Managing various SettingsData objects --------------------------------------
Settings &Settings::undo()
{
   return updateFromData( backup);
}

// Called when the window gets closed; everything gets updated
Settings &Settings::commit()
{
   SettingsData currentSettings;

   // fill in currentSettings from dialogs
   PageSet::cursor c( data->pages);

   for_cursor( c)
      (*c.current())->dlg->dumpToData( currentSettings);

   mgr.settings = currentSettings;
   backup = currentSettings;
   return *this;
}

Settings &Settings::updateFromData( const SettingsData &d)
{
   // go thru dlgs and set them up
   PageSet::cursor c( data->pages);

   for_cursor( c)
      (*c.current())->dlg->initFromData( d);

   return *this;
}

// scan the list for the right page. Ought not to get notfounds ---------------
Settings &Settings::turnToPalettePage()
{
   SPage *pg = *(data->pages.elementWithKey( dlg_pal_settings));
   notebook.turnToPage( *(pg->pg));
   return *this;
}

BOOL Settings::closing()
{
   removeFromTasklist();
   mgr.focus();
   hide();
   commit();
   return true;
}

BOOL Settings::command( const JCmdEvent &e)
{ return closing(); }

// !! WTF is this doing ???
BOOL Settings::handleEvents( JEventData *e)
{
// if( e->msg == 2)
//    return true;
   return false;
}

// Settings dialog class ------------------------------------------------------
SettingsDialog::SettingsDialog( const JResID &r, Settings *s, JNotebook *nb)
               : settings( s), JDialog( nb, r, s)
{
   setautodelete( false);
}

SettingsDialog::~SettingsDialog()
{}

BOOL SettingsDialog::closing()
{ return true; }

// Palette page ---------------------------------------------------------------
PaletteSettings::PaletteSettings( Settings *s, JNotebook *b)
                : SettingsDialog( dlg_pal_settings, s, b),
                  c1( this, id_chk_disp),
                  c2( this, id_chk_float),
                  c3( this, id_chk_asave),
                  cbox( this, id_cbox_cmod)
{
   colourMgr.fillCBox( &cbox);
}

PaletteSettings &PaletteSettings::initFromData( const SettingsData &s)
{
   c1.check( s.showPalette());
   c2.check( s.floatPalette());
   c3.check( s.autosavePalette());
   colourMgr.fillCBox( &cbox);
// char m = s.colourModel();
// JRadioButton *p = 0;
// switch( m) {
//    case 0: p = &r1; break;
//    case 1: p = &r2; break;
//    case 2: p = &r3; break;
//    case 3: p = &r4; break;
// }
// p->check();
   return *this;
}

PaletteSettings &PaletteSettings::dumpToData( SettingsData &s)
{
   s.showPalette( c1.isChecked());
   s.floatPalette( c2.isChecked());
   s.autosavePalette( c3.isChecked());
   s.colourModel( cbox.entryfield().getText());
   colourMgr.setDefault( s.colourModel());
   return *this;
}

BOOL PaletteSettings::command( const JCmdEvent &e)
{
   if( id_butt_undo == e.value())
      settings->undo();
   return true;
}

// Data buffer ----------------------------------------------------------------
SettingsData::SettingsData() : fOne( 0)
{}
                                             //10987654321098765432109876543210
#define PAL_FLOAT      1                     //00000000000000000000000000000001
#define PAL_SHOW       2                     //00000000000000000000000000000010
#define PAL_ASAVE      4                     //00000000000000000000000000000100

BOOL SettingsData::floatPalette() const
{ return fOne & PAL_FLOAT; }

SettingsData &SettingsData::floatPalette( BOOL yn)
{ if( yn) fOne |= PAL_FLOAT; else fOne &= ~PAL_FLOAT; return *this; }

BOOL SettingsData::showPalette() const
{ return fOne & PAL_SHOW; }

SettingsData &SettingsData::showPalette( BOOL yn)
{ if( yn) fOne |= PAL_SHOW; else fOne &= ~PAL_SHOW; return *this; }

BOOL SettingsData::autosavePalette() const
{ return fOne & PAL_ASAVE; }

SettingsData &SettingsData::autosavePalette( BOOL yn)
{ if( yn) fOne |= PAL_ASAVE; else fOne &= ~PAL_ASAVE; return *this; }

JStr SettingsData::colourModel() const
{ return nmCModel; }

SettingsData &SettingsData::colourModel( char *c)
{ nmCModel = c; return *this; }

SettingsData &SettingsData::operator = ( const SettingsData &op2)
{
   fOne = op2.fOne;
   nmCModel = op2.nmCModel;
   return *this;
}

SettingsData &SettingsData::saveToProfile( JProfile *p)
{
   p->write( fOne, "D1", "Config");
   p->write( nmCModel, "DCM");
   return *this;
}

SettingsData &SettingsData::loadFromProfile( JProfile *p)
{
   try {
      fOne = p->readLong( "D1", "Config");
      nmCModel = p->readString( "DCM");
   } catch( JException *)
   {}
   if( JPM::current()->exn) jlib_catch();
   return *this;
}
