// Spline investigator program
// Demonstrates: asynchronous drawing
//               double buffering
//               correlation
//               thread coordination
//               pointer capture
//
//  MJF Christmas 1996

// A bitmap is maintained containing a spline and markers at its control
// points. This is blitted onto the screen as required.
// Mouseclicks on the canvas trigger correlation in the bitmap; if a click is
// on one of the middle control points, a 'tracking' flag is set and the
// pointer is captured. When a mousemove condition is raised, the image data
// (coordinates of the points) are updated, and a redraw of the bitmap is
// triggered. This happens in the second thread; when the redraw is complete,
// the window is invalidated, performing a copyup.

#include <JLib.h>
#include "Test.h"
#include "Test.hpp"

// Main program ---------------------------------------------------------------
#include <JExcept.hpp>
#include <JPM.hpp>
#include <JResID.hpp>

int main()
{
   JException::debugEnabled = true;

   splineDlg dlg;

   dlg.show().activate();

   JPM::current()->process();

   return 0;
}

// spline canvas --------------------------------------------------------------
#include <JGPrim.hpp>
#include <JPaintH.hpp>
#include <JPSpace.hpp>
#include <JGLPrim.hpp>
#include <JColour.hpp>

splineCanvas::splineCanvas( splineDlg *dlg) : JCanvas( dlg, id_canvas_spline),
                                              tracking( 0),
                                              handler( this),
                                              memPS( dc),
                                              bmp( this->size()),
                                              sem( true),
                                              quit( false)
{
   pts[ 0] = JPoint( 80, 80);
   pts[ 1] = JPoint( 100, 280);
   pts[ 2] = JPoint( 240, 40);
   pts[ 3] = JPoint( 360, 300);

   memPS.select( bmp);

   JLineSettings st;
   st.setFGColour( JColour::red);
   st.setType( JGfx::line::type::dashDotDot);
   memPS.render( st);

   JMarkerSettings sts;
   sts.setFGColour( JColour::green);
   sts.setSymbol( JGfx::marker::square);
   memPS.render( sts);

   attach( &handler);

   thread.start( &splineCanvas::drawingThread, this);
}

splineCanvas::~splineCanvas()
{
   memPS.deselectBitmap();
   quit = true;
   sem.post();
   thread.wait();
}

// Rats, ought to add this to the library...
#define INCL_PM
#define INCL_GPI
#include <os2.h>

void splineCanvas::buttonDown( const JPoint &p)
{
   GpiSetPickAperturePosition( memPS.handle(), p);
   GpiSetDrawControl( memPS.handle(), DCTL_CORRELATE, DCTL_ON);

   JMarker marker( pts[ 1]);

   BOOL rc2 = false, rc1 = marker.renderIn( memPS);
   if( !rc1) {
      JMarker marker2( pts[ 2]);
      rc2 = marker2.renderIn( memPS);
   }
   GpiSetDrawControl( memPS.handle(), DCTL_CORRELATE, DCTL_OFF);

   if( rc1 || rc2) {
      if( rc1) tracking = 1;
      if( rc2) tracking = 2;
      capturePointer();
   }
}

void splineCanvas::buttonUp( const JPoint &p)
{
   tracking = 0;
   releasePointer();
}

void splineCanvas::mouseMove( const JPoint &p)
{
   if( tracking && JRect( 0, 0, size()).contains( p)) {
      // update the image data
      pts[ tracking] = p;
      // if the picture isn't being redrawn, tell it to redraw.
      // when complete, it'll be copied up
      if( sem.postCount() == 0) sem.post();
   }
}

BOOL splineCanvas::paint( JPaintEvent &e)
{
   // just blit the required area from memory onto the screen
   e.ps().render( JBmpImage( memPS, e.updateRect(), e.updateRect()));
   return true;
}

// this member runs as a separate thread to update the buffered image
ulong splineCanvas::drawingThread()
{
   BOOL firstTime = true;
   for(;;) {                // do while do
      if( !firstTime) sem.wait();

      if( quit) break;

      JPointList points;

      for( int i = 0; i < 4; i++)
         points.append( pts[ i]);

      memPS.clear();
      memPS.render( JMarkers( points));
      memPS.render( JMove( pts[ 0]));
      points.removeAt( 0);
      memPS.render( JSpline( points));

      // this does the screen copy-up, indirectly...
      invalidate();

      if( firstTime) firstTime = false;
      else sem.reset();
   };
   return 0;
}

// spline dialog ( very boring) -----------------------------------------------
splineDlg::splineDlg() : JDialog( &JWindow::theDesktopWindow, id_dlg_spline),
                         canvas( this)
{
   showInTasklist();
   centreOver( &JWindow::theDesktopWindow);
}

BOOL splineDlg::closing()
{ JPM::current()->quit(); return false;}

// spline handler class (boring) ----------------------------------------------
splineHandler::splineHandler( splineCanvas *c) : cvs( c)
{}

BOOL splineHandler::button2down( const JClickEvent &e)
{
   cvs->buttonDown( e.pos());
   return false;
}

BOOL splineHandler::button2up( const JClickEvent &e)
{
   cvs->buttonUp( e.pos());
   return false;
}

BOOL splineHandler::mouseMove( const JMouseEvent &e)
{
   cvs->mouseMove( e.pos());
   return false;
}
