/**
 * CHandler for JamochaMUD will act as our "proxy" between
 * the core of JamochaMUD and our different sockets, if we
 * have simultaneous connections to other MU*s
 * $Id: CHandler.java 1.12 2001/11/05 03:18:12 jeffnik Exp $
 */

/* JamochaMUD, a Muck/Mud client program
 * Copyright (C) 1998-2001 Jeff Robinson
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

package anecho.JamochaMUD;

import java.awt.Color;
import java.awt.Font;

import java.util.Vector;

import anecho.gui.JMText;

/** To allow an easier time of dealing with multiple connections,
 * any input or output from JamochaMUD will pass through the CHandler
 * class.  This class will then determine which MU* the outbound
 * traffic will go out (eg. the &quot;active" MU* if from the DataIn
 * window) or the appropriate window to send incoming traffic
 * based on the socket it is received from */
public class CHandler {

    private Vector connections;   // A list of our active sockets/MUs
    private int activeMU;               // which is our active MU*
    private JMConfig settings;          // a pointer to our settings
    private MuckMain mainWindow;

    /** Our constructor.  Since this should only be called once
     * per invocation of JamochaMUD, we will set up the &quot;frames"
     * for our MU* client. */
    public CHandler(JMConfig mainSettings, MuckMain mainWindow) {
        this.settings = mainSettings;
        this.mainWindow = mainWindow;

        connections = new Vector(0, 1);

        // Since this has just been called, chances are we're
        // not connected to anything.  So let's start it off!
        // listConnections();
        connectToNewMU();

    }

    /** Create a new socket and add a &quot;window" for it */
    public void OpenSocket(String name, String address, int port) {
        // create a new MuSocket, and then populate it with the proper info
        MuSocket msock = new MuSocket(settings);
        
        msock.resetTitle();
        msock.setAddress(address);
        msock.setPort(port);

        // We'll set a timestamp to this MU* to differentiate incase the
        // user hooks up to the same MU* multiple times.
        msock.setTimeStamp(System.currentTimeMillis());

        // Now add the new Socket to the connections vector
        connections.addElement(msock);

        activeMU = connections.size() - 1;

        // Add this new TextWindow to the existing main frame.
        // We're assuming, since this is the newest connection,
        // that this is the one we'll want visible
        // mainWindow.setActiveMU(msock.getTextWindow());
        mainWindow.addNewMU(msock);
        mainWindow.setVisibleMU();

        // set the layout if it hasn't been done already.  We do this from here
        // so that we don't try to set a layout before we have at least one JMText area
        if (connections.size() == 1) {
            mainWindow.setMainLayout();
        }

        // Now we'll start our socket-listening thread
        msock.start();

    }

    /** make the next MU* in the list the active MU*
     * The process that makes the call to this method
     * is responsible for updating the display appropriately
     */
    public void nextMU() {
        // Check to see if we have more than one MU, otherwise this is pointless
        if (connections.size() < 1) return;

        activeMU++;
        if (activeMU > (connections.size() - 1)) {
            activeMU = 0;
        }
    }

    /** Make the previous MU* in the list the active MU*.
     * As with nextMU, the calling process is responsible
     * for updating the display appropriately.
     */
    public void previousMU() {
        // Don't do anything if we only have one connection
        if (connections.size() < 1) return;

        activeMU--;
        if (activeMU < 0) {
            activeMU = connections.size() - 1;
        }
    }

    /** Close the active MU* */
    public void closeActiveMU() {
        MuSocket active = getActiveMUHandle();
        closeSocket(active);
    }

    public void removeActiveMU() {
        // Disconnect the MU* if still connected
        if (isActiveMUDConnected()) {
            closeActiveMU();
        }

        // "destroy" any physical entities
        // uhm... somehow.  We could rely on garbage collection but
        // I'd feel better actually knowing it was gone.
        // Fix this XXX

        // remove the MU* from our hashtable
        connections.removeElementAt(activeMU);

        // Now make certain that we're pointing at the next logical MU*
        if (activeMU > 0) {
            activeMU--;
        }

    }
    /** Close and &quot;destroy" an existing socket.
     * This allows any of our clases to shut things down
     */
    public void closeSocket(MuSocket socket) {
        // Clean up the existing socket
        socket.closeSocket();

        // Set the menu state
        MuckMain main = settings.getMainWindowVariable();
        main.setConnectionMenu();
    }

    /** Return the 'address' for the active JMText object */
    public JMText getActiveMUDText() {
        MuSocket msock = (MuSocket)connections.elementAt(activeMU);
        return msock.getTextWindow();
    }

    /** Return the proper MuSocket for the requested mu */
    public MuSocket getMUHandle(int mu) {
        if (mu > (connections.size() - 1) || mu < 0) {
            // Can't give you info about something that doesn't exist!
            return null;
        }
        return (MuSocket)connections.elementAt(mu);
    }

    /** Return the MuSocket for the currently &quot;active" MU* */
    public MuSocket getActiveMUHandle() {
        // Check to see if we have any connections yet
        if (connections.size() < 1) {
            return null;
        }
        return (MuSocket)connections.elementAt(activeMU);
    }

    /** Return the status of the active MU* */
    public boolean isActiveMUDConnected() {
        if (connections.size() < 1) {
            // We hve no MU*'s listed, so obviously it isn't connectioned
            return false;
        }
        MuSocket msock = (MuSocket)connections.elementAt(activeMU);
        return msock.isConnectionActive();
    }

    /** Send a string to the active MU*, using the
     * proper encoding method (ASCII or Unicode) */
    public void sendText(String send) {
        MuSocket msock = (MuSocket)connections.elementAt(activeMU);
        msock.sendText(send);
    }

    /** Change the font face and size on all the connections */
    public void setAllAttribs(Font fontFace, Color fg, Color bg) {
        MuSocket msock;       // our temporary socket
        JMText text;          // our temporary text object

        int total = connections.size();

        if (total < 1) {
            // There are no active MU*s
            return;
        }

        for (int i = 0; i < total; i++) {
            // Loop through our connections and change the fonts
            msock = (MuSocket)connections.elementAt(i);
            text = msock.getTextWindow();
            if (fontFace != null) {
                text.setFont(fontFace);
            }
            if (fg != null) {
                text.setForeground(fg);
                text.setBackground(bg);
            }
        }
    }

    public void setAllColours(Color fg, Color bg) {
        setAllAttribs(null, fg, bg);
    }

    public void setAllFonts(Font newStyle) {
        setAllAttribs(newStyle, null, null);
    }

    /** Set-up a connection to a new MU* */
    // public void connectToNewMU() {
    public synchronized void connectToNewMU() {
        // Show our list of connections (the MuckConnector)
        // listConnections();
        MuckConn muChoice = new MuckConn(this, settings);

        if (muChoice.getName() == null) {
            // A MU* was not selected
            return;
        }

        String name, address;
        int port;

        // Gather information about our chosen connection
        name = muChoice.getName();
        address = muChoice.getAddress();
        port = muChoice.getPort();

        // Open our new connection
        OpenSocket(name, address, port);

    }

    /** The total number of connections we're wrangling */
    public int totalConnections() {
        return connections.size();
    }

    /** Allow another program to set which of our MU*s are active */
    public void setActiveMU(int mu) {
        // First, check to see if it fits our parameters
        if (mu < 0 || mu > (connections.size() -1)) {
            System.out.println(mu + " is not a valid MU number");
        }

        activeMU = mu;
    }

    /** Return the title of the active MU* */
    public String getActiveTitle() {
        if (connections.size() < 1) {
            // We have no connections, thus no title
            return "";
        }

        MuSocket msock = (MuSocket)connections.elementAt(activeMU);
        return msock.getTitle();
    }

    /** Change the drawing type for the JMText... either
     * single buffered (false) or double buffered (true)
     */
    public void setDoubleBuffer(boolean state) {
        MuSocket msock;
        for (int i = 0; i < connections.size(); i++) {
            msock = (MuSocket)connections.elementAt(i);
            (msock.getTextWindow()).setDoubleBuffer(state);
        }
    }
}
