/**
 * DataIn, text input window
 * $Id: DataIn.java 1.14 2001/11/05 03:18:12 jeffnik Exp $
 */

/* JamochaMUD, a Muck/Mud client program
 * Copyright (C) 1998-2000  Jeff Robinson
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

package anecho.JamochaMUD;

import java.awt.*;
import java.awt.event.*;

import anecho.gui.SyncFrame;
import anecho.gui.SyncFrameGroup;
import anecho.gui.ResReader;

	/**
	 * This is the input window where the user may type to send info
	 * to the MUCK/MUD.  In addition, if features scrolling of long messages
	 * and a 10 command 'recall' feature implemented by a right mouse-click.
         * @version $Id: DataIn.java 1.14 2001/11/05 03:18:12 jeffnik Exp $
         * @author Jeff Robinson
	 */
public class DataIn extends SyncFrame implements ActionListener, KeyListener, MouseListener, WindowListener{

    private PopupMenu backTrack;
    private String typedLine, bCPopup[];
    private String backComm[];
    private TextArea dataText;
    private JMConfig settings;
    private CHandler connHandler;

    public DataIn(JMConfig mainSettings){

        super(RB("dataIn.title"));
        this.settings = mainSettings;
        this.connHandler = settings.getConnectionHandler();

        addWindowListener(this);

        // Set Gridbag layout
        GridBagLayout dILayout = new GridBagLayout();
        GridBagConstraints constraints = new GridBagConstraints();

        backTrack = new PopupMenu();
        backComm = new String[10];

        // Add the text field to the box.
        constraints.gridwidth = 1;
        constraints.gridheight = 1;
        constraints.gridx = 0;
        constraints.gridy = 0;
        constraints.fill = GridBagConstraints.BOTH;
	dataText = new TextArea("", 3, 70, TextArea.SCROLLBARS_VERTICAL_ONLY);
	
	dataText.addKeyListener(this);
        add(dataText);
	
	Menu testMenu = new Menu("TestMenu!");
	backTrack.addActionListener(this);
	dataText.add(backTrack); 		// Adds PopupMenu to data entry area
	dataText.addMouseListener(this);	// Adds MouseListener to data entry area
	
        pack();

        // Sure, this is a bit of work, but we so love our icons, don't we?
        MediaTracker tracker = new MediaTracker(this);
        Image jamochaMUDImage = Toolkit.getDefaultToolkit().getImage("kehza.gif");
        tracker.addImage(jamochaMUDImage, 0);
        try {
            tracker.waitForAll();
        } catch (Exception e) {
            System.out.println("Icon load error.  Exception " + e);
        }
        this.setIconImage(jamochaMUDImage);

	// Display macroBar
        dataText.setForeground(settings.getForegroundColour());
        dataText.setBackground(settings.getBackgroundColour());
        dataText.setFont(settings.getFontFace());
    }

    public void keyPressed(KeyEvent event){

        int arg = event.getKeyCode();

        if (arg == KeyEvent.VK_ENTER) {
            JMSendText(dataText.getText());
            event.consume();
        }

        if (arg == KeyEvent.VK_ALT)
            (settings.getMainWindowVariable()).requestFocus();

        if (arg == KeyEvent.VK_UP) {
            // Check to see if the cursor is allowed to move up
            if (dataText.getCaretPosition() <= dataText.getColumns()) {
                // Caret is on first line, can't move up
                event.consume();
            }

        }

        if ((arg == KeyEvent.VK_X) && (event.isControlDown())) {
            // 'Supervisor' key was called.
            // Minimize all the windows
            HideFramesQuick();
        }

        if ((arg == KeyEvent.VK_Z) && (event.isControlDown())) {
            // restore minimized windows
            RestoreMinimizedFrames();

        }

        if (event.isControlDown()) {
            // Check for any "editor" controls, such as CTRL+SHIFT+ARROW
            if (arg == KeyEvent.VK_LEFT || arg == KeyEvent.VK_RIGHT) {
                // Nothing to see here, let the system handle it!
                return;
            }

            // First, check to see if this is for TinyFugue emulation
            if (settings.getTFKeyEmu()) {
                // Check this for applicable TinyFugue keys
                // and if it is, it will probably consume the event
                // STUB

                // I don't like how this is called.  Fix Me XXX!
                if (JMTFKeys.JMTFKeyStroke(settings, event.getKeyCode())) {
                    // Well, boys and girls, it really was a TF keystroke!
                    // let us make sure this event goes no further!
                    event.consume();
                }
            }

            // We'll send this to MuckMain, to see if
            // it is supposed to be a menu shortcut
            MuckMain tempMain = settings.getMainWindowVariable();
            tempMain.dispatchEvent(event);

        }

        return;
    }

    public void keyReleased(KeyEvent event){}

    public void keyTyped(KeyEvent event){
        // We must make one exception, if the user is using TinyFugue emulation
        // so that we do not spool out text if they're releasing the 'pause text' key
        if (settings.getTFKeyEmu() && event.isControlDown()) {
            event.consume();
        } else {
            checkPause();
        }
    }

    public void actionPerformed(ActionEvent event){
        String arg = event.getActionCommand();
        if (arg == "^") {
            typedLine = dataText.getText();
        }

        // Determine if a popupmenu item was selected (until a better way is found)
        for (int i = 0; i < backTrack.getItemCount(); i++) {
            if (arg.equals((backTrack.getItem(i).getLabel()))) {
                JMSetFromScrollBack(i);
            }
        };
        return;
    }

    // Check for mouse actions

    public void mousePressed(MouseEvent e) {
        checkPopup(e) ;
    }

    public void mouseReleased(MouseEvent e) {
        checkPopup(e) ;
    }

    public void mouseClicked(MouseEvent e) {
        // This spools out any paused text
        checkPause();
    }

    public void mouseEntered(MouseEvent e) {
    }

    public void mouseExited(MouseEvent e) {
    }

    /**
     * Check for PopupTrigger to display command history
     */
    void checkPopup(MouseEvent e) {
        if (e.isPopupTrigger()) {
            ShowPopup(this, e.getX(), e.getY());
            e.consume();
        }
    }

    /** Display the pop-up menu */
    public void ShowPopup(Component origin, int x, int y) {

        // if x and y come in as -1, then we set the popup location from just
        // the frame as this method was probably called by an outside class
        if (x == -1 && y == -1) {
            origin = this;
            x = (int)(this.getSize().width / 2);
            y = (int)(this.getSize().height / 2);

        }

        backTrack.show(origin, x, y);
    }

    /**
     * Bring the frame into focus so the user may type
     */
    public void windowActivated(WindowEvent event) {
        // First we insert a 6 millisecond kludge to make
        // up for the requestFocus timing problem in Java 1.1.7+
        long startTime = java.lang.System.currentTimeMillis();
        long currentTime = java.lang.System.currentTimeMillis();

        while ((currentTime - startTime) < 7) {
            // Just an empty loop
            currentTime = java.lang.System.currentTimeMillis();
        }

        // Now request focus
        dataText.requestFocus();

    }

    public void windowClosed(WindowEvent event) {}

    public void windowClosing(WindowEvent event) {}

    public void windowDeactivated(WindowEvent event) {}

    public void windowDeiconified(WindowEvent event) {}

    public void windowIconified(WindowEvent event) {
        // record the frame into the hashtable
        settings.setDataBar(getBounds());
    }

    public void windowOpened(WindowEvent event) {}

//    public void processComponentEvent(ComponentEvent event) {
//        super.processComponentEvent(event); // Handle listeners
//
//        if (event.getID() == ComponentEvent.COMPONENT_MOVED || event.getID() == ComponentEvent.COMPONENT_RESIZED) {
//            settings.setDataBar(this.getBounds());
//            System.out.println("Databar set new bounds.");
//        }
//
//    }

    /**
     * This is a generic bit to access the
     * ResReader.class, for localization
     * (Multi-language support)
     */
    private static String RB(String itemTarget) {
        ResReader reader = new ResReader();
        // return reader.LangString("JamochaMUDBundle", itemTarget);
        return reader.LangString(JMConfig.BUNDLEBASE, itemTarget);
    }

    /**
     * Spool out any text that may have been paused
     */
    public void SpoolText() {

        System.out.println("DataIn.SpoolText() was called");
        // resetTitles();

        // Fix this XXX
        // MuckMain.pauseStatus = false;
        // while (MuckMain.heldResponse.size() > 0) {
        //     MuckMain.mainText.append(MuckMain.heldResponse.elementAt(0).toString());
        //     MuckMain.heldResponse.removeElementAt(0);
        // }
    }

    /**
     * Method to hide all the frames.  The 'supervisor' function
     */
    // public static void HideFramesQuick() {
    public void HideFramesQuick() {
        // Hide all the frames,
        // the supervisor's here!
        setTitle(".");
        toBack();

        // Fix this XXX
        // MuckMain.textWindow.setVisible(false);
        // MuckConn.jmMacros.setVisible(false);

        // We'll unsync the databar, otherwise bringing it up to the
        // front may result in all the other windows showing up again.  Eeek!
        setSync(false);
    }

    /**
     * Restore frames hidden by the 'Supervisor' function
     */
    //	public static void RestoreMinimizedFrames() {
    public void RestoreMinimizedFrames() {
        // Restore the frames that
        // were minimized by the 'supervisor' key
        // Fix this XXX
        // MuckMain.textWindow.setVisible(true);

        // Restore frame icons
        // dataBar.setIconImage(MuckConn.jamochaMUDImage);
        // MuckMain.textWindow.setIconImage(MuckConn.jamochaMUDImage);

        // Querry the MuckMain **** menu, to
        // see if we should set the Macros visible
        // Fix this XXX
        // if (MuckMain.tWMacro.getState()) {
        // tWMacro is set to 'true', so show the Macro Frame
        // MuckConn.jmMacros.setVisible(true);
        //}

        // Fix this XXX
        // if (MuckMain.tWSyncWindowsItem.getState()) {
        // setSync(true);
        // }
        // now reset the title
        // resetTitles();
    }

    /**
     * Add an entry to the 'command history' pop-up
     */
    private void addPopUpEntry(String str) {
        if (backTrack.getItemCount() > 9) {
            backTrack.remove(0);  // Removes first entry to make room
            for(int i = 0; i < backTrack.getItemCount(); i++) {
                backComm[i] = backComm[i + 1];
            }
        }

        backComm[backTrack.getItemCount()] = str;
        if (str.length() > 30) {
            backTrack.add(str.substring(0, 30) + "...");
        } else {
            backTrack.add(str);
        }
    }

    /**
     * This is an &quot;empty" method that allows other
     * classes to request that we send our existing text
     * to the MU*.  This doesn't happen often, but can
     * be a handy ability to have */
    public void JMSendText() {
        JMSendText(dataText.getText());
    }

    /**
     * Package up the data to be sent to the MU*, parsing off any
     * yucky characters such as 'new lines', etc., also determining
     * via user-preference whether they should be sent in Unicode
     * or standard ASCII format
     */
    public void JMSendText(String str) {
        // Safety first!

        if (connHandler == null) {
            connHandler = settings.getConnectionHandler();
        }

        // try {
        // try to extract any tabs, form feeds
        // backspaces, new lines, and carraige returns
        // str.replace('\t', '*');
        // str.replace('\f', '*');
        // str.replace('\b', '*');
        // str.replace('\n', '*');
        // str.replace('\r', '*');

        // First, check for TinyFugue emulation, and parse any of
        // those commands first
        if (str.startsWith("@/")) {
            // if (MuckMain.tWTFKeysItem.getState()) {
            if (settings.getTFKeyEmu()) {
                // We'll check this line for actual TF commands
                // STUB
            }
        }

        // Send the text to the appropriate MU* via its connection handler
        connHandler.sendText(str);

        // If local echo is active, append it to the user's screen as well
        if (settings.isLocalEchoEnabled()) {
            anecho.gui.JMText text;
            text = connHandler.getActiveMUDText();
            text.append("> " + str + '\n');
        }

        // return the cursor to the starting position and empty the text area
        dataText.setCaretPosition(0);
        dataText.setText("");

        // Now add the command to the Popupmenu
        addPopUpEntry(str);

        // Send string to lag timer if active
        // if (MuckMain.tWTimers.getState()) {
            // Timers are running, send the string
            // Fix this XXX
            // Timers.AddOutput(str);
        // }

    }

    /** Set the current text from the scrollback menu,
     * the item indicated by int pos */
    public void JMSetFromScrollBack(int pos) {
        // If the integer comes in as '-1', that means to use the last string
        if (pos == -1) pos = backTrack.getItemCount() - 1;

        // First, check to see if the integer is valid
        if (pos > backTrack.getItemCount() || pos < 0) return;

        // Now we'll set the text from the menu item
        dataText.setText(backComm[pos]);
        // Explicitly set the cursor location, as this is
        // not automatic on some operating systems.
        dataText.setCaretPosition(backComm[pos].length());
    }

    /** Some text methods */
    public void setText(String text) {
        dataText.setText(text);
    }

    public void setFont(Font newFont) {
        dataText.setFont(newFont);
    }

    public void append(String text) {
        dataText.append(text);
    }

    public String getText() {
        return dataText.getText();
    }

    public void setCaretPosition(int pos) {
        dataText.setCaretPosition(pos);
    }

    public int getCaretPosition() {
        return dataText.getCaretPosition();
    }

        public int getColumns() {
            return dataText.getColumns();
        }

    public void setForegroundColour(Color fg) {
        dataText.setForeground(fg);
    }

    public void setBackgroundColour(Color bg) {
        dataText.setBackground(bg);
    }

    /** Set the title of this frame on the basis of if the
     * active MU* (the one visible at the moment) is
     * connected or not.  This method differs from the main
     * window's setWindowTitle() method in that it takes
     * the text being paused into consideration as well
     */
    public synchronized void setWindowTitle() {
        // This resets the window's title, depending if the                                    setWindowTitle();
        // client is connected to a MU* or not
        CHandler connection = settings.getConnectionHandler();
        boolean active = connection.isActiveMUDConnected();

        if (active) {
            MuSocket mSock = connection.getActiveMUHandle();

            if (mSock.isPaused()) {
                this.setTitle(RB("outputPaused"));
            } else {
                this.setTitle(connection.getActiveTitle());
            }
        } else {
            // Connection is inactive, just have program's title
            this.setTitle("JamochaMUD - " + RB("notConnected"));
        }
    }

    /** We export our TextArea for use in a &quot;combined"
     * JamochaMUD configuration, where both the input and
     * output are shown in the same frame
     */
    public TextArea exportText() {
        this.setVisible(false);
        this.setSync(false);

        // We should also remove this frame from the sync-group?
        return dataText;
    }


    /** We are having our TextArea &quot;returned" to us from
     * the main window.  We'll add it back to our frame and
     * then set everything visible again
     */
    public void restoreText() {
        this.add(dataText);
        this.pack();

        // Check for sync value to restore us to our previous glory
        if (settings.getSyncWindows()) {
            this.setSync(true);
        }

        this.setVisible(true);
    }

    /** Remove the paused status from JamochaMUD because of some
     * action that we have received
     */
    private void checkPause() {
        CHandler target = settings.getConnectionHandler();
        MuSocket mSock = target.getActiveMUHandle();
        if (mSock.isPaused()) {
            // This spools out any paused text
            mSock.SpoolText();
        }
    }


}
