/**
 * A MuSocket is used to hold all information important to
 * a single socket connection
 * $Id: MuSocket.java 1.14 2001/11/05 03:18:12 jeffnik Exp $
 */

/* JamochaMUD, a Muck/Mud client program
 * Copyright (C) 1998-2001 Jeff Robinson
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

package anecho.JamochaMUD;

import anecho.gui.JMText;
import anecho.gui.ResReader;

import anecho.extranet.Socks5socket;
import anecho.extranet.MUDBufferedReader;

import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;

import java.io.DataOutputStream;
import java.io.InputStreamReader;
import java.io.IOException;

import java.net.ConnectException;
import java.net.InetAddress;
import java.net.NoRouteToHostException;
import java.net.Socket;
import java.net.UnknownHostException;

import java.util.Vector;

public class MuSocket extends Thread implements MouseListener {

    private String name, address;
    private int port;
    private long timeStamp;
    private boolean connected;        // The state of the connection
    private boolean paused;             // Has our output been paused?
    private JMText mainText;            // The text-area belonging to this connection
    private JMConfig settings;          // A pointer to our settings
    private Socket connSock;
    private String fromMU;              // a string to hold output from the MU*
    private MUDBufferedReader inStream; // input to us from the MU*
    private DataOutputStream outStream; // for sending info to the MU*
    private Vector heldResponse;        // A vector to hold any text incase we are "paused"
    private int minimisedLineCount = 0; // the number of minimised lines


    /** The constructor.  Behold!! */
    public MuSocket(JMConfig settings) {
        this.settings = settings;
        // Set up a new text area
        // mainText = new JMText();
        // I don't think we should have to give all this info, should we?
        // Fix me XXX
        mainText = new JMText("", 50, 80, JMText.SCROLLBARS_VERTICAL_ONLY);
        // Set the attributes of our text area
        mainText.setEditable(false);
        // pauseStatus = false;
        heldResponse = new Vector(0, 1);

        /* if (settings.getDoubleBuffer()) {
            mainText.setDoubleBuffer(true);
        } else {
            mainText.setDoubleBuffer(false);
        }  */

        mainText.setDoubleBuffer(settings.getDoubleBuffer());

        // tempVector = new Vector(0, 1);
        // mainText.addKeyListener(this);
        // textWindow.addKeyListener(this);
        mainText.addMouseListener(this);

        setMainColours();
    }

    // Check for mouse actions

    public void mousePressed(MouseEvent e) {
        setPaused(true);
    }

    public void mouseReleased(MouseEvent e) {
    }

    public void mouseClicked(MouseEvent e) {
        // Determine number of mouse clicks
        int a = e.getClickCount();

        if (a >= 2) {
            // Call outside routines (plug-ins) to handle the selected text
            e.consume();
            SpoolText();

            // grab the 'selected' URL
            StringBuffer tentativeURL = new StringBuffer((mainText.getSelectedText()).trim());
            // System.out.println("Muckmain... using " + tentativeURL + " for external program.");

            // Fix this XXX
            // The frame is sent to anchor any potential dialogues
            // send the 'URL' to launch the correct program
            // ExternalProgs.LaunchProgram(textWindow, tentativeURL);

            // Deselect the text
            // mainText.select(mainText.getSelectionEnd(), mainText.getSelectionEnd());
            System.out.println("MuckMain: Deselect text?");
        } else {
            // A single click signals a pause
            // This will pause the output window,
            // the incoming lines held in queue
            setPaused(true);
            // PauseText();
        }
    }

    public void mouseEntered(MouseEvent e) {
    }

    public void mouseExited(MouseEvent e) {
    }

    /** The name of our connection */
    public synchronized void setTitle(String name) {
        this.name = name;
        setFrameTitles();
    }

    /** getTitle does a small bit of book-keeping, returning a title based
     * on whether we're connected to the MU* or not.  If we're not connected
     * then we'll return &quot;JamochaMUD" instead of the name
     */
    public String getTitle() {
        if (!connected) {
            return "Not Connected: " + address;
        }

        if (name == null) {
            this.name = address;
        }
        return "JamochaMUD (" + name +")";
    }

    /** This returns the true name of the MU*, regardless of our connection
     */
    public String getMUName() {
        return this.name;
    }

    /** Call this to reset the frame's title to the &quot;standard" title.
     * This is usually just the location */
    public void resetTitle() {
        // this.name = "JamochaMUD (" + address + ")";
        this.name = name;
        setFrameTitles();
    }

    /** The Address of our connection */
    public void setAddress(String address) {
        this.address = address;
    }

    public String getAddress() {
        return address;
    }

    /** The port our connection will connect to */
    public void setPort(int port) {
        this.port = port;
    }

    public int getPort() {
        return port;
    }

    /** The timestamp of our connection, for internal use mostly */
    public void setTimeStamp(long time) {
        this.timeStamp = time;
    }

    public long getTimeStamp() {
        return timeStamp;
    }

    /** This is the main method for our thread.  Veryvery important!
     *  Basically, it'll be a big loop that only gets broken when our
     * connection disappears... either from the socket's end or the
     * user's end */
    public void run() {
        String tempAddy = this.getAddress();

        // Give some visual notification that we're attempting a connection
        setVisuals(tempAddy);
        // mainText.append("Attempting connection to " + tempAddy + "..." + '\n');
        // Write the location to the main window's title, as well

        // Try to establish a connection
        int port = this.getPort();
        // String tempAddy = this.getAddress();
        InetAddress serverAddy;
        // MUDBufferedReader inStream;
        // DataOutputStream outStream;
        // We'll give sockPort and sockHost initial values to satisfy later conditions
        int sockPort = 0;
        String sockHost = "";
        boolean socks = settings.getProxy();

        // Fix me XXX
        // if (sP.get("proxySet").equals("true")) {
        if (socks) {
            sockHost = settings.getProxyHost();
            sockPort = settings.getProxyPort();
        }

        try {
            serverAddy = InetAddress.getByName(tempAddy);
            if (socks) {
                System.out.println("Attempt to use Socks5socket.");
                connSock = (Socket)new Socks5socket(tempAddy, port, sockHost, sockPort);
                System.out.println("Socks5socket successfully connected");
            } else {
                connSock = new Socket(serverAddy, port);
            }
            outStream = new DataOutputStream(connSock.getOutputStream());
        } catch (NoRouteToHostException oops) {
            // No route to host, or operation timed out
            disconnectMenu();

            String tempString = new String(oops + "");
            if (tempString.endsWith("unreachable")) {
                // Host unreachable
                mainText.append(RB("noRouteToHostException") + '\n');
            } else {
                mainText.append(RB("operationTimedOutException") + '\n');
            }
        } catch (UnknownHostException oops) {
            disconnectMenu();
            mainText.append(RB("unknownHostException") + '\n');
        } catch (ConnectException oops) {
            disconnectMenu();
            mainText.append(RB("connectException") + '\n');
        } catch (Exception oops) {
            disconnectMenu();
            System.out.println("From Net, no socket " + oops);

            mainText.append(RB("exception") + '\n');
            // Fix this XXX
            // closeConnection();
        }

        // Create a MUDBufferedReader for our input stream from the MU*
        InputStreamReader inStreamReader;

        try {
            inStreamReader = new InputStreamReader(connSock.getInputStream());
        } catch (Exception e) {
            // System.out.println("Error creating InputStreamReader.");
            return;
        }

        try {
            inStream = new MUDBufferedReader(inStreamReader);
            // inStream = new MUDBufferedReader(new InputStreamReader(connSock.getInputStream()));
        // } catch (IOException oops) {
        } catch (Exception e) {
            // System.out.println("No inStream");
            return;
        }

        // Try and setup a MUDBufferedReader for our "data-pipe"
        try {
            inStream = new MUDBufferedReader(new InputStreamReader(connSock.getInputStream()));
        } catch (IOException oops) {
            System.out.println("No inStream");
        }

        // Okay, we're connected.  Do our little set-up stuff

        // Read from the socket until we get tired and fall down
        connected = true; // This notifies the program of true connection

        // Update the main window menus to reflect our connected state XXX

        // Change the titles of our frames to show we're connected
        setFrameTitles();
        connectMenu();

        while (connected) {
            try {
                fromMU = inStream.readLine();
            } catch (Exception e) {
                // We've run into some sort've problem, so chances are
                // our connection has been terminated in some way.
                // System.out.println("Error at FromNet inStream.readLine " + e);
                // e.printStackTrace();
                // closeConnection();
                connected = false;
                break;
            }
            // Check for output plugins
            // Fix this XXX
            // we should probably not be making a "static" call like this

            // Check to see if we somehow got disconnected.  (It can happen!)
            if (fromMU == null) {
                // sets out menu to the correct state as well as closes the socket
                mainText.append("---> " + RB("connectionClosed") + " <---" + '\n');
                closeSocket();
                connected = false;
                disconnectMenu();
                return;
            }

            fromMU = EnumPlugIns.CallPlugin(fromMU, "output", this);

            // Send the current line to our Text-field
            if (!paused) {
                mainText.append(fromMU);

                // Check to see if the main frame is minimised.  If so, we add
                // the new text to the title bar
                if (settings.isMainWindowIconified()) {
                    addTitleText(fromMU);
                } else {
                    minimisedLineCount = 0;
                }

            } else {
                // System.out.println("paused: " + fromMU);
                heldResponse.addElement(fromMU);
            }

        }

    }

    private static String RB(String itemTarget) {
        ResReader reader = new ResReader();
        // return reader.LangString("JamochaMUDBundle", itemTarget);
        return reader.LangString(JMConfig.BUNDLEBASE, itemTarget);
    }

    // Return our text area to who-ever wants to know! */
    public JMText getTextWindow() {
        return mainText;
    }

    /** Send text out to our MU* */
    public void sendText(String send) {
        try {
            if (settings.getUseUnicode()) {
                // Send the string as Unicode
                outStream.writeChars(send + "\r\n");
            } else {
                // Send the string as regular ANSI text
                outStream.writeBytes(send + "\r\n");
            }
        } catch (Exception e) {
            System.out.println("MuSocket.sendText exception: " + e);
        }
    }

    /** Set the proper colours, fonts, etc */
    public void setMainColours() {
        mainText.setForeground(settings.getForegroundColour());
        mainText.setBackground(settings.getBackgroundColour());
        mainText.setFont(settings.getFontFace());
    }

    /** This will set up some of the initial indicators
     * that we're trying to make a connection */
    private void setVisuals(String title) {
        // mainText.append("Attempting connection to " + title + "..." + '\n');
        mainText.append(RB("attemptingConnectionTo") + " " + title + "..." + '\n');
    }

    private void setFrameTitles() {
        // We should only be able to change the frame titles it this
        // is the active MU*
        CHandler connHandler = settings.getConnectionHandler();
        if (connHandler.getActiveMUHandle() == this) {
            MuckMain target = settings.getMainWindowVariable();
            target.setWindowTitle();

            DataIn inWindow = settings.getDataInVariable();
            inWindow.setWindowTitle();
        }
    }

    /** Return the status of our connection:<br>
     * <pre>true</pre> - Connected<br>
     * <pre>false</pre> - Not connected
     */
    public boolean isConnectionActive() {
        return connected;
    }

    /** Set the proper states of our windows, etc.
     * 'cause, woe is us, we are disconnected.  Most often
     * this would be called internally by our class
     */
    private void disconnectMenu() {
        // This is where we actually disconnect the socket.
        // Is there a nicer way of saying good-bye?
        // closeSocket();
        // closeSocket() is called from the CHandler class.
        CHandler connHandler = settings.getConnectionHandler();

        // This guarantees that we've cleaned up after ourselves, too
        connHandler.closeSocket(this);

        // Change the connection notice on the main window if applicable
        MuckMain window = settings.getMainWindowVariable();
        window.CheckDisconnectMenu();

    }

    /** Allow other classes to close our socket */
    public void closeSocket() {
        // Put a notification on the screen for the user
        // Fix this XXX it should come from the resource bundle
        // mainText.append("---> Connection Closed <---" +  '\n');
        // mainText.append("---> " + RB("connectionClosed") + " <---" + '\n');
        try {
            connSock.close();
        } catch (Exception closeError) {
            // The only reason we should get here is if the socket was never openned
            // in the first place.  This can happen if we tried to open a bad address
            // System.out.println("MuSocket hit an error (.disconnectMenu): " + closeError);
        }

        // Change the titles of the frames if applicable
        resetTitle();
        setFrameTitles();

        return;
    }

    /** Set the proper states of our windows as
     * we've probably just connected
     */
    private void connectMenu() {
        MuckMain main = settings.getMainWindowVariable();
        main.setConnectionMenu();
    }

    /** Set the paused status of our text area */
    public void setPaused(boolean option) {
        paused = option;

        if (option) {
            setFrameTitles();
        } else {
            SpoolText();
        }
    }

    /** Return the fact that our connection is either
     * paused - <pre>true</pre><br>
     * or not paused - <pre>false</pre><br>
     */
    public boolean isPaused() {
        return paused;
    }

    public synchronized void SpoolText() {
        // We'll reset this variable, just in case
        paused = false;

        // Reset our frame titles
        resetTitle();

        // Do a little bounds checking first
        if (heldResponse.size() < 1) {
            // System.out.println("MuSocket: SpoolText, nothing to spool.");
            return;
        }

        // System.out.println("MuSocket.SpoolText: we have " + heldResponse.size() + " elements.");
        while (heldResponse.size() > 0) {
            // System.out.println("MuSocket.SpoolText: " + heldResponse.elementAt(0).toString());
            mainText.append(heldResponse.elementAt(0).toString());
            heldResponse.removeElementAt(0);
        }
    }

    /** Trim down the string we've received from the MU*, add
     * the minimised line count, and the set the new title */
    private synchronized void addTitleText(String newText) {
        minimisedLineCount++;
        // System.out.println("MuSocket.addTitleText: " + minimisedLineCount + " " + newText);

        String str = StripEscapes(newText);

        // Trim the str down to a managable length
        String trimmedString;
        if (str.length() > 80) {
            trimmedString = str.substring(0, 80) + "...";
        } else {
            trimmedString = str;
        }

        MuckMain target = settings.getMainWindowVariable();
        target.setWindowTitle("(" + minimisedLineCount + ") " + str);

    }

    /** This method returns the value of a string
     * once it has had any escape characters stripped
     * from it.
     */
    private String StripEscapes(String token) {
        StringBuffer workString = new StringBuffer("");
        boolean loop = true;
        int start = 0;
        int end = 0;

        while (loop) {
            end = token.indexOf('\u001b', start);

            if (end < start) {
                // There are no escapes left
                workString.append(token.substring(start));
                loop = false;
                break;
            }

            if (end > 0) {
                workString.append(token.substring(start, end));
            }

            // Now set the new 'start'
            start = token.indexOf('m', end) + 1;
            if (start <= end) {
                loop =false;
                System.out.println("Not a proper ANSI escape");
                break;
            }
        }

        return workString.toString();
    }

}
