/**
 * JMText, a multi-colour text-area for Java 1.1 (originally developed for JamochaMUD)
 * $Id: JMText.java 1.38 2002/01/30 04:35:47 jeffnik Exp $
*/

/* JMText, a multi-colour TextArea for Java 1.1
 * Copyright (C) 1999-2000 Jeff Robinson
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2, as published by the Free Software Foundation; 
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

package anecho.gui;

import java.awt.*;
import java.awt.event.*;

import java.lang.Integer;

import java.util.Vector;
import java.util.StringTokenizer;

/** A widget to display multicoloured text in a Java 1.1 environment
 * with an effort made to clone the TextArea API as close as possible.
 * @version $Id: JMText.java 1.38 2002/01/30 04:35:47 jeffnik Exp $
 * @author Jeff Robinson
 */
public class JMText extends Container implements AdjustmentListener, KeyListener, MouseListener, MouseMotionListener, ComponentListener, FocusListener {

	// Variables
	public static final int SCROLLBARS_BOTH = 0;

	public static final int SCROLLBARS_VERTICAL_ONLY = 1;

	public static final int SCROLLBARS_HORIZONTAL_ONLY = 2;

	public static final int SCROLLBARS_NONE = 3;

        private LittleCanvas drawText;
        private Graphics g;             // The "graphics" used to display the characters
        private Image offScreenImg;     // Our off-screen image for double-buffering
        private Color bgColour, fgColour;  // colours we'll use for our foreground/background
	private char[][] textChars;	// The actual text (characters) received
	private String[][] markings;	// The mark-ups for the associated characters in 'text'
	private boolean minimised = true;	// Whether this component is minimized or not
        // Starting "true" forces us to calculate the size
        private boolean softReturn = false;
        private StringTokenizer workString;       // for parsing of tokens in append(str)
	private Dimension widgetSize = new Dimension(0, 0);	// A size to be used when minimized
	private boolean notified = false;         // Help to determine if we have an 'object' that
        // we can query for information
        private boolean sizeSet = false;
        private boolean doubleBuffer = true;      // Double-buffer output graphics (smoother
                                                  // draw but slower)
        private int columns, rows;	// The number of rows and columns in the JMText area
        private int prevCols = 0;       // Used for tracking changes against new columns
        private int prevRows = 0;       // Used for tracking changes against new rows
	private int maxRows = 6000;	// The maximum number of rows available at once
	private int halfMax = 2500;	// The amount of rows we send to history at once
	private Scrollbar vertScroll;	// Vertical scrollbar
	private AdjustmentListener vertListener;	// Vertical scrollbar Listener
	private Scrollbar horzScroll;	// Horizontal scrollbar
	private BorderLayout layout;	// The layout manager for this component
	private int viewBase;		// The 'base' line that is viewable to the user
	private Vector pendingText = new Vector(0, 1);	// a vector for text that can't be
								// presented at the time of receipt
	private Vector historyText = new Vector(0, 1);	// This vector will keep all text
								// that is 'spooled off' the active
								// text Array.
	private String currentAttrib = "";	// Keeps track of the "active" character attributes
	private Point cursorPos;		// The current cursor position
	private static Point selectBegin = new Point(0, 0);	// Starting mouse position (for selections)
	private static Point selectEnd = new Point(0, 0);	// End point for mouse selections
	private static Point textBegin = new Point(0, 0);	// Actual text selected
	private static Point textEnd = new Point(0, 0);	// End point of actual text

	private KeyListener keyL;		// A keylistener that can 'talk' to other classes
	private TextListener textL;		// A textlistener
	private MouseListener mListener;	// A mouselistener
        // private FocusListener fListener;

	// Constructor Methods
	public JMText() {
		this("", -1, -1);
	}

	public JMText(String text) {
		this(text, -1, -1);
	}

	public JMText(int rows, int columns) {
		this("", rows, columns);
	}

	public JMText(String text, int rows, int columns) {
		this(text, rows, columns, SCROLLBARS_NONE);
	}

	public JMText(String text, int rows, int columns, int scrollbars) {
		// set up the variables and then add the components
		layout = new BorderLayout();
		setLayout(layout);

		drawText = new LittleCanvas();
		add(drawText, BorderLayout.CENTER);
		drawText.addKeyListener(this);
		drawText.addKeyListener(keyL);
                drawText.addComponentListener(this);		// This listener catches updates
		drawText.addFocusListener(this);
		drawText.addMouseListener(this);
		drawText.addMouseMotionListener(this);

		// Check to see if scrollbars are needed
		switch (scrollbars) {
			case SCROLLBARS_VERTICAL_ONLY: {
				vertScroll = new Scrollbar(Scrollbar.VERTICAL);
				vertScroll.addAdjustmentListener(this);
				add(vertScroll, BorderLayout.EAST);
				break;
			}
			case SCROLLBARS_HORIZONTAL_ONLY: {
				horzScroll = new Scrollbar(Scrollbar.HORIZONTAL);
				add(horzScroll, BorderLayout.SOUTH);
				break;
			}
			case SCROLLBARS_BOTH: {
				vertScroll = new Scrollbar(Scrollbar.VERTICAL);
				add(vertScroll, BorderLayout.EAST);
				horzScroll = new Scrollbar(Scrollbar.HORIZONTAL);
				add(horzScroll, BorderLayout.SOUTH);
				break;
			}
		}

		// We'll store the rows and columns for now.  If the user hasn't specified
		// their size, then we can't calculate it until AddNotify is called
		this.rows = rows;
                this.columns = columns;

                pendingText.addElement(text);

        }

	/** This method is used to initialise all the variables in JMText 
	 * that could not be done when it was initially created. */
        private synchronized void InitializeJMText() {

            if (minimised) {
                // System.out.println("InitializeJMText() called, but we're minimized.");
                return;
            }
            if (checkNewRowsSame()) {
                return;
            }

            notified = true;	// We don't want to accidentally re-initialise the variables
            currentAttrib = "";     // make sure no attributes were left over from a previous session

            // Set up our character arrays
            setCharacterArrays();

            // Set up the scrollbars
            resetScrollbars();

            cursorPos = new Point(0, 0);		// set the cursor to its starting position

            // If there is any text pending, we will write it out now
            if (pendingText.size() > 0) {
                for (int i = 0; i < pendingText.size(); i++) {
                    append(pendingText.elementAt(i).toString());
                }

                pendingText.removeAllElements();	// Make sure the vector is empty
            }

            if (historyText != null) {
                historyText.removeAllElements();	// Empty the history text vector
            }
        }

        /** Set the proper rows and columns for this page.  Admittedly, this is ugly. */
        private synchronized void getRowsAndColumns() {
            Dimension size = getSize();
            // Dimension size = drawText.getSize();
            String test ="l";
            int tempRows, tempCols;
            int total = 0;
            int fontWidth = 0;
            FontMetrics fm = getFontMetrics(getFont());

            try {
                total = fm.stringWidth(test);
            } catch (Exception e) {
                System.out.println("JMText exception getting average character width");
            }

            fontWidth = (int)(total /= test.length());

            if (rows > 0 && size.width == 0 && size.height == 0) {
                // Set the size and width via the rows and columns supplied
                setSize(columns * fontWidth, fm.getHeight() * rows);
            } else {
                if (rows < 1 && size.width == 0 && size.height == 0) {
                    size = this.getParent().getSize();
                }
                tempRows = (int)size.height / fm.getHeight();
                tempCols = (int)size.width / fontWidth;

                if (tempRows < 1 || tempCols < 1) {
                    // For some reason our rows and columns don't stack up
                    // We're going to assume that we're minimised
                    minimised = true;
                    return;
                }

                rows = tempRows;
                columns = tempCols;

            }

            // Now we can set up the character arrays
            widgetSize.setSize(this.getSize());
        }

        /** Append text to JMText, just like the TextArea API call */
        public synchronized void append(String str) {
            if (!notified || minimised) {
                // This object has no size right now, so save this String
                pendingText.addElement(str);
                return;
            }

            // Experimental line-wrap code ***WARNING***
            // First, break the string into tokens
            workString = new StringTokenizer(str, " ", true);

            FontMetrics fontWidth = getFontMetrics(drawText.getFont());
            int displayWidth = fontWidth.charsWidth(textChars[cursorPos.y], 0, cursorPos.x);
            int pelWidth = 0;
            int len = 0;
            boolean loop = true;
            String token, strippedToken;

            while (workString.hasMoreTokens()) {
                token = workString.nextToken();
                strippedToken = StripEscapes(token);


                // We now have the token we want to send out...
                len = strippedToken.length();
                pelWidth = fontWidth.stringWidth(strippedToken.trim() + " ");

                // now evaluate the length of the token, and see if it
                // will fit on the current line.  If not, send a 'new-line'
                if ((displayWidth + pelWidth) > getSize().width || (len + cursorPos.x) > columns - 2) {
                    // Check to see if the token is too large to fit on a line.
                    // If it is, we'll let it slip through to be broken up.
                    if (pelWidth < getSize().width) {
                        cursorPos.setLocation(0, cursorPos.y + 1);	// Go to next line
                        ScrollDown();
                        displayWidth = 0;
                    } else {
                        // System.out.println("This is too wide to fit entirely on a new line.");
                        // Check to see if we are minimised.  If so, we'll stuff this
                        // back into the pending text after resetting the minimised attribute.
                        getRowsAndColumns();
                        if (minimised) {
                            System.out.println("JMText is minimsed, pending: " + str);
                            pendingText.addElement(str);
                            return;
                        }
                    }
                }

                displayWidth += pelWidth;
                for(int i = 0; i < token.length(); i++) {
                    // Check to see if this is an ANSI escape
                    while (token.charAt(i) == '\u001b') {
                        i = checkEscape(token, i);
                        if ( i >= token.length()) {
                            break;
                        }
                    }

                    // write the token to the screen
                    if (i < token.length()) {
                        // This "try" statement was looking for null characters
                        // but we shouldn't have to do this anymore, as we do some
                        // checking prior to this
                        appendChar(token.charAt(i), currentAttrib);
                    }
                }
            }

            // End linewrap code
            // System.out.print("Append.repaint() ");
            repaint();

        }

        /** Append characters to JMText, one at a time, including the current attribute */
        // public synchronized void appendChar(char jchar, String attrib) {
        public void appendChar(char jchar, String attrib) {
            FontMetrics fm = getFontMetrics(getFont());
            int width = fm.charsWidth(textChars[cursorPos.y], 0, cursorPos.x);
            width += fm.charWidth(jchar);

            if (width >= getSize().width || cursorPos.x >= columns) {
                cursorPos.setLocation(0, cursorPos.y + 1);
                softReturn = true;
                ScrollDown();

                if (cursorPos.x >= columns) {
                    System.out.println("Return: Too wide or ran out of columns at: " + jchar);
                }
                // else {
                    // System.out.println("Return: Too wide for screen at: " + jchar);
                // }
            }

            textChars[cursorPos.y][cursorPos.x] = jchar;
            markings[cursorPos.y][cursorPos.x] = attrib;

            if (jchar == '\n') {
                if (!softReturn) {
                    cursorPos.setLocation(0, cursorPos.y + 1);
                    ScrollDown();	// Check to see if we need to scroll down
                              		// No need to update the screen on a newline
                }
                return;
            }

            cursorPos.setLocation(cursorPos.x + 1, cursorPos.y);

            // Notify our text-listener that something has happened
            FauxTextListener();
            softReturn = false;

        }

        /** the paint method will be over-ridden to give us complete control over this event */
        public void paint(Graphics z) {
            if (!notified) {
                // System.out.println("Not notified yet.");
                return;
            }

            // if (!this.isVisible()) {
            // System.out.println("This component isn't visible right now, sorry");
            // }

            // System.out.println("JMComponentResized() called from paint()");
            JMComponentResized();

            int tWidth = drawText.getSize().width;
            int tHeight = drawText.getSize().height;
            if (tWidth < 1 || tHeight < 1) return;          // we're not visible!

            bgColour = getBackground();
            fgColour = getForeground();
            offScreenImg = createImage(tWidth, tHeight);

            if (doubleBuffer) {
                // We're using double-buffering.  It removes the "flicker" but
                // redraws somewhat slower
                g = offScreenImg.getGraphics();
                g.setFont(getFont());
                g.setColor(bgColour);
                g.fillRect(0, 0, tWidth, tHeight);
            } else {
                // No double-buffering.  The display "flickers" more
                // but is redrawn faster than the double-buffered type
                g = drawText.getGraphics();
            }

		int lineWidth;		// Keep track of the line width instead of recalculating
		int lineHeight;					// Height of the line
		FontMetrics fm = getFontMetrics(getFont());	// Fontmetrics for this character
                int charHeight = fm.getHeight();		// Height of this character
                int charDescent = fm.getDescent();

		// Only display the lines which may be visible for faster refreshes
		int baseLine = 0;	// the 'top' line of text to view
		if (vertScroll != null) {
			baseLine = vertScroll.getValue();
		}

		int endRead = 0;
		if (cursorPos.y + 1 < rows) {
			endRead = cursorPos.y;
		} else {
			endRead = baseLine + rows;
		}

		// Kludgily, we loop through what *should* be visible
                char tChar;   // the variable we'll be checking
                String tLine; // Our entire line of text
                String tMarks;
                int tLength;

                for (int y = baseLine; y <= endRead; y++) {
                    // By dumping the multi-dimensional array to a string
                    // we can save a lot of overhead 'cause we don't have
                    // to keep checking the area.
                    tLine = new String(textChars[y]);  // By not having to check an array
                    tLength = tLine.length();

                    lineWidth = 0;
                    lineHeight = (y - baseLine) * charHeight;

                    g.setColor(bgColour);
                    g.fillRect(0, lineHeight + charDescent, tWidth, charHeight);

                    for (int x = 0; x < tLength; x++) {
                        tChar = tLine.charAt(x);
                        tMarks = markings[y][x];

                        if (tMarks == null) break;	// No characters here!

                        if (tChar == '\n') {
                            break;
                        }
                        // Now draw the character
                        // if (markings[y][x].indexOf("fg:") >= 0) {
                        if (tMarks.indexOf("fg:") >= 0) {
                            // We'll set the colour from the attrib
                            g.setColor(CalculateColour(tMarks, tMarks.indexOf("fg:")));
                        } else {
                            g.setColor(fgColour);
                        }

                        g.drawString(tChar + "", lineWidth, lineHeight + charHeight);
                        lineWidth += fm.charWidth(tChar);
                    }

                    // Now we clear the remainder of the line
                    if (cursorPos.y - baseLine < rows) {
                        g.setColor(bgColour);
                    }

                    // Now we'll redraw any selections if they exist
                    if (selectBegin.x != selectEnd.x && selectBegin.y != selectEnd.y) {
                        paintSelection(selectBegin, selectEnd, bgColour);
                    }

                }

                if (doubleBuffer) {
                    // Double buffer must draw the graphics onto the active area now
                    Graphics onScreen = drawText.getGraphics();
                    onScreen.drawImage(offScreenImg, 0, 0, null);
                }
        }

        public void update(Graphics g) {
            // System.out.println("update() -> paint() ");
            paint(g);
	}

	// Handle scrollbar methods
	public void adjustmentValueChanged(AdjustmentEvent e) {
            // Scrollbar has been adjusted
            // System.out.print("adjustmentValueChanged() -> paint() ");
            repaint();
	}

	/** Bump the scrollbar down one notch if necessary (scrolling the 'screen' up) */
	private void ScrollDown() {
            // Now we'll also check to see if we're getting too much text for this array.
            // If this is the case, we'll write some of the lines to historyText and chop
            // them off the top of the array.  They may appear to be 'gone' from the viewer
            // but if a text-dump is done, they will still be included.
            if (cursorPos.y > maxRows - 2) {
                // Write the strings out to the historyText vector
                String work;
                StringBuffer tempString = new StringBuffer("");
                int returnColumns = textChars[0].length;


                for (int y = 0; y < halfMax; y++) {
                    work = "";
                    try {
                        work = textChars[y] + "";
                    } catch (Exception e) {
                        System.out.println("Error in JMText.ScrollDown historyText " + e);
                    }
                    try {
                        historyText.addElement(work.substring(0, work.lastIndexOf('\n') + 1));
                    } catch (Exception e) {
                        System.out.println("Error in Scrolldown " + e);
                    }
                }

                // We'll copy these into temporary arrays before erasing the old
                char tempChars[][] = new char[cursorPos.y + 1][columns];
                System.arraycopy(textChars, halfMax, tempChars, 0, cursorPos.y - halfMax);
                textChars = new char[maxRows][columns];
                System.arraycopy(tempChars, 0, textChars, 0, cursorPos.y - halfMax);

                // copy the markings
                String tempMarkings[][] = new String[cursorPos.y + 1][columns];
                System.arraycopy(markings, halfMax, tempMarkings, 0, cursorPos.y - halfMax);
                markings = new String[maxRows][columns];
                System.arraycopy(tempMarkings, 0, markings, 0, cursorPos.y - halfMax);

                // Set the new cursor position and scrollbar information
                cursorPos.setLocation(cursorPos.x, cursorPos.y - halfMax);

            }

            if (vertScroll != null && cursorPos.y + 1> rows) {
                if (cursorPos.y + 1 > rows) {
                    vertScroll.setMaximum((cursorPos.y + 1) - rows);
                    vertScroll.setValue(vertScroll.getMaximum());
                }
                // System.out.println("Scrolldown() -> paint()");
                repaint();
            }

        }

        /** Set the text in JMText to whatever we are sent in the string,
         * erasing everything else */
        public synchronized void setText(String str) {
            // first, check it the JMText area is valid yet
            if (!notified) {
                // We don't have an actual JMText are yet, so we'll erase
                // any previously stored lines, and set this one as the new one
                pendingText.removeAllElements();
                System.out.println("JMText.setText adding pending: " + str);
                pendingText.addElement(str);

                return;		// we can't really do any more right now
            }

            // now we'll erase anything that may be on the widget's text area right now
            Graphics g = drawText.getGraphics();
            g.setColor(drawText.getBackground());
            g.fillRect(0, 0, drawText.getSize().width, drawText.getSize().height);

            // Set up our character arrays
            setCharacterArrays();

            // Set up the scrollbars
            resetScrollbars();

            cursorPos = new Point(0, 0);		// set the cursor to its starting position

            append(str);		// Now add the new string that we were passed

        }

	/** Finish this method, damn't! */
	public void setCaretPosition(int position) {
		System.out.println("setCaretPosition of JMText was called, but it is still just a stub.");
	}

	/** Return a string of the text-content of JMText (does not return the markings) */
	public synchronized String getText() {
		String work;
		StringBuffer tempString = new StringBuffer("");
		int returnRows = cursorPos.y;
		int returnColumns = textChars[0].length;
		
		// add any historyText, if there is
		if (historyText != null) {
			for (int i = 0; i < historyText.size(); i++) {
				tempString.append(historyText.elementAt(i));
			}
		}

		for (int y = 0; y < returnRows; y++) {
			work = "";
			try {
				work = textChars[y] + "";
			} catch (Exception e) {
				System.out.println("Error in JMText.getText " + e);
			}

                        // Smack any 'null's off the end *WHACK WHACK WHACK!*
                        if (work.indexOf('\u0000') == 0) {
                            tempString.append('\n');
                        } else {
                            if (work.indexOf('\u0000') > 0) {
                                tempString.append(work.substring(0, work.indexOf('\u0000')));
                                if (work.indexOf('\n') < 1) {
                                    // There is no newline here, we'll add one
                                    tempString.append('\n');
                                }
                            } else {
				tempString.append(work);
                            }
                        }
		}

		return tempString.toString().trim();
		
	}

        /** Set our redraw status to use double buffering for our display redraws */
        public synchronized void setDoubleBuffer(boolean status) {
            doubleBuffer = status;
        }

        /** Report if we use double buffering for our display redraws */
        public synchronized boolean isDoubleBuffer() {
            return doubleBuffer;
        }

        public synchronized void select(int selectionStart, int selectionEnd) {
            if (selectionStart == 0 && selectionEnd == 0) {
                // Deselect any selections
                selectBegin.x = 0;
                selectBegin.y = 0;
                selectEnd.x = 0;
                selectEnd.y = 0;
            }
            System.out.println("Stub: JMText.select(int, int) was called.");

            // Refresh the screen with out changes
            repaint();
            return;
        }

	public synchronized int getSelectionEnd() {
		return TranslateSelect(textBegin);
	}

	public synchronized int getSelectionStart() {
		return TranslateSelect(textEnd);
	}

	/** Translate the selected location into a integer representing
	 * the number of characters from the beginning of the JMText
	 * to the selected point */
	private int TranslateSelect(Point location) {
		int total = 0;

		if (location.y > 0) {
			for (int i = 0; i < location.y; i++) {
				total += textChars[i].toString().length();
			}
		}

		total += location.x;

		return total;
	}

	/** return a String of the text currently selected in the JMText area */
        public synchronized String getSelectedText() {
            StringBuffer totalString = new StringBuffer("");
            String workString = new String("");

            int baseLine = 0;	// the 'top' line of text to view
            if (vertScroll != null) {
                baseLine = vertScroll.getValue();
            }

            TestSelection();

            // First, we'll nab the start and end points based on the selection
            for (int i = 0; i <= (textEnd.y - textBegin.y); i++) {
                // First we grab the entire, upper-most line
                workString = textChars[textBegin.y + i + baseLine] + "";

                // We'll make sure we're not containing any 'newline' characters
                // The first one we encounter, that is the *end* of our line
                if (workString.indexOf('\n') > 0) {
                    workString = workString.substring(0, workString.indexOf('\n'));
                } else {
                    if (workString.indexOf('\u0000') > 0) {
                        workString = workString.substring(0, workString.indexOf('\u0000'));
                    }
                }

                // Now we'll trim down the workString if it happens to be the first
                // or last row (this can be "cumulative")
                if (i == 0) {
                    workString = workString.substring(textBegin.x);
                }
                if (i == (textEnd.y - textBegin.y)) {
                    if (textEnd.y == textBegin.y) {
                        // Beginning and end are on the same line
                        int end = textEnd.x - textBegin.x;
                        if (workString.length() < end || end < 0) {
                            end = workString.length();
                        }
                        workString = workString.substring(0, end);
                    } else {
                        int end = textEnd.x;
                        if (workString.length() < end || end < 0) {
                            end = workString.length();
                        }
                        workString = workString.substring(0, end);
                    }
                }
                totalString.append(workString);
            }

            String finalString = totalString.toString();
            // return finalString.substring(0, finalString.length() - 1); // Chop of any extra white space at the end.
            return finalString;
        }

        public synchronized void setEditable(boolean edit) {
            System.out.println("Stub: JMText.setEditable(boolean) was called.  Currently, JMText does not have the capacity to be editable.");
        }

        /** This is the over-ridden TextListener for JMText.  Since we lack the
	 * pre-made text-listener of a normal TextArea, we'll have to do a little
	 * work-around.
	 */
	public void addTextListener(TextListener listener) {
		textL = AWTEventMulticaster.add(textL, listener);
	}

	// KeyListener events
	public void addKeyListener(KeyListener listener) {
		keyL = AWTEventMulticaster.add(keyL, listener);
		enableEvents(AWTEvent.KEY_EVENT_MASK);
	}

	public void removeKeyListener(KeyListener listener) {
		keyL = AWTEventMulticaster.remove(keyL, listener);
	}

	public void keyPressed(KeyEvent event) {
		if(keyL != null) {
			keyL.keyPressed(event);
		}
	}

	public void keyReleased(KeyEvent event) {
		if(keyL != null) {
			keyL.keyReleased(event);
		}
	}

	public void keyTyped(KeyEvent event) {
		if(keyL != null) {
			keyL.keyTyped(event);
		}
	}

        public synchronized void componentHidden(ComponentEvent event) {
            minimised = true;
	}

	public void componentMoved(ComponentEvent event) {
	}

        public synchronized void componentResized(ComponentEvent event) {
            // if (!notified) {
            if (notified) {
                return;
            }

            if (this.getSize().width < 1 || this.getSize().height < 1) {
                // This item has a negative dimension, which we'll
                // take as being the same a "minimized"
                minimised = true;
                // System.out.println("JMText set minimised, returning.");
                return;
            } else {
                if ((widgetSize.height != drawText.getSize().height || widgetSize.width != drawText.getSize().width) && getSize().height > 0) {
                    minimised = false;
                    if (!notified) {
                        InitializeJMText();
                    }
                }

                // this item has a positive size, so we'll save its dimensions
                widgetSize.setSize(getSize());
                System.out.println("JMText.ComponentResized() called, would spool text.");
                // spoolText();
            }
            minimised = false;
            // }
        }

        private void JMComponentResized() {

            if (this.getSize().width < 1 || this.getSize().height < 1) {

                // This item has a negative dimension, which we'll
                // take as being the same a "minimized"
                minimised = true;
                return;
            } else {
                // if ((widgetSize.height != drawText.getSize().height || widgetSize.width != drawText.getSize().width) && getSize().height > 0) {
                if (widgetSize.height != drawText.getSize().height || widgetSize.width != drawText.getSize().width) {
                    minimised = false;
                    if (!notified) {
                        InitializeJMText();
                    }
                    // Check to see if it affects the columns & rows before calling a resize
                    // if (checkNewRowsSame()) {
                    // return;
                    // }

                    // Calculate new columns & rows
                    resizeDisplay();

                    // this item has a positive size, so we'll save its dimensions
                    // System.out.println("resizeDisplay() setting widgetSize to current size:");
                    // System.out.println(widgetSize + " -> " + drawText.getSize());
                    // widgetSize.setSize(getSize());  // This returns a slightly different value than drawText
                    widgetSize.setSize(drawText.getSize());
                    spoolText();
                    //System.out.println("JMText.JMComponentResized() called, would spool text.");
                }

            }
            minimised = false;
        }

        public synchronized void componentShown(ComponentEvent event) {
            minimised = false;
            widgetSize.setSize(getSize());
            spoolText();
        }

        /** We need to make our own mouse-listener that'll report back
	 * to any listeners that may've registered to this component.  Or
	 * something.  This may not make a lotta sense, I'm tired.
	 */
	public synchronized void addMouseListener(MouseListener mouse) {
		mListener = AWTEventMulticaster.add(mListener, mouse);
		enableEvents(AWTEvent.MOUSE_EVENT_MASK);
	}

	public synchronized void removeMouseListener(MouseListener mouse) {
		if (mListener != null) {
			mListener = AWTEventMulticaster.remove(mListener, mouse);
		}
	}

	// Mouse events
	public void mouseClicked(MouseEvent event) {
		if (event.getClickCount() >= 2) {
			// Select the "word" that the mouse was double-clicked on
			doubleClickSelect(event.getX(), event.getY());
		}

		if (mListener != null) {
			mListener.mouseClicked(event);
		}
	}

	public void mouseEntered(MouseEvent event) {
		// Change the mouse cursor to an I-beam
		setCursor(new Cursor(Cursor.TEXT_CURSOR));
	}

	public void mouseExited(MouseEvent event) {
		// Return mouse cursor back to standard pointer
		setCursor(new Cursor(Cursor.DEFAULT_CURSOR));
	}

	public void mousePressed(MouseEvent event) {
		// The mouse has been pressed, we will record its array position
		textBegin = pointToArray(new Point(event.getX(), event.getY()));
		textEnd = new Point(textBegin.x, textBegin.y);

                // Check to see if we have to deselect any old selections
                if (selectBegin.x > 0 || selectBegin.y > 0 || selectEnd.x > 0 || selectEnd.y > 0) {
                    paintSelection(selectBegin, selectEnd, drawText.getBackground());
                }

		// And now we'll record the 'corrected position'; the character's position
                selectBegin = arrayToPoint(textBegin);
		selectEnd = new Point(selectBegin.x, selectBegin.y);
	}

	public void mouseReleased(MouseEvent event) {
            if(selectBegin.x == selectEnd.x && selectBegin.y == selectEnd.y) {
                repaint();
                // No need for selection... start and end points are the same
                return;
            }
	}

	public void mouseDragged(MouseEvent event) {
            // First we'll nab our selected character location
            Point tempPoint = pointToArray(new Point(event.getX(), event.getY()));
            int selectedColumn = tempPoint.x;
            int selectedRow = tempPoint.y;

            // First we'll grab our FontMetrics
            FontMetrics fm = getFontMetrics(getFont());

            int oldx = selectEnd.x, oldy = selectEnd.y;
            // Figure out the coordinates of this selection

            tempPoint = arrayToPoint(new Point(selectedColumn, selectedRow));
            int x = tempPoint.x;
            int y = tempPoint.y;

            if((x < selectBegin.x && y < selectBegin.y) &&
               (x < selectEnd.x && y < selectEnd.y)) {

                selectBegin.x = x;
                selectBegin.y = y;
                textBegin.x = selectedColumn;
                textBegin.y = selectedRow;
            } else {
                selectEnd.x = x;
                selectEnd.y = y;
                textEnd.x = selectedColumn;
                textEnd.y = selectedRow;
            }

            if(oldx != x || oldy != y) {
                // paintSelection();
                // erase our old selection first
                paintSelection(selectBegin, new Point(oldx, oldy), drawText.getBackground());
                // Then draw our new selection
                paintSelection(selectBegin, selectEnd, drawText.getBackground());
            }
        }

	public void mouseMoved(MouseEvent e) {
	}
	
	public void focusGained(FocusEvent event) {
            if (!notified) {
                InitializeJMText();
            }
        }

        public void focusLost(FocusEvent event) {
	}

	/** Here, we check for an ANSI colour escape in the 'token', starting at position 'i'.
	 * If this is indeed a real colour escape, we will return i as the end of the escape,
	 * modifying the 'currentAttrib' and "eating" the escape */
        // private synchronized int checkEscape(String token, int i) {
        private int checkEscape(String token, int i) {
		int end = 0;
                int split = token.indexOf('\u001b') + 2;
                int newSplit = 0;
		int token1 = -1;
                int token2 = -1;
                boolean loop = true;

                end = token.indexOf('m', i);	// look for the closing 'm' from the offset of 'i'

                if (end == 0) {
			return i; 	// This token doesn't qualify as a colour
                }
                int mPlace = token.indexOf('m', i);
                if (mPlace < 0) {
                    return token.length();
                }
                String lilToken = token.substring(i, mPlace + 1);

                // This should continue a loop until our last element, which does
                // not end with a ';' character.  We'll grab that last.
                try {
                    while (loop) {
                        newSplit = lilToken.indexOf(';', split);

                        if (newSplit > end || newSplit < 1 || split >= newSplit) {
                            loop = false;
                            break;
                        }

                        token1 = Integer.parseInt(lilToken.substring(split, newSplit));

                        BuildColourAttr(token1);
                        split = lilToken.indexOf(';', split);
                        split++;
                    }
                } catch (Exception e) {
                    System.out.println("Token-loop caught: " + e);
                    System.out.println("Start (split) " + split + " end split (newSplit) " + newSplit);
                    loop = false;
                }

                if (lilToken.indexOf(';') > 0) {
                    int endStart = lilToken.lastIndexOf(';') + 1;
                    int endEnd = lilToken.lastIndexOf('m');
                    try {
                        if (endStart > 0) {
                            token1 = Integer.parseInt(lilToken.substring(endStart, endEnd));
                        } else {
                            // This is just a single colour token
                            token1 = Integer.parseInt(token.substring(i + 2, end));
                            System.out.println("We think we've just got a little token.");
                        }

                        if (token1 == 0) token1 = 10;
                    } catch (NumberFormatException nfe) {
                        System.out.println(nfe);
                        System.out.println("lilToken " + lilToken);
                        System.out.println("Token " + token);
                        System.exit(0);
                        return (endEnd + 1);
                    } catch (Exception e) {
                        System.out.println("Error trying to get closing token." + e);
                        // This seems to happen if we're missing a final 'm'
                        System.out.println("Our entire token was: " + token);
                        System.out.println("The lilToken was: " + lilToken);
                        int start = lilToken.lastIndexOf(';');
                        int stop = lilToken.indexOf('m', start);
                        token1 = Integer.parseInt(lilToken.substring(start + 1, stop));
                    }
                } else {
                    token1 = Integer.parseInt(token.substring(i + 2, end));
                }
                BuildColourAttr(token1);

                return (end + 1);
	}

        /** A method to sift through ANSI colour numbers and assign
         * them to the proper "area" (foreground, background, etc...)
         */
        private void BuildColourAttr(int ca) {
            if (ca == 0) {
                ChangeAttrib("reset");
                return;
            }

            if (ca > 0 && ca < 40) {
                ChangeAttrib("fg:" + ca);
                return;
            }

            if (ca > 39 && ca < 50) {
                ChangeAttrib("bg:" + ca);
                return;
            }

            return;
        }

	/** We'll use this method to 'intelligently' add and remove attributes */
	private void ChangeAttrib(String attrib) {
		attrib = attrib.toLowerCase();

		// first, if we just need to reset the values, we don't need to continue
		if (attrib.equals("reset")) {
                    // reset attributes
                    currentAttrib = "";
                    return;
                }

		// Break up the currentAttribs into a vector for ease of manipulation
		Vector tokens = new Vector(0, 1);
		StringTokenizer workString = new StringTokenizer(currentAttrib, "|", false);
		for (int i = 0; i < workString.countTokens(); i++) {
			tokens.addElement(workString.nextToken());
		}

		// It is okay to have "nested" attributes, except for foreground and
		// background colours, which we will deal with here.
		if (attrib.startsWith("fg:")) {
                    // Change the foreground colour
                    for (int i = 0; i < tokens.size(); i++) {
                        if (((String)tokens.elementAt(i)).startsWith("fg:")) {
                            tokens.removeElementAt(i);
                            break;
                        }
                    }
                    // Now add the new foreground colour to the end of the vector
                }

		if (attrib.startsWith("bg:")) {
                    // Change the background colour
                    for (int i = 0; i < tokens.size(); i++) {
                        if (((String)tokens.elementAt(i)).startsWith("bg:")) {
                            tokens.removeElementAt(i);
                            break;
                        }
                    }
                    // Now add the new background colour to the end of the vector
                }

                // Now we reassemble the attributes
                StringBuffer newAttribs = new StringBuffer("");
                for (int i = 0; i < tokens.size(); i++) {
                    newAttribs.append((String)tokens.elementAt(i) + "|");
                }

                newAttribs.append(attrib + "|");	// Now add the newest attribute
                currentAttrib = newAttribs.toString();

        }

        /** Calculate the proper colour to return from either ANSI or Hex codes */
        private Color CalculateColour(String markings, int offset) {
            String c = markings.substring(offset + 3, markings.indexOf('|', offset));
            Color rC = getForeground();	// Just in case
            // Now determine what the colour is
            if (c.equals("0") || c.equals("30") || c.equals("40")) rC = Color.black;
            if (c.equals("10")) rC = Color.lightGray;
            if (c.equals("1") || c.equals("31") || c.equals("41")) rC = Color.red;
            if (c.equals("2") || c.equals("32") || c.equals("42")) rC = Color.green;
            if (c.equals("3") || c.equals("33") || c.equals("43")) rC = Color.yellow;
            if (c.equals("3")) rC = Color.lightGray;
            if (c.equals("4") || c.equals("34") || c.equals("44")) rC = Color.blue;
            if (c.equals("5") || c.equals("35") || c.equals("45")) rC = Color.magenta;
            if (c.equals("6") || c.equals("36") || c.equals("46")) rC = Color.cyan;
            if (c.equals("7") || c.equals("37") || c.equals("47")) rC = Color.white;

            return rC;
        }

        /** This method returns "feedback" to our registered TextListener.
         * Right now this is very simplified, simply alerting the Listener that
         * the text value has changed @see java.awt.event.TextEvent
         * In the future this method may be expanded upon.
         */
        private void FauxTextListener() {
            // Here, we'll process our FauxText Event and package it up to the listener
            if (textL != null) {
                textL.textValueChanged(new TextEvent(this, TextEvent.TEXT_VALUE_CHANGED));
            }

        }

        /** We override 'getSize()' to ensure that we always send a valid
         * size back to the component.  (This compensates for weirdness
         * when the component is minimised and stuff)
         */
        public Dimension getSize() {
            return super.getSize();
        }

	/** With the displayed area being resized (on the screen),
	 * we will have to handle copying our arrays to a temporary
	 * array, resizing the original array and then put the info back
         */
        // We'll temporarily name this method "oldresizeDisplay" to try
        // a new method of resizing the Display (just below this one)
	private synchronized void resizeDisplay() {
            if (!notified) {
                return;
            }

            if (checkNewColumnsSame()) {
                // Only the height has changed, so we do not need to reformat
                // Adjust the scrollbars to their proper maximum
                resizeScrollbars();
                return;
            }

            if (checkNewRowsSame()) {
                return;
            }

            if (rows < 1 || columns < 1) {
                minimised = true;
                return;
            }

            if (cursorPos == null) return;  // If there is no cursor, we don't do this

            char[][] tempChars = new char[cursorPos.y + 1][columns];
            String[][] tempMarkings = new String[cursorPos.y + 1][columns];
            int tempRows = cursorPos.y;		// The bottom of the array
            int tempColumns = columns;		// The "right edge" of the array

            // Copy the array into a new temporary array
            try {
                System.arraycopy(textChars, 0, tempChars, 0, cursorPos.y + 1);
                System.arraycopy(markings, 0, tempMarkings, 0, cursorPos.y + 1);
            } catch (Exception e) {
                System.out.println("Trouble copying arrays in resizeDisplay. " + e);
                System.out.println("Current cursor position: " + cursorPos.y);
                return;
            }

            // Clear the old array
            setCharacterArrays();

            // Reset the attributes
            InitializeJMText();

            // Now we'll (in an ugly way) loop through the array and re-write
            // the old information into the format of the new array

            StringBuffer chopString;
            String curAt = new String("");; // Current attribute for our purposes here
            String workString = new String("");
            String markLine[];
            int atLoc; // cummulative length of attributes added to the line
            int tLength;
            int lastNewLine;

            // This initial loop goes through the array line by line,
            // converting each line to a string (with proper termination)
            for (int i = 0; i < tempRows; i++) {
                atLoc = 0;
                workString = new String(tempChars[i]);
                // chopString = new StringBuffer(workString.trim());
                lastNewLine = workString.lastIndexOf('\n');
                lastNewLine++;  // This gives us the number of characters till the last new line
                if (lastNewLine > 0) {
                    chopString = new StringBuffer(workString.substring(0, lastNewLine));
                    tLength = lastNewLine;
                } else {
                    tLength = workString.lastIndexOf(' ');
                    tLength++;
                    // chopString = new StringBuffer(workString.trim());
                    chopString = new StringBuffer(workString.substring(0, tLength));
                    if (tLength < chopString.length()) {
                        tLength = chopString.length();
                    }
                }
/*                if (workString.lastIndexOf('\n') > 0) {
                    // There was a hard-break at the end of the original line
                    // which .trim() removed, so we'll replace it.
                    chopString.append('\n');
                    }
                    */

                // tLength = (workString.trim()).length();
                // tLength = chopString.length();
                markLine = tempMarkings[i];
                for (int j = 0; j < tLength; j++) {
                    if (markLine[j] != null) {
                        if (!curAt.equals(markLine[j])) {
                            curAt = markLine[j];
                            if (curAt.equals("")) {
                                // Add a tag to reset colouring
                                chopString.insert(j + atLoc, '\u001b' + "[0m");
                                atLoc += 4;
                            } else {
                                // Insert a tag to start colouring of proper attributes
                                chopString.insert(j + atLoc, '\u001b' + "[" + curAt.substring(3, curAt.length() - 1) + "m");
                                atLoc += curAt.length() - 1;
                            }
                        }
                    }
                }
                workString = chopString.toString();
                append(workString);
            }

            Point tempCursor = new Point(0, 0); // To keep track of where we are...
        }

        /** With the displayed area being resized (on the screen),
	 * we will have to handle copying our arrays to a temporary
	 * array, resizing the original array and then put the info back
         */
        // This is the new experimental-type version of resizeDisplay
	private void newresizeDisplay() {

            if (cursorPos == null) return;  // If there is no cursor, we don't do this

            char[][] tempChars = new char[cursorPos.y + 1][columns];
            String[][] tempMarkings = new String[cursorPos.y + 1][columns];
            int tempRows = cursorPos.y;		// The bottom of the array
            int tempColumns = columns;		// The "right edge" of the array

            // Copy the array into a new temporary array
            try {
                System.out.println("Our cursor position is: " + cursorPos.y);
                System.arraycopy(textChars, 0, tempChars, 0, cursorPos.y + 1);
                System.arraycopy(markings, 0, tempMarkings, 0, cursorPos.y + 1);
            } catch (Exception e) {
                System.out.println("Trouble copying arrays in resizeDisplay. " + e);
                System.out.println("Current cursor position: " + cursorPos.y);
                return;
            }

            // Reset the attributes
            InitializeJMText();

            // Now we'll (in an ugly way) loop through the array and re-write
            // the old information into the format of the new array

            StringTokenizer chopString;
            int count = 0;
            for (int i = 0; i < tempRows; i++) {
                // Loop through the rows

            }

            Point tempCursor = new Point(0, 0); // To keep track of where we are...
        }

        private void paintSelection(Point begin, Point end, Color XORColour) {
            // Here, we keep the "selection" updated for the paint method
            FontMetrics fm = getFontMetrics(getFont());
            int height = fm.getHeight();
            int descent = fm.getMaxDescent();

            Graphics tempGraph = drawText.getGraphics();

            if (begin.y == end.y) {
                tempGraph.setXORMode(XORColour);
                tempGraph.fillRect(begin.x,
                                   begin.y + descent,
                                   end.x - begin.x,
                                   end.y - begin.y + height);
            }

            if (begin.y < end.y) {
                tempGraph.setXORMode(XORColour);
                tempGraph.fillRect(begin.x,
                                   begin.y + descent,
                                   getSize().width - begin.x,
                                   height);
                tempGraph.fillRect(0,
                                   begin.y + height + descent,
                                   getSize().width,
                                   end.y - (begin.y + height));
                tempGraph.fillRect(0,
                                   end.y + descent,
                                   end.x,
                                   height);
            }

        }

        /** This translates a point on the screen into a location
         * on our array of characters !  (This method is usually used
         * in conjunction with marking text)
         */
        public Point pointToArray(Point pos) {
            // First we'll grab our FontMetrics
            FontMetrics fm = getFontMetrics(getFont());
            int height = fm.getHeight();
            int descent = fm.getDescent();

            // Figure out where we're dragging right now
            int selectedColumn = -1;
            int selectedRow = (int)((pos.y - descent) / height);
            if ((pos.y - descent) / height > selectedRow) {
                // We'll need to round this up by one
                selectedRow++;
            }

            int x = 0;
            // Figure out the coordinates of this selection
            int count = 0;
            int baseLine = 0;	// the 'top' line of text to view

            if (vertScroll != null) {
                baseLine = vertScroll.getValue();
            }

            for (int i = 0; i < columns; i++) {
                count += fm.charWidth(textChars[selectedRow + baseLine][i]);
                if (count > pos.x || (textChars[selectedRow + baseLine][i] == '\u0000')) {
                    selectedColumn = i;
                    break;
                }
            }
            if (selectedColumn > columns) selectedColumn = columns;

            return new Point(selectedColumn, selectedRow);
        }

        private Point arrayToPoint(Point pos) {
            // First we'll grab our FontMetrics
            FontMetrics fm = getFontMetrics(getFont());

            // Figure out where we're dragging right now
            int selectedColumn = pos.x;
            int selectedRow = pos.y;

            int x = 0;
            // Figure out the coordinates of this selection
            int count = 0;
            int baseLine = 0;	// the 'top' line of text to view

            if (vertScroll != null) {
                baseLine = vertScroll.getValue();
            }

            for (int i = 0; i < selectedColumn; i++) {
                count += fm.charWidth(textChars[selectedRow + baseLine][i]);
            }


            int y = selectedRow * fm.getHeight();
            return new Point(count, y);
        }

        /** This method returns the value of a string
         * once it has had any escape characters stripped
         * from it.
         */
        private String StripEscapes(String token) {
            StringBuffer workString = new StringBuffer("");
            boolean loop = true;
            int start = 0;
            int end = 0;

            do {
                end = token.indexOf('\u001b', start);

                if (end < start) {
                    // There are no escapes left
                    if (start < 0) {
                        System.out.println("We break to avoid an index error.");
                        break;
                    }
                    workString.append(token.substring(start));
                    loop = false;
                    break;
                }

                if (end > 0 && start >= 0) {
                    workString.append(token.substring(start, end));
                }

                // Now set the new 'start'
                start = token.indexOf('m', end) + 1;

                if (start <= end) {
                    loop = false;
                    System.out.println("Not a proper ANSI escape");
                    System.out.println(token);
                    break;
                }

                // Check to see if we've reached the end of our token.  If this is
                // the case, then we can end the loop and return what we've got so far.
                if (start >= token.length()) {
                    loop = false;
                    break;
                }
            } while (loop);

            return workString.toString();
        }

        /** Select the 'token' that has been double-clicked,
         * based on the x and y position of the cursor
         */
	private void doubleClickSelect(int x, int y) {
            // First, determine the start-point...
            // we want the 'point' translated to a location in our array
            Point cursor;
            cursor = pointToArray(new Point(x, y));
            int sx, sy;	// The 'start points'
            int ex, ey;	// The 'end points'
            int baseLine = 0;	// The 'baseline' to start reading from
            if (vertScroll != null) {
                baseLine = vertScroll.getValue();
            }
            sx = cursor.x;
            sy = cursor.y + baseLine;

            // Now we'll read backwards to find the beginning of the token
            while (sy > -1 && textChars[sy][sx] !=' ') {
                sx--;
                if (sx < 0) {
                    sy--;
                    sx = (textChars[sy] + "").length() - 1;
                    if (textChars[sy][sx] == '\n' || textChars[sy][sx] == '\r' || textChars[sy][sx] == ' ') {
                        sx = 0;
                        sy++;
                        break;
                    }
                }
            }
            sx++;	// This moves us off the 'space' character

            // And forwards to the end of the token.
            ex = sx;
            ey = sy;

            while (ey < cursorPos.y && textChars[ey][ex] != ' ') {
                if (ey == cursorPos.y && ex >= (cursorPos.x - 1)) {
                    break;
                }
                ex++;
                if (ex >= (textChars[ey] + "").length() || textChars[ey][ex] == '\u0000' || textChars[ey][ex] == '\n' || textChars[ey][ex] == '\r') {
                    if (ey < cursorPos.y - 1) {
                        ey++;
                        ex = 0;
                    } else {
                        break;
                    }
                }

                if (textChars[ey][ex] == ' ') {
                    // Normal termination check here
                    break;
                }

            }

            // We now have our selection!
            textBegin = new Point(sx, sy - baseLine);
            textEnd = new Point(ex, ey - baseLine);
        }

        /** Test the selected area to make sure parameters don't fall
         * into illegal bounds, as it were.  Y'know, like -1 and such */
        private void TestSelection() {
            int length = (textChars[textBegin.y] + "").length() - 1;
            if (textBegin.x < 0) textBegin.x = 0;
            if (textEnd.x > length) textBegin.x = length;
            if (textBegin.y < 0) textBegin.y = 0;
            // There should be a test for the end of 'y', but I digress
        }

        public void addNotify() {
            super.addNotify();

            if (!notified && this.getParent().isVisible()) {
                Dimension test = new Dimension(this.getParent().getSize());
                if (getSize().width == 0 || getSize().height == 0) {
                    setSize(getPreferredSize());
                }
                this.validate();
            }
        }

        /** Over-ride the setFont so that we can readjust our display
         * for the new font-metrics */
        public synchronized void setFont(Font f) {
            super.setFont(f);
            resizeDisplay();
        }

        /** This tests the current view for the number of rows and columns.
         * If the columns and rows are the same, <pre>true</pre> is
         * returned.  Otherwise, we receive a <pre>false</pre>
         */
        private boolean checkNewRowsSame() {
            // Check to see if the change in size is suitable
            // int prevRows = rows;
            // int prevCols = columns;

            getRowsAndColumns();
            if (prevRows == rows && prevCols == columns) {
                return true;
            }

            // Our new rows are different
            prevRows = rows;
            prevCols = columns;

            // Our size has changed, so make certain that our scrollbars keep up!
            resizeScrollbars();

            return false;
        }

        /** Much like checkNewRowsSame(), though this function only checks to
         * see if when resized we still maintain the same number of columns.
         * This returns <pre>true</pre> if the number of hasn't changed, or
         * <pre>false</pre> if they have.  A function like this is useful as
         * reformating text due to a width-change is very time consuming; but
         * if only the height need be adjusted, it is a trivial matter of setting
         * a new &quot;base-line" for the window.
         */
        private boolean checkNewColumnsSame() {
            getRowsAndColumns();

            if (prevCols == columns) {
                // The number of columns haven't changed
                return true;
            }

            // Yes, our number of columns have changed.  You poor CPU, you.
            return false;
        }

        private void setCharacterArrays() {
            // Now we can set up the character arrays
            textChars = new char[maxRows][columns];
            markings = new String[maxRows][columns];
            cursorPos = new Point(0, 0);
        }

        /** Set the initial values for any scrollbars we have */
        private void resetScrollbars() {
            // Set up the scrollbars
            if (vertScroll != null) {
                vertScroll.setMaximum(0);
            }

            if (horzScroll != null) {
                horzScroll.setMaximum(0);
            }
        }

        /** This method calculates the new size of the scrollbars;
         * handy incase we change the size of our display
         */
        private void resizeScrollbars() {
            if (vertScroll != null && cursorPos != null) {
                // Fix this XXX, right now we just hammer things to the bottom of the screen
                // vertScroll.setValue(vertScroll.getMaximum());
                int max = (cursorPos.y + 1) - rows;
                if (vertScroll.getMaximum() != max); {
                    vertScroll.setMaximum(max);
                    vertScroll.setValue(max);
                }
            }
        }

        /** Spool out any text that may be pending.  Most likely this would be
         * called from an event that displays our JMText after being minimised
         * or hidden. */
        private synchronized void spoolText() {
            if (pendingText.size() == 0) {
                // There is no pending text
                return;
            }

            for (int i = 0; i < pendingText.size(); i++) {
                append(pendingText.elementAt(i).toString());
            }

            pendingText.removeAllElements();	// Make sure the vector is empty

        }
}

/** An inner class simply to subclass the canvas (to catch paint() events) */
class LittleCanvas extends Canvas {

    public void paint(Graphics g) {
        this.getParent().paint(g);
    }

}

