/**
 * The JamochaMUD Frame synchroniser:
 * SyncFrame.java handles the synchronising of
 * frame minimize/maximize, and potentially, movement events
 * $Id: SyncFrame.java 1.15 2001/12/23 02:07:58 jeffnik Exp $
 */

/* JamochaMUD, a Muck/Mud client program
 * Copyright (C) 1998-2000  Jeff Robinson
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version. *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

package anecho.gui;

import java.awt.*;
import java.awt.event.*;

import anecho.gui.SyncFrameGroup;

	/**
	 * The JamochaMUD Frame synchroniser:
	 * SyncFrame.java handles the synchronising of
	 * frame minimize/maximize, and potentially, movement events
         * @version $Id: SyncFrame.java 1.15 2001/12/23 02:07:58 jeffnik Exp $
         * @author Jeff Robinson
	 */
public class SyncFrame extends Frame {

	// Variables for the frame
	private SyncFrameGroup frameGroup;		/* The group for this frame */
	private boolean inSync = false;		/* The frame's synchronisation status */
	private Point compPos;			/* A comparison location to judge
							  *  movement of the frame */
	private boolean endProgram = false;		/* Closing of this frame terminates JVM */
	private boolean locked = false;		/* Lock status of this frame (for use
							 * with SyncFrameGroup */

  
	public SyncFrame(String title) {
		this(title, null);
	}

	public SyncFrame(String title, SyncFrameGroup group) {
		this(title, false, group);
	}

	public SyncFrame(String title, boolean sync, SyncFrameGroup group) {
		super(title);
		if (group != null) {
			this.frameGroup = group;
			frameGroup.add(this);
			
			// group.add(this);
		}

		inSync = sync;
		compPos = new Point(0, 0);

		enableEvents(AWTEvent.WINDOW_EVENT_MASK);
		enableEvents(AWTEvent.COMPONENT_EVENT_MASK);
	}

	/**
	 * Set the synchronisation of this frame
	 */
	public void setSync(boolean sync) {
		inSync = sync;
	}

	/**
	 * Set the synchronisation of the entire frame group
	 */
	public void setGroupSync(boolean sync) {
		// Go through the Syncframes of this group and set their status
		if (frameGroup == null) return;
	}

	/**
	 * Get the synchronisation status of this frame
	 * @return	returns the boolean state of this SyncFrame's
	 *			synchronisation status
	 * @see 	gui.SyncFrame#setSync
	 */
	public boolean isSynced() {
		return inSync;
	} 

	/** Set the SyncFrameGroup for this frame */
        public void setSyncFrameGroup(SyncFrameGroup group) {
            frameGroup = group;
	}

	/** Set the "termination state" of this frame (whether it kills the JVM or not) 
	 *@param	state	A <code>true</code> state indicates that closing
	 *				this SyncFrame will terminate the JVM.
	 *			A <code>false</code> indicates this SyncFrame will
	 *				simply terminate itself, leaving the JVM to run
	 */
	public void setCloseState(boolean state) {
		endProgram = state;
	}

	/** Report the "termination state" of this frame
	 * @return	A <code>true</code> state indicates that the closing
	 *			of this frame will terminate the JVM, 
	 *			a <code>false</code> will simply close the frame 
	 */
	public boolean getCloseState() {
		return endProgram;
	}

	// WINDOW EVENTS
	public void processWindowEvent(WindowEvent event) {
            super.processWindowEvent(event); // Handle listeners

            if (event.getID() == WindowEvent.WINDOW_CLOSING) {
                super.processWindowEvent(event);
                if (endProgram) {
                    // Now we kill the JVM
                    System.exit(0);
                }
            }

            if (event.getID() == WindowEvent.WINDOW_ICONIFIED) {
                hideComponents();

                if (frameGroup != null && this.isLocked()) {
                    this.setLock(false);
                    return;
                }

                // hideComponents();
            }

            if (event.getID() == WindowEvent.WINDOW_DEICONIFIED) {
                showComponents();
                if (frameGroup != null && this.isLocked()) {
                    this.setLock(false);
                    return;
                }

            }

            if (event.getID() == WindowEvent.WINDOW_ACTIVATED) {
            }

        }

        public void processComponentEvent(ComponentEvent event) {
            super.processComponentEvent(event); // Handle listeners

            if (event.getID() == ComponentEvent.COMPONENT_MOVED) {
                moveComponents(event);
                if (frameGroup != null && this.isLocked()) {
                    this.setLock(false);
                    return;
                }
            }

            if (event.getID() == ComponentEvent.COMPONENT_RESIZED) {
                super.processComponentEvent(event); // Handle listeners
            }

        }

        /**
         * Handle the calculation of how far the active frame
         * has moved, and adjust the rest of the group accordingly
         */
        private void moveComponents(ComponentEvent event) {
            // Our component has to be visible.  Newer versions of Java
            // get very cranky about this feature
            if (compPos == null || !this.isVisible()) {
                // System.out.println("Checking for visibility (" + this.isVisible() + ") and compPos " + compPos);
                // System.out.println("We failed on " + this.getTitle());
                return;
            }

            // First, we check to see if this frame is sync'd.
            // If not, then we don't need to check the rest of this method
            if (!inSync) {
                return;
            }

            // Original position of the component
            int xOrg = compPos.x;
            int yOrg = compPos.y;
            // New position of the component
            int xNew = event.getComponent().getLocationOnScreen().x;
            int yNew = event.getComponent().getLocationOnScreen().y;

            // Now we'll set the new comparison position
            compPos.setLocation(xNew, yNew);

            // Cycle through all the SyncFrames in this group and apply the position change
            if (frameGroup != null) {
                frameGroup.moveFrames(this, (xNew - xOrg), (yNew - yOrg));
            }
        }

	/** Hide all the associated frames if in sync */
	private void hideComponents() {
		if (!inSync) return;

		if (frameGroup != null) {
			frameGroup.changeState(this, SyncFrameGroup.HIDE);
		}
	}

	/** Show all the associated frames if in sync */
	private void showComponents() {
		if (!inSync) return;

		if (frameGroup != null) {
			frameGroup.changeState(this, SyncFrameGroup.SHOW);
		}
	}

	/** Set the state of all the frames in this group to be the same 
	 * @param state	If <code>true</code>, all the frames in
	 *				the group will be synchronised
	 *			If <code>false</code> the frames will act
	 *				independantly of each other.
	 */
	public void setAllStates(boolean state) {
		inSync = state;

		if (frameGroup != null) {
			if (state) {
				frameGroup.changeState(this, SyncFrameGroup.SET_ALL_TRUE);
			} else {
				frameGroup.changeState(this, SyncFrameGroup.SET_ALL_FALSE);
			}
		}
	}

	// Overridden methods of Frame
	/** This method is overridden to keep us up to date with the component's location */
	public void setBounds(int x, int y, int width, int height) {
		// Grab the dimension information
		
		compPos.setLocation(x, y);
		super.setBounds(x, y, width, height);
	}

	/** setBounds is overridden to keep us up to date with the component's location */
	public void setBounds(Rectangle r) {
		compPos.setLocation(r.x, r.y);
		super.setBounds(r);
	}

	/** setLocation is overridden to keep us up to date with the component's location */
        public void setLocation(int x, int y) {
            if (compPos == null) {
                return;
            }
            compPos.setLocation(x, y);
            super.setLocation(x, y);
	}

	/** setBounds is overridden to keep us up to date with the component's location */
        public void setLocation(Point p) {
            if (compPos == null) {
                return;
            }
            compPos.setLocation(p);
            super.setLocation(p);
	}

	/** dispose of public class Window is handled to remove this frame from it's *
	 * registered SyncFrameGroups first before letting the super-class handle the rest */
	public void dispose() {
            super.dispose();
            if (frameGroup != null) {
                frameGroup.remove(this);
            }
	}

	/** A paramString so that other classes can identify individual SyncFrames.
	 * @return	The name of the class, plus the name of the class
	 */
	public String paramString() {
		return new String(getClass() + ", " + getName());
	}

	/** Sets the "lock" status of this frame.  This is needed when dealing with
	 * SyncFrameGroups to prevent "stuttering".  A SyncFrame becomes "locked" 
	 * <code>true</code> by default if another SyncFrame is moved, hidden, or shown.
	 * Without the lock, certain OSes activate the other SyncFrame's events after the
	 * original event has been completed, making terrible loops.  Once a locked frame has
	 * received an event, it is unlocked (set <code>false</code>) again.
	 * @see	gui.SyncFrameGroup
	 * @param	status	the SyncFrame's lock status
	 */
	public synchronized void setLock(boolean status) {
		locked = status;
	}

	/** Returns the "lock" state of this frame.
	 * @see 	gui.SyncFrame#setLock
	 * @return	the state of the lock as a boolean variable
	 */
	public synchronized boolean isLocked() {
		return locked;
	}
}
