/*
 * (C) Copyright IBM Corp. 1997-1998  All rights reserved.
 *
 * US Government Users Restricted Rights Use, duplication or
 * disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * IBM will not be liable for any damages suffered by you as a result
 * of using the Program. In no event will IBM be liable for any
 * special, indirect or consequential damages or lost profits even if
 * IBM has been advised of the possibility of their occurrence. IBM
 * will not be liable for any third party claims against you.
 */

package samples.XJParse;

import com.ibm.xml.parser.Child;
import com.ibm.xml.parser.DTD;
import com.ibm.xml.parser.FormatPrintVisitor;
import com.ibm.xml.parser.MIME2Java;
import com.ibm.xml.parser.NonRecursivePreorderTreeTraversal;
import com.ibm.xml.parser.Parent;
import com.ibm.xml.parser.Parser;
import com.ibm.xml.parser.Source;
import com.ibm.xml.parser.Stderr;
import com.ibm.xml.parser.TXDocument;
import com.ibm.xml.parser.TXElement;
import com.ibm.xml.parser.TXText;
import com.ibm.xml.parser.Util;
import com.ibm.xml.parser.Version;
import com.ibm.xml.xpointer.XPointer;

import java.io.FileInputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.io.Writer;
import java.io.Reader;
import java.net.URL;
import java.util.Locale;
import java.util.StringTokenizer;

/**
 * XML Syntax Checker.
 *
 * @version Revision: 91 1.9 samples/XJParse/XJParse.java, xml4jsamples, xml4j-jtcsv, xml4j_1_1_16 
 * @author TAMURA Kent &lt;kent@trl.ibm.co.jp&gt;
 */
public class XJParse
{
    private XJParse()
    {
    }

    private static void reportMemory(Runtime rt) {
        long total = rt.totalMemory();
        long free = rt.freeMemory();
        long using = total-free;
        XJParse.printer.println(free+"/"+total+": use "+using+" Byte");
    }

    public static void main(String[] argv) {
        Locale.setDefault(new Locale(System.getProperty("user.language", "en"),
                                     System.getProperty("user.region", "")));
        int ret = process(argv);
        XJParse.printer.flush();
        System.exit(ret);
    }

    static boolean s_reportmemory = false;
    static Runtime s_rt = Runtime.getRuntime();
    static PrintWriter printer = new PrintWriter(System.err);

    public static int process(String[] argv) {
        boolean isprint = false;
        boolean isprintns = false;
        boolean isformatprint = false;
        String encname = null;
        boolean isprintclass = false;
        boolean isprintxpointer = false;
        boolean isallowj = false;
        boolean iskeepcomment = true;
        boolean donamespace = false;
        boolean warnnoxml = false;
        boolean stoponerror = false;
        boolean dodtd = false;
        boolean didversion = false;
        boolean processextdtd = true;
        boolean printwarning = true;
        int expandreferences = 0;
        int loop = 0;
        String catalog = null;
        String[] fnames = new String[argv.length];
        int noffnames = 0;

        for (int i = 0;  i < argv.length;  i ++) {
            if ('-' != argv[i].charAt(0)) {
                fnames[noffnames++] = argv[i];

            } else if ("-memory".equals(argv[i])) {
                s_reportmemory = true;
                XJParse.printer.print("Initial free/total memory:\n\t");
                reportMemory(s_rt);

            } else if ("-print".equals(argv[i]) || "-d".equals(argv[i])) {
                isprint = true;
                    
            } else if ("-nsprint".equals(argv[i])) {
                isprintns = true;

            } else if ("-format".equals(argv[i])
                       || "-print2".equals(argv[i])
                       || "-d2".equals(argv[i])) {
                isformatprint = true;

            } else if ("-expandreferences".equals(argv[i])) {
                expandreferences = 1;

            } else if ("-expandreferences2".equals(argv[i])) {
                expandreferences = 2;

            } else if ("-class".equals(argv[i])) {
                isprintclass = true;

            } else if ("-xpointer".equals(argv[i])) {
                isprintxpointer = true;

            } else if ("-nodtd".equals(argv[i])) {
                processextdtd = false;

            } else if ("-nowarn".equals(argv[i])) {
                printwarning = false;

            } else if ("-dtd".equals(argv[i])) {
                dodtd = true;

            } else if ("-n".equals(argv[i]) || "-t".equals(argv[i])) {
                if (i+1 < argv.length) {
                    int v = Integer.parseInt(argv[++i]);
                    if (0 >= v) {
                        XJParse.printer.println("-n/-t requires a number more than 0.");
                    } else
                        loop = v;
                } else {
                    XJParse.printer.println("-n/-t requires a number.");
                }

            } else if ("-c".equals(argv[i])) {
                if (i+1 < argv.length) {
                    catalog = argv[++i];
                } else {
                    XJParse.printer.println("-c requires a file name.");
                }

            } else if ("-outencoding".equals(argv[i])) {
                if (i+1 < argv.length) {
                    encname = argv[++i];
                } else {
                    XJParse.printer.println("-outencoding requires an encoding name.");
                }

            } else if ("-allowjavaencoding".equals(argv[i])) {
                isallowj = true;

            } else if ("-namespace".equals(argv[i])) {
                donamespace = true;

            } else if ("-removecomment".equals(argv[i])) {
                iskeepcomment = false;

            } else if ("-warnnoxml".equals(argv[i])) {
                warnnoxml = true;
                    
            } else if ("-stoponerror".equals(argv[i])) {
                stoponerror = true;

            } else if ("-version".equals(argv[i])) {
                XJParse.printer.println("XML4J version " + Version.S_VERSION);
                didversion = true;

            } else if ("-stdout".equals(argv[i])) {
                XJParse.printer.flush();
                XJParse.printer = new PrintWriter(System.out);
                Stderr.printer = XJParse.printer;
            } else {
                XJParse.printer.println("Warning: Unknown option: "+argv[i]);
            }
        }

        if (noffnames <= 0) {
            if (didversion)
                return 0;
            else  { 
                XJParse.printer.println("XML Syntax Checker");
                XJParse.printer.println("Usage:");
                XJParse.printer.println("    java XJParse [options] <URL or filename>");
                XJParse.printer.println("Options:");
                XJParse.printer.println("   -d                     Print parsed document in XML format");
                XJParse.printer.println("   -format                Print parsed document in XML format with formatting");
                XJParse.printer.println("   -nsprint               Print parsed document, including namespace substitutions");
                XJParse.printer.println("   -expandreferences      Expand entity references");
                XJParse.printer.println("   -nodtd                 Process no external DTD subset");
                XJParse.printer.println("   -nowarn                Print no warnings");
                XJParse.printer.println("   -outencoding mimename  Specify encoding when print");
                XJParse.printer.println("   -namespace             Enable namespace feature");
                XJParse.printer.println("   -n <n>                 Parse n times and print time");
                XJParse.printer.println("   -memory                Report memory usage");
                XJParse.printer.println("   -version               Display XML4J version number.");
                XJParse.printer.println("   -stdout                Display messages to the standard output");
                XJParse.printer.println("   -c <filename>          Catalog file");
                XJParse.printer.println("External-DTD parsing:");
                XJParse.printer.println("   -dtd");
                XJParse.printer.println("   -d                     Print parsed DTD");
                XJParse.printer.println("");
                XJParse.printer.println("To know other options, See the source :-)");
                return 1;
            }
        }

        for (int i = 0;  i < noffnames;  i ++) {
            String fname = fnames[i];

            try {
                TXDocument doc = null;
                DTD dtd = null;
                if (0 == loop) {
                    Stderr se = new Stderr(fname);
                    if (catalog != null) {
                        FileReader rr = new FileReader(catalog);
                        se.loadCatalog(rr);
                        rr.close();
                    }
                
                    Source src;
                    try {
                        src = se.getInputStream(fname, null, Stderr.file2URL(fname).toString());
                    } catch (java.io.FileNotFoundException e) {
                        XJParse.printer.println(fname + ": File not found.");
                        return 1;
                    }

                    se.setPrintWarning(printwarning);
                    Parser pc = new Parser(fname, se, se);
                    pc.setProcessExternalDTD(processextdtd);
                    pc.setAllowJavaEncodingName(isallowj);
                    pc.setKeepComment(iskeepcomment);
                    pc.setWarningNoDoctypeDecl(false);
                    pc.setProcessNamespace(donamespace);
                    pc.setWarningNoXMLDecl(warnnoxml);
                    pc.setEndBy1stError(stoponerror);
                    pc.setExpandEntityReferences(expandreferences == 1);
                    if (dodtd) {
                        dtd = pc.readDTDStream(src);
                        if (isprint) {
                            if (null == encname) {
                                dtd.printExternal(new PrintWriter(System.out), null);
                            } else {
                                String enc = MIME2Java.convert(encname);
                                if (null == enc)   enc = encname;
                                dtd.setEncoding(encname);
                                dtd.printExternal(new PrintWriter(new OutputStreamWriter(System.out, enc)),
                                                  enc);
                            }
                        }
                        return 0;
                    } else {
                        doc = pc.readStream(src);
                    }
                    pc = null;
                    if (expandreferences == 2)
                        doc.expandEntityReferences();

                    if (null == doc) {
                        XJParse.printer.println("No document");
                    } else if (isformatprint) {
                        if (null == encname) {
                            doc.printWithFormat(new OutputStreamWriter(System.out));
                        } else {
                            String enc = MIME2Java.convert(encname);
                            if (null == enc)   enc = encname;
                            doc.printWithFormat(new OutputStreamWriter(System.out, enc), enc, 2);

                        }
                    } else if (isprint) {
                        if (null == encname)
                            doc.print(new OutputStreamWriter(System.out));
                        else {
                            String enc = MIME2Java.convert(encname);
                            if (null == enc)   enc = encname;
                            doc.print(new OutputStreamWriter(System.out, enc), enc);
                        }
                    }
                
                    if (isprintns) {
                        if (null == encname)
                            toNSXMLString(doc, new OutputStreamWriter(System.out), null);
                        else {
                            String enc = MIME2Java.convert(encname);
                            if (null == enc)   enc = encname;
                            toNSXMLString(doc, new OutputStreamWriter(System.out, enc), enc);
                        }
                    }

                    if (isprintclass)
                        printClassTree(doc, new PrintWriter(new OutputStreamWriter(System.out)), 0);
                    if (isprintxpointer)
                        printXPointers(doc, new PrintWriter(new OutputStreamWriter(System.out)), 0);

                } else {
                    long total = 0;
                    s_rt.gc();
                    s_rt.runFinalization();
                    s_rt.gc();
                    for (int j = 0;  j < loop;  j ++) {
                        Stderr se = new Stderr(fname);
                        if (catalog != null) {
                            Reader rr = new FileReader(catalog);
                            se.loadCatalog(rr);
                            rr.close();
                        }
                        se.setPrintWarning(printwarning);
                        Parser pc = new Parser(fname, se, se);
                        long t = System.currentTimeMillis();
                        pc.setProcessExternalDTD(processextdtd);
                        pc.setAllowJavaEncodingName(isallowj);
                        pc.setKeepComment(iskeepcomment);
                        pc.setWarningNoDoctypeDecl(false);
                        pc.setProcessNamespace(donamespace);
                        pc.setWarningNoXMLDecl(warnnoxml);
                        pc.readStream(new FileInputStream(fname));
                        t = System.currentTimeMillis()-t;
                        XJParse.printer.println(t+" [msec]");
                        s_rt.gc();
                        s_rt.runFinalization();
                        s_rt.gc();
                        total += t;
                    }
                    XJParse.printer.println("Total: "+total+" [msec],  Average: "+(total/loop)+" [msec]");
                }
            } catch (IOException e) {
                e.printStackTrace();
            }
            if (s_reportmemory) {
                XJParse.printer.print("Free/total memory after processing:\n\t");
                reportMemory(s_rt);
                XJParse.printer.println("Garbage Collecting ...");
                XJParse.printer.flush();
                s_rt.gc();
                s_rt.runFinalization();
                s_rt.gc();
                XJParse.printer.print("Free/total memory after gc:\n\t");
                reportMemory(s_rt);
                XJParse.printer.flush();
            }
        }
        return 0;
    }

    public static String getLastName(String cl) {
        String ret = cl;
        StringTokenizer st = new StringTokenizer(cl, " .");
        while (st.hasMoreTokens())
            ret = st.nextToken();
        return ret;
    }
    public static void printClassTree(Parent par, PrintWriter pw, int indent) throws IOException {
        for (Child ch = (Child)par.getFirstChild();
             ch != null;  ch = (Child)ch.getNextSibling()) {
            Util.printSpace(pw, indent);
            pw.print(getLastName(ch.getClass().toString()));
            pw.println(" \""+TXText.makePrintable(ch.getText())+"\"");
            if (ch instanceof Parent) {
                printClassTree((Parent)ch, pw, indent+2);
            }
        }
        pw.flush();
    }

    public static void printXPointers(Parent par, PrintWriter pw, int indent) throws IOException {
        for (Child ch = (Child)par.getFirstChild();
             ch != null;  ch = (Child)ch.getNextSibling()) {
            if (!(par instanceof TXDocument) || ch instanceof TXElement) {
                Util.printSpace(pw, indent);
                pw.print(getLastName(ch.getClass().toString()));
                pw.print("\t");
                pw.println(ch.makeXPointer());
                if (ch instanceof Parent) {
                    printXPointers((Parent)ch, pw, indent+2);
                }
            }
        }
        pw.flush();
    }
    
    
    public static void toNSXMLString(TXDocument doc, Writer writer, String encoding)  {
        try {
            FormatPrintVisitor v = new NSFormatPrintVisitor(writer, encoding, 2);
            new NonRecursivePreorderTreeTraversal(v).traverse(doc);
        } catch (Exception e) {
            XJParse.printer.println("Unexpected exception in toNSXMLString: " + e.toString());
        }
    }
    
}
