/*
 * (C) Copyright IBM Corp. 1997-1998  All rights reserved.
 *
 * US Government Users Restricted Rights Use, duplication or
 * disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * IBM will not be liable for any damages suffered by you as a result
 * of using the Program. In no event will IBM be liable for any
 * special, indirect or consequential damages or lost profits even if
 * IBM has been advised of the possibility of their occurrence. IBM
 * will not be liable for any third party claims against you.
 */
package samples.XMLTreeViewer;

import java.awt.*;
import java.awt.event.*;
import java.io.*;
import java.util.*;
import javax.swing.*;
import javax.swing.event.*;
//import javax.swing.plaf.basic.BasicTreeCellRenderer;
import javax.swing.tree.*;

import com.ibm.xml.parser.TXElement;
import com.ibm.xml.parser.TXText;
import com.ibm.xml.parser.TXAttribute;
import com.ibm.xml.parser.Parser;
import com.ibm.xml.parser.Stderr;
import org.w3c.dom.Node;


/**
 * The XMLTreeView is significantly modified from TreeView.
 *
 * XMLTreeView has a Tree view, a Source view and a Message view.
 * Errors are shown in the Tree view on the left in red. When you 
 * click on an error in the Tree view, the Message shows the error
 * text and the XML Source View displays the error line. The various 
 * views can be resized, through the use of JSplitPane.
 *
 * The XMLTreeView is used in the Processing XML Tutorial to 
 * demonstrate the collecting of errors (ErrorFlagger), associating
 * errors with Nodes (ErrorTreeFactory), and associating errors
 * with source lines. (this XMLTreeView#nodeSelected).
 *
 * The XMLTreeCellRenderer is an inner class which enables the 
 * highlighting of errors in the tree and shows the gender values
 * as different icons.
 *
 * @version Revision: 55 1.2 samples/XMLTreeViewer/XMLTreeView.java, xml4jsamples, xml4j-jtcsv, xml4j_1_1_16 
 * @author PFEIFFER Ralf &lt;rpfeiffe@us.ibm.com&gt; (modified from TreeView for error detection)
 */
public class XMLTreeView extends JFrame implements ActionListener, TextListener {

    /**
     *  If argv[0]is present it is considered to be a filename
     */

    static void main(String[] argv) {
        
        JFrame frame = null;
        if (argv.length < 1) {
            // null behaviour is blank screen - eg no JTree, or file dispalyed
            frame = new XMLTreeView();
        } else {
            frame = new XMLTreeView(title+": "+argv[0], argv[0]);
        }
        frame.addWindowListener(new java.awt.event.WindowAdapter() {
         public void windowClosing(java.awt.event.WindowEvent e) {
             System.exit(0);
         }
        });
        frame.setSize(790, 590);
        frame.setVisible(true);
    }

    //
    // Constants
    //

    public static final boolean DEBUG = true;

    static final String title = "XMLTreeViewer";
    static final String openString = "Open";
    static final String quitString = "Quit";
    static final String reloadString = "Reload current XML file";
    static final String expandString = "Expand Tree";
    static final String collapseString = "Collapse Tree";

    //
    // Data
    //

    ErrorFlagger ef;
    String fname;
    JTree m_tree;
    JTextArea sourceText, messageText;
    Vector textLine;
    FileNameInput fni;

    /**
     *  Constructor
     */
    public XMLTreeView() {
        this(title, null);
    }

    /**
     *  Constructor
     */
    public XMLTreeView(String title, String filename) {
        super(title);
        fname = filename;
        JMenuBar jmb = new JMenuBar();
        JMenu fileMenu = new JMenu("File");
        JMenuItem item;

            item = new JMenuItem(openString);
            fileMenu.add(item);
            item.addActionListener(this);

        item = new JMenuItem(quitString);
        fileMenu.add(item);
        item.addActionListener(this);

        JMenu shortcutMenu = new JMenu("Shortcuts");

            item = new JMenuItem(expandString);
            shortcutMenu.add(item);
            item.addActionListener(this);

        item = new JMenuItem(collapseString);
        shortcutMenu.add(item);
        item.addActionListener(this);

        item = new JMenuItem(reloadString);
        shortcutMenu.add(item);
        item.addActionListener(this);

        jmb.add(fileMenu);
        jmb.add(shortcutMenu);
        setJMenuBar(jmb);

        getContentPane().add(createUI(fname));
    }

    /** create and return the entire UI from the root TreeNode
     */
    JComponent createUI(String filename) {
        if (DEBUG) System.out.println("START createUI:"+filename);
 
        // create the message panel first so we can send messages to it...
        messageText = new JTextArea(3,40);
        messageText.setFont(new Font("Helvetica", Font.PLAIN, 18));
        JPanel messagePanel = new JPanel(new BorderLayout());
        messagePanel.add(new JScrollPane(messageText) {
            public Dimension getPreferredSize(){ 
                Dimension size = XMLTreeView.this.getSize();
                return new Dimension(size.width, size.height / 4); 
                }
            public Dimension getMinimumSize(){ 
                return new Dimension(100, 100); 
                }
            }, 
            BorderLayout.CENTER);
        messagePanel.setBorder(BorderFactory.createCompoundBorder(
            BorderFactory.createTitledBorder("Messages"),
            BorderFactory.createEmptyBorder(4, 4, 4, 4)
            ));
       
        // create the TextArea for XML source
        sourceText = new JTextArea();
        sourceText.setFont(new Font("Courier", Font.PLAIN, 18));
        sourceText.setBackground(Color.white);
        sourceText.setForeground(Color.black);
        sourceText.setSelectedTextColor(Color.black);
        sourceText.setSelectionColor(Color.red);
        //sourceText.setOpaque(true);
        sourceText.setEditable(false);
        //sourceText.setEnabled(true);
        JPanel sourcePanel = new JPanel(new BorderLayout());
        sourcePanel.add(new JScrollPane(sourceText){
            public Dimension getPreferredSize(){ 
                Dimension size = XMLTreeView.this.getSize();
                return new Dimension(size.width / 2, size.height * 3 / 5); 
                }
            public Dimension getMinimumSize(){ 
                return new Dimension(100, 100); 
                }
            }, 
            BorderLayout.CENTER);
        sourcePanel.setBorder(BorderFactory.createCompoundBorder(
            BorderFactory.createTitledBorder("Source View"),
            BorderFactory.createEmptyBorder(4, 4, 4, 4)
            ));
       
        // create the JTree and scroll pane.
        JPanel treePanel = new JPanel(new BorderLayout());
        treePanel.add(new JScrollPane(m_tree = createJTree()) {
            public Dimension getPreferredSize(){ 
                Dimension size = XMLTreeView.this.getSize();
                return new Dimension(size.width / 2, size.height * 3 / 5); 
                }
            public Dimension getMinimumSize(){ 
                return new Dimension(100, 100); 
                }
            }, 
            BorderLayout.CENTER);

        treePanel.setBorder(BorderFactory.createCompoundBorder(
            BorderFactory.createTitledBorder("Tree View"),
            BorderFactory.createEmptyBorder(4, 4, 4, 4)
            ));
          
        // refreshUI loads everthything!    
        refreshUI(filename);
        
        // use the new JSplitPane to dynamically resize...
        JComponent split = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT, 
            true, treePanel, sourcePanel);

        JComponent mainSplitPane = 
            new JSplitPane(JSplitPane.VERTICAL_SPLIT, 
                           true, split, messagePanel);

        if (DEBUG) System.out.println("END createUI:"+filename);
        return mainSplitPane;
    }

    /** refreshUI is called when we have a new filename to parse.
     */
    void refreshUI(String filename) {
        if (DEBUG) System.out.println("START refreshUI:"+filename);
 
        messageText.selectAll();
        messageText.cut();
        
        if (filename == null || filename.equals("")) {
            messageText.setForeground(Color.red);
            messageText.append("No input XML filename specified:"+filename+"\n");
            return;           
        }
        
        fname = filename;
        // new JTree Model
        TreeNode newRoot = getRoot(filename);
        if (newRoot == null) {
            messageText.setForeground(Color.red);
            messageText.append("Unable to get new Tree for:"+filename+"\n");
            return;           
        }
        m_tree.setModel(new DefaultTreeModel(newRoot));
        
        // new Source 
        sourceText.selectAll();
        sourceText.cut();
        readXMLFile(fname, sourceText);
        sourceText.select(0, 0 );

        setTitle(title+": "+filename);
 
        if (m_tree!= null)
            expandTree();
            
        if (ef != null && ef.getErrorNodes()!=null && ef.getErrorNodes().size()>0) {
            messageText.setForeground(Color.red);
            messageText.append("XML source, "+fname+" has errors.\n");
            messageText.append("Please click on red Tree View items for details.\n");
        }
        if (DEBUG) System.out.println("END refreshUI:"+filename);
    }

    /**
     *  Invoke the Parser on fname and return the root TreeNode.
     */
    public TreeNode getRoot(String filename) {
        if (DEBUG) System.out.println("START getRoot:"+filename);
        
        if (filename == null || filename.equals(""))
        return null;
        
        try {
            ErrorTreeFactory etf = new ErrorTreeFactory();
            ef = new ErrorFlagger(etf);
            Parser p = new Parser(fname, ef, null);
            p.setElementFactory(etf);
            p.setEndBy1stError(false);
            return (TreeNode)p.readStream(
                    p.getInputStream(fname, null, Stderr.file2URL(fname).toString()));
        } catch (Exception e) {
            e.printStackTrace();
        }
        return null;
    }
   
    /** read the xml file from filename and append it to the JTextArea
     */
    synchronized void readXMLFile(String filename, JTextArea ta) {

        if (DEBUG) System.out.println("START readXMLFile"+filename);
        if (filename == null || filename.equals(""))
            return;
        File file = new File(filename);
        FileInputStream fis = null;
        BufferedReader dis = null;

        try {
            fis = new FileInputStream(file);
            /**
             * By this way, we can not read XML documents in encodings other than
             * default encoding of our platform.
             */
            dis = new BufferedReader(new InputStreamReader(fis));
        } catch (FileNotFoundException fnf) {
            System.err.println("ERROR: XML4J.readXMLFile: "+fnf);
            return;
        }

        String line;
        int i = 0;

        int len = 0;
        textLine = new Vector();
        String nl = "\n";
        int nllen = nl.length();
        StringBuffer sb = new StringBuffer();

        try{
            while ((line = dis.readLine()) != null) {
                sb.append(line+nl);
                //ta.append(line+nl);
                textLine.addElement(new Integer(len));
                len += line.length()+nllen;
            }
            textLine.addElement(new Integer(len));
        } catch (IOException io) {
            System.err.println(io);
            return;
        }
        
        ta.append(sb.toString());
        
        // relayout because contents have changed
        //ta.revalidate();

        if (DEBUG) System.out.println("END readXMLFile"+filename);
        return;

    }

    /** create a JTree from the root TreeNode and return it.
     */
    JTree createJTree() {
        JTree tree = new JTree();
        if (tree == null)
            return null; // should never happen.

        tree.setCellRenderer(new XMLTreeCellRenderer());
        tree.getSelectionModel().setSelectionMode
            (TreeSelectionModel.SINGLE_TREE_SELECTION);

        // Listen for when the selection changes, call nodeSelected(node)
        tree.addTreeSelectionListener(
            new TreeSelectionListener() {
                public void valueChanged(TreeSelectionEvent e) {
                    TreeNode node = (TreeNode)
                        (e.getPath().getLastPathComponent());

                    nodeSelected(node);
                }
            }
        );
        tree.setRowHeight(24);
        tree.setFont(new Font("Helvetica", Font.PLAIN, 18));
        return tree;

    }

    /** called when our JTree's nodes are selected.
     */
    void nodeSelected(TreeNode node) {

        StringBuffer sb = new StringBuffer();
        messageText.selectAll();
        messageText.cut();

        Object errorObject = ef.getError((Node)node);
        if (errorObject != null) {
            // There *is* an error in this node.
            messageText.setForeground(Color.red);
            ErrorInParsing eip = (ErrorInParsing)errorObject;
            sb.append("Error: "+eip.getMsg()+"\n");
            int lineNo = eip.getLineNo();
            int pos = ((Integer)textLine.elementAt(lineNo-1)).intValue();
            int next = (((Integer)textLine.elementAt(lineNo)).intValue());
            sourceText.select(pos, next );
            //m_textScrollPane.repaint();
        } else {
            messageText.setForeground(Color.black);
            sourceText.select(0, 0 );
        }
        
        if (node instanceof TXElement
            || node instanceof TXText) {
            sb.append(node.toString());
        }
        
        messageText.append(sb.toString());
    }

    /** called when a the text value has changed in the FileNameInput.
     *  read in new XML file.
     */
    public void textValueChanged(TextEvent e) {
        try {
            if (fni != null)
                fni.setVisible(false);
            fname = ((JTextField)e.getSource()).getText();
            if (DEBUG) System.out.println("textValueChanged:"+fname);
            refreshUI(fname);

        } catch (Exception ex) {
            ex.printStackTrace();
        }
    }

    /** called to handle menu actions.
     */
    public void actionPerformed(java.awt.event.ActionEvent e) {
        if (DEBUG) System.err.println("ACTION: "+e.getActionCommand()+", "+e.paramString());
  
        if (e.getActionCommand().equals(quitString)) {
          System.exit(0);
        }
        else if (e.getActionCommand().equals(openString)) {

            fni = new FileNameInput("Open File");
            fni.addTextListener(this);
            fni.setVisible(true);
        }
        else if (e.getActionCommand().equals(expandString)) {
            expandTree();
        }
        else if (e.getActionCommand().equals(collapseString)) {
            int rows = m_tree.getRowCount();
            for (int i = 0; i < rows; i++) {
                m_tree.collapseRow(i);
            }
        }
        else if (e.getActionCommand().equals(reloadString)) {
            refreshUI(fname);
        }
    }

    void expandTree() {
        int rows = 0;
        for (int levels=0; levels <= 4; levels++) {
            rows=m_tree.getRowCount();
            for (int i = 0; i < rows; i++) {
                m_tree.expandRow(i);
            }
        }
    }

    /*
     * The XMLTreeCellRenderer is an inner class which enables the 
     * highlighting of errors in the tree and shows the gender values
     * as different icons.
     */
    class XMLTreeCellRenderer extends DefaultTreeCellRenderer
    {
        
        public Component getTreeCellRendererComponent(JTree tree, Object value,
                                          boolean selected, boolean expanded,
                                          boolean leaf, int row,
                                                  boolean hasFocus)
        {
            Component comp = super.getTreeCellRendererComponent(tree, value,
                                           selected,  expanded, leaf,  row, hasFocus);
            if (selected) {
                comp.setBackground(Color.blue);
            }
            if (value != null && value instanceof Node
            && ef.getErrorNodes().containsKey( (Node)value )) {  
                comp.setForeground(Color.red);
            }

            if (value != null && value instanceof TXElement) {
                TXElement txNode = (TXElement)value;
                Enumeration en = txNode.attributeElements();
                while (en.hasMoreElements()) {
                    TXAttribute txAtt = (TXAttribute)en.nextElement();
                    if (txAtt.getName().equals("gender")) {
                        if (txAtt.getValue().equals("male")) {
                            setIcon(new ImageIcon("male.gif"));
                        } else
                        if (txAtt.getValue().equals("female")) {
                            setIcon(new ImageIcon("female.gif"));
                        } 
                        break;
                    }
                }
            }

            return comp;
        }
    }
    
    /*
     * The FileNameInput is an inner class which allows the user
     * to enter a filename. It exists due to a Swing bug which
     * has problems with the real file input panel.
     */
    class FileNameInput extends JFrame implements ActionListener {


        JLabel fileLabel;
        JTextField textField;
        JButton ok;
        JButton cancel;
        Vector textListeners;

        public FileNameInput() {
            this("");
        }

        public FileNameInput(String title) {

            super(title);

            fileLabel = new JLabel("Enter XML file name:");
            textField = new JTextField();
            textField.addActionListener(this);
            ok = new JButton("ok");
            cancel = new JButton("cancel");
            JPanel buttonPanel = new JPanel();
            buttonPanel.add(ok);
            buttonPanel.add(cancel);
            ok.addActionListener(this);
            cancel.addActionListener(this);
            getContentPane().add(fileLabel, BorderLayout.NORTH);
            getContentPane().add(textField, BorderLayout.CENTER);
            getContentPane().add(buttonPanel, BorderLayout.SOUTH);
            setSize(400,100);
        }

        public void actionPerformed(ActionEvent e) {

            if (e.getSource() == ok || e.getSource() == textField) {
                System.out.println("FileNameInput: pressed OK");
                    TextEvent event = new TextEvent(textField, TextEvent.TEXT_VALUE_CHANGED);
                    deliverEvent(event);
                    setVisible(false);
            } else
            if (e.getSource() == cancel) {
                System.out.println("FileNameInput: pressed cancel");
                    setVisible(false);
            }
        }

        /**
         * Adds a TextListener event listener.
         *
         * @param listener  The listener to add.
         *
         * @see #removeTextListener
         */
        public void addTextListener(TextListener listener) {

            // is there anything to do?
            if (listener == null)
                return;

            if (textListeners == null)
               textListeners = new Vector();

            // add listener
            textListeners.addElement(listener);
            }

        /**
         * Removes a TextListener event listener.
         *
         * @param listener  The listener to remove.
         *
         * @see #addTextListener
         */
        public void removeTextListener(TextListener listener) {

            // is there anything to do?
            if (listener == null || textListeners == null)
                return;

            // add listener
            textListeners.removeElement(listener);
            }


        /**
         * This function delivers TextListener events, when the ok
         * button is clicked.
         *
         * @param evt The event to deliver.
         */
        protected void deliverEvent(EventObject evt) {

            if (evt instanceof TextEvent) {
                TextEvent event = (TextEvent)evt;

                Vector l;
                synchronized (textListeners) { l = (Vector)textListeners.clone(); }

                int size = l.size();
                for (int i = 0; i < size; i++)
                    ((TextListener)l.elementAt(i)).textValueChanged(event);
                }
            }
    }
    
}
