/*
 * (C) Copyright IBM Corp. 1997-1998  All rights reserved.
 *
 * US Government Users Restricted Rights Use, duplication or
 * disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * IBM will not be liable for any damages suffered by you as a result
 * of using the Program. In no event will IBM be liable for any
 * special, indirect or consequential damages or lost profits even if
 * IBM has been advised of the possibility of their occurrence. IBM
 * will not be liable for any third party claims against you.
 */

package com.ibm.xml.parser;

import java.util.Enumeration;
import java.util.Vector;
import org.w3c.dom.Node;

/**
 * <P>This class provides support for a single attribute object that is declared as part of an
 * Element's attribute list declaration in the DTD.  Refer to Attlist for a detailed description.</P>
 *
 * @version Revision: 86 1.10 src/com/ibm/xml/parser/AttDef.java, xml4jsrc, xml4j-jtcsv, xml4j_1_1_16 
 * @author TAMURA Kent &lt;kent@trl.ibm.co.jp&gt;
 * @see com.ibm.xml.parser.Attlist
 * @see com.ibm.xml.parser.Child
 */
public class AttDef extends Child {
    static final long serialVersionUID = 3041229546206616963L;
    // DeclaredValueType
    public static final int      CDATA                = 1;
    public static final int      ID                   = 2;
    public static final int      IDREF                = 3;
    public static final int      IDREFS               = 4;
    public static final int      ENTITY               = 5;
    public static final int      ENTITIES             = 6;
    public static final int      NMTOKEN              = 7;
    public static final int      NMTOKENS             = 8;
    public static final int      NOTATION             = 9;
    public static final int      NAME_TOKEN_GROUP     = 10;
    public static final String[] S_TYPESTR   = {
        "*UNKNOWN*",
        "CDATA",
        "ID",
        "IDREF",
        "IDREFS",
        "ENTITY",
        "ENTITIES",
        "NMTOKEN",
        "NMTOKENS",
        "NOTATION",
        "ENUMERATION",
    };

    // DefaultValueType
    public static final int         FIXED       = 1;
    public static final int         REQUIRED    = 2;
    public static final int         IMPLIED     = 3;
    public static final int         NOFIXED     = -1;
    public static final int         UNKNOWN     = 0;
    
    String      name                =   null;       
    int         declaredValueType   =   UNKNOWN; // Must be DeclaredValueType
    String      value               =   null;
    Vector      tokens              =   null;   // Only used for enumerated atttribute types:
                                                //   <var>NOTATION</var> and <var>NAME_TOKEN_GROUP</var>.
    int         defaultValueType    =   NOFIXED;// Must be DefaultValueType
            
    /**
     * Constructor.
     * @param   name        Name of this attribute as defined by the DTD.
     */
    public AttDef(String name) {
        this.name = name;
    }

    /**
     * Clone this attribute definition using the appropriate factory.  
     * <p>This method is defined by Child.
     * @return          Cloned attribute definition.
     * @see com.ibm.xml.parser.Child#clone
     */
    public synchronized Object clone() {
        checkFactory();
        AttDef ad = factory.createAttDef(this.name);
        ad.setFactory(getFactory());
        ad.value = this.value;
        ad.defaultValueType = this.defaultValueType;
        ad.declaredValueType = this.declaredValueType;
        if (null != this.tokens)
            ad.tokens = (Vector)this.tokens.clone();
        return ad;
    }

    /**
     *
     * @param deep ignored.
     */
    public synchronized boolean equals(Node arg, boolean deep) {
        if (arg.getNodeType() != Child.ATTDEF)  return false;
        AttDef ad = (AttDef)arg;
        if (!ad.getName().equals(this.getName()))  return false;
        if (ad.getDefaultType() != this.defaultValueType)  return false;
        if (ad.getDeclaredType() != this.declaredValueType)  return false;
        if (!((ad.getDefaultStringValue() == null && this.value == null)
              || (this.value != null && this.value.equals(ad.getDefaultStringValue()))))
            return false;
        if (ad.size() == 0 && this.size() == 0)  return true;
        if (ad.size() != this.size())  return false;
        for (int i = 0;  i < ad.size();  i ++) {
            String elem = ad.elementAt(i);
            if (!elem.equals(this.elementAt(i)))
                return false;
        }
        return true;
    }

    /**
     * Returns that this object is an attribute definition Node. 
     * <p>This method is defined by DOM.
     * @return          Attribute definition Node indicator.
     */
    public short getNodeType() {
        return Child.ATTDEF;
    }

    /**
     *
     */
    public String getNodeName() {
        return Child.NAME_ATTDEF;
    }

    /**
     * Returns the name of this attribute in the DTD definition.  
     * @return          Name of this attribute in the DTD definition.
     * @see #setName
     */
    public String getName() {
        return this.name;
    }
    
    /**
     * Sets the name of this attribute in the DTD definition.
     * @param   name    Name of this attribute in the DTD definition.
     * @see #getName
     */
    public void setName(String name) {
        this.name = name;
    }

    /**
     * Returns the declared type of this attribute in the DTD definition.
     * @return          Declared type of this attribute in the DTD definition.
     * @see #setDeclaredType
     */
    public int getDeclaredType() {
        return this.declaredValueType;
    }
    
    /**
     * Sets the declared type of this attribute in the DTD definition.
     * @param declaredValueType Declared type of this attribute in the DTD definition.
     * @see #getDeclaredType
     */
    public void setDeclaredType(int declaredValueType) {
        this.declaredValueType = declaredValueType;
    }

    /**
     * Returns the default value of this attribute in the DTD definition.  
     * This is useful if this attribute was not given an explicit value in the document 
     * instance.  The returned value is only meaningful for attribute types defined as
     * <var>FIXED</var> or <var>NOFIXED</var>.
     * @return          The default value of the attribute, or <var>null</var> if none specified.
     * @see #setDefaultStringValue
     */
    public String getDefaultStringValue() {
        return this.value;
    }

    /**
     * Sets the default value of this attribute in the DTD definition.  
     * This is useful if this attribute was not given an explicit value in the document 
     * instance.  The specified <var>value</var> is only meaningful for attribute types defined as
     * <var>FIXED</var> or <var>NOFIXED</var>.
     * @param   value   The default value of the attribute in the DTD definition.      
     * @see #getDefaultStringValue
     */
    public void setDefaultStringValue(String value) {
        this.value = TXAttribute.normalize(getDeclaredType(), value);
    }

    /**
     * Adds the specified <var>token</var> to the end of the list of tokens that can be used as values 
     * for this enumerated attribute type; the specified <var>token</var> has no meaning for 
     * other attribute types. 
     * Enumerated attribute types are: <var>NOTATION</var> and <var>NAME_TOKEN_GROUP</var>.
     * @param token     Value to add to the list of valid tokens for this enumerated attribute.
     * @return          =true if the token does not already exist in this token list; otherwise, =false.
     * @see #elementAt
     * @see #contains
     * @see #size
     * @see #elements
     */
    public boolean addElement(String token) {
        if (null == this.tokens)  this.tokens = new Vector();
        if (0 <= this.tokens.indexOf(token))  return false;
        this.tokens.addElement(token);
        return true;
    }
    
    /**
     * Returns the token at the specified <var>index</var> in the list of tokens that can be used as values 
     * for this enumerated attribute type; the returned token has no meaning for other attribute types. 
     * Enumerated attribute types are: <var>NOTATION</var> and <var>NAME_TOKEN_GROUP</var>.
     * @param index     Index into the list of tokens that can be used as values
     *                  for this enumerated attribute type.
     * @return          Token at the specified index, or <var>null</var> if an invalid index.           
     * @see #addElement
     * @see #contains
     * @see #size
     * @see #elements
     */
    public String elementAt(int index) {
        return null == this.tokens ? null : (String)this.tokens.elementAt(index);
    }
    
    /**
     * Returns whether the specified <var>token</var> is in the list of tokens that can be 
     * used as values for this enumerated attribute type; the returned value has no meaning 
     * for other attribute types. 
     * Enumerated attribute types are: <var>NOTATION</var> and <var>NAME_TOKEN_GROUP</var>.
     * @param token     Value to check in the list of valid tokens for this enumerated attribute.
     * @return          =true if <var>token</var> is defined; otherwise, =false.           
     * @see #addElement
     * @see #elementAt
     * @see #size
     * @see #elements
     */
    public boolean contains(String token) {
        return null == this.tokens ? false : 0 <= this.tokens.indexOf(token);
    }
    
    /**
     * Returns the number of tokens in the list of tokens that can be used as values for 
     * this enumerated attribute type; the returned value has no meaning for other attribute types. 
     * Enumerated attribute types are: <var>NOTATION</var> and <var>NAME_TOKEN_GROUP</var>.
     * @return          Number of tokens, or <var>null</var> if no tokens defined.           
     * @see #addElement
     * @see #elementAt
     * @see #contains
     * @see #elements
     */
    public int size() {
        return null == this.tokens ? 0 : this.tokens.size();
    }

    /**
     * Returns a list of all tokens that can be used as values for this enumerated attribute type;
     * the returned value has no meaning for other attribute types.
     * Enumerated attribute types are: <var>NOTATION</var> and <var>NAME_TOKEN_GROUP</var>.
     * @return          The list of allowed tokens for the attribute value as defined by
     *                  the DTD definition, or <var>null</var> if none specified.
     * @see #addElement
     * @see #elementAt
     * @see #size
     * @see #contains
     */
    public Enumeration elements() {
        return null == this.tokens ? null : this.tokens.elements();
    }
    
    /**
     * Returns the default type of this attribute's value in the DTD definition.  
     * @return          The default type of this attribute in the DTD definition.
     * @see #setDefaultType
     */
    public int getDefaultType() {
        return this.defaultValueType;
    }
    
    /**
     * Sets the default type of this attribute's value in the DTD definition.  
     * @param   defaultValueType    The default type of this attribute in the DTD definition.
     * @see #getDefaultType
     */
    public void setDefaultType(int defaultValueType) {
        this.defaultValueType = defaultValueType;
    }
    
    /**
     * Implements the accept operation of the visitor design pattern when the start of
     * an AttDef Node is recognized when traversing the document object tree. 
     * @param   visitor The implemention of the Visitor operation (toXMLString, digest, ...)
     * @exception Exception Thrown if this Node can not be visitted, or traversal modification is requested.
     * @see com.ibm.xml.parser.Visitor
     * @see com.ibm.xml.parser.TreeTraversal
     * @see com.ibm.xml.parser.NonRecursivePreorderTreeTraversal
     * @see com.ibm.xml.parser.TreeTraversalException
     */
    public void acceptPre(Visitor visitor) throws Exception {
        visitor.visitAttDefPre(this);
    }

    /**
     * Implements the accept operation of the visitor design pattern when the end of
     * an AttDef Node is recognized when traversing the document object tree. 
     * @param   visitor The implemention of the Visitor operation (toXMLString, digest, ...)
     * @exception Exception Thrown if this Node can not be visitted, or traversal modification is requested.
     * @see com.ibm.xml.parser.Visitor
     * @see com.ibm.xml.parser.TreeTraversal
     * @see com.ibm.xml.parser.NonRecursivePreorderTreeTraversal
     * @see com.ibm.xml.parser.TreeTraversalException
     */
    public void acceptPost(Visitor visitor) throws Exception {
        visitor.visitAttDefPost(this);
    }
}
