/*
 * (C) Copyright IBM Corp. 1997-1998  All rights reserved.
 *
 * US Government Users Restricted Rights Use, duplication or
 * disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * IBM will not be liable for any damages suffered by you as a result
 * of using the Program. In no event will IBM be liable for any
 * special, indirect or consequential damages or lost profits even if
 * IBM has been advised of the possibility of their occurrence. IBM
 * will not be liable for any third party claims against you.
 */

package com.ibm.xml.parser;

import java.io.IOException;
import java.io.Serializable;
import java.io.Writer;
import org.w3c.dom.DOMException;
import org.w3c.dom.Document;
import org.w3c.dom.EntityReference;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * The Child class implements the Node interface as defined by the Document
 * Object Model (DOM).
 * <p>
 * Node is the base type of most objects in the Document Object Model. It 
 * may have an arbitrary number (including zero) of sequentially ordered
 * Child Nodes. It usually has a parent Node; the exception being that the
 * root Node in a document hierarchy has no parent.
 * <p>
 * The child node has an added property, <i>userData</i>, that allows
 * the programmer to attach application specific data to any node in the
 * document tree that extends Child.
 *
 * @version Revision: 88 1.13 src/com/ibm/xml/parser/Child.java, xml4jsrc, xml4j-jtcsv, xml4j_1_1_16 
 *
 * @author TAMURA Kent &lt;kent@trl.ibm.co.jp&gt;
 * @author Andy Clark, IBM
 *
 * @see org.w3c.dom.Node
 * @see com.ibm.xml.parser.TXElement
 * @see com.ibm.xml.parser.TXText
 * @see com.ibm.xml.parser.TXComment
 * @see com.ibm.xml.parser.TXPI
 * @see com.ibm.xml.parser.TXCDATASection
 * @see com.ibm.xml.parser.TXNotation
 * @see com.ibm.xml.parser.TXAttribute
 * @see com.ibm.xml.parser.Parent
 */
public abstract class Child implements Node, Cloneable, Serializable, Visitee {

    static final long serialVersionUID = 6737260867707773565L;
    public static final int ELEMENT_DECL        = 20;
    public static final int ATTLIST             = 21;
    public static final int ATTDEF              = 22;
    public static final int PSEUDONODE          = 23;

                                                // Defined in DOM spec.
    public static final String NAME_DOCUMENT    = "#document";
    public static final String NAME_COMMENT     = "#comment";
    public static final String NAME_TEXT        = "#text";
    public static final String NAME_CDATA       = "#cdata-section";
    public static final String NAME_DOCFRAGMENT = "#document-fragment";
                                                // Not defined in DOM.
    public static final String NAME_ATTDEF      = "#attribute-definition";
    public static final String NAME_ATTLIST     = "#attribute-definition-list";
    public static final String NAME_ELEMENT_DECL= "#element-declaration";
    public static final String NAME_PSEUDONODE  = "#pseudo-node";
    
            Node             parent      = null;
            Node             prevSibling = null;
            Node             nextSibling = null;
            transient byte[] digest      = null;
            TXDocument       factory     = null;

    /** Application specific user data. */
    private Object userData;

    /**
     * Clone this object.
     * @return          Copy of this object.
     */
    abstract public Object clone();

    /**
     * <p>This method is defined by DOM.
     *
     * @param deep ignored.
     */
    public Node cloneNode(boolean deep) {
        return (Node)clone();
    }

    /**
     *
     */
    abstract public boolean equals(Node node, boolean deep);

    /**
     * <p>This method is defined by DOM.
     *
     */
    public String getNodeValue() {
        return null;
    }

    /**
     * <p>This method is defined by DOM.
     *
     */
    public void setNodeValue(String arg) {
        throw new TXDOMException(DOMException.NO_MODIFICATION_ALLOWED_ERR, "setNodeValue(String) isn't supported in this class.");
    }

    /**
     * Returns the parent of the given Node instance. If this Node is the root of the document object tree,
     * or if the Node has not been added to a document tree, <var>null</var> is returned.
     * <p>This method is defined by DOM.
     * @return          Parent Node associated with this object, or <var>null</var> if no parent.
     */
    public Node getParentNode() {
        return this.parent;
    }

    /**
     *
     */
    public Node getParentWithoutReference() {
        Node parent = getParentNode();
        while (parent != null && parent.getNodeType() == Node.ENTITY_REFERENCE_NODE)
            parent = parent.getParentNode();
        return parent;
    }

    /**
     * Returns an empty NodeList object because Child Nodes are leaf Nodes.  
     * If the Child class is extended by non-leaf Nodes, this method should be overridden.
     * <p>This method is defined by DOM.
     *
     * @return          Always returns an empty NodeList.
     * @see com.ibm.xml.parser.Parent#getChildNodes
     */
    public NodeList getChildNodes() {
        return TXNodeList.emptyNodeList;
    }

    /**
     * Returns <var>false</var> because Child Nodes are leaf Nodes.  
     * If the Child class is extended by non-leaf Nodes, this method should be overridden.
     * <p>This method is defined by DOM.
     *
     * @return          Always returns <var>false</var>.
     * @see com.ibm.xml.parser.Parent#hasChildNodes
     */
    public boolean hasChildNodes() {
        return false;
    }

    /**
     * Returns <var>null</var> because Child Nodes are leaf Nodes.
     * If the Child class is extended by non-leaf Nodes, this method should be overridden.
     * <p>This method is defined by DOM.
     *
     * @return          Always returns <var>null</var>.
     * @see com.ibm.xml.parser.Parent#getFirstChild
     */
    public Node getFirstChild() {
        return null;
    }

    /**
     * Return <VAR>null</vAR>
     *
     * @return          Always returns <var>null</var>.
     * @see com.ibm.xml.parser.Parent#getFirstWithoutReference
     */
    public Node getFirstWithoutReference() {
        return null;
    }

    /**
     * Returns <var>null</var> because Child Nodes are leaf Nodes.
     * If the Child class is extended by non-leaf Nodes, this method should be overridden.
     * <p>This method is defined by DOM.
     *
     * @return          Always returns <var>null</var>.
     * @see com.ibm.xml.parser.Parent#getLastChild
     */
    public Node getLastChild() {
        return null;
    }

    /**
     * Return <VAR>null</vAR>
     *
     * @return          Always returns <var>null</var>.
     * @see com.ibm.xml.parser.Parent#getLastWithoutReference
     */
    public Node getLastWithoutReference() {
        return null;
    }

    /**
     * Returns the Node immediately preceding this Node in a breadth-first traversal of the tree. If
     * there is no such Node, <var>null</var> is returned.
     * <p>This method is defined by DOM.
     *
     * @return          The Child Node immediately preceding this object in the Parent's Child list,
     *                    or <var>null</var> if this object is the first sibling.
     * @see #getNextSibling
     */
    public Node getPreviousSibling() {
        return this.prevSibling;
    }

    /*
     * Returns the Node preceding this Node. This method ignore GeneralReference instances.
     * If there is no such Node, <var>null</var> is returned.
     *
     * @return          The child Node preceding this object,
     *                    or <var>null</var> if this object is the last sibling.
     * @see #getNextWithoutReference
     * @see #getPreviousSibling
     */
    public Node getPreviousWithoutReference() {
        Node ret = this.getPreviousSibling();
        if (ret == null) {
            Node parent = this.getParentNode();
            if (parent == null)   return null;
            if (parent.getNodeType() == Node.ENTITY_REFERENCE_NODE)
                ret = ((Child)parent).getPreviousWithoutReference();
        } else {                                // not null
            while (ret.getNodeType() == Node.ENTITY_REFERENCE_NODE) {
                Node last = ret.getLastChild();
                if (last == null) {
                    ret = ((Child)ret).getPreviousWithoutReference();
                    break;
                } else
                    ret = last;
            }
        }
        return ret;
    }

    /**
     * Returns the Node immediately following this Node in a breadth-first traversal of the tree. If
     * there is no such Node, <var>null</var> is returned.
     * <p>This method is defined by DOM.
     *
     * @return          The Child Node immediately following this object in the parent's Child list,
     *                    or <var>null</var> if this object is the last sibling.
     * @see #getPreviousSibling
     */
    public Node getNextSibling() {
        return this.nextSibling;
    }

    /*
     * Returns the Node following this Node. This method ignore GeneralReference instances.
     * If there is no such Node, <var>null</var> is returned.
     *
     * @return          The child Node following this object,
     *                    or <var>null</var> if this object is the last sibling.
     * @see #getPreviousWithoutReference
     * @see #getNextSibling
     */
    public Node getNextWithoutReference() {
        Node ret = this.getNextSibling();
        if (ret == null) {
            Node parent = this.getParentNode();
            if (parent == null)   return null;
            if (parent.getNodeType() == Node.ENTITY_REFERENCE_NODE)
                ret = ((Child)parent).getNextWithoutReference();
        } else {                                // not null
            while (ret.getNodeType() == Node.ENTITY_REFERENCE_NODE) {
                Node first = ret.getFirstChild();
                if (first == null) {
                    ret = ((Child)ret).getNextWithoutReference();
                    break;
                } else
                    ret = first;
            }
        }
        return ret;
    }

    /**
     * Returns <VAR>null</VAR>.  
     * <p>This method is defined by DOM.
     * @see com.ibm.xml.parser.TXElement#getAttributes
     */
    public NamedNodeMap getAttributes() {
        return null;
    }

    /**
     * Throws a <var>DOMException</var> because Child Nodes are leaf Nodes.
     * If the Child class is extended by non-leaf Nodes, this method should be overridden.
     * <p>This method is defined by DOM.
     *
     * @param   newChild    Not used.
     * @param   refChild    Not used.
     * @return              Nothing is ever returned (an exception is thrown).
     * @exception org.w3c.dom.DOMException Thrown to indicate that no children exist in leaf Nodes.
     * @see com.ibm.xml.parser.Parent#insertBefore
     */
    public Node insertBefore(Node newChild, Node refChild) throws DOMException {
        throw new TXDOMException(DOMException.HIERARCHY_REQUEST_ERR,
                                 "com.ibm.xml.parser.Child#insertBefore(): Can't insert any nodes to this.");
    }

    /**
     * Throws a <var>DOMException</var> because Child Nodes are leaf Nodes.
     * If the Child class is extended by non-leaf Nodes, this method should be overridden.
     * <p>This method is defined by DOM.
     *
     * @param   oldChild    Not used.
     * @param   newChild    Not used.
     * @return              Nothing is ever returned (an exception is thrown).
     * @exception DOMException Thrown to indicate that no children exist in leaf Nodes.
     * @see com.ibm.xml.parser.Parent#replaceChild
     */
    public Node replaceChild(Node oldChild, Node newChild) throws DOMException {
        throw new TXDOMException(DOMException.HIERARCHY_REQUEST_ERR,
                               "com.ibm.xml.parser.Child#replaceChild(): Can't insert any nodes to this.");
    }

    /**
     * Throws a <var>DOMException</var> because Child Nodes are leaf Nodes.
     * If the Child class is extended by non-leaf Nodes, this method should be overridden.
     * <p>This method is defined by DOM.
     *
     * @param   oldChild    Not used.
     * @return              Nothing is ever returned (an exception is thrown).
     * @exception org.w3c.dom.DOMException Thrown to indicate that no children exist in leaf Nodes.
     * @see com.ibm.xml.parser.Parent#removeChild
     */
    public Node removeChild(Node oldChild) throws DOMException {
        throw new TXDOMException(DOMException.NOT_FOUND_ERR,
                               "com.ibm.xml.parser.Child#removeChild(): Can't insert any nodes to this.");
    }

    /**
     * Throws a <var>DOMException</var> because Child Nodes are leaf Nodes.
     * If the Child class is extended by non-leaf Nodes, this method should be overridden.
     * <p>This method is defined by DOM.
     *
     * @see com.ibm.xml.parser.Parent#removeChild
     */
    public Node appendChild(Node newChild) throws DOMException {
        throw new TXDOMException(DOMException.HIERARCHY_REQUEST_ERR,
                               "com.ibm.xml.parser.Child#appendChild(): Can't insert any nodes to this.");
    }

    /**
     * Returns all text associated with this Node without considering entities.                                           
     * This method is intended to be overridden by DOM-defined Node types.
     * @return          Always returns <var>""</var>.
     * @see com.ibm.xml.parser.TXText#getText
     * @see #toXMLString
     */
    public String getText() {
        return "";
    }

    /**
     * Returns the first <var>TXElement</var> match through all ancestors.
     * Searching is done recursively, not just for the immediate Parent Node.
     * The specified qualified name refers to the Element tag name (see Namespace for details).
     * @param   qName   Qualified name to match against in all ancestors.
     * @return          The first matching <var>TXElement</var> Node, or <var>null</var> if no matches.
     * @see com.ibm.xml.parser.Namespace
     */
    public TXElement searchAncestors(String qName) {
        return searchAncestors(Match.QNAME, null, qName);
    }

    /**
     * Returns the first <var>TXElement</var> match through all ancestors.
     * All specified input parameters must match for an Element to be judged as matched (see Namespace for details).
     * Various namespace matching algorithms are supported: qualified name, local name & URI, or URI.
     * Searching is done recursively, not just for the immediate Parent Node.
     * @param   matchType           Namespace match type: <code>Match.QNAME, Match.NSLOCAL, Match.NS</code>
     * @param   uri                 When matching a URI, specify the value to match; otherwise, specify <var>null</var>.
     * @param   qNameOrLocalName    When matching a qualified name or local name, specify the value to match; otherwise, specify <var>null</var>.
     * @return                      The first matching <var>TXElement</var> Node, or <var>null</var> if no matches.
     * @see com.ibm.xml.parser.Namespace
     */
    public TXElement searchAncestors(int matchType, String uri, String qNameOrLocalName) {
        // This method is not used by the parser.
        Node parent = this;
        while (null != (parent = parent.getParentNode())) {
            if (!(parent instanceof TXElement))  return null;
            if (Match.matchName((Namespace)parent, matchType, uri, qNameOrLocalName))
                return (TXElement)parent;
        }
        return null;
    }

    /**
     * Returns this Node and any children in XML format using the specified character 
     * <var>encoding</var>.
     * @param writer        The character output stream to use.
     * @param encoding      Java character encoding in use by <VAR>writer</VAR>.
     * @see #getText
     * @exception IOException       Thrown if a Node can not be visitted because of an invalid <var>witer</var>.
     * @exception LibraryException  Thrown if the document object hierarchy is in an unknown state.
     */
    public void toXMLString(Writer writer, String encoding) throws IOException, LibraryException {
        try {
            ToXMLStringVisitor toXMLStringVisitor = new ToXMLStringVisitor(writer, encoding);
            new NonRecursivePreorderTreeTraversal(toXMLStringVisitor).traverse(this);
        } catch (IOException ioe) {
            throw ioe;
        } catch (Exception e) {
            throw new LibraryException("com.ibm.xml.parser.Child#toXMLString(): Unexpected Exception: "+e.toString());
        }
    }

    /**
     * Returns this Node and any children in XML format using the default character encoding.
     * @param writer  The character output stream to use.
     * @see #getText
     * @exception IOException       Thrown if a Node can not be visitted because of an invalid <var>witer</var>.
     * @exception LibraryException  Thrown if the document object hierarchy is in an unknown state.
     */
    public void toXMLString(Writer writer) throws IOException, LibraryException {
        toXMLString(writer, null);
    }

    /**
     * Print this Node and any children in XML format, using the specified character 
     * <var>encoding</var>.
     * @param writer        The character output stream to use.
     * @param encoding      Java character encoding in use by <VAR>writer</VAR>.
     * @see com.ibm.xml.parser.TXDocument#printWithFormat
     * @exception IOException       Thrown if a Node can not be visitted because of an invalid <var>witer</var>.
     * @exception LibraryException  Thrown if the document object hierarchy is in an unknown state.
     */
    public void print(Writer writer, String encoding) throws IOException, LibraryException {
        toXMLString(writer, encoding);
    }

    /**
     * Print this Node and any children in XML format, using the default character encoding.
     * @param writer   The character output stream to use.
     * @see com.ibm.xml.parser.TXDocument#printWithFormat
     * @exception IOException       Thrown if a Node can not be visitted because of an invalid <var>witer</var>.
     * @exception LibraryException  Thrown if the document object hierarchy is in an unknown state.
     */
    public void print(Writer writer) throws IOException, LibraryException {
        print(writer, null);
    }

    /**
     * <p>This method is defined by DOM.
     *
     */
    public Document getOwnerDocument() {
        return this.factory;
    }

    /**
     * Returns the factory used for creating this Node.
     * @return          The factory used for creating this Node, or <var>null</var> if no factory.
     * @see #setFactory
     */
    public TXDocument getFactory() {
        return this.factory;
    }

    /**
     * Sets the factory to be used in creating this Node.
     * @param factory   The factory to be used in creating this Node.
     * @see #getFactory
     */
    public void setFactory(TXDocument factory) {
        this.factory = factory;
    }

    /**
     * Returns the digest value for this Node.  If no digest value currently exists, 
     * calculate and store it.
     * @return          The array of bytes for the resulting hash value.
     * @see #clearDigest
     * @exception LibraryException  Thrown if the document object hierarchy is in an unknown state.
     */
    public byte[] getDigest() throws LibraryException {
        if (null == this.digest) {
            checkFactory();
            try {
                MakeDigestVisitor makeDigestVisitor = new MakeDigestVisitor(this.getFactory().createMessageDigest());
                new NonRecursivePreorderTreeTraversal(makeDigestVisitor).traverse(this);
            } catch (Exception e) {
                throw new LibraryException("com.ibm.xml.parser.Child#getDigest(): Unexpected Exception: "+e.toString());
            }
        }
        return this.digest;
    }

    /**
     * Clears the digest value for this Node and any ancestors.
     * @see #getDigest
     */
    public void clearDigest() {
        if (null != this.digest) {
            this.digest = null;
            if (null != this.parent)
                ((Child)this.parent).clearDigest();
        }
    }

    /**
     * Returns an XPointer instance representing this Node.
     * @return  XPointer instance, or <VAR>null</VAR> if this Node can not be represented
     *          by an XPointer.
     * @see com.ibm.xml.xpointer.XPointer
     */
    public com.ibm.xml.xpointer.XPointer makeXPointer() {
        return com.ibm.xml.xpointer.XPointer.makeXPointer(this);
    }

    void checkFactory() {
        if (null == this.factory)
            this.factory = TXDocument.getInstance();
    }

    void setParentNode(Node parent) {
        this.parent = parent;
    }
    
    void setPreviousSibling(Node prevSibling) {
        this.prevSibling = prevSibling;
    }
    
    void setNextSibling(Node nextSibling) {
        this.nextSibling = nextSibling;
    }
    
    byte[] getVisitorDigest() {
        return this.digest;
    }
    
    void setVisitorDigest(byte[] digest) {
        this.digest = digest;
    }

    //
    // Public methods
    //

    /**
     * Sets the user data associated to this child node. This 
     * method is useful for attaching application specific 
     * information to any node in the document tree.
     *
     * @param data  The user data to attach to this child node.
     *              This value may be null.
     *
     * @see #getUserData
     */
    public void setUserData(Object data) {
        userData = data;
        }

    /**
     * Returns the user data associated to this child node. The
     * returned value can be null if no user data is associated.
     *
     * @see #setUserData
     */
    public Object getUserData() {
        return userData;
        }

}
