/*
 * (C) Copyright IBM Corp. 1997-1998  All rights reserved.
 *
 * US Government Users Restricted Rights Use, duplication or
 * disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * IBM will not be liable for any damages suffered by you as a result
 * of using the Program. In no event will IBM be liable for any
 * special, indirect or consequential damages or lost profits even if
 * IBM has been advised of the possibility of their occurrence. IBM
 * will not be liable for any third party claims against you.
 */

package com.ibm.xml.parser;

import java.io.IOException;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Vector;
import org.w3c.dom.DOMException;
import org.w3c.dom.DocumentType;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.w3c.dom.Notation;

/**
 * The DTD class implements the DocumentType interface as defined by the Document Object Model (DOM).
 * <p>The DTD defines the document class (type) and indicates where to find its syntactic grammar
 * definition.
 * <p>The external DTD subset is outside the XML document.  Typically, it is a standardized
 * document.  The internal DTD subset forms part of the XML document itself.  The internal
 * DTD subset often contains entity declarations that are used throughout the XML document
 * as well as other document-specific declarations.
 * <p>When an XML document has both internal and external DTD subsets, the internal subset
 * definitions are processed first, and take precedence over external subset definitions.
 * <p>The following examples illustrate an external DTD reference, an internal DTD subset, 
 * and an external DTD reference combined with an internal DTD subset:
 * <pre>
 *
 * &lt;!DOCTYPE PGML SYSTEM &quot;http://www.ibm.com/pgml.xml&quot;&gt;
 *
 * &lt;!DOCTYPE PGML [ <var>internal DTD subset</var> ]&gt;
 *
 * &lt;!DOCTYPE PGML SYSTEM &quot;http://www.ibm.com/pgml.xml&quot; [ <var>internal DTD subset ]</var>&gt;
 * </pre>
 *
 * @version Revision: 95 1.25 src/com/ibm/xml/parser/DTD.java, xml4jsrc, xml4j-jtcsv, xml4j_1_1_16 
 * @author TAMURA Kent &lt;kent@trl.ibm.co.jp&gt;
 * @see org.w3c.dom.DocumentType
 * @see com.ibm.xml.parser.Parent
 * @see com.ibm.xml.parser.Child
 * @see com.ibm.xml.parser.ExternalID
 */
public class DTD extends Parent implements DocumentType {

    static final long serialVersionUID = 8746509003209143695L;
    /**
     * A constant symbol for use by <code>getInsertableElements()</code> and <code>getAppendableElements()</code> methods.
     */
    public static final String CM_EOC = " *EOC* ";
    /**
     * A constant symbol for use by <code>getInsertableElements()</code> and <code>getAppendableElements()</code> methods.
     */
    public static final String CM_ERROR = " *ERROR* ";
    /**
     * A constant symbol for use by <code>getInsertableElements()</code> and <code>getAppendableElements()</code> methods.
     */
    public static final String CM_PCDATA = "#PCDATA";
 
    private String      name            =   null;
    private String      xmlEncoding     =   null;
    private TXNodeList  internalChildren=   null;
    private TXNodeList  externalChildren=   new TXNodeList();
    private ExternalID  externalID      =   null;
    private Hashtable   elementDecls    =   new Hashtable();
    private Hashtable   attListDecls    =   new Hashtable();
    private Hashtable   notationDecls   =   new Hashtable();
    private boolean     parsingExternal =   false;
            boolean     isPrintInternalDTD = true;
    private Hashtable   idHash          =   null;
            EntityPool  entityPool      =   null;

    /**
     * Constructor.
     */
    public DTD(){
        this.internalChildren = this.children;
    }
    
    /**
     * Constructor.
     * @param name          The name of this DTD.  This value is also known as the <code>DOCTYPE</code>
     *                      and the root Element Name.
     * @param externalID    The external ID associated with this DTD.
     * @see #getName
     * @see #setName
     * @see com.ibm.xml.parser.ExternalID
     */
    public DTD(String name, ExternalID externalID) {
        this.name       = name;
        this.externalID = externalID;
        this.internalChildren = this.children;
    }

    /**
     * Clone this DTD Node and its children using the appropriate factories.
     * <strong>Note:</strong> checkID() isn't working in the clone.
     * <p>This method is defined by Child.
     * @return          Cloned DTD Node.
     * @see com.ibm.xml.parser.Child#clone
     * @see #checkID(java.lang.String)
     */
    public Object clone() {
        return cloneNode(true);
    }

    /**
     *
     */
    public synchronized Node cloneNode(boolean deep) {
        checkFactory();
        DTD d = factory.createDTD(this.name, this.externalID);
        d.setFactory(getFactory());
        d.isPrintInternalDTD = this.isPrintInternalDTD;
        d.xmlEncoding = this.xmlEncoding;
        EntityPool ep = new EntityPool();
        // The predefined entities are added by the parser (but aren't children), 
        // so they must be added here...
        ep.add(factory.createEntityDecl("lt", "&#60;", false));
        ep.add(factory.createEntityDecl("gt", "&#62;", false));
        ep.add(factory.createEntityDecl("amp", "&#38;", false));
        ep.add(factory.createEntityDecl("apos", "&#39;", false));
        ep.add(factory.createEntityDecl("quot", "&#34;", false));
        d.setEntityPool(ep);
        if (deep) {
            d.setParsingExternal(true);
            d.externalChildren.ensureCapacity(this.externalChildren.getLength());
            for (int i = 0;  i < this.externalChildren.getLength();  i ++)
                d.appendChild((Child)((Child)this.externalChildren.item(i)).clone());
            d.setParsingExternal(false);
            d.internalChildren.ensureCapacity(this.internalChildren.getLength());
            for (int i = 0;  i < this.internalChildren.getLength();  i ++)
                d.appendChild((Child)((Child)this.internalChildren.item(i)).clone());
        }
        d.setParsingExternal(this.parsingExternal);
        return d;
    }

    /**
     *
     */
    public synchronized boolean equals(Node arg, boolean deep) {
        if (arg == null)  return false;
        if (!(arg instanceof DTD))  return false;
        DTD dtd = (DTD)arg;
        if (!((dtd.name == null && this.name == null)
              || dtd.name != null && dtd.name.equals(this.name)))  return false;
        if (!((dtd.externalID == null && this.externalID == null)
              || dtd.externalID != null && dtd.externalID.equals(this.externalID)))
            return false;
        if (!((dtd.xmlEncoding == null && this.xmlEncoding == null)
              || dtd.xmlEncoding != null && dtd.xmlEncoding.equalsIgnoreCase(this.xmlEncoding)))
            return false;
        if (deep) {
            if (!dtd.internalChildren.equals(this.internalChildren, deep))
                return false;
            if (!dtd.externalChildren.equals(this.externalChildren, deep))
                return false;
        }
        return true;
    }

    /**
     * Returns that this object is a DTD Node. 
     * <p>This method is defined by DOM.
     * @return          DTD Node indicator.
     */
    public short getNodeType() {
        return Node.DOCUMENT_TYPE_NODE;
    }

    /**
     * <p>This method is defined by DOM.
     */
    public String getNodeName() {
        return this.name;
    }

    /**
     * Returns this DTD's name.  This value is also known as the <code>DOCTYPE</code>
     * and the root Element Name. 
     * <p>This method is defined by DOM.
     * @return      This DTD's name, or <var>null</var> if no name.
     */
    public String getName() {
        return this.name;
    }
      
    /**
     * Sets this DTD's name.  This value is also known as the <code>DOCTYPE</code>
     * and the root Element Name. 
     * @param   name    This DTD's name.
     * @see #getName
     */
    public void setName(String name) {
        this.name = name;
    }

    /**
     * Insert a Child Node into the specified position in this Node's list of children.
     * The <var>newChild</var> is added to the internal or external DTD subset depending on the
     * value of <var>parsingExternal</var>.
     * @param newChild  The Node being inserted.
     * @param index     0-based index into the list of children.
     * @exception org.w3c.dom.DOMException Thrown if <var>index</var> is not valid.
     * @see #setParsingExternal
     * @see #isParsingExternal
     * @see com.ibm.xml.parser.Parent#insert
     */
    protected void realInsert(Node newChild, int index) throws DOMException {
        if (newChild == null)  return;
        super.realInsert(newChild, index);

        if (newChild instanceof ElementDecl) {
            ElementDecl ed = (ElementDecl)newChild;
            this.elementDecls.put(ed.getName(), ed);
        } else if (newChild instanceof Attlist) {
            Attlist al = (Attlist)newChild;
            String en = al.getName();
            Hashtable atthash = (Hashtable)this.attListDecls.get(en);
            if (null == atthash) {
                atthash = new Hashtable();
                this.attListDecls.put(en, atthash);
            }
            for (int i = 0;  i < al.size();  i ++) {
                AttDef ad = al.elementAt(i);
                if (null == atthash.get(ad.getName()))
                    atthash.put(ad.getName(), ad);
            }
        } else if (newChild instanceof TXNotation) {
            TXNotation nn = (TXNotation)newChild;
            this.notationDecls.put(nn.getNodeName(), nn);
        } else if (newChild instanceof EntityDecl) {
            EntityDecl e = (EntityDecl)newChild;
            this.entityPool.add(e);
        }
    }

    /**
     * Sets the flag which indicates if the internal of external DTD subset is currently
     * being parsed.
     * @param   flag    =true if parsing the external DTD subset; =false if parsing in the 
     *                  internal DTD subset.
     * @see #isParsingExternal
     * @see #addElement
     */
    public void setParsingExternal(boolean flag) {
        this.parsingExternal = flag;
        this.children = flag ? this.externalChildren : this.internalChildren;
    }
    
    /**
     * Returns whether the internal of external DTD subset is currently being parsed.
     * @return          =true if parsing the external DTD subset; =false if parsing in the 
     *                  internal DTD subset.
     * @see #setParsingExternal
     * @see #addElement
     */
    public boolean isParsingExternal() {
        return this.parsingExternal;
    }

    /**
     * Returns an Enumeration instance of all internal subset children of this DTD.  
     * @return          An enumeration of all internal subset children of this DTD.
     * @see com.ibm.xml.parser.Parent#getChildNodes
     * @see com.ibm.xml.parser.Parent#elements
     * @see #addElement
     * @see #externalElements
     */
    public Enumeration internalElements() {
        return this.internalChildren.elements();
    }
    
    /**
     * Returns an Enumeration instance of all external subset children of this DTD.  
     * @return          An enumeration of all external subset children of this DTD.
     * @see #addElement
     * @see #internalElements
     */
    public Enumeration externalElements() {
        return this.externalChildren.elements();
    }

    /**
     * Returns the external ID associated with this DTD.  An external ID contains system,
     * and optionally, public identifiers.
     * @return          The external ID associated with this DTD, or <var>null</var> if no
     *                  external DTD reference.
     * @see #setExternalID
     * @see com.ibm.xml.parser.ExternalID
     */
    public ExternalID getExternalID() {
        return this.externalID;
    }

    /**
     * Sets the external ID associated with this DTD.  An external ID contains system,
     * and optionally, public identifiers.
     * @param externalID    The external ID associated with this DTD.
     * @see #getExternalID
     * @see com.ibm.xml.parser.ExternalID
     */
    public void setExternalID(ExternalID externalID) {
        this.externalID = externalID;
    }

    /**
     * Return an Enumeration instance of all attribute list declarations for the
     * the specified <var>elementName</var> in this DTD's internal and external subsets. 
     * @param   elementName The Element name to match in the internal and external DTD subsets.
     * @return              An enumeration of all attribute list declarations.
     * @see #addElement
     * @see #getAttributeDeclaration
     * @see #isAttributeDeclared
     */
    public Enumeration getAttributeDeclarations(String elementName) {
        Hashtable atthash = (Hashtable)this.attListDecls.get(elementName);
        if (null == atthash)  atthash = new Hashtable();
        return atthash.elements();
    }

    /**
     * Return an <var>AttDef</var> instance that matches the specified <var>elementName</var>
     * and <var>attributeName</var> in this DTD's internal and external subsets. 
     * @param   elementName   The Element name to match in the internal and external DTD subsets.
     * @param   attributeName The Attribute name to match in <var>elementName</var>.
     * @return                The matching attribute definition, or <var>null</var> if no match.
     * @see #addElement
     * @see #getAttributeDeclarations
     * @see #isAttributeDeclared
     */
    public AttDef getAttributeDeclaration(String elementName, String attributeName) {
        Hashtable atthash = (Hashtable)this.attListDecls.get(elementName);
        if (null == atthash)  return null;
        return (AttDef)atthash.get(attributeName);
    }

    /**
     * Return whether an attribute definition exists that matches the specified <var>elementName</var>
     * and <var>attributeName</var> in this DTD's internal and external subsets. 
     * @param   elementName   The Element name to match in the internal and external DTD subsets.
     * @param   attributeName The Attribute name to match in <var>elementName</var>.
     * @return                =true if the attribute definition exists; otherwise, =false.
     * @see #addElement
     * @see #getAttributeDeclarations
     * @see #getAttributeDeclaration
     */
    public boolean isAttributeDeclared(String elementName, String attributeName) {
        return null != getAttributeDeclaration(elementName, attributeName);
    }

    /**
     * Return an Enumeration instance of all element declarations in this DTD's internal 
     * and external subsets. 
     * @return              An enumeration of all element declarations.
     * @see #addElement
     * @see #getElementDeclaration
     * @see #isElementDeclared
     * @see #makeContentElementList
     */
    public Enumeration getElementDeclarations() {
        return this.elementDecls.elements();
    }

    /**
     * Return an <var>ElementDecl</var> instance that matches the specified <var>elementName</var>
     * in this DTD's internal and external subsets. 
     * @param   elementName   The Element name to match in the internal and external DTD subsets.
     * @return                The matching element definition, or <var>null</var> if no match.
     * @see #addElement
     * @see #getElementDeclarations
     * @see #isElementDeclared
     * @see #makeContentElementList
     */
    public ElementDecl getElementDeclaration(String elementName) {
        return (ElementDecl)this.elementDecls.get(elementName);
    }

    /**
     * Return whether an element definition exists that matches the specified <var>elementName</var>
     * in this DTD's internal and external subsets. 
     * @param   elementName   The Element name to match in the internal and external DTD subsets.
     * @return                =true if the element definition exists; otherwise, =false.
     * @see #addElement
     * @see #getElementDeclarations
     * @see #getElementDeclaration
     * @see #makeContentElementList
     */
    public boolean isElementDeclared(String elementName) {
        return this.elementDecls.containsKey(elementName);
    }

    /**
     * Returns a <var>Vector</var> of the children of the specified <var>elementDeclName</var>
     * as defined by this DTD's internal and external subset.
     * @param   elementDeclName The element definition name to match in the internal and
     *                          external DTD subsets.
     * @return                  The children of the element definition, or <var>null</var>
     *                          if the element is not defined or the element's content model
     *                          is not defined as <var>MODEL_GROUP</var>.
     * @see #addElement
     * @see #getElementDeclarations
     * @see #getElementDeclaration
     * @see com.ibm.xml.parser.ElementDecl
     */
    public Vector makeContentElementList(String elementDeclName) {
        ElementDecl ed = (ElementDecl)this.elementDecls.get(elementDeclName);
        if (null == ed)  return null;
        return ed.getXML4JContentModel().getChildrenOrder();
    }

    /**
     * Return an Enumeration instance of all notation declarations (<CODE>TXNotation</CODE>)
     * in this DTD's internal and external subsets. 
     * @return              An enumeration of all notations.
     * @see #getNotation
     */
    public Enumeration getNotationEnumeration() {
        return this.notationDecls.elements();
    }

    /**
     * Return an <var>Notation</var> instance that matches the specified <var>notationName</var>
     * in this DTD's internal and external subsets. 
     * @param   notationName  The Notation name to match in the internal and external DTD subsets.
     * @return                The matching Notation, or <var>null</var> if no match.
     * @see #getNotations
     * @see #getNotationEnumeration
     */
    public Notation getNotation(String notationName) {
        TXNotation not = (TXNotation)this.notationDecls.get(notationName);
        return not == null ? null : not.getNotationImpl();
    }

    /**
     * Returns the <var>NamedNodeMap</var> which has the set of notations
     * that were defined within this DTD's external and internal subset.
     * <p>This method is defined by DOM.
     * @return          NamedNodeMap whose children are the notations for this DTD.
     */
    public NamedNodeMap getNotations(){
        return new HashNamedNodeMap(this.notationDecls);
    }

    /**
     * for Hashtable consisted of TXNotations, or EntityDecls.
     */
    static class HashNamedNodeMap implements NamedNodeMap {
        Hashtable hash;
        Node[] data;
        HashNamedNodeMap(Hashtable hash) {
            this.hash = hash;
            makeArray();
        }
        private void makeArray() {
            this.data = new Node[this.hash.size()];
            Enumeration en = this.hash.elements();
            int i = 0;
            while (en.hasMoreElements())
                this.data[i++] = (Node)en.nextElement();
        }
        public Node getNamedItem(String name) {
            Object obj = this.hash.get(name);
            if (obj == null) {
                return null;
            } else if (obj instanceof TXNotation) {
                return ((TXNotation)obj).getNotationImpl();
            } else if (obj instanceof EntityDecl) {
                return ((EntityDecl)obj).getEntityImpl();
            } else {
                throw new RuntimeException("XML4J internal error: non-supported hash.");
            }
        }
        public Node setNamedItem(Node arg) {
            throw new TXDOMException(DOMException.NO_MODIFICATION_ALLOWED_ERR,
                                     "This NamedNodeMap is read-only.");
            /*
            Node exists = (Node)this.hash.get(arg.getNodeName());
            this.hash.put(arg.getNodeName(), arg);
            makeArray();
            return exists;
            */
        }
        public Node removeNamedItem(String name) {
            throw new TXDOMException(DOMException.NO_MODIFICATION_ALLOWED_ERR,
                                     "This NamedNodeMap is read-only.");
            /*
            Node node = (Node)this.hash.get(name);
            if (node != null) {
                this.hash.remove(name);
                makeArray();
            }
            return node;
            */
        }
        public Node item(int index) {
            return this.data[index];
        }
        public int getLength() {
            return this.data.length;
        }
    }

    /**
     * Sets the value of the XML encoding parameter from the XML prolog declaration (e.g. <code>&lt;?xml encoding="..."&gt;</code>).
     * This value is used when printing the DTD external subset.
     * <p>The supported XML encodings are the intersection of XML-supported code sets and 
     * those supported in JDK 1.1:
     * <ul>
     * <li>UTF-16
     * <li>ISO-10646-UCS-2
     * <li>ISO-10646-UCS-4
     * <li>UTF-8
     * <li>US-ASCII
     * <li>ISO-8859-1 ... ISO-8859-9
     * <li>ISO-2022-JP
     * <li>Shift_JIS
     * <li>EUC-JP
     * <li>GB2312
     * <li>Big5
     * <li>EUC-KR
     * <li>ISO-2022-KR
     * <li>KOI8-R
     * </ul>
     * @param   xmlEncoding Value of the XML encoding parameter.  
     * @see com.ibm.xml.parser.MIME2Java#convert
     * @see #printExternal
     * @see com.ibm.xml.parser.TXDocument#setEncoding
     */
    public void setEncoding(String xmlEncoding) {
        this.xmlEncoding = xmlEncoding;
    }

    /**
     * Print this DTD's external subset in XML format using the specified character encoding.
     * @param pw        The character output stream to use.
     * @param encoding  Java character encoding in use by <VAR>pw</VAR>.
     */
    public void printExternal(java.io.Writer pw, String encoding) throws IOException {
        if (null != this.xmlEncoding) {
            pw.write("<?xml encoding=\"" + this.xmlEncoding + "\"?>");
        }
        for (int i = 0;  i < this.externalChildren.getLength();  i ++) { // Can't use TreeTraversal
            ((Child)this.externalChildren.item(i)).print(pw, encoding);
        }
        pw.flush();
    }

    /**
     * Returns whether the specified <var>element</var> is declared in this DTD's internal and external
     * subsets, AND whether it currently adheres to its defined content model (see ElementDecl for details).
     * @param   element The Element to check in this DTD's internal and external subset.
     * @return          =true if <var>element</var> exists AND it adheres to its content
     *                  model; otherwise =false.
     * @see com.ibm.xml.parser.ElementDecl
     * @see #getContentModel
     * @see #prepareTable
     * @see #getInsertableElements
     * @see #getAppendableElements
     */
    public boolean checkContent(TXElement element) {
        ElementDecl ed = (ElementDecl)this.elementDecls.get(element.getNodeName());
        if (null == ed)  return false;
        return ed.getXML4JContentModel().check(element);
    }
    
    /**
     * Returns the <var>ContentModel</var> for the specified <var>elementName</var>
     * in this DTD's internal and external subset (see ElementDecl for details).
     * @param   elementName The name of element definition to check in this DTD's
     *                      internal and external subset.
     * @return              The element definition's content model, or <var>null</var>
     *                      if the element definition does not exist.
     * @see com.ibm.xml.parser.ElementDecl
     * @see #checkContent
     * @see #getContentType
     * @see #prepareTable
     * @see #getInsertableElements
     * @see #getAppendableElements
     */
    public ContentModel getContentModel(String elementName) {
        ElementDecl ed = (ElementDecl)this.elementDecls.get(elementName);
        if (null == ed)  return null;
        return ed.getXML4JContentModel();
    }

    /**
     * Returns the content model type for the specified <var>elementName</var>
     * in this DTD's internal and external subset (see ElementDecl for details).
     * @param   elementName The name of element definition to check in this DTD's
     *                      internal and external subset.
     * @return              The element definition's content model type, or
     *                      <var>-1</var> if the element definition does not exist.
     *                      Must be one of org.w3c.dom.ElementDefinition#ContentType.
     * @see com.ibm.xml.parser.ElementDecl
     * @see #checkContent
     * @see #getContentModel
     * @see #prepareTable
     * @see #getInsertableElements
     * @see #getAppendableElements
     */
    public int getContentType(String elementName) {
        ElementDecl ed = (ElementDecl)this.elementDecls.get(elementName);
        if (null == ed)  return -1;
        return ed.getXML4JContentModel().getType();
    }
    
    /**
     * Prepare a hash table for use by the <code>getInsertableElements()</code> and
     * <code>getAppendableElements()</code> methods.  Refer to <code>getInsertableElements</code> 
     * for details on how this table is used.
     * @param   elementName The name of Element to use in building the hash table.
     * @return              The constructed hash table of acceptable Elements that may
     *                      be inserted; acceptable Elements are defined by the element
     *                      definition defined in this DTD's internal and external subset.
     * @see #getInsertableElements
     * @see #getAppendableElements
     * @see com.ibm.xml.parser.InsertableElement
     */
    public Hashtable prepareTable(String elementName) {
        ElementDecl ed = (ElementDecl)this.elementDecls.get(elementName);
        if (null == ed)  return null;
        return ed.getXML4JContentModel().prepareTable();
    }
    
    /**
     * Returns a hash table which defines what kind of Element you can insert into the specified <var>index</var> of <var>element</var>
     * according to the document grammar specified by this DTD's internal and external subset.
     * <p>Usage:</p>
     * <blockquote><code>
     *     DTD dtd = ...;<br>
     *     Hashtable table = dtd.prepareTable(element.getName());<br>
     *     dtd.getInsertableElements(element, index, table);
     * </code></blockquote>
     * <p>The resulting <var>table</var> instance consists of Element names as keys and <var>InsertableElement</var> as values.
     * The following information is available about <var>table</var>:</p>
     * <dl>
     *   <dt><code>insertableElement = (InsertableElement)hashtable.get("<var>elementName</var>")</code>
     *   <dd>If <code>insertableElement != null</code> AND <code>insertableElement.status == true</code>
     *       you can insert <var>elementName</var> Elements under <var>element</var>.  
     *       Note that this does not mean that <var>element</var> will be guaranteed to have 
     *       correct contents after the insertion. Otherwise, the <var>elementName</var> 
     *       may not be inserted.
     *   <dd>When the <var>element</var>'s model group is <var>ANY</var> AND <code>insertableElement == null</code>,
     *       <var>elementName</var> may be inserted.  Otherwise, <var>elementName</var> may 
     *       not be inserted.
     *   <dt><code>insertableElement = (InsertableElement)hashtable.get(DTD.CM_PCDATA)</code>
     *   <dd>When <code>insertableElement.status == true</code>, you can insert any
     *       a TXText Node under <var>element</var>.  Note that this does not mean
     *       that <var>element</var> will be guaranteed to have correct contents after the 
     *       insertion.   Otherwise, a TXText Node may not be inserted.
     *   <dt><code>insertableElement = (InsertableElement)hashtable.get(DTD.CM_EOC)</code>
     *   <dd><code>insertableElement.status</code> indicates whether <var>element</var> has 
     *       correct and complete contents in the subsequence of children [0, <var>index</var>-1].
     *   <dt><code>insertableElement = (InsertableElement)hashtable.get(DTD.CM_ERROR)</code>
     *   <dd>When <var>insertableElement.status</var> is <code>=true</code>,
     *       <var>insertableElement.index</var> is the index of the first incorrect Child 
     *       Node of the specified <var>element</var>.  Note that you might still be able to 
     *       insert Elements under <var>element</var> even if 
     *       <code>insertableElement.status</code> is <code>=true</code>.
     *  </dl>
     * @param   element     The Element to use in building the hash table.
     * @param   index       The 0-based index of <var>element</var>'s children which are
     *                      to be examined for validity in preparation for insertion.
     * @param   hashtable   The hash table built by <code>prepareTable</code>.
     * @return              The constructed hash table, or <var>null</var> if unable
     *                      to build the hash table with the specified <var>elementName</var>.
     * @see #prepareTable
     * @see #getAppendableElements
     * @see #getInsertableElementsForValidContent
     * @see com.ibm.xml.parser.InsertableElement
     */
    public Hashtable getInsertableElements(Element element, int index, Hashtable hashtable) {
        // This method is not used by the parser.
        ContentModel cm = getContentModel(element.getNodeName());
        if (null == cm)  return null;
        return cm.getInsertableElements(element, index, hashtable, false);
    }
    
    /**
     * Returns a hash table which defines what kind of Element you can append onto specified <var>element</var>
     * according to the document grammar specified by this DTD's internal and external subset.
     * Refer to <code>getInsertableElements</code> for details on how this table is used.
     * @param   element     The Element to use in building the hash table.
     * @param   hashtable   The hash table built by <code>prepareTable</code>.
     * @return              The constructed hash table, or <var>null</var> if unable
     *                      to build the hash table with the specified <var>elementName</var>.
     * @see #prepareTable
     * @see #getInsertableElements
     * @see com.ibm.xml.parser.InsertableElement
     */
    public Hashtable getAppendableElements(Element element, Hashtable hashtable) {
        return getInsertableElements(element, element.getChildNodes().getLength(), hashtable);
    }

    /**
     * Returns a hash table which defines what kind of Element you can insert into the specified <var>index</var> of <var>element</var>
     * according to the document grammar specified by this DTD's internal and external subset.
     * <p>This method can be used when <VAR>element</VAR> already has valid content.
     * After insertion of returned element to <VAR>element</VAR>, <VAR>element</VAR> will have
     * valid content.
     *
     * @param   element     The Element to use in building the hash table.
     * @param   index       The 0-based index of <var>element</var>'s children which are
     *                      to be examined for validity in preparation for insertion.
     * @param   hashtable   The hash table built by <code>prepareTable</code>.
     * @return              The constructed hash table, or <var>null</var> if unable
     *                      to build the hash table with the specified <var>elementName</var>.
     * @see #prepareTable
     * @see #getInsertableElements
     * @see com.ibm.xml.parser.InsertableElement
     */
    public Hashtable getInsertableElementsForValidContent(Element element,
                                                          int index, Hashtable hashtable) {
        // This method is not used by the parser.
        ContentModel cm = getContentModel(element.getNodeName());
        if (null == cm)  return null;
        return cm.getInsertableElements(element, index, hashtable, true);
    }

    /**
     * Register the identifier attribute associated with the specified <var>element</var> in the XML document.
     * Registered identifiers must be unique within an XML document.
     * An existing identifier attribute must be unregistered before it can
     * be registered with another element.
     * <p>The XML4J parser automatically invokes this method when parsing a 
     * document with DTD. After parsing, the responsibility for calling 
     * registID IDs lies with the programmer adding, deleting or modifiying 
     * elements.
     *
     * @param   element     Element to be registered.
     * @param   id          Identifier to be registered.
     * @return              =true if successfully register; =false if <var>id</var> already
     *                      is registered in the XML document. Then unregistID must be called
     *                      before register.
     * @see #unregistID
     * @see #checkID
     * @see #IDs
     */
    public boolean registID(Element element, String id) {
        if (null == this.idHash)  this.idHash = new Hashtable();
        if (this.idHash.containsKey(id))  return false; // already exists
        this.idHash.put(id, element);
        return true;
    }
    
    /**
     * Unregister the identifier attribute associated with the specified 
     * <var>element</var> in the XML document. 
     * Registered identifiers must be unique within an XML document.
     * An existing identifier attribute must be unregistered before it can
     * be registered with another element.
     * <p>The responsibility for calling unregistID IDs lies with the 
     * programmer adding, deleting or modifiying elements.
     *
     * @param   id          Identifier to be unregistered.
     * @return              =true if successfully unregister; =false if <var>id</var> 
     *                      wasn't registered in the XML document.
     * @see #registID
     * @see #checkID
     * @see #IDs
     */
    public boolean unregistID(String id) {
        if (null == this.idHash) return false;
        if (!this.idHash.containsKey(id))  return false; // doesn't exist
        this.idHash.remove(id);
        return true;
    }

    /**
     * Returns the <var>Element</var> associated with the specified <var>id</var> in the
     * XML document.
     * @param   id          Identifier to be matched.
     * @return              Matching Element, or <var>null</var> if <var>id</var> is not registered.
     * @see #registID
     * @see #IDs
     */
    public Element checkID(String id) {
        if (null == this.idHash)  return null;
        return (Element)this.idHash.get(id);
    }

    /**
     * Returns an <var>Enumeration</var> of all Element IDs currently registered in the XML document.
     * @return              An enumeration of <var>String</var> representing all currently
     *                      registered Element IDs.
     * @see #checkID
     * @see #IDs
     */
    public Enumeration IDs() {
        return null == this.idHash ? new Hashtable().keys() : this.idHash.keys();
    }
    
    /**
     * Returns an <var>Enumeration</var> of all entities defined in this DTD's internal and
     * external subsets except paramete entities.
     * @return              An enumeration of <var>EntityDecl</var> representing all defined
     *                      entities in this DTD's internal and external subsets.
     * @see com.ibm.xml.parser.EntityDecl
     */
    public Enumeration getEntityEnumeration() {
        return this.entityPool.elements();
    }

    /**
     * Returns an <var>Enumeration</var> of all parameter entities defined in this DTD's internal and
     * external subsets.
     * @return              An enumeration of <var>EntityDecl</var> representing all defined
     *                      entities in this DTD's internal and external subsets.
     * @see com.ibm.xml.parser.EntityDecl
     */
    public Enumeration getParameterEntityEnumeration() {
        return this.entityPool.parameterEntityElements();
    }

    /**
     * Returns an EntityDecl in this DTD's internal and external subsets that matches the
     * specified <var>name</var> and is of the specified <var>isParameter</var> type.
     * @param   name        The entity name to match against in this DTD's internal and external
     *                      DTD subset.
     * @param   isParameter =true if a parameter entity; otherwise =false.
     * @return              The matching EntityDecl, or <var>null</var> if no match.
     * @see #getEntityEnumeration
     */
    public EntityDecl getEntityDecl(String name, boolean isParameter) {
        EntityDecl ret;
        if (isParameter) {
            ret = this.entityPool.referPara(name);
        } else {
            ret = this.entityPool.refer(name);
        }
        return ret;
    }

    /**
     * <p>This method is defined by DOM.
     *
     * @see #getEntityEnumeration
     */
    public NamedNodeMap getEntities() {
        return new HashNamedNodeMap(this.entityPool.getEntityHash());
    }

    /**
     * Return a number of Nodes in an internal subset.
     */
    public int getInternalSize() {
        return this.internalChildren.getLength();
    }

    /**
     * Return a number of Nodes in an external subset.
     */
    public int getExternalSize() {
        return this.externalChildren.getLength();
    }

    /**
     * Sets the flag which indicates if the <CODE>print()</CODE> method prints the internal 
     * subset.  By default, the internal DTD subset is printed.
     *
     * @see #isPrintInternalDTD
     * @see com.ibm.xml.parser.Child#print
     */
    public void setPrintInternalDTD(boolean flag) {
        this.isPrintInternalDTD = flag;
    }
    /**
     * Returns whether the <CODE>print()</CODE> method prints internal subse.
     *
     * @see #setPrintInternalDTD
     * @see com.ibm.xml.parser.Child#print
     */
    public boolean isPrintInternalDTD() {
        return this.isPrintInternalDTD;
    }
    
    /**
     * Implements the accept operation of the visitor design pattern when the start of
     * a DTD Node is recognized when traversing the document object tree. 
     * @param   visitor The implemention of the Visitor operation (toXMLString, digest, ...)
     * @exception Exception Thrown if this Node can not be visited, or traversal modification is requested.
     * @see com.ibm.xml.parser.Visitor
     * @see com.ibm.xml.parser.TreeTraversal
     * @see com.ibm.xml.parser.NonRecursivePreorderTreeTraversal
     * @see com.ibm.xml.parser.TreeTraversalException
     */
    public void acceptPre(Visitor visitor) throws Exception {
        visitor.visitDTDPre(this);
    }

    /**
     * Implements the accept operation of the visitor design pattern when the end of
     * a DTD Node is recognized when traversing the document object tree. 
     * @param   visitor The implemention of the Visitor operation (toXMLString, digest, ...)
     * @exception Exception Thrown if this Node can not be visited, or traversal modification is requested.
     * @see com.ibm.xml.parser.Visitor
     * @see com.ibm.xml.parser.TreeTraversal
     * @see com.ibm.xml.parser.NonRecursivePreorderTreeTraversal
     * @see com.ibm.xml.parser.TreeTraversalException
     */
    public void acceptPost(Visitor visitor) throws Exception {
        visitor.visitDTDPost(this);
    }

    /**
     * Called by Parser.
     */
    void setEntityPool(EntityPool ent) {
        this.entityPool = ent;
    }

    /**
     * Check whether <VAR>child</VAR> is allowed to be insered in this node or not.
     * When not allowed, a DOMException with HIERARCHY_REQUEST_ERR is thrown.
     */
    protected void checkChildType(Node child) throws DOMException {
        switch (child.getNodeType()) {
          case Node.NOTATION_NODE:
          case Node.ENTITY_NODE:
          case Node.PROCESSING_INSTRUCTION_NODE:
          case Node.COMMENT_NODE:
          case Node.TEXT_NODE:
          case Child.ELEMENT_DECL:
          case Child.ATTLIST:
          case Child.PSEUDONODE:
            break;
          default:
            throw new TXDOMException(DOMException.HIERARCHY_REQUEST_ERR,
                                     "Specified node type ("+child.getNodeType()
                                     +") can't be a child of DocumentType.");
        }
    }
}
