/*
 * (C) Copyright IBM Corp. 1997-1998  All rights reserved.
 *
 * US Government Users Restricted Rights Use, duplication or
 * disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * IBM will not be liable for any damages suffered by you as a result
 * of using the Program. In no event will IBM be liable for any
 * special, indirect or consequential damages or lost profits even if
 * IBM has been advised of the possibility of their occurrence. IBM
 * will not be liable for any third party claims against you.
 */

package com.ibm.xml.parser;

import org.w3c.dom.*;

/**
 * <p>Element declarations in the DTD determine the possible structure of the XML document.
 * The following structures, known as <var>content models</var>, are supported for element declaration:
 * <dl>
 * <dt>EMPTY
 * <dd>This Element can not have any content, and must appear as an empty Element in the
 * XML document.  For example, the <var>EMPTY</var> element declaration 
 * <CODE>&lt;!ELEMENT graphic EMPTY&gt;</CODE> would have the Element tag <CODE>&lt;graphic/&gt;</CODE>
 * appear in the XML document.
 * <dt>ANY
 * <dd>This Element can have any combination of character data and element types allow by
 * the DTD.
 * <dt>#PCDATA
 * <dd>This Element must only contain character data.  For example, the <var>#PCDATA</var> 
 * element declaration <CODE>&lt;!ELEMENT text (#PCDATA)&gt;</CODE> could have the Element 
 * tag <CODE>&lt;text&gt;a group of character data&lt;/text&gt;</CODE> appear in the XML document.
 * <dt>MODEL_GROUP
 * <dd>This Element allows a mixture of #PCDATA and other Elements as their children.
 * The following language primitives are used to help express this relationship:
 *   <dl>
 *   <dt>( and )
 *   <dd>Surround a sequence of alternatives
 *   <dt>,
 *   <dd>Separates each Element type in a sequence
 *   <dt>|
 *   <dd>Separates each Element type in a list of alternatives
 *   <dt>? and *
 *   <dd>Indicates that an Element or group occurs zero or more times
 *   <dt>+
 *   <dd>Indicates that an Element or group occurs one or more times
 *   </dl>
 *   <p>To illustrate this point, the element declaration 
 *   <CODE>&lt;!ELEMENT memo (from, to+, cc*, date?, content)&gt;</CODE> could have the Element 
 *   tag <CODE>&lt;memo&gt; &lt;from&gt;me&lt;/from&gt; &lt;to&gt;you&lt;/to&gt; &lt;to&gt;and you&lt;/to&gt; &lt;content&gt;hello world&lt;/content&gt; &lt;/memo&gt;</CODE> appear in the XML document.
 * </dl>
 *
 * @version Revision: 96 1.10 src/com/ibm/xml/parser/ElementDecl.java, xml4jsrc, xml4j-jtcsv, xml4j_1_1_16 
 * @author TAMURA Kent &lt;kent@trl.ibm.co.jp&gt;
 * @see com.ibm.xml.parser.Child
 */
public class ElementDecl extends Child {

    static final long serialVersionUID = 1933367195772929307L;
    public static final int EMPTY = 1;
    public static final int ANY = 2;
    public static final int MODEL_GROUP = 4;
    
            String          name            =   null;
            ContentModel    contentModel    =   null;

    /**
     * Constructor.
     * @param name          This element definition's name. 
     * @param contentModel  The content model to associate with this element definition,
     *                      or <var>null</var> if the content model is to be set later.
     */
    public ElementDecl(String name, ContentModel contentModel) {
        this.name = name;
        this.contentModel = contentModel;
    }

    /**
     * Clone this element definition Node using the appropriate factory.
     * <p>This method is defined by Child.
     * @return          Cloned element definition Node.
     * @see com.ibm.xml.parser.Child#clone
     */
    public synchronized Object clone() {
        checkFactory();
        ElementDecl ed = factory.createElementDecl(this.name, null);
        ed.setFactory(getFactory());
        ed.contentModel = (ContentModel)this.contentModel.clone();
        return ed;
    }

    /**
     * @param deep ignored.
     */
    public synchronized boolean equals(Node arg, boolean deep) {
        if (arg == null)  return false;
        if (!(arg instanceof ElementDecl))  return false;
        ElementDecl ed = (ElementDecl)arg;
        if (!ed.getName().equals(this.getName()))  return false;
        return ed.contentModel.equals(this.contentModel);
    }

    /**
     * Returns that this object is an element definition Node.
     * <p>This method is defined by DOM.
     * @return          Element definition Node indicator.
     */
    public short getNodeType() {
        return Child.ELEMENT_DECL;
    }

    /**
     *
     */
    public String getNodeName() {
        return Child.NAME_ELEMENT_DECL;
    }

    /**
     * Returns this element definition's name.
     * @return          The string that is this Element's name, or <var>null</var> if no name. 
     * @see #setName
     */
    public String getName() {
        return this.name;
    }

    /**
     * Sets this element definition's name. 
     * @param   name    The string that is this element definition's name. 
     * @see #getName
     */
    public void setName(String name) {
        this.name = name;
    }

    /**
     * Returns the content type which represents the content model used by this element definition. 
     * @return          The content type for this element's definition, or <var>null</var> if 
     *                  no content type has been defined. 
     *                  Must be one of EMPTY/ANY/MODEL_GROUP.
     * @see #setContentType
     */
    public int getContentType() {
        return getXML4JContentModel().getType();
    }

    /**
     * Sets the content type which represents the content model used by this element definition. 
     * @param contentType   The content type for this element's definition.
     *                      Must be one of EMPTY/ANY/MODEL_GROUP.
     * @see #getContentType
     */
    public void setContentType(int contentType) {
        getXML4JContentModel().setType(contentType);
    }

    /**
     * Implements the accept operation of the visitor design pattern when the start of
     * an ElementDecl Node is recognized when traversing the document object tree. 
     * @param   visitor The implemention of the Visitor operation (toXMLString, digest, ...)
     * @exception Exception Thrown if this Node can not be visitted, or traversal modification is requested.
     * @see com.ibm.xml.parser.Visitor
     * @see com.ibm.xml.parser.TreeTraversal
     * @see com.ibm.xml.parser.NonRecursivePreorderTreeTraversal
     * @see com.ibm.xml.parser.TreeTraversalException
     */
    public void acceptPre(Visitor visitor) throws Exception {
        visitor.visitElementDeclPre(this);
    }

    /**
     * Implements the accept operation of the visitor design pattern when the end of
     * an ElementDecl Node is recognized when traversing the document object tree. 
     * @param   visitor The implemention of the Visitor operation (toXMLString, digest, ...)
     * @exception Exception Thrown if this Node can not be visitted, or traversal modification is requested.
     * @see com.ibm.xml.parser.Visitor
     * @see com.ibm.xml.parser.TreeTraversal
     * @see com.ibm.xml.parser.NonRecursivePreorderTreeTraversal
     * @see com.ibm.xml.parser.TreeTraversalException
     */
    public void acceptPost(Visitor visitor) throws Exception {
        visitor.visitElementDeclPost(this);
    }
    
    ContentModel getXML4JContentModel() {
        return this.contentModel;
    }
}
