/*
 * (C) Copyright IBM Corp. 1997-1998  All rights reserved.
 *
 * US Government Users Restricted Rights Use, duplication or
 * disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * IBM will not be liable for any damages suffered by you as a result
 * of using the Program. In no event will IBM be liable for any
 * special, indirect or consequential damages or lost profits even if
 * IBM has been advised of the possibility of their occurrence. IBM
 * will not be liable for any third party claims against you.
 */

package com.ibm.xml.parser;

/**
 * An interface for receiving control when pre-registered element tag names are recognized
 * by the XML4J parser.
 * <p>This interface is implemented by filter programs which do not want to deal with the
 * complexities of manipulating an entire XML document hierarchy.  Rather, filter programs
 * only want to recieve control when a pre-registered set of tags are recognized by the parser.
 * <p>For example, a filter program may want to convert all <code>&lt;email&gt;foo@bar&lt;/email&gt;</code>
 * tags to <code>&lt;url href="mailto:foo@bar"/&gt;</code>.  In this case, the <var>email</var>
 * tag is registered with the XML4J parser instance, prior to parsing the input stream, by using
 * the <code>addElementHandler</code> method.  Subsequently, the implemented <code>handleElement</code>
 * method will be invoked by the XML4J parser instance when the input stream is read; the
 * implemented <code>handleElement</code> method is responsible for manipulating the child
 * objects of the <code>&lt;email&gt;</code> element.  This logic is implemented in the 
 * following code fragment:
 * <pre>
 *
 * public class EMAILFilter implements ElementHandler {
 *   public TXElement handleElement(TXElement originalElement) { 
 *       String addr;
 *       if (0 != (addr = originalElement.getText()).length()) {
 *           TXElement newElement = new TXElement("URL");
 *           newElement.setAttribute("HREF", "mailto:"+addr);
 *           originalElement = newElement;
 *       } 
 *       return originalElement;
 *   }
 *
 *   public static void main(String[] argv) {
 *       ...
 *       Parser p = new Parser(fname);
 *       p.addElementHandler(new EMAILFilter(), "EMAIL");   // Watch for EMAIL tag
 *       // Note: If the above line is changed to <code>p.addElementHandler(new EMAILFilter());</code>,
 *       //       EMAILFilter is invoked when EACH end tag is recognized by the parser.
 *       TXDocument doc = p.readStream(p.getInputStream(fname, null, fname));
 *       // doc instance now contains updated EMAIL Elements.
 *       ...
 *   }
 *
 * </pre>
 *
 * @version Revision: 98 1.3 src/com/ibm/xml/parser/ElementHandler.java, xml4jsrc, xml4j-jtcsv, xml4j_1_1_16 
 * @author TAMURA Kent &lt;kent@trl.ibm.co.jp&gt;
 * @see com.ibm.xml.parser.Parser#addElementHandler(com.ibm.xml.parser.ElementHandler)
 * @see com.ibm.xml.parser.Parser#addElementHandler(com.ibm.xml.parser.ElementHandler,java.lang.String)
 */
public interface ElementHandler {

    /**
     * Interface to be implemented for receiving control when pre-registered element tag names
     * are recognized by the XML4J parser.  Control is transferred after calling any TagHandlers
     * when the end tag is parsed.
     * @param element   An Element that matches the specified name on a prior call to
     *                  <code>parser.addElementHandler</code>.  Note that namespace names
     *                  may also be included in Element names.
     * @return The modified element, or <var>null</var> if the Element is to be removed from
     *         the document.
     * @see com.ibm.xml.parser.Namespace
     * @see com.ibm.xml.parser.TXElement
     */
    public TXElement handleElement(TXElement element);
}

