/*
 * (C) Copyright IBM Corp. 1997-1998  All rights reserved.
 *
 * US Government Users Restricted Rights Use, duplication or
 * disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * IBM will not be liable for any damages suffered by you as a result
 * of using the Program. In no event will IBM be liable for any
 * special, indirect or consequential damages or lost profits even if
 * IBM has been advised of the possibility of their occurrence. IBM
 * will not be liable for any third party claims against you.
 */

package com.ibm.xml.parser;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.CharArrayReader;
import java.io.CharArrayWriter;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.Reader;
import org.w3c.dom.DOMException;
import org.w3c.dom.Document;
import org.w3c.dom.Entity;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * The EntityDecl class represents entity declrations.
 * Entities are useful in:
 * <ul compact>
 * <li>Representing non-standard characters within an XML document (Example: an e-acute symbol)
 * <li>Holding sections or chapters from a large XML document
 * <li>Organizing DTDs into logical sub-units
 * <li>Representing shorthand notation for frequently used phrases
 * <li>Holding markup common to multiple XML documents
 * </ul>
 * <p>Internal entities have a <var>value</var> that is directly given in the entity declaration.
 * For example:
 * <p><center><code>&lt;!ENTITY IBM &quot;International Business Machines&quot;&gt;</code></center>
 * <p>External entities refer to declarations to a storage unit by means of a SYSTEM or
 * PUBLIC identifier.  For example:
 * <p><center><code>&lt;!ENTITY IBM SYSTEM &quot;ibm.xml&quot;&gt;</code></center>
 * <p>..associates the name <code>IBM</code> with the URL <code>&quot;ibm.xml&quot;</code>.
 * XML4J must read the file referenced in order to find out the content of that entity.
 * <p>As you might expect, external Entities contain <var>External IDs</var> in order to provide
 * access to the resource.  External entities can can be either text or binary resources.
 * Text resources are XML-encoded resources, and are illustrated by this example:
 * <p><center><code>&lt;!ENTITY IBM SYSTEM &quot;ibm.xml&quot;&gt;</code></center>
 * <p>Binary entities are anything that is not XML-encoded.  Binary entities always include
 * a notation (<var>ndata</var>) which describes the type of resource.  For example:
 * <p><center><code>&lt;!ENTITY IBMLogo SYSTEM &quot;ibm.jpg&quot; NDATA JPEG&gt;</code></center>
 * <!--p>Only internal Entities can be <var>parameter</var> Entities; this special
 * type of entity is only used within an internal DTD.-->
 *
 * @version Revision: 71 1.9 src/com/ibm/xml/parser/EntityDecl.java, xml4jsrc, xml4j-jtcsv, xml4j_1_1_16 
 * @author TAMURA Kent &lt;kent@trl.ibm.co.jp&gt;
 * @see com.ibm.xml.parser.ExternalID
 * @see com.ibm.xml.parser.Parent
 * @see org.w3c.dom.Entity
 */
public class EntityDecl extends Parent {

        static final long serialVersionUID = 1661279924518265405L;
        String      name        =   null;
        boolean     isParameter =   false;
        String      value       =   null;   // Always <var>null</var> for external Entities
        ExternalID  externalID  =   null;   // Always <var>null</var> for internal Entities
        String      ndata       =   null;   // Always <var>null</var> for internal Entities

        byte[]      rawByteStream   =   null;
        String      encoding        =   null;   // for rawByteStream
        char[]      rawCharStream   =   null;

        boolean     parsed      =  false;

    /**
     * Constructor for internal Entities.
     * @param   name        Name of this Entity.
     * @param   value       The XML-encoded value that was directly assigned to the Entity.
     * @param   isParameter =true if a parameter Entity; otherwise =false.
     */
    public EntityDecl(String name, String value, boolean isParameter) {
        this.name           = name;
        this.value          = value;
        this.isParameter    = isParameter;
    }

    /**
     * Constructor for external Entities.
     * @param   name        Name of the Entity.
     * @param   externalID  The reference(s) to the external entity to retrieve.
     * @param   isParameter =true if a parameter Entity; otherwise =false.
     * @param   ndata       The notation associated with the binary Entity, or <var>null</var> if
     *                      the Entity is a text Entity.
     * @see com.ibm.xml.parser.ExternalID
     */
    public EntityDecl(String name, ExternalID externalID, boolean isParameter, String ndata) {
        this.name           = name;
        this.externalID     = externalID;
        this.isParameter    = isParameter;
        this.ndata          = ndata;
    }

    /**
     * Clone this EntityDecl using the appropriate factory.
     * <p>This method is defined by Child.
     * @return          Cloned EntityDecl.
     * @see com.ibm.xml.parser.Child#clone
     */
    public synchronized Object clone() {
        return cloneNode(true);
    }

    /**
     * Clone the EntityDecl node.
     */
    public synchronized Node cloneNode(boolean deep) {
        checkFactory();
        EntityDecl e = null;
        if (externalID != null) {
            e = factory.createEntityDecl(
                    name, 
                    new ExternalID(externalID.publicID, 
                                   externalID.systemID),
                    isParameter,
                    ndata);
        } else {
            e = factory.createEntityDecl(name, value, isParameter);
        }
        e.setFactory(getFactory());
        if (deep) {
            e.children.ensureCapacity(children.getLength());
            for (int i = 0;  i < children.getLength();  i ++)
                e.appendChild(children.item(i).cloneNode(true));
            e.parsed = this.parsed;
        }
        return e;
    }

    /**
     * Compare two EntityDecl nodes.
     */
    public boolean equals(Node arg, boolean deep) {
        if (arg == null)  return false;
        if (!(arg instanceof EntityDecl))  return false;
        
        EntityDecl ed = (EntityDecl)arg;
        
        if (!name.equals(ed.name)) return false;
        if (!((ed.value == null && this.value == null)
              || ed.value != null && ed.value.equals(this.value)))
            return false;
        if (externalID!=null) 
        {
            if (!externalID.equals(ed.externalID))
                return false;
            if (!((ed.externalID.systemID == null && externalID.systemID == null)
                  || ed.externalID.systemID != null && ed.externalID.systemID.equals(externalID.systemID)))
                return false;
            if (!((ed.externalID.publicID == null && externalID.publicID == null)
                  || ed.externalID.publicID != null && ed.externalID.publicID.equals(externalID.publicID)))
                return false;
        } else if (ed.externalID != null) return false;
        
        if (!((ed.ndata == null && ndata == null)
              || ed.ndata != null && ed.ndata.equals(ndata)))
            return false;
            
        if (deep) {
            if (!ed.children.equals(this.children, deep))
                return false;
        }        
        
        return true;
    }

    /**
     * Returns that this object is an Entity Node.
     * <p>This method is defined by DOM.
     * @return          Entity Node indicator.
     */
    public short getNodeType() {
        return Node.ENTITY_NODE;
    }

    /**
     * Returns the name associated with this Entity.
     * <p>This method is defined by DOM.
     */
    public String getNodeName() {
        return this.name;
    }

    /**
     * Returns the name associated with this Entity.
     * @deprecated Use getNodeName()
     * @return          Name of this entity.
     */
    public String getName() {
        return this.name;
    }

    /**
     * Returns whether this Entity is a parameter Entity.
     * @return          =true if an internal parameter Entity; otherwise =false.
     */
    public boolean isParameter() {
        return this.isParameter;
    }

    /**
     * Returns the value of this Entity.
     * @return              The XML-encoded value that was directly assigned to the internal Entity;
     *                      otherwise, <var>null</var>.
     */
    public String getValue() {
        return this.value;
    }

    /**
     * Returns the system identifier of the Notation.
     * A system identifier is a URI, which may be used to retrieve an external entity's content.
     * <p>This method is defined by DOM.
     * @return          The system identifier, or <var>null</var> if the identifier is not defined.
     * @see com.ibm.xml.parser.ExternalID#getSystemLiteral
     */
    public String getSystemId() {
        return this.externalID.getSystemLiteral();
    }

    /**
     * Sets the system identifier of the Notation.
     * A system identifier is a URI, which may be used to retrieve an external entity's content.
 * <p>This method is defined by DOM.
     *
     * @param   systemIdentifier    The system identifier.
     * @see com.ibm.xml.parser.ExternalID
    public void setSystemId(String systemIdentifier) {
        this.externalID = new ExternalID(this.externalID.getPubidLiteral(), systemIdentifier);
    }
     */

    /**
     * Returns the public identifier of the Notation.  This value is only valid if the
     * identifier is defined as <var>public</var> (as opposed to <var>system</var>).
     * Public identifiers may be used to try to generate an alternative URI in order to
     * retrieve the an external entities content.  If retrieval fails using
     * the public identifier, an attempt must be made to retrieve content using the system
     * identifier.
     * <p>This method is defined by DOM.
     * @return          The public identifier, or <var>null</var> if the identifier is not defined.
     * @see com.ibm.xml.parser.ExternalID
     */
    public String getPublicId() {
        return this.externalID.getPubidLiteral();
    }

    /**
     * Sets the public identifier of the Notation.  This value is only valid if the
     * identifier is defined as <var>public</var> (as opposed to <var>system</var>).
     * Public identifiers may be used to try to generate an alternative URI in order to
     * retrieve the an external entities content.  If retrieval fails using
     * the public identifier, an attempt must be made to retrieve content using the system
     * identifier.
     * <p>This method is defined by DOM.
     *
     * @param   publicIdentifier    The public identifier.
     * @see #getPublicId
     * @see com.ibm.xml.parser.ExternalID
    public void setPublicId(String publicIdentifier) {
        this.externalID = new ExternalID(publicIdentifier, this.externalID.getSystemLiteral());
    }
     */

    /**
     * Returns the external ID of this Entity.
     * @return              The reference(s) to the external entity to retrieve; otherwise,
     *                      <var>null</var>.
     * @see com.ibm.xml.parser.ExternalID
     */
    public ExternalID getExternalID() {
        return this.externalID;
    }

    /**
     * Returns whether this entity value is external.
     * @return              =true if entity is external; otherwise, =false.
     * @see com.ibm.xml.parser.ExternalID
     */
    public boolean isExternal() {
        return this.externalID != null;
    }

    /**
     * Returns the notation associated with this Entity.
     * @deprecated This method will be removed in future release. Use getNotationName().
     * @return              The notation associated with the external binary Entity, otherwise,
     *                      <var>null</var>.
     */
    public String getNDATAType() {
        return this.ndata;
    }

    /**
     * Returns the notation associated with this Entity.
     * <p>This method is defined by DOM.
     * @return              The notation associated with the external binary Entity, otherwise,
     *                      <var>null</var>.
     */
    public String getNotationName() {
        return this.ndata;
    }

    /**
     * <p>This method is defined by DOM.
     *
    public void setNotationName(String arg) {
        this.ndata = arg;
    }
     */

    /**
     * Returns whether there is a  notation associated with this entity value.
     * @return              =true if the external binary entity contains a notation; otherwise, =false.
     * @deprecated This method will be removed in future release.
     */
    public boolean isNDATA() {
        return this.ndata != null;
    }

    /**
     * Implements the accept operation of the visitor design pattern when the start of
     * an Entity Node is recognized when traversing the document object tree.
     * @param   visitor The implemention of the Visitor operation (toXMLString, digest, ...)
     * @exception Exception Thrown if this Node can not be visitted, or traversal modification is requested.
     * @see com.ibm.xml.parser.Visitor
     * @see com.ibm.xml.parser.TreeTraversal
     * @see com.ibm.xml.parser.NonRecursivePreorderTreeTraversal
     * @see com.ibm.xml.parser.TreeTraversalException
     */
    public void acceptPre(Visitor visitor) throws Exception {
        visitor.visitEntityDeclPre(this);
    }

    /**
     * Implements the accept operation of the visitor design pattern when the end of
     * an Entity Node is recognized when traversing the document object tree.
     * @param   visitor The implemention of the Visitor operation (toXMLString, digest, ...)
     * @exception Exception Thrown if this Node can not be visitted, or traversal modification is requested.
     * @see com.ibm.xml.parser.Visitor
     * @see com.ibm.xml.parser.TreeTraversal
     * @see com.ibm.xml.parser.NonRecursivePreorderTreeTraversal
     * @see com.ibm.xml.parser.TreeTraversalException
     */
    public void acceptPost(Visitor visitor) throws Exception {
        visitor.visitEntityDeclPost(this);
    }

    void load(Parser pa) throws IOException {
        if (getExternalID() != null && (null == this.rawByteStream || null == this.rawCharStream)) {
            try {
                Source src = pa.streamProducer.getInputStream(this.name, this.externalID.getPubidLiteral(),
                                                   this.externalID.getSystemLiteral());
                int ch;
                if (null != src.getInputStream()) {
                    ByteArrayOutputStream baos = new ByteArrayOutputStream();
                    InputStream is = src.getInputStream();
                    while (0 <= (ch = is.read()))
                        baos.write(ch);
                    is.close();
                    baos.close();
                    this.rawByteStream = baos.toByteArray();
                    this.encoding = src.getEncoding();
                } else {
                    CharArrayWriter caw = new CharArrayWriter();
                    Reader rr = src.getReader();
                    while (0 <= (ch = rr.read()))
                        caw.write(ch);
                    rr.close();
                    caw.close();
                    this.rawCharStream = caw.toCharArray();
                }
            } catch (IOException e) {
                this.rawByteStream = new byte[0];
                this.rawCharStream = new char[0];
                throw e;
            }
        }
    }

    Source getInputStream() {
        return null != this.rawByteStream
            ? new Source(new ByteArrayInputStream(this.rawByteStream), this.encoding)
            : new Source(new CharArrayReader(this.rawCharStream));
    }

    void setValue(String s) {
        this.value = s;
    }

    void setParsed(boolean p) {
        this.parsed = p;
    }

    boolean getParsed() {
        return this.parsed;
    }

    Token createToken(Parser p, Token tok) throws IOException {
        Token toks;
        if (getExternalID() != null) {
            try {
                load(p);
            } catch (FileNotFoundException e) {
                p.format1(tok, "E_IO0", getExternalID().getSystemLiteral());
            }
            toks = new Token(p, getExternalID().getSystemLiteral(), getInputStream());
        } else {
            toks = new Token(p, getValue());
            toks.getReading().setNext(tok.getReading());
            toks.setFollow(false);
        }
        return toks;
    }

    /**
     * Check whether <VAR>child</VAR> is allowed to be insered in this node or not.
     * When not allowed, a DOMException with HIERARCHY_REQUEST_ERR is thrown.
     */
    protected void checkChildType(Node child) throws DOMException {
        switch (child.getNodeType()) {
          case Node.ELEMENT_NODE:
          case Node.PROCESSING_INSTRUCTION_NODE:
          case Node.COMMENT_NODE:
          case Node.TEXT_NODE:
          case Node.CDATA_SECTION_NODE:
          case Node.ENTITY_REFERENCE_NODE:
          case Child.PSEUDONODE:
            break;
          default:
            throw new TXDOMException(DOMException.HIERARCHY_REQUEST_ERR,
                                     "Specified node type ("+child.getNodeType()
                                     +") can't be a child of EntityReference.");
        }
    }

    /**
     * Wrapper for DOM.
     */
    protected Entity getEntityImpl() {
        if (this.isParameter())
            throw new RuntimeException("XML4J internal error: EntityImpl for parameter entity.");
        return new EntityImpl(this);
    }

    static class EntityImpl implements Entity {
        EntityDecl decl;
        EntityImpl(EntityDecl ed) {
            this.decl = ed;
        }
        public Node getParentNode() {
            return null;
        }
        public String getPublicId() {
            return this.decl.getPublicId();
        }
        public String getSystemId() {
            return this.decl.getSystemId();
        }
        public String getNotationName() {
            return this.decl.getNotationName();
        }
        public String getNodeName() {
            return this.decl.getNodeName();
        }
        public String getNodeValue() {
            return null;
        }
        public void setNodeValue(String data) {
            throw new TXDOMException(DOMException.NO_MODIFICATION_ALLOWED_ERR,
                                     "This Entity is read-only.");
        }
        public short getNodeType() {
            return Node.ENTITY_NODE;
        }
        public NodeList getChildNodes() {
            return this.decl.getChildNodes();
        }
        public Node getFirstChild() {
            return this.decl.getFirstChild();
        }
        public Node getLastChild() {
            return this.decl.getLastChild();
        }
        public Node getPreviousSibling() {
            return this.decl.getPreviousSibling();
        }
        public Node getNextSibling() {
            return this.decl.getNextSibling();
        }
        public NamedNodeMap getAttributes() {
            return null;
        }
        public Document getOwnerDocument() {
            return this.decl.getOwnerDocument();
        }
        public Node insertBefore(Node c1, Node c2) {
            throw new TXDOMException(DOMException.NO_MODIFICATION_ALLOWED_ERR,
                                     "This Entity is read-only.");
        }
        public Node replaceChild(Node c1, Node c2) {
            throw new TXDOMException(DOMException.NO_MODIFICATION_ALLOWED_ERR,
                                     "This Entity is read-only.");
        }
        public Node removeChild(Node c1) {
            throw new TXDOMException(DOMException.NO_MODIFICATION_ALLOWED_ERR,
                                     "This Entity is read-only.");
        }
        public Node appendChild(Node c1) {
            throw new TXDOMException(DOMException.NO_MODIFICATION_ALLOWED_ERR,
                                     "This Entity is read-only.");
        }
        public boolean hasChildNodes() {
            return this.decl.hasChildNodes();
        }
        public Node cloneNode(boolean deep) {
            return ((EntityDecl)this.decl.cloneNode(deep)).getEntityImpl();
        }
    }
}
