/*
 * (C) Copyright IBM Corp. 1997-1998  All rights reserved.
 *
 * US Government Users Restricted Rights Use, duplication or
 * disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * IBM will not be liable for any damages suffered by you as a result
 * of using the Program. In no event will IBM be liable for any
 * special, indirect or consequential damages or lost profits even if
 * IBM has been advised of the possibility of their occurrence. IBM
 * will not be liable for any third party claims against you.
 */

package com.ibm.xml.parser;

/**
 * The ExternalID class provides support for external XML entities.  External IDs are used, 
 * for example, by DOM TXNotation Nodes.  External IDs always contain a <var>system ID</var> 
 * which is a URI.  However, external IDs may also include a <var>public ID</var> which may 
 * be used to try to generate an alternative URI. This can be very useful when a particular 
 * entity is already widely available, and the XML processor is can use the <var>public ID</var>
 * to index into a local catalog of resources to retrieve the entity value.  If valid, 
 * <var>public IDs</var> are used in preference to <var>system IDs</var>.
 *
 * @version Revision: 02 1.8 src/com/ibm/xml/parser/ExternalID.java, xml4jsrc, xml4j-jtcsv, xml4j_1_1_16 
 * @author TAMURA Kent &lt;kent@trl.ibm.co.jp&gt;
 * @see com.ibm.xml.parser.TXNotation
 */

public class ExternalID implements java.io.Serializable {
    
    static final long serialVersionUID = 4611817161873481189L;
    private static final int T_SYSTEM    =   0;
    private static final int T_PUBLIC    =   1;
                
                int         type        =   T_PUBLIC;
                String      publicID    =   null;
                String      systemID    =   null;

    /**
     * Constructor for system IDs.
     * @param   systemID    URI, which may be used to retrieve an external entity's content.
     */
    public ExternalID(String systemID) {
        this.type       = T_SYSTEM;
        this.publicID   = null;
        this.systemID   = systemID;
    }

    /**
     * Constructor for public and system IDs.
     * @param   publicID    Identifier to be used to try to generate an alternative URI in order to
     *                      retrieve the external entity's content, or <var>null</var> if
     *                      a system identitier is to be constructed.
     * @param   systemID    URI, which may be used to retrieve an external entity's content.
     */
    public ExternalID(String publicID, String systemID) {
        this.type       = T_PUBLIC;
        this.publicID   = publicID;
        this.systemID   = systemID;
        if (null == this.publicID)
            this.type = T_SYSTEM;
    }
    
    /**
     * Returns if this external ID is a system ID (or public ID).
     * @return          System ID=true, Public ID=false.
     * @see #isPublic         
     */
    public boolean isSystem() {
        return this.type == T_SYSTEM;
    }
    
    /**
     * Returns if this external ID is a public ID (or system ID).
     * @return          Public ID=true, System ID=false.
     * @see #isSystem         
     */
    public boolean isPublic() {
        return this.type == T_PUBLIC;
    }
    
    /**
     * Returns the system identifier of this external ID.  
     * A system identifier is a URI, which may be used to retrieve an external entity's content. 
     * @return          The system identifier, or <var>null</var> if the identifier is not defined.
     * @see #getPubidLiteral         
     */
    public String getSystemLiteral() {
        return this.systemID;
    }
    
    /**
     * Returns the public identifier of this external ID.  This value is only valid if the
     * identifier is defined as <var>public</var> (as opposed to <var>system</var>).
     * Public identifiers may be used to try to generate an alternative URI in order to
     * retrieve an external entity's content.  If retrieval fails using
     * the public identifier, an attempt must be made to retrieve content using the system
     * identifier.
     * @return          The public identifier, or <var>null</var> if the identifier is not defined.
     * @see #getSystemLiteral         
     */
    public String getPubidLiteral() {
        return this.publicID;
    }
    
    /**
     * Returns this external ID in the format it was declared: <CODE>SYSTEM &quot;<VAR>systemID</VAR>&quot;</CODE> or
     * <CODE>PUBLIC &quot;<VAR>publicID</VAR>&quot; &quot;<VAR>systemID</VAR>&quot;</CODE>.
     * @return          XML string representing the content of the external ID (never <var>null</var>).
     */
    public String toString() {
        String ret;
        if (isSystem()) {
            ret = "SYSTEM \""+getSystemLiteral()+"\"";
        } else if (null != getSystemLiteral()) {
            ret = "PUBLIC \""+getPubidLiteral()+"\" \""+getSystemLiteral()+"\"";
        } else {
            ret = "PUBLIC \""+getPubidLiteral()+"\""; // for NOTATION
        }            
        return ret;
    }

    /**
     *
     */
    public boolean equals(Object obj) {
        if (obj == null)  return false;
        if (!(obj instanceof ExternalID))  return false;
        ExternalID eid = (ExternalID)obj;
        if (!((eid.publicID == null && this.publicID == null)
              || eid.publicID != null && eid.publicID.equals(this.publicID)))
            return false;
        if (!((eid.systemID == null && this.systemID == null)
              || eid.systemID != null && eid.systemID.equals(this.systemID)))
            return false;
        return true;
    }

    
    /**
     *
     */
    public int hashCode() {
        int retval = 0;
        if (publicID != null)
            retval = publicID.hashCode();
        if (systemID != null)
            retval += systemID.hashCode();
        return retval;
    }

    
}
