/*
 * (C) Copyright IBM Corp. 1998  All rights reserved.
 *
 * US Government Users Restricted Rights Use, duplication or
 * disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * IBM will not be liable for any damages suffered by you as a result
 * of using the Program. In no event will IBM be liable for any
 * special, indirect or consequential damages or lost profits even if
 * IBM has been advised of the possibility of their occurrence. IBM
 * will not be liable for any third party claims against you.
 */

package com.ibm.xml.parser;

import java.io.Writer;
import java.util.Enumeration;
import java.util.Stack;
import org.w3c.dom.CDATASection;
import org.w3c.dom.Node;

/**
 * FormatPrintVisitor implements the Visitor interface in the visitor design pattern for the
 * purpose of supporting <code>TXDocument#printWithFormat</code> on the various DOM- and
 * XML4J-defined Nodes.
 * <p>Formatted printing is defined as:
 * <ul>
 * <li>Indenting all elements of an internal DTD subset definition
 * <li>Indenting nested TXElement nodes
 * <li>Allowing a single Node type per line of output (except for TXText nodes)
 * <li>Space is respected according to TXElement#isPreserveSpace and TXText#getIsIgnorableWhitespace
 * </ul>
 *
 * <p>The following sample code uses the FormatPrintVisitor on a hierarchy of nodes:
 * <pre>
 *
 * Visitor visitor = new FormatPrintVisitor(printWriter, encoding, indent);
 * new NonRecursivePreorderTreeTraversal(visitor).traverse(this);
 *
 * </pre>
 *
 * @version Revision: %M% %I% %W% %Q%
 * @author TAMURA Kent &lt;kent@trl.ibm.co.jp&gt;
 * @see com.ibm.xml.parser.TXDocument#printWithFormat
 * @see com.ibm.xml.parser.TXElement#isPreserveSpace
 * @see com.ibm.xml.parser.TXText#getIsIgnorableWhitespace
 * @see com.ibm.xml.parser.Visitor
 * @see com.ibm.xml.parser.TreeTraversal
 * @see com.ibm.xml.parser.NonRecursivePreorderTreeTraversal
 * @see com.ibm.xml.parser.ToXMLStringVisitor
 * @see com.ibm.xml.parser.util.HTMLPrintVisitor
 */
public class FormatPrintVisitor extends ToXMLStringVisitor implements Visitor {

    protected int     currentIndent =   0;
    protected int     indent        =   2;
    protected boolean ispreserve    =   false;
    protected Stack   preserves     =   new Stack();
    protected boolean isprevtext    =   false;

    /**
     * Constructor for customized encoding and indent.
     * @param writer        The character output stream to use.
     * @param encoding      Java character encoding in use by <VAR>writer</VAR>.
     * @param indent        Number of spaces to indent at each nesting level.
     */
    public FormatPrintVisitor(Writer writer, String encoding, int indent) {
        super(writer, encoding);
        this.indent   =   indent;
    }

    /**
     * Constructor for customized encoding.
     * @param writer        The character output stream to use.
     * @param encoding      Java character encoding in use by <VAR>writer</VAR>.
     */
    public FormatPrintVisitor(Writer writer, String encoding) {
        this(writer, encoding, 2);
    }

    /**
     * Constructor for default encoding.
     * @param writer  The character output stream to use.
     */
    public FormatPrintVisitor(Writer writer) {
        this(writer, null, 2);
    }

    /**
     * Flush the writer.
     * @param   document        Node to print with format.
     * @exception Exception     Thrown if this Node can not be visitted because of an invalid character output stream.
     * @see com.ibm.xml.parser.TXDocument
     */
    public void visitDocumentPost(TXDocument document) throws Exception {
        this.writer.write('\n');
        super.visitDocumentPost(document);
    }

    /**
     * Creates a formatted string representation of the start of the specified <var>element</var> Node
     * and its associated attributes in XML format, and directs it to the print writer.
     * @param   element         Node to print with format.
     * @exception Exception     Thrown if this Node can not be visitted because of an invalid character output stream.
     * @see com.ibm.xml.parser.TXElement
     */
    public void visitElementPre(TXElement element) throws Exception {
        if (!this.ispreserve && !this.isprevtext)  Util.indent(this.writer, this.currentIndent);
        super.visitElementPre(element);
        this.currentIndent += this.indent;
        if (element.hasChildNodes()) {
            Child ch = (Child)element.getFirstChild();
            this.preserves.push(new Boolean(this.ispreserve));
            if (1 == element.children.getLength() && ch.getNodeType() == Node.TEXT_NODE) {
                this.ispreserve = true;
            } else {
                this.ispreserve = element.isPreserveSpace();
            }
        }
        this.isprevtext = false;
    }

    /**
     * Creates a formatted string representation of the end of the specified <var>element</var> Node
     * in XML format, and directs it to the print writer.
     * @param   element         Node to print with format.
     * @exception Exception     Thrown if this Node can not be visitted because of an invalid character output stream.
     * @see com.ibm.xml.parser.TXElement
     */
    public void visitElementPost(TXElement element) throws Exception {
        this.currentIndent -= this.indent;
        if (element.hasChildNodes()) {
            if (!this.ispreserve && !this.isprevtext)  Util.indent(this.writer, this.currentIndent);
            this.writer.write("</");
            this.writer.write(element.getTagName());
            this.writer.write(">");
            this.ispreserve = ((Boolean)this.preserves.pop()).booleanValue();
        }
        this.isprevtext = false;
    }

    /**
     * Creates a formatted string representation of the specified <var>pi</var> Node in XML 
     * format, and directs it to the print writer.
     * @param     pi            Node to print with format.
     * @exception Exception     Thrown if this Node can not be visitted because of an invalid character output stream.
     * @see com.ibm.xml.parser.TXPI
     */
    public void visitPIPre(TXPI pi) throws Exception {
        if (!this.ispreserve && !this.isprevtext)  Util.indent(this.writer, this.currentIndent);
        super.visitPIPre(pi);
    }

    /**
     * Creates a formatted string representation of the specified <var>comment</var> Node in 
     * XML format, and directs it to the print writer.
     * @param     comment       Node to print with format.
     * @exception Exception     Thrown if this Node can not be visitted because of an invalid character output stream.
     * @see com.ibm.xml.parser.TXComment
     */
    public void visitCommentPre(TXComment comment) throws Exception {
        if (!this.ispreserve && !this.isprevtext)  Util.indent(this.writer, this.currentIndent);
        super.visitCommentPre(comment);
    }

    /**
     * Creates a formatted string representation of the specified <var>text</var> Node in XML 
     * format, and directs it to the print writer.  CDATASections are respected.
     * @param     text          Node to print with format.
     * @exception Exception     Thrown if this Node can not be visitted because of an invalid character output stream.
     * @see com.ibm.xml.parser.TXText
     */
    public void visitTextPre(TXText text) throws Exception {
        if (!this.ispreserve && text.getIsIgnorableWhitespace()) {
            this.isprevtext = false;
            return;
        }
        if (text instanceof CDATASection) {
            if (!this.ispreserve && !this.isprevtext)
                Util.indent(this.writer, this.currentIndent);
            this.writer.write("<![CDATA[");
            this.writer.write(text.getData());
            this.writer.write("]]>");
        } else {
            this.writer.write(Util.backReference(text.getData(), this.encoding));
            this.isprevtext = true;
        }
    }

    /**
     * Creates a formatted string representation of the start of the specified <var>dtd</var> Node
     * in XML format, and directs it to the print writer.
     * The internal DTD subset will be included based on the value of <code>isPrintInternalDTD()</code>.
     * @param     dtd           Node to print with format.
     * @exception Exception     Thrown if this Node can not be visitted because of an invalid character output stream.
     * @see com.ibm.xml.parser.DTD#isPrintInternalDTD
     * @see com.ibm.xml.parser.DTD
     */
    public void visitDTDPre(DTD dtd) throws Exception {
        Util.indent(this.writer, this.currentIndent);
        this.writer.write("<!DOCTYPE ");
        this.writer.write(dtd.getName());
        this.writer.write(" ");
        if (null != dtd.getExternalID()) {
            this.writer.write(dtd.getExternalID().toString());
            this.writer.write(" ");
        }
        this.currentIndent += this.indent;
        if (dtd.isPrintInternalDTD() && 0 < dtd.getInternalSize()) {
            this.writer.write("[\n");
            for (Node node = dtd.getFirstChild();
                 node != null;
                 node = node.getNextSibling()) {
                if (node.getNodeType() == Node.TEXT_NODE) continue;
                try {
                    ((Child)node).acceptPre(this);
                } catch (ToNextSiblingTraversalException tnste) {
                                                // visitEntityPre() always throws it.
                } // Ignore.
            }
        }
        throw new ToNextSiblingTraversalException();
    }

    /**
     * Creates a formatted string representation of the end of the specified <var>dtd</var> Node
     * in XML format, and directs it to the print writer.
     * @param     dtd           Node to print with format.
     * @exception Exception     Thrown if this Node can not be visitted because of an invalid character output stream.
     * @see com.ibm.xml.parser.DTD
     */
    public void visitDTDPost(DTD dtd) throws Exception {
        this.currentIndent -= this.indent;
        if (dtd.isPrintInternalDTD() && 0 < dtd.getInternalSize()) {
            Util.printSpace(this.writer, this.currentIndent);
            this.writer.write("]");
        }
        this.writer.write(">");
    }

    /**
     * Creates a formatted string representation of the specified <var>elementDecl</var> Node 
     * in XML format, and directs it to the print writer.
     * @param     elementDecl   Node to print with format.
     * @exception Exception     Thrown if this Node can not be visitted because of an invalid character output stream.
     * @see com.ibm.xml.parser.ElementDecl
     */
    public void visitElementDeclPre(ElementDecl elementDecl) throws Exception {
        Util.printSpace(this.writer, this.currentIndent);
        super.visitElementDeclPre(elementDecl);
        this.writer.write('\n');
    }

    /**
     * Creates a formatted string representation of the specified <var>attlist</var> Node and any
     * associated AttDefs in XML format, and directs it to the print writer.
     * @param     attlist       Node to print with format.
     * @exception Exception     Thrown if this Node can not be visitted because of an invalid character output stream.
     * @see com.ibm.xml.parser.Attlist
     */
    public void visitAttlistPre(Attlist attlist) throws Exception {
        Util.printSpace(this.writer, this.currentIndent);
        super.visitAttlistPre(attlist);
        this.writer.write('\n');
    }

    /**
     * Creates a formatted string representation of the specified <var>entityDecl</var> Node in 
     * XML format, and directs it to the print writer.
     * @param     entityDecl    Node to print with format.
     * @exception Exception     Thrown if this Node can not be visitted because of an invalid character output stream.
     * @see com.ibm.xml.parser.EntityDecl
     */
    public void visitEntityDeclPre(EntityDecl entityDecl) throws Exception {
        Util.printSpace(this.writer, this.currentIndent);
        try {
            super.visitEntityDeclPre(entityDecl);
        } catch (ToNextSiblingTraversalException tnste) {} // ignore
        this.writer.write('\n');
        throw new ToNextSiblingTraversalException();
    }

    /**
     * Creates a formatted string representation of the specified <var>notation</var> Node in 
     * XML format, and directs it to the print writer.
     * @param     notation      Node to print with format.
     * @exception Exception     Thrown if this Node can not be visitted because of an invalid character output stream.
     * @see com.ibm.xml.parser.TXNotation
     */
    public void visitNotationPre(TXNotation notation) throws Exception {
        Util.printSpace(this.writer, this.currentIndent);
        super.visitNotationPre(notation);
        this.writer.write('\n');
    }
}  //FormatPrintVisitor
