/*
 * (C) Copyright IBM Corp. 1997-1998  All rights reserved.
 *
 * US Government Users Restricted Rights Use, duplication or
 * disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * IBM will not be liable for any damages suffered by you as a result
 * of using the Program. In no event will IBM be liable for any
 * special, indirect or consequential damages or lost profits even if
 * IBM has been advised of the possibility of their occurrence. IBM
 * will not be liable for any third party claims against you.
 */

package com.ibm.xml.parser;

import org.w3c.dom.DOMException;
import org.w3c.dom.EntityReference;
import org.w3c.dom.Node;

/**
 * GeneralReference implements EntityReference as defined by the Document Object Model (DOM).
 * This class represents general entity references (&amp;foo;) occured in character data
 * or attribute values.
 *
 * <p>GeneralReferences do NOT include character references(e.g. &quot;&lt;&quot; is 
 * referenced as &quot;&amp;#60;&quot;).
 *
 * <p>The object tree produced by the XML4J parser doesn't include GeneralReference Nodes
 * if the parser is configured as <CODE>setExpandEntityReferences(false);</CODE>.
 *
 * @version Revision: 05 1.14 src/com/ibm/xml/parser/GeneralReference.java, xml4jsrc, xml4j-jtcsv, xml4j_1_1_16 
 * @author TAMURA Kent &lt;kent@trl.ibm.co.jp&gt;
 * @see com.ibm.xml.parser.Parent
 * @see com.ibm.xml.parser.Child
 * @see com.ibm.xml.parser.Parser#setExpandEntityReferences
 * @see com.ibm.xml.parser.ReferenceHandler
 */
public class GeneralReference extends Parent implements EntityReference {

    static final long serialVersionUID = 3555190050476479048L;
    private String  name    =   null;

    /**
     * Constructor.
     * @param name  This General Reference's name.  This is also the name of the entity
     *              being referred to by the General Reference.
     */
    public GeneralReference(String name) {
        this.name = name;
    }

    /**
     * Clone this General Reference Node and its children using the appropriate factories.
     * <p>This method is defined by Child.
     * @return          Cloned General Reference Node.
     * @see com.ibm.xml.parser.Child#clone
     */
    public Object clone() {
        return cloneNode(true);
    }

    /**
     * <p>This method is defined by DOM.
     *
     */
    public synchronized Node cloneNode(boolean deep) {
        checkFactory();
        GeneralReference gr = (GeneralReference)factory.createEntityReference(getName());
        if (deep) {
            gr.children.ensureCapacity(children.getLength());
            for (int i = 0;  i < children.getLength();  i ++)
                gr.appendChild(children.item(i).cloneNode(true));
        }
        return gr;
    }

    /**
     * <p>This method is defined by DOM.
     *
     */
    public synchronized boolean equals(Node arg, boolean deep) {
        if (arg == null)  return false;
        if (!(arg instanceof GeneralReference))  return false;
        GeneralReference gr = (GeneralReference)arg;
        if (!gr.getName().equals(this.getName()))  return false;
        if (deep) {
            if (!gr.children.equals(this.children, deep))
                return false;
        }
        return true;
    }

    /**
     * Returns that this object is a EntityReference Node.
     * <p>This method is defined by DOM.
     * @return          EntityReference Node indicator.
     */
    public short getNodeType() {
        return Node.ENTITY_REFERENCE_NODE;
    }

    /**
     * <p>This method is defined by DOM.
     *
     */
    public String getNodeName() {
        return getName();
    }

    /**
     * Returns this General Reference's name.
     * @return          The string that is this General Reference's name (should never
     *                  be <var>null</var>).
     */
    public String getName() {
        return this.name;
    }

    /**
     * Returns the XML language ID (the value of the <code>xml:lang</code> Attribute) in
     * use for this General Reference Node.  If this General Reference Node has no explicitly
     * defined language Attribute, Parent Nodes will be recursively checked.
     * @return          The XML language ID, or <var>null</var> if all Parents have no language ID.
     */
    public String getLanguage() {
        if (null == parent)  return null;
        if (parent instanceof TXElement)
            return ((TXElement)parent).getLanguage();
        if (parent instanceof GeneralReference)
            return ((GeneralReference)parent).getLanguage();
        return null;
    }
    
    /**
     * Implements the accept operation of the visitor design pattern when the start of
     * a General Reference Node is recognized when traversing the document object tree. 
     * @param   visitor The implemention of the Visitor operation (toXMLString, digest, ...)
     * @exception Exception Thrown if this Node can not be visitted, or traversal modification is requested.
     * @see com.ibm.xml.parser.Visitor
     * @see com.ibm.xml.parser.TreeTraversal
     * @see com.ibm.xml.parser.NonRecursivePreorderTreeTraversal
     * @see com.ibm.xml.parser.TreeTraversalException
     */
    public void acceptPre(Visitor visitor) throws Exception {
        visitor.visitGeneralReferencePre(this);
    }

    /**
     * Implements the accept operation of the visitor design pattern when the end of
     * a General Reference Node is recognized when traversing the document object tree. 
     * @param   visitor The implemention of the Visitor operation (toXMLString, digest, ...)
     * @exception Exception Thrown if this Node can not be visitted, or traversal modification is requested.
     * @see com.ibm.xml.parser.Visitor
     * @see com.ibm.xml.parser.TreeTraversal
     * @see com.ibm.xml.parser.NonRecursivePreorderTreeTraversal
     * @see com.ibm.xml.parser.TreeTraversalException
     */
    public void acceptPost(Visitor visitor) throws Exception {
        visitor.visitGeneralReferencePost(this);
    }

    /**
     * Check whether <VAR>child</VAR> is allowed to be insered in this node or not.
     * When not allowed, a DOMException with HIERARCHY_REQUEST_ERR is thrown.
     */
    protected void checkChildType(Node child) throws DOMException {
        switch (child.getNodeType()) {
          case Node.ELEMENT_NODE:
          case Node.PROCESSING_INSTRUCTION_NODE:
          case Node.COMMENT_NODE:
          case Node.TEXT_NODE:
          case Node.CDATA_SECTION_NODE:
          case Node.ENTITY_REFERENCE_NODE:
          case Child.PSEUDONODE:
            break;
          default:
            throw new TXDOMException(DOMException.HIERARCHY_REQUEST_ERR,
                                     "Specified node type ("+child.getNodeType()
                                     +") can't be a child of EntityReference.");
        }
    }

    /**
     * Call <CODE>collectNamespaceAttributes()</CODE> for each <CODE>TXElement</CODE> child.
     * @see com.ibm.xml.parser.TXDocument#isProcessNamespace
     * @see com.ibm.xml.parser.TXElement#collectNamespaceAttributes
     */
    public void collectNamespaceAttributes() {
        this.collectNamespaceAttributes(this.getParentNode());
    }

    /**
     * Call <CODE>collectNamespaceAttributes()</CODE> for each <CODE>TXElement</CODE> child.
     * @see com.ibm.xml.parser.TXDocument#isProcessNamespace
     * @see com.ibm.xml.parser.TXElement#collectNamespaceAttributes
     */
    public void collectNamespaceAttributes(Node parent) {
        if (!getFactory().isProcessNamespace())  return;
        for (Node node = this.getFirstWithoutReference();
             node != null;
             node = ((Child)node).getNextWithoutReference()) {
            if (node.getNodeType() == Node.ELEMENT_NODE)
                ((TXElement)node).collectNamespaceAttributes(parent);
        }
    }

    /**
     * Call <CODE>removeOverlappedNamespaceAttributes()</CODE> for each <CODE>TXElement</CODE> child.
     * @see com.ibm.xml.parser.TXDocument#isProcessNamespace
     * @see com.ibm.xml.parser.TXElement#removeOverlappedNamespaceAttributes
     */
    public void removeOverlappedNamespaceAttributes() {
        if (!getFactory().isProcessNamespace())  return;
        for (Node node = this.getFirstWithoutReference();
             node != null;
             node = ((Child)node).getNextWithoutReference()) {
            if (node.getNodeType() == Node.ELEMENT_NODE)
                ((TXElement)node).removeOverlappedNamespaceAttributes();
        }
    }
}
