/*
 * (C) Copyright IBM Corp. 1998  All rights reserved.
 *
 * US Government Users Restricted Rights Use, duplication or
 * disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * IBM will not be liable for any damages suffered by you as a result
 * of using the Program. In no event will IBM be liable for any
 * special, indirect or consequential damages or lost profits even if
 * IBM has been advised of the possibility of their occurrence. IBM
 * will not be liable for any third party claims against you.
 */

package com.ibm.xml.parser;

import java.io.Writer;
import java.util.Enumeration;
import java.io.ByteArrayOutputStream;
import java.io.DataOutputStream;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Hashtable;
import java.util.Vector;
import org.w3c.dom.Node;
import org.w3c.dom.Text;

/**
 * MakeDigestVisitor implements the Visitor interface in the visitor design pattern for the
 * purpose of calculating and storing a digest on the various DOM- and XML4J-defined Nodes as the document
 * object tree is traversed.  Digests are calculated usng the message digest passed on the constructor.
 * <p>Digests can be made for the following Node types:
 * <ul>
 * <li>TXElement
 * <li>TXAttribute
 * <li>TXPI
 * <li>TXComment
 * <li>TXText
 * </ul>
 * <p>The following sample code uses the MakeDigestVisitor on a hierarchy of nodes:
 * <pre>
 *
 * MakeDigestVisitor makeDigestVisitor = new MakeDigestVisitor(this.getFactory().createMessageDigest());
 * new NonRecursivePreorderTreeTraversal(makeDigestVisitor).traverse(this);
 *
 * </pre>
 *
 * @version Revision: 45 1.12 src/com/ibm/xml/parser/MakeDigestVisitor.java, xml4jsrc, xml4j-jtcsv, xml4j_1_1_16 
 * @author ROWE Tom &lt;trowe@us.ibm.com&gt;
 * @see com.ibm.xml.parser.Visitor
 * @see com.ibm.xml.parser.NOOPVisitor
 * @see com.ibm.xml.parser.TreeTraversal
 * @see com.ibm.xml.parser.NonRecursivePreorderTreeTraversal
 */
public class MakeDigestVisitor extends NOOPVisitor implements Visitor {

    protected MessageDigest messageDigest   =   null;

    /**
     * Constructor.
     * @param   messageDigest    Message digest instance to use when visitting.
     * @see com.ibm.xml.parser.TXDocument#createMessageDigest
     */
    public MakeDigestVisitor(MessageDigest messageDigest) {
        this.messageDigest = messageDigest;
    }

    /**
     * If the specified <var>element</var> already has a calculated digest, throw an
     * exception to avoid unnecessary traversal of children.
     * @param   element     Node to calculate and set a digest.
     * @exception Exception  Thrown if this Node already has a digest.
     * @see com.ibm.xml.parser.TXElement
     * @see #visitElementPost
     */
    public void visitElementPre(TXElement element) throws Exception {
        if (element.getVisitorDigest() != null) {
            throw new ToNextSiblingTraversalException();
        }
    }

    /**
     * Calculates and sets a digest for the specified <var>element</var>.
     * <p>A digest of a TXElement Node consists of its:
     * <ul>
     * <li>Node type
     * <li>Name
     * <li>Number of attributes
     * <li>Attribute digests (orderred according to the attribute's name (using <CODE>String#compareTo()</CODE>).
     * <li>Number of immediate children (consecutive Child Text Nodes are counted as 1).
     * <li>Immediate children digests (digests for consecutive Child Text Nodes are combined into a single digest).
     * </ul>
     * @param   element     Node to calculate and set a digest.
     * @exception Exception Thrown if this Node can not be visitted because of a problem
     *                      in calculating the digest.  Often, this is because the
     *                      message digest specified on the constructor is invalid,
     *                      the JVM does not support <code>UnicodeBigUnmarked</code> encoding,
     *                      or the JVM does not have the required security providers
     *                      for the message digest.
     * @see com.ibm.xml.parser.TXElement
     * @see #visitElementPre
     */
    public void visitElementPost(TXElement element) throws Exception {
        if (element.getVisitorDigest() == null) {
            synchronized (element) {
                try {
                    ByteArrayOutputStream baos = new ByteArrayOutputStream();
                    DataOutputStream dos = new DataOutputStream(baos);
                    dos.writeInt(Node.ELEMENT_NODE); // Node type
                    dos.write(getExpandedName(element).getBytes("UnicodeBigUnmarked")); // Name
                    dos.write((byte)0);
                    dos.write((byte)0);
                    
                    if (null == element.attributes) { // Number of attributes
                        dos.writeInt(0);
                    } else {
                        int atlen = element.attributes.getLength();
                        int lenWithoutXmlns = 0;
                        Hashtable hash = new Hashtable(atlen);
                        
                        String[] as = new String[atlen]; // Attribute digests orderred by expanded name
                        for (int i = 0;  i < atlen;  i ++) {
                            Node attr = element.attributes.item(i);
                            String qname = attr.getNodeName();
                            if (!qname.equals("xmlns") && !qname.startsWith("xmlns:")) {
                                String expandedName = getExpandedName(attr);
                                hash.put(expandedName, qname);
                                as[lenWithoutXmlns++] = expandedName;
                            }
                        }
                        dos.writeInt(lenWithoutXmlns);
                        Util.heapSort(as, lenWithoutXmlns);
                        for (int i = 0;  i < lenWithoutXmlns;  i ++) {
                            this.messageDigest.reset();
                            TXAttribute txa = (TXAttribute)element.getAttributeNode((String)hash.get(as[i]));
                            visitAttributePre(txa);
                            dos.write(txa.getVisitorDigest());
                        }
                    }
                    
                                                // Collect children digests
                    Vector v = new Vector();
                    StringBuffer sb = new StringBuffer();
                    for (Child n = (Child)element.getFirstWithoutReference();
                         n != null;
                         n = (Child)n.getNextWithoutReference()) {
                                        // Child digests should all be set because this is ElementPost
                        byte[] digest0 = n.getVisitorDigest();
                                        // This condition isn't n.getNodeType() == Node.TEXT_NODE.
                                        // Includes CDATA
                        if (n instanceof Text) {
                            sb.append(n.getNodeValue());
                            while (null != n.getNextWithoutReference()
                                   && n.getNextWithoutReference() instanceof Text) {
                                n = (Child)n.getNextWithoutReference();
                                digest0 = n.getVisitorDigest(); 
                                sb.append(n.getNodeValue());
                            }
                            if (0 < sb.length()) {
                                TXText tempText = new TXText(sb.toString());
                                this.messageDigest.reset();
                                visitTextPre(tempText);
                                digest0 = tempText.getVisitorDigest();
                            }
                            sb.setLength(0);
                        }
                        if (n.getNodeType() != Node.COMMENT_NODE) //Include no Comment (1999.2.23 kent)
                            v.addElement(digest0);
                    }
                    dos.writeInt(v.size()); // Number of children
                    for (int i = 0;  i < v.size();  i ++)
                        dos.write((byte[])v.elementAt(i)); // Children digests
                    dos.close();

                    this.messageDigest.reset();
                    element.setVisitorDigest(this.messageDigest.digest(baos.toByteArray()));
                    
                } catch (Exception exception) {
                    throw new LibraryException("com.ibm.xml.parser.MakeDigestVisitor#visitElementPost(): "+exception.toString());
                } // try-catch
            } // synchronized
        } // hasDigest
    }

    /**
     * Calculates and sets a digest for the specified <var>attribute</var>.
     * <p>A digest of a TXAttribute Node consists of its Node type, its name, and its value.
     * <p>Note that TXAttribute Nodes are not parsed into the document object hierarchy by the
     * XML4J parser; attributes exist as part of a TXElement Node.
     * @param   attribute   Node to calculate and set a digest.
     * @exception Exception Thrown if this Node can not be visitted because of a problem
     *                      in calculating the digest.  Often, this is because the
     *                      message digest specified on the constructor is invalid,
     *                      the JVM does not support <code>UnicodeBigUnmarked</code> encoding,
     *                      or the JVM does not have the required security providers
     *                      for the message digest.
     * @see com.ibm.xml.parser.TXAttribute
     */
    public void visitAttributePre(TXAttribute attribute) throws Exception {
        if (attribute.getVisitorDigest() == null) {
            String qname = attribute.getNodeName();
            if (qname.equals("xmlns") || qname.startsWith("xmlns:"))
                return;
            synchronized (attribute) {
                try {
                    this.messageDigest.reset();
                    this.messageDigest.update((byte)((Node.ATTRIBUTE_NODE >> 24) & 0xff));
                    this.messageDigest.update((byte)((Node.ATTRIBUTE_NODE >> 16) & 0xff));
                    this.messageDigest.update((byte)((Node.ATTRIBUTE_NODE >> 8) & 0xff));
                    this.messageDigest.update((byte)(Node.ATTRIBUTE_NODE & 0xff));
                    this.messageDigest.update(getExpandedName(attribute).getBytes("UnicodeBigUnmarked"));
                    this.messageDigest.update((byte)0);
                    this.messageDigest.update((byte)0);
                    attribute.setVisitorDigest(this.messageDigest.digest(attribute.getValue().getBytes("UnicodeBigUnmarked")));
                } catch (Exception exception) {
                    throw new LibraryException("com.ibm.xml.parser.MakeDigestVisitor#visitAttributePre(): "+exception.toString());
                } // try-catch
            } // synchronized
        } // hasDigest
    }

    /**
     * Calculates and sets a digest for the specified <var>pi</var>.
     * <p>A digest of a TXPI Node consists of its Node type, its name, and its data.
     * @param   pi          Node to calculate and set a digest.
     * @exception Exception Thrown if this Node can not be visitted because of a problem
     *                      in calculating the digest.  Often, this is because the
     *                      message digest specified on the constructor is invalid,
     *                      the JVM does not support <code>UnicodeBigUnmarked</code> encoding,
     *                      or the JVM does not have the required security providers
     *                      for the message digest.
     * @see com.ibm.xml.parser.TXPI
     */
    public void visitPIPre(TXPI pi) throws Exception {
        if (pi.getVisitorDigest() == null) {
            synchronized (pi) {
                try {
                    this.messageDigest.reset();
                    this.messageDigest.update((byte)((Node.PROCESSING_INSTRUCTION_NODE >> 24) & 0xff));
                    this.messageDigest.update((byte)((Node.PROCESSING_INSTRUCTION_NODE >> 16) & 0xff));
                    this.messageDigest.update((byte)((Node.PROCESSING_INSTRUCTION_NODE >> 8) & 0xff));
                    this.messageDigest.update((byte)(Node.PROCESSING_INSTRUCTION_NODE & 0xff));
                    this.messageDigest.update(pi.getNodeName().getBytes("UnicodeBigUnmarked"));
                    this.messageDigest.update((byte)0);
                    this.messageDigest.update((byte)0);
                    pi.setVisitorDigest(this.messageDigest.digest(pi.getData().getBytes("UnicodeBigUnmarked")));
                } catch (Exception exception) {
                    throw new LibraryException("com.ibm.xml.parser.MakeDigestVisitor#visitPIPre(): "+exception.toString());
                } // try-catch
            } // synchronized
        } // hasDigest
    }

    /**
     * Calculates and sets a digest for the specified <var>comment</var>.
     * <p>A digest of a TXComment Node consists of its Node type and its data.
     * @param   comment     Node to calculate and set a digest.
     * @exception Exception Thrown if this Node can not be visitted because of a problem
     *                      in calculating the digest.  Often, this is because the
     *                      message digest specified on the constructor is invalid,
     *                      the JVM does not support <code>UnicodeBigUnmarked</code> encoding,
     *                      or the JVM does not have the required security providers
     *                      for the message digest.
     * @see com.ibm.xml.parser.TXComment
     */
    public void visitCommentPre(TXComment comment) throws Exception {
        if (comment.getVisitorDigest() == null) {
            synchronized (comment) {
                try {
                    this.messageDigest.reset();
                    this.messageDigest.update((byte)((Node.COMMENT_NODE >> 24) & 0xff));
                    this.messageDigest.update((byte)((Node.COMMENT_NODE >> 16) & 0xff));
                    this.messageDigest.update((byte)((Node.COMMENT_NODE >> 8) & 0xff));
                    this.messageDigest.update((byte)(Node.COMMENT_NODE & 0xff));
                    comment.setVisitorDigest(this.messageDigest.digest(comment.getData().getBytes("UnicodeBigUnmarked")));
                } catch (Exception exception) {
                    throw new LibraryException("com.ibm.xml.parser.MakeDigestVisitor#visitCommentPre(): "+exception.toString());
                } // try-catch
            } // synchronized
        } // hasDigest
    }

    /**
     * Calculates and sets a digest for the specified <var>text</var>.
     * <p>A digest of a TXText Node consists of its Node type and its data.
     * @param   text        Node to calculate and set a digest.
     * @exception Exception Thrown if this Node can not be visitted because of a problem
     *                      in calculating the digest.  Often, this is because the
     *                      message digest specified on the constructor is invalid,
     *                      the JVM does not support <code>UnicodeBigUnmarked</code> encoding,
     *                      or the JVM does not have the required security providers
     *                      for the message digest.
     * @see com.ibm.xml.parser.TXText   
     */
    public void visitTextPre(TXText text) throws Exception {
        if (text.getVisitorDigest() == null) {
            synchronized (text) {
                try {
                    this.messageDigest.reset();
                    this.messageDigest.update((byte)((Node.TEXT_NODE >> 24) & 0xff));
                    this.messageDigest.update((byte)((Node.TEXT_NODE >> 16) & 0xff));
                    this.messageDigest.update((byte)((Node.TEXT_NODE >> 8) & 0xff));
                    this.messageDigest.update((byte)(Node.TEXT_NODE & 0xff));
                    text.setVisitorDigest(this.messageDigest.digest(text.getData().getBytes("UnicodeBigUnmarked")));
                } catch (Exception exception) {
                    throw new LibraryException("com.ibm.xml.parser.MakeDigestVisitor#visitTextPre(): "+exception.toString());
                } // try-catch
            } // synchronized
        } // hasDigest
    }

    /**
     * Do nothing.
     */
    public void visitGeneralReferencePre(GeneralReference generalReference) throws Exception {
    }

    /**
     *
     * @exceptin LibraryException node is neither TXElement nor TXAttribute.
     */
    public static String getExpandedName(Node node) {
        String ret = null;
        if (node.getNodeType() == Node.ELEMENT_NODE) {
            TXDocument owner = (TXDocument)node.getOwnerDocument();
            synchronized (owner) {
                String originalSeparator = owner.expandedNameSeparator;
                owner.expandedNameSeparator = ":";
                ret = ((TXElement)node).createExpandedName();
                owner.expandedNameSeparator = originalSeparator;
            }
        } else if (node.getNodeType() == Node.ATTRIBUTE_NODE) {
            String name = node.getNodeName();
            int index = name.indexOf(':');
            if (index < 0) {
                ret = name;
            } else {
                ret = ((TXAttribute)node).getNSName()+":"+((TXAttribute)node).getNSLocalName();
            }
        } else {
            throw new LibraryException("com.ibm.xml.parser.MakeDigestVisitor#getExpandedName(): Unsupported node type: "+node.getNodeType());
        }
        if (ret != null)  ret = ret.intern();
        return ret;
    }
}  //MakeDigestVisitor
