/*
 * (C) Copyright IBM Corp. 1998  All rights reserved.
 *
 * US Government Users Restricted Rights Use, duplication or
 * disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * IBM will not be liable for any damages suffered by you as a result
 * of using the Program. In no event will IBM be liable for any
 * special, indirect or consequential damages or lost profits even if
 * IBM has been advised of the possibility of their occurrence. IBM
 * will not be liable for any third party claims against you.
 */
 
package com.ibm.xml.parser;

import org.w3c.dom.Node;

/**
 * NonRecursivePreorderTreeTraversal defines a specific document object tree traversal algorithm
 * for use by the visitor design pattern. This algorithm visits the Parent before visiting
 * its children.
 *
 * @version Revision: %M% %I% %W% %Q%
 * @author ROWE Tom &lt;trowe@us.ibm.com&gt;
 * @see com.ibm.xml.parser.TreeTraversal
 * @see com.ibm.xml.parser.util.RecursivePreorderTreeTraversal
 */
public class NonRecursivePreorderTreeTraversal extends TreeTraversal {
    
    /**
     * Constructor.
     * @param   visitor The implemention of the Visitor operation (toXMLString, digest, ...)
     * @see com.ibm.xml.parser.Visitor
     */
    public NonRecursivePreorderTreeTraversal(Visitor visitor) {
        super(visitor);
    }

    /**
     * Traverses the document object tree at the specified <var>startNode</var>.
     * <p>The numbered nodes in the trees below indicate the order of traversal given
     * the specified <code>startNode</code> of &quot;1&quot;.
     * <pre>
     *
     *                 1              x              x
     *                / \            / \            / \
     *               2   6          1   x          x   x
     *              /|\   \        /|\   \        /|\   \
     *             3 4 5   7      2 3 4   x      x 1 x   x
     *
     * </pre>
     * @param   startNode   The starting point to begin traversing the document object tree.
     * @exception Exception Thrown if the document hierarchy can not be visitted.
     */
    public void traverse(Node startNode) throws Exception {
        Node    tempNode1       = startNode;
        Node    tempNode2       = null;
        try {
            outer: while (tempNode1 != null) {
                try {
                    ((Visitee) tempNode1).acceptPre(getVisitor());  // Always pre-visit
                    tempNode2 = tempNode1.getFirstChild();
                } catch (ToNextSiblingTraversalException exception) {
                    tempNode2 = null;                           // Treat this node like a leaf.
                }
                if (tempNode2 == null) {                        // If current node is a leaf node..
                    //if (!(tempNode1.equals(startNode))) {       // If not moving past stop node.. 
                    if (tempNode1 != startNode) {               // If not moving past stop node.. 
                        tempNode2 = tempNode1.getNextSibling(); // ..get leaf node's sibling 
                        if (tempNode2 != null) {                // If more siblings..
                            try {
                                ((Visitee) tempNode1).acceptPost(getVisitor()); // ..post visit this leaf.
                            } catch (ToNextSiblingTraversalException exception) {
                                // Do nothing because we're a leaf.
                            }
                        }
                    }
                    else {                                      // This leaf is the final node.
                        try {
                            ((Visitee) tempNode1).acceptPost(getVisitor()); // ..post visit final node.
                        } catch (ToNextSiblingTraversalException exception) {
                            // Do nothing because we're finished anyway.
                        }
                        break outer;                                             
                    }
                } // Current node is a leaf node
                while (tempNode2 == null && tempNode1 != null) {// While leaf node is the last sibling..
                    tempNode2 = tempNode1.getParentNode();      // Get last leaf's parent
                    try {
                        ((Visitee) tempNode1).acceptPost(getVisitor()); // Post visit last leaf
                    } catch (ToNextSiblingTraversalException exception) {
                        // Do nothing because we're moving to our parent.
                    }
                    tempNode1 = tempNode2;
                    if (tempNode1 != null) {                    // If not at document root..
                        //if (!(tempNode1.equals(startNode))) {   // If not moving past stop node..
                        if (tempNode1 != startNode) {           // If not moving past stop node..
                            tempNode2 = tempNode1.getNextSibling(); // ..get parent's sibling.
                            if (tempNode2 != null) {            // If more siblings..
                                try {
                                    ((Visitee) tempNode1).acceptPost(getVisitor()); // ..post visit this node.
                                } catch (ToNextSiblingTraversalException exception) {
                                    // Do nothing because we're moving to next sibling.
                                }
                            }
                        }
                        else {                                  // This node is the final node.
                            try {
                                ((Visitee) tempNode1).acceptPost(getVisitor()); // ..post visit final node.
                            } catch (ToNextSiblingTraversalException exception) {
                                // Do nothing because we're finished anyway.
                            }
                            break outer;        
                        }
                    }
                } 
                tempNode1 = tempNode2;
            } // outer: while tempNode1 != null
        } catch (EndTraversalException exception) {
            //  Suppress propogation of this condition
        }
    }

}  //NonRecursivePreorderTreeTraversal
