/*
 * (C) Copyright IBM Corp. 1997-1998  All rights reserved.
 *
 * US Government Users Restricted Rights Use, duplication or
 * disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * IBM will not be liable for any damages suffered by you as a result
 * of using the Program. In no event will IBM be liable for any
 * special, indirect or consequential damages or lost profits even if
 * IBM has been advised of the possibility of their occurrence. IBM
 * will not be liable for any third party claims against you.
 */

package com.ibm.xml.parser;

import org.w3c.dom.DOMException;
import org.w3c.dom.DocumentFragment;
import org.w3c.dom.EntityReference;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import java.util.Vector;

/**
 * The Parent class extends Child and provides services in support of non-leaf Nodes.
 *
 * @version Revision: 15 1.22 src/com/ibm/xml/parser/Parent.java, xml4jsrc, xml4j-jtcsv, xml4j_1_1_16 
 * @author TAMURA Kent &lt;kent@trl.ibm.co.jp&gt;
 * @see com.ibm.xml.parser.TXElement
 * @see com.ibm.xml.parser.TXDocument
 * @see com.ibm.xml.parser.DTD
 * @see com.ibm.xml.parser.Child
 * @see org.w3c.dom.Node
 */
public abstract class Parent extends Child {

            static final long serialVersionUID = -5379012622242611200L;
            TXNodeList children =   new TXNodeList();

    /**
     * Returns a NodeList object that will enumerate all children of this Node. If there 
     * are no children, an empty NodeList is returned. The content of the 
     * returned NodeList is "live" in the sense that changes to the children of the Node 
     * object that it was created from will be immediately reflected in the Nodes returned by 
     * the NodeList; it is not a static snapshot of the content of the Node. Similarly, changes 
     * made to the Nodes returned by the NodeList will be immediately reflected in the tree, 
     * including the set of children of the Node that the NodeList was created from.
     * <p>This method is defined by DOM.
     * @return          A NodeList of all children of this Parent Node.
     * @see #hasChildNodes
     * @see #elements
     * @see #getFirstChild
     * @see #getLastChild
     * @see #getChildrenArray
     */
    public NodeList getChildNodes() {
        return this.children;
    }

    /**
     * Returns <var>true</var> if this Node has any children, or <var>false</var> if this Node has no children at all. 
     * This method exists both for convenience as well as to allow implementations to be able 
     * to bypass object allocation, which may be required for implementing getChildNodes().
     * <p>This method is defined by DOM.
     * @return          <var>True</var> if any children exist; otherwise, returns <var>false</var>.
     * @see #getChildNodes
     * @see #elements
     * @see #getFirstChild
     * @see #getLastChild
     * @see #getChildrenArray
     */
    public boolean hasChildNodes() {
        return 0 < this.children.getLength();
    }

    /**
     * Return an Enumeration instance of all children of this Node.  
     * @return          An enumeration of all children of this Parent Node, or <var>null</var> if no children.
     * @see #hasChildNodes
     * @see #getChildNodes
     * @see #getFirstChild
     * @see #getLastChild
     * @see #getChildrenArray
     */
    public java.util.Enumeration elements() { 
        return this.children.elements();
    }

    /**
     * Make an array of <var>Child</var> Nodes for all children of this Node.  
     * @return          An array of all children of this Parent Node, or <var>null</var> if no children.
     * @see #hasChildNodes
     * @see #getChildNodes
     * @see #getFirstChild
     * @see #elements
     */
    public Child[] getChildrenArray() {
        Child[] ac = new Child[this.children.getLength()];
        this.children.nodes.copyInto(ac);
        return ac;
    }

    /**
     * Returns the first Child of this Node. If there are no children, <var>null</var> is returned.
     * <p>This method is defined by DOM.
     * @return          The first Child of this Parent Node, or <var>null</var> if no children.
     * @see #getLastChild
     * @see #hasChildNodes
     * @see #getChildNodes
     * @see #elements
     */
    public Node getFirstChild() {
        return 0 < this.children.getLength() ? this.children.item(0) : null;
    }

    /**
     *
     * @see #getFirstChild
     * @see #getLastWithoutReference
     */
    public Node getFirstWithoutReference() {
        Node ret = getFirstChild();
        while (ret != null && ret.getNodeType() == Node.ENTITY_REFERENCE_NODE) {
            Node first = ret.getFirstChild();
            if (first == null) {
                ret = ((Child)ret).getNextWithoutReference();
                break;
            }
            ret = first;
        }
        return ret;
    }

    /**
     * Returns the last Child of this Node. If there are no children, <var>null</var> is returned.
     * <p>This method is defined by DOM.
     *
     * @return          The last Child of this Parent Node, or <var>null</var> if no children.
     * @see #getFirstChild
     * @see #hasChildNodes
     * @see #getChildNodes
     * @see #elements
     */
    public Node getLastChild() {
        int size = this.children.getLength();
        return 0 < size ? this.children.item(size-1) : null;
    }

    /**
     * @see #getLastChild
     * @see #getFirstWithoutReference
     */
    public Node getLastWithoutReference() {
        Node ret = getLastChild();
        while (ret != null && ret.getNodeType() == Node.ENTITY_REFERENCE_NODE) {
            Node last = ret.getLastChild();
            if (last == null) {
                ret = ((Child)ret).getPreviousWithoutReference();
                break;
            }
            ret = last;
        }
        return ret;
    }

    /**
     * Check whether <VAR>child</VAR> is allowed to be insered in this node or not.
     * When not allowed, a DOMException with HIERARCHY_REQUEST_ERR is thrown.
     */
    abstract protected void checkChildType(Node child) throws DOMException;

    /**
     * Insert a Child Node into the specified position in this Node's list of children.
     * @param child     The Node being inserted. Must not DocumentFragment.
     * @param index     0-based index into the list of children.
     * @exception org.w3c.dom.DOMException Thrown if <var>index</var> is not valid.
     * @see #insertBefore
     * @see #insertAfter
     * @see #insertFirst
     * @see #insertLast
     */
    protected void realInsert(Node child, int index) throws DOMException {
        if (child.getParentNode() != null)
            child.getParentNode().removeChild(child);
        checkChildType(child);
        if (child == this)
            throw new TXDOMException(DOMException.HIERARCHY_REQUEST_ERR,
                                     "Can't have itself as child.");

        TXDocument document = getFactory();
        if (null != document) {
            if (document.isCheckOwnerDocument() && document != child.getOwnerDocument())
                throw new TXDOMException(DOMException.WRONG_DOCUMENT_ERR,
                                         "Specified child was created from a different document. The parent is \""+this.getNodeName()+"\", the child is \""+child.getNodeName()+"\".");

            if (document.isCheckNodeLoop()) {
                // Check whether the child is one of ancestors of this.
                Node an = this;
                while (null != (an = an.getParentNode())) {
                    if (an == child)
                        throw new TXDOMException(DOMException.HIERARCHY_REQUEST_ERR,
                                                 "Can't have an ancestor as child");
                }
            }
        }
        this.children.insert(index, child);
        ((Child)child).setParentNode(this);
        clearDigest();
    }

    /**
     * Insert a Child Node into the specified position in this Node's list of children.
     * @param child     The Node being inserted.
     * @param index     0-based index into the list of children.
     * @exception org.w3c.dom.DOMException Thrown if <var>index</var> is not valid.
     * @see #insertBefore
     * @see #insertAfter
     * @see #insertFirst
     * @see #insertLast
     */
    public synchronized void insert(Node child, int index) throws DOMException {
        if (child instanceof DocumentFragment) {
            Node node;
            while ((node = child.getLastChild()) != null) {
                child.removeChild(node);
                insert(node, index);
            }
        } else
            realInsert(child, index);
    }

    /**
     * Inserts a Child Node (newChild) before the existing Child Node (refChild). If <var>refChild</var> is <var>null</var>, insert
     * <var>newChild</var> at the end of the list of children. If <var>refChild</var> is not a Child of this Node,
     * a <var>DOMException</var> is thrown.
     * <p>This method is defined by DOM.
     * @param newChild  The Child Node being inserted.
     * @param refChild  The Child Node to insert <var>newChild</var> before, or <var>null</var> if <var>newChild</var> is to be inserted at the end of the Child list.
     * @return          The Child Node being inserted.
     * @exception org.w3c.dom.DOMException Thrown if <var>refChild</var> is not a Child of this object.
     * @see #insert
     * @see #insertAfter
     * @see #insertFirst
     * @see #insertLast
     */
    public synchronized Node insertBefore(Node newChild, Node refChild) throws DOMException {
        if (null == refChild) {
            insert(newChild, this.children.getLength());
        } else {
            int index = this.children.indexOf(refChild);
            if (0 > index) {
                throw new TXDOMException(DOMException.NOT_FOUND_ERR,
                                       "com.ibm.xml.parser.Parent#insertBefore(): Node "
                                       +refChild+" is not found in the child list.");
            }
            insert(newChild, index);
        }
        return newChild;
    }
    
    /**
     * Inserts a Child Node (newChild) after the existing Child Node (refChild). If <var>refChild</var> is <var>null</var>, insert
     * <var>newChild</var> at the beginning of the list of children. If <var>refChild</var> is not a Child of this Node,
     * a <var>DOMException</var> is thrown.
     * @param newChild  The Child Node being inserted.
     * @param refChild  The Child Node to insert <var>newChild</var> after, or <var>null</var> if <var>newChild</var> is to be inserted at the beginning of the Child list.
     * @return          The Child Node being inserted.
     * @exception org.w3c.dom.DOMException Thrown if <var>refChild</var> is not a Child of this object.
     * @see #insertBefore
     * @see #insert
     * @see #insertFirst
     * @see #insertLast
     */
    public synchronized Node insertAfter(Node newChild, Node refChild) throws DOMException {
        if (null == refChild) {
            insert(newChild, 0);
        } else {
            insertBefore(newChild, refChild.getNextSibling());
        }
        return newChild;
    }
    
    /**
     * Inserts the specified Node as the first Child in this Node's list of children. 
     * @param newChild  The Child Node being inserted.
     * @return          The Child Node being inserted.
     * @see #insertBefore
     * @see #insertAfter
     * @see #insert
     * @see #insertLast
     */
    public synchronized Node insertFirst(Node newChild) {
        insert(newChild, 0);
        return newChild;
    }

    /**
     * Inserts the specified Node as the last Child in this Node's list of children. 
     * @param newChild  The Child Node being inserted.
     * @return          The Child Node being inserted.
     * @see #insertBefore
     * @see #insertAfter
     * @see #insertFirst
     * @see #insert    
     * @see #appendChild
     */
    public synchronized Node insertLast(Node newChild) {
        insert(newChild, this.children.getLength());
        return newChild;
    }
    
    /**
     * Inserts the specified Node as the last Child in this Node's list of children.
     * @deprecated Use appendChild()
     * @param newChild  The Child Node being inserted.
     * @return          The Child Node being inserted.
     */
    public synchronized void addElement(Child newChild) {
        if (newChild != null)
            insert(newChild, this.children.getLength());
    }

    /**
     * Inserts the specified Node as the last Child in this Node's list of children. 
     * @param newChild  The Child Node being inserted.
     * @return          The Child Node being inserted.
     * @see #insertBefore
     * @see #insertAfter
     * @see #insertFirst
     * @see #insert    
     * @see #insertLast
     */
    public synchronized Node appendChild(Node newChild) {
        if (newChild != null)
            insert(newChild, this.children.getLength());
        return newChild;
    }

    /**
     * Replaces the Child Node <var>oldChild</var> with <var>newChild</var> in this Node's list of children, and return
     * the <var>oldChild</var> Node. If <var>oldChild</var> was not already a Child of this Node,
     * a <var>DOMException</var> is thrown.
     * <p>This method is defined by DOM.
     * <p><p><B>Note: </B>The parameters are not intuitively positioned. The first parameter
     * is the new child node which replaces the child node specified in the second parameter.
     * This definition departs from the conventional "replace a by b" representation.<p><p>
     *
     * @param newChild  The Child Node to replace with.
     * @param oldChild  The Child Node being replaced.
     * @return          The Child Node being replaced.
     * @exception org.w3c.dom.DOMException Thrown if <var>oldChild</var> is not a Child of this object.
     * @see #removeChild
     */
    public synchronized Node replaceChild(Node newChild, Node oldChild) throws DOMException {
        int index = this.children.indexOf(oldChild);
        if (0 > index) {
            throw new TXDOMException(DOMException.NOT_FOUND_ERR,
                                     "com.ibm.xml.parser.Parent#replaceChild(): Node "
                                     +oldChild+" is not found in the child list.");
        }
        if (newChild == oldChild)  return newChild;
        if (newChild.getParentNode() != null)
            newChild.getParentNode().removeChild(newChild);
        Child child = (Child)newChild;
        this.children.replace(index, child);
        child.setParentNode(this);
        clearDigest();
        // this.processAfterRemove(oldChild);
        return oldChild;
    }
    
    /**
     * Removes the Child Node indicated by <var>oldChild</var> from this Nodes list of children,
     * and returns it. If <var>oldChild</var>
     * was not a Child of this Node, a <var>DOMException</var> is thrown.
     * <p>This method is defined by DOM.
     * @param oldChild  The Child Node being removed.
     * @return          The Child Node being removed.
     * @exception org.w3c.dom.DOMException Thrown if <var>oldChild</var> is not a Child of this object.
     * @see #replaceChild 
     */

    public synchronized Node removeChild(Node oldChild) throws DOMException {
        int index = this.children.indexOf(oldChild);
        if (0 > index) {
            throw new TXDOMException(DOMException.NOT_FOUND_ERR,
                                     "com.ibm.xml.parser.Parent#removeChild(): Node "
                                     +oldChild+" is not found in the child list.");
        }
        this.children.remove(index);
        clearDigest();
        // this.processAfterRemove(oldChild);
        return oldChild;
    }

    /**
     * @see com.ibm.xml.parser.TXElement#collectNamespaceAttributes
     * @see com.ibm.xml.parser.GeneralReference#collectNamespaceAttributes
     */
    protected void processAfterRemove(Node oldChild) {
        int type = oldChild.getNodeType();
        if (type == Node.ELEMENT_NODE) {
            ((TXElement)oldChild).collectNamespaceAttributes(this);
        } else if (type == Node.ENTITY_REFERENCE_NODE) {
            ((GeneralReference)oldChild).collectNamespaceAttributes(this);
        }
    }
    

    /**
     *
     */
    public void expandEntityReferences() {
        Node node = getFirstChild();
        while (node != null) {
            if (node instanceof Parent) {
                ((Parent)node).expandEntityReferences();
                if (node instanceof EntityReference) {
                    Node child;
                    while ((child = node.getLastChild()) != null) {
                        node.removeChild(child);
                        insertAfter(child, node);
                    }
                    child = node.getNextSibling(); // First child of an EntityReference
                    removeChild(node);          // Remove an EntityReference
                    node = child;
                    continue;
                }
            }
            node = node.getNextSibling();
        }
    }

    /**
     * Return all text associated with this Node and its children without considering entities.
     * <p>This method is defined by Child.
     * @return          Text associated with all children, or <var>""</var> if no children.
     * @see com.ibm.xml.parser.Child#toXMLString
     */
    public String getText() {
        int length;
        if (children == null || (length = children.getLength()) == 0)
            return "";
        if (length == 1)
            return ((Child)children.item(0)).getText();
        StringBuffer sb = new StringBuffer(128);
        synchronized (children) {
            for (int i = 0;  i < length;  i ++)
                sb.append(((Child)children.item(i)).getText());
        }
        return sb.toString().intern();
    }
}
