/*
 * (C) Copyright IBM Corp. 1997-1998  All rights reserved.
 *
 * US Government Users Restricted Rights Use, duplication or
 * disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * IBM will not be liable for any damages suffered by you as a result
 * of using the Program. In no event will IBM be liable for any
 * special, indirect or consequential damages or lost profits even if
 * IBM has been advised of the possibility of their occurrence. IBM
 * will not be liable for any third party claims against you.
 */

package com.ibm.xml.parser;

import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;
import java.io.Reader;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.ListResourceBundle;
import java.util.ResourceBundle;
import java.util.Vector;
import org.w3c.dom.Attr;
import org.w3c.dom.Comment;
import org.w3c.dom.EntityReference;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.w3c.dom.ProcessingInstruction;

/**
 * Parser is the XML4J parser that examines an XML document and generates an object tree
 * of Document Object Model (DOM) Nodes.  Details of these Nodes are provided by the
 * references below.
 * 
 * @version Revision: 17 1.48 src/com/ibm/xml/parser/Parser.java, xml4jsrc, xml4j-jtcsv, xml4j_1_1_16 
 * @author TAMURA Kent &lt;kent@trl.ibm.co.jp&gt;
 * @see com.ibm.xml.parser.TXDocument
 * @see com.ibm.xml.parser.TXElement
 * @see com.ibm.xml.parser.TXText
 * @see com.ibm.xml.parser.TXComment
 * @see com.ibm.xml.parser.TXPI
 * @see com.ibm.xml.parser.StylesheetPI
 * @see com.ibm.xml.parser.TXCDATASection
 * @see com.ibm.xml.parser.TXNotation
 * @see com.ibm.xml.parser.TXAttributeList
 * @see com.ibm.xml.parser.TXAttribute
 * @see com.ibm.xml.parser.DTD
 * @see com.ibm.xml.parser.ElementDecl
 * @see com.ibm.xml.parser.Attlist
 * @see com.ibm.xml.parser.AttDef
 * @see com.ibm.xml.parser.EntityDecl
 */
public class Parser {

    private Vector unparsedEntities = new Vector();
    
            StreamProducer      streamProducer              =   null;
            EntityPool          entityPool                  =   null;
            int                 errorCount                  =   0;
            int                 validityFailureCount        =   0;
            int                 warnCount                   =   0;
            boolean             doExternalDTD               =   true;
            boolean             isKeepComment               =   true;
            boolean             isErrorNoByteMark           =   true;
            boolean             isAllowJavaEncodingName     =   false;
            boolean             isPreserveSpace             =   false;
            boolean             isEndBy1stError             =   true;
            boolean             isExpandEntityReferences    =   false;
            TXDocument          document                    =   null;
            DTD                 dtd                         =   null;
            Hashtable           attributeErrors             =   null;
            Hashtable           contentModelErrors          =   null;
            Vector              generalReferences           =   new Vector();
            Vector              parameterEntityReferences   =   new Vector();
            ReferenceHandler    referenceHandler            =   null;
            int                 readerBufferSize            =   32*1024;
            StringPool          stringPool                  =   null;
    
    private static final String s_enameall                  =   " ALL ";
    
    private ResourceBundle      resourceBundle              =   null;
    private ErrorListener       errorListener               =   null;
    private TagHandler          tagHandler                  =   null;
    private Vector              preRootHandlers             =   null;
    private Vector              piHandlers                  =   null;
    private Hashtable           elementHandlerHash          =   null;
    private Vector              requiredAttributeHandlers   =   null;
    private boolean             isProcessNamespace          =   false;
    private boolean             isWarningRedefinedEntity    =   true;
    private boolean             isWarningNoXMLDecl          =   true;
    private boolean             isWarningNoDoctypeDecl      =   false;
    private Vector              idRefPool                   =   null;
    private Vector              notationNames               =   null;
    private CMNode              contentModelPCData          =   null;   
    private int                 parenLevel                  =   -1;
    private int                 condSectLevel               =   0;
    private boolean             stopRequest                 =   false;
    private SAXDriver           saxdriver                   =   null;
    private boolean             isProcessExternalDTD        =   true;
    

    /**
     * Constructor when using the default <code>ErrorListener</code> and <code>StreamProducer</code>.
     * The default <code>ErrorListener</code> and <code>StreamProducer</code> are defined by
     * the <code>Stderr</code> class.
     * @param   name    URL or filespec to use as the default input stream; if a filespec is
     *                  provided, this value can include a drive and directory spec.
     *                  This value is also used to associate a name with errors reported to the
     *                  default error listener that have a <var>file</var> parameter 
     *                  <code>=null</code>.
     * @see com.ibm.xml.parser.Stderr
     * @see com.ibm.xml.parser.ErrorListener
     * @see com.ibm.xml.parser.StreamProducer
     */
    public Parser(String name) {
        this(name, null, null);
    }

    /**
     * Constructor when NOT using the default <CODE>ErrorListener</CODE> or <CODE>StreamProducer</CODE>.
     * The default <code>ErrorListener</code> and <code>StreamProducer</code> are defined by
     * the <code>Stderr</code> class.
     * @param   name    This parameter is only used if either <var>errorListener</var> or
     *                  <var>streamProducer</var> are allowed to default.  If <var>streamProducer</var>
     *                  is allowed to default, this value is the URL or filespec to use as 
     *                  the default input stream; if a filespec is provided, this value can 
     *                  include a drive and directory spec.  If <var>errorListener</var> is
     *                  allowed to default, this value is used to associate a name with 
     *                  errors reported to the default error listener that have a 
     *                  <var>file</var> parameter <code>=null</code>.
     * @param errorListener Handler for error and warning events that are recognized by this 
     *                  parser, or <var>null</var> if to use the default listener.
     * @param streamProducer Mechanism for manipulating this parser's input stream, or
     *                  <var>null</var> if to use the default stream producer.
     * @see com.ibm.xml.parser.Stderr
     * @see com.ibm.xml.parser.ErrorListener
     * @see com.ibm.xml.parser.StreamProducer
     */
    public Parser(String name, ErrorListener errorListener, StreamProducer streamProducer) {
        Stderr se = null;
        if (null == errorListener || null == streamProducer)  se = new Stderr(name);
        this.errorListener = null == errorListener ? se : errorListener;
        this.streamProducer = null == streamProducer ? se : streamProducer;
        init();
    }
    
   /**
     * Sets a locale for error and warning messages.
     * <p>By default, the current user locale's resource bundle is used.  If no messages
     * exist for the requested <var>locale</var>, US English messages are used.
     * @param   locale  A Java Locale object.
     */
    public void setLocale(java.util.Locale locale) {
        this.resourceBundle = ListResourceBundle.getBundle("com.ibm.xml.parser.r.Message", locale);
    }

    /**
     * Sets the code to receive control when the various components of the Document Object
     * Model (DOM) and XML4J need to be created and initialized.  By default, a instance of the
     * <code>TXDocument</code> is used.
     * <P>When this method is called, readStream() returns the <VAR>elementFactory</VAR>.
     * @param   elementFactory  Factory to use to create and initialize objects.  This must be an instance of TXDocument or a subclass of TXDocument.
     * @see com.ibm.xml.parser.TXDocument
     */
    public void setElementFactory(TXDocument elementFactory) {
        this.document = elementFactory;
    }

   /**
     * <p>Returns the source of the input stream (could be a character stream or a byte stream) 
     * based on the entity specified by the system ID and/or public ID, and the 
     * <var>streamProducer</var> specified when this parser was constructed.
     * @param   name        The name to be associated with the input stream.  For example, 
     *                      this name could be the public ID of an external ID, or a local filename.
     * @param   publicID    Entity's public ID, or <var>null</var> if no public ID (see ExternalID for details).    
     * @param   systemID    Entity's system ID.    
     * @return              The resolved source of the input stream, or <var>null</var> if unable to resolve.
     * @exception IOException   Thrown if unable to open the source defined by the specified IDs.
     * @see com.ibm.xml.parser.ExternalID
     * @see #closeInputStream
     * @see com.ibm.xml.parser.StreamProducer#getInputStream
     */
    public Source getInputStream(String name, String publicID, String systemID) throws IOException {
        return this.streamProducer.getInputStream(name, publicID, systemID);
    }
    
   /**
     * <p>Removes the input stream currently in use. 
     * @param   source      Source of the input stream to be closed.
     * @see #getInputStream
     * @see com.ibm.xml.parser.StreamProducer#closeInputStream
     */
    public void closeInputStream(Source source) {
        this.streamProducer.closeInputStream(source);
    }
    
    /**
     * Loads a catalog which provides mapping between public IDs and system IDs.
     * @param       reader      Character input stream reader.
     * @exception   IOException Thrown if <var>reader</var> is invalid.
     * @see #getInputStream
     * @see com.ibm.xml.parser.ExternalID
     * @see com.ibm.xml.parser.Stderr#loadCatalog
     * @see com.ibm.xml.parser.StreamProducer#loadCatalog
     */
    public void loadCatalog(Reader reader) throws IOException {
        this.streamProducer.loadCatalog(reader);
    }

    /**
     * Register a handler for receiving control when tag start and tag end events are 
     * recognized by this parser.
     * By default, no tag handler is registered to this parser.
     * @param   tagHandler  Handler to be registered.
     * @see com.ibm.xml.parser.TagHandler
     */
    public void setTagHandler(TagHandler tagHandler) {
        this.tagHandler = tagHandler;
    }

   /**
     * Register a handler for receiving control when a General Reference is recognized by 
     * this parser.
     * By default, no reference handler is registered to this parser.
     * @param   referenceHandler    Handler to be registered.
     * @see com.ibm.xml.parser.ReferenceHandler
     */
    public void setReferenceHandler(ReferenceHandler referenceHandler) {
        this.referenceHandler = referenceHandler;
    }
    
   /**
     * Register a handler for receiving control from this parser after the internal and/or 
     * external DTD subsets are parsed, but before the document's root Element is parsed.
     * By default, no preroot handlers are registered to this parser.
     * @param   preRootHandler  Handler to be registered.
     * @see com.ibm.xml.parser.PreRootHandler
     */
    public void addPreRootHandler(PreRootHandler preRootHandler) {
        if (null == this.preRootHandlers)  this.preRootHandlers = new Vector();
        this.preRootHandlers.addElement(preRootHandler);
    }

   /**
     * Register a handler for receiving control when PIs are recognized by this parser.
     * By default, no PI handlers are registered to this parser.
     * @param   piHandler   Handler to be registered.
     * @see com.ibm.xml.parser.PIHandler
     */
    public void addPIHandler(PIHandler piHandler) {
        if (null == this.piHandlers)  this.piHandlers = new Vector();
        this.piHandlers.addElement(piHandler);
    }
    
   /**
     * Register a handler for receiving control when ANY element tag is recognized by this
     * parser.
     * By default, no element handlers are registered to this parser.
     * @param   elementHandler  Handler to be registered.
     * @see com.ibm.xml.parser.ElementHandler
     */
    public void addElementHandler(ElementHandler elementHandler) {
        addElementHandler(elementHandler, s_enameall);
    }
    
   /**
     * Register a handler for receiving control when the specified <var>elementTagName</var>
     * element tag is recognized by this parser.
     * By default, no element handlers are registered to this parser.
     * @param   elementHandler  Handler to be registered for the specified <var>elementTagName</var>.
     * @param   elementTagName  Element name to be watched for by this parser.  If the 
     *                          Element's name has a namespace prefix, that prefix should
     *                          be included.   
     * @see com.ibm.xml.parser.ElementHandler
     */
    public void addElementHandler(ElementHandler elementHandler, String elementTagName) {
        if (null == this.elementHandlerHash)  this.elementHandlerHash = new Hashtable();
        Vector v = (Vector)this.elementHandlerHash.get(elementTagName);
        if (null == v) {
            v = new Vector();
            this.elementHandlerHash.put(elementTagName, v);
        }
        v.addElement(elementHandler);
    }

   /**
     * Register a handler for receiving control when a "no required attribute" event is 
     * recognized by this parser.  A "no required attribute" event occurs when an Element 
     * is parsed and a required Attribute for this Element is not seen; Attributes are 
     * declared as required as part of the internal and/or external DTD declaration of the 
     * XML document.
     * <p>By default, no required attribute handlers are registered to this parser.
     * @param   noRequiredAttributeHandler  Handler to be registered.
     * @see com.ibm.xml.parser.NoRequiredAttributeHandler
     */
    public void addNoRequiredAttributeHandler(NoRequiredAttributeHandler noRequiredAttributeHandler) {
        if (null == this.requiredAttributeHandlers)  this.requiredAttributeHandlers = new Vector();
        this.requiredAttributeHandlers.addElement(noRequiredAttributeHandler);
    }

    /**
     *
     */
    void setSAXDriver(SAXDriver sax) {
        this.saxdriver = sax;
    }

    /**
     * Returns the number of errors and exceptions currently encountered by this parser.
     * Warning messages do not count in this number.
     * @return      Number of errors and exceptions encountered by this parser.
     * @see com.ibm.xml.parser.ErrorListener
     */
    public int getNumberOfErrors() {
        return this.errorCount + this.validityFailureCount;
    }

    /**
     * Returns the number of warnings currently encountered by this parser.
     * @return      Number of warnings encountered by this parser.
     * @see com.ibm.xml.parser.ErrorListener
     */
    public int getNumberOfWarnings() {
        return this.warnCount;
    }

    /**
     * Sets whether <code>TXComment</code> Nodes are created by this parser when XML
     * comments are encountered.
     * <p>By default, <code>TXComment</code> Nodes are created by this parser.
     * @param   isKeepComment   <code>=true</code> means maintain parsed comments; 
     *                          <code>=false</code> means drop comments when creating object tree.
     * @see com.ibm.xml.parser.TXComment
     */
    public void setKeepComment(boolean isKeepComment) {
        this.isKeepComment = isKeepComment;
    }

    /**
     * Sets whether <code>EntityReference</code> Nodes are expanded by this parser.
     * <p>By default, <code>EntityReference</code> Nodes are kept.
     * @param   isKeepComment   <code>=true</code> means expanding entity references;
     *                          <code>=false</code> means maintain entity references.
     * @see org.w3c.dom.EntityReference
     * @see com.ibm.xml.parser.GeneralReference
     */
    public void setExpandEntityReferences(boolean isExpandEntityReferences) {
        this.isExpandEntityReferences = isExpandEntityReferences;
    }

    /**
     * Sets whether namespaces are respected by this parser.  Namespace notation can occur
     * on the names of <code>TXElement, TXPI, and TXAttribute</code> Nodes.  If namespaces
     * are respected, the methods defined by the <code>Namespace</code> interface will
     * function properly in these Nodes.
     * <p>By default, namespaces are NOT respected by this parser.
     * @param   isProcessNamespace  <code>=true</code> means parse for namespaces; 
     *                              <code>=false</code> means ignore namespaces when parsing.
     * @see com.ibm.xml.parser.Namespace
     * @see com.ibm.xml.parser.TXElement
     * @see com.ibm.xml.parser.TXAttribute
     */
    public void setProcessNamespace(boolean isProcessNamespace) {
        this.isProcessNamespace = isProcessNamespace;
    }

    /**
     * Sets whether this parser allows Java encoding names to be specified in place of
     * MIME charset names on <var>xmlEncoding</var> parameters.  The <code>MIME2Java</code>
     * class is provided to facilitate conversion when Java encoding names are not allowed.
     * <p>By default, Java encoding names are NOT allowed on <var>xmlEncoding</var> parameters
     * by this parser.
     * @param   isAllowJavaEncodingName <code>=true</code> means Java encoding names are allowed; 
     *                                  <code>=false</code> means MIME charset names must be used.
     * @see com.ibm.xml.parser.TXDocument#setEncoding
     * @see com.ibm.xml.parser.DTD#setEncoding
     * @see com.ibm.xml.parser.MIME2Java
     */
    public void setAllowJavaEncodingName(boolean isAllowJavaEncodingName) {
        this.isAllowJavaEncodingName = isAllowJavaEncodingName;
    }

    /**
     * Sets whether this parser reads an external subset of DTD, pointed in an ExternalID
     * in DOCTYPE declaration.
     * <P>By default, this parser reads an external subset of DTD.
     * @param isProcessExternalDTD <code>=true</code> means process an external DTD;
     *                             <code>=false</code> means ignore an external DTD.
     * @see com.ibm.xml.parser.DTD
     */
    public void setProcessExternalDTD(boolean isProcessExternalDTD) {
        this.isProcessExternalDTD = isProcessExternalDTD;
    }
    /**
     * Sets, at the document level, whether space is to be preserved in the parsed document.  
     * Regardless of the value of <var>isPreserveSpace</var>, this parser will preserve 
     * whitespace.  This value is used, for example, to determine if space is to be preserved
     * in Text Nodes during printWithFormat() operations.  Note that the document-level value will be 
     * overridden at the Node level if this parser recognizes a <code>xml:space</code> 
     * attribute.
     * <p>By default, space is not to be preserved in the parsed document.
     * @param   isPreserveSpace <code>=true</code> space is to be preserved; 
     *                          <code>=false</code> space is to be ignored.
     * @see com.ibm.xml.parser.TXElement#setPreserveSpace
     * @see com.ibm.xml.parser.TXText#setIsIgnorableWhitespace
     * @see com.ibm.xml.parser.TXDocument#printWithFormat
     */
    public void setPreserveSpace(boolean isPreserveSpace) {
        this.isPreserveSpace = isPreserveSpace;
    }

    /**
     * Sets whether this parser treats missing byte marks in the input stream as an error,
     * and reports this condition through the defined <code>ErrorListener</code>.
     * <p>By default, this condition is reported by this parser as an error.
     * @param   isErrorNoByteMark   <code>=true</code> means report condition as an error; 
     *                              <code>=false</code> means ignore this condition.
     */
    public void setErrorNoByteMark(boolean isErrorNoByteMark) {
        this.isErrorNoByteMark = isErrorNoByteMark;
    }

    /**
     * Sets whether this parser treats redefined entities as a warning,
     * and reports this condition through the defined <code>ErrorListener</code>.
     * <p>By default, this condition is reported by this parser as a warning.
     * @param   isWarningRedefinedEntity    <code>=true</code> means report condition as a warning; 
     *                                      <code>=false</code> means ignore this condition.
     * @see com.ibm.xml.parser.EntityDecl
     */
    public void setWarningRedefinedEntity(boolean isWarningRedefinedEntity) {
        this.isWarningRedefinedEntity = isWarningRedefinedEntity;
    }

    /**
     * Sets whether this parser treats a missing XML prolog PIs as a warning,
     * and reports this condition through the defined <code>ErrorListener</code>.
     * <p>By default, this condition is reported by this parser as a warning.
     * @param   isWarningNoXMLDecl  <code>=true</code> means report condition as a warning; 
     *                              <code>=false</code> means ignore this condition.
     * @see com.ibm.xml.parser.TXPI
     */
    public void setWarningNoXMLDecl(boolean isWarningNoXMLDecl) {
        this.isWarningNoXMLDecl = isWarningNoXMLDecl;
    }

    /**
     * Sets whether this parser treats a missing DTD declaration as a warning,
     * and reports this condition through the defined <code>ErrorListener</code>.
     * <p>By default, this condition is NOT reported by this parser as a warning.
     * @param   isWarningNoXMLDecl  <code>=true</code> means report condition as a warning; 
     *                              <code>=false</code> means ignore this condition.
     * @see com.ibm.xml.parser.DTD
     */
    public void setWarningNoDoctypeDecl(boolean isWarningNoDoctypeDecl) {
        this.isWarningNoDoctypeDecl = isWarningNoDoctypeDecl;
    }

    /**
     * Sets whether this parser terminates processing when an initial error or exception
     * occurs.
     * <P>By default, this parser stops parsing after an error occurs, as required by
     * the XML specification.
     * @param   isEndBy1stError     <code>=true</code> means initial error or exception ends processing;
     *                              <code>=false</code> means processing continues until document is fully parsed.
     */
    public void setEndBy1stError(boolean isEndBy1stError) {
        this.isEndBy1stError = isEndBy1stError;
    }

    /**
     * Stops a <code>Parser.readStream()</code> operation.
     * The calling thread will signal that the thread executing in <code>this.readStream()</code>
     * should clean up and return as soon as practical.  The calling thread will return immediately.
     * The effect of <code>stop()</code> is sticky, meaning that after <code>stop()</code> has been 
     * called, a thread entering readStream() will return immediately.  
     * (This helps in the case where the UI thread invokes <code>stop</code> before the processing 
     * thread actually enters <code>readStream</code>
     * <p> <code>stop</code> affects only the specific instance of the parser (this) on which
     * it is invoked.  It is not global; <code>readStream</code> on other parser instances
     * will not be affected.
     * 
     */
    public void stop() {
         stopRequest=true;
    }
    
    class StopSignal extends IOException {
    }
    
    private final void pollForStop() throws StopSignal {
        if (stopRequest)  {
            throw new StopSignal();
        }
    }
        
        
    /**
     * Returns a parsed XML document based on the specified <var>inputStream</var>.
     * The returned document can then be accessed by methods such as <code>TXDocument#getDocumentElement</code>.
     * <p>Once this method is invoked, the parser instance is of no further use, and should NOT be reused.
     * @param   inputStream Byte-stream-oriented data source.
     * @return              Parsed XML document (should never be <var>null</var>).
     * @see com.ibm.xml.parser.Source
     * @see com.ibm.xml.parser.TXDocument#getDocumentElement
     * @see #readDTDStream
     */
    public TXDocument readStream(InputStream inputStream) {
        return readStream(new Source(inputStream));
    }
    
    /**
     * Returns a parsed XML document based on the specified <var>reader</var>.
     * The returned document can then be accessed by methods such as <code>TXDocument#getDocumentElement</code>.
     * <p>Once this method is invoked, the parser instance is of no further use, and should NOT be reused.
     * @param   reader      Character-stream-oriented data source.
     * @return              Parsed XML document (should never be <var>null</var>).
     * @see com.ibm.xml.parser.Source
     * @see com.ibm.xml.parser.TXDocument#getDocumentElement
     * @see #readDTDStream
     */
    public TXDocument readStream(Reader reader) {
        return readStream(new Source(reader));
    }
    
    /**
     * Returns a parsed XML document based on the specified <var>source</var>.
     * The returned document can then be accessed by methods such as <code>TXDocument#getDocumentElement</code>.
     * <p>Once this method is invoked, the parser instance is of no further use, and should NOT be reused.
     * @param   source      Byte-stream-oriented or character-stream-oriented data source.
     * @return              Parsed XML document (should never be <var>null</var>).
     * @see com.ibm.xml.parser.Source
     * @see com.ibm.xml.parser.TXDocument#getDocumentElement
     * @see #readDTDStream
     */
    public TXDocument readStream(Source source) {
        if (this.document == null)
            this.document = new TXDocument();
        this.document.setProcessNamespace(this.isProcessNamespace);
        init2();

        if (stopRequest)
            return this.document;
 
        Token tok = null;
        boolean isCheckNodeLoop = this.document.isCheckNodeLoop();
        this.document.setCheckNodeLoop(false);

        try {
            tok = new Token(this, null, source);

            /*
             * document ::= XMLDecl? Misc* (doctypedecl Misc*)? elemnt Misc*
             * Misc ::= Comment | PI | S
             */
            int state = 0;
            int ch;
            while (0 <= (ch = tok.m_next)) {
                if ('<' == ch) {
                    tok.getChar();              // '<'
                    ch = tok.m_next;
                    if ('?' == ch) {            // PI (including ?XML)
                        ProcessingInstruction p = parsePI(tok, 0 == state);
                        if (null != p) {
                            this.document.appendChild(p);
                            applyPIHandlers(p);
                        }
                    
                    } else if ('!' == ch) {     // comment or DOCTYPE
                        tok.getChar();          // !
                                                // <!-- | <![CDATA[
                        ch = tok.m_next;
                        if ('-' == ch) {
                            tok.getChar();      // 1st '-'
                            ch = tok.m_next;
                            if ('-' == ch) {    // comment
                                tok.getChar();  // 2nd '-'
                                if (this.isKeepComment) {
                                    Comment com = this.document.createComment(tok.getComment());
                                    if (com != null)  this.document.appendChild(com);
                                } else
                                    tok.getComment();
                            } else {
                                format0(tok, "E_COM0");
                                tok.skipTagEnd();
                            }
                        } else {
                            String n = tok.getName();
                            if (null == n || !"DOCTYPE".equals(n)) {
                                format0(tok, "E_DOCTYPE0");
                                tok.skipTagEnd();
                            } else {
                                if (2 > state)
                                    state = 2;
                                else
                                    format0(tok, "E_STRUCT3");
                                parseDOCTYPE(tok);
                            }
                        }

                    } else if (3 > state) {     // root element
                        Exception ex = null;
                        TXElement t = null;
                        try {
                            t = parseElement(tok, this.document, this.isPreserveSpace,
                                             true, true);
                        } catch (Exception e) {
                            ex = e;
                        }
                        if (null != t) {
                            t.setParentNode(null);
                            this.document.appendChild(t);
                        }
                        state = 3;
                        if (null != ex)  throw ex;
                    }

                } else if (XMLChar.isSpace((char)ch)) {
                    tok.getWhitespaces();
                    this.document.appendChild(this.document.createTextNode(tok.m_tstrbuf, 0, tok.m_tbufpos, true));

                } else {                        // !Char
                    format0(tok, "E_STRUCT0");
                    tok.nextTagStart();
                }
                if (0 == state)  state = 1;
            }

            if (isWarningNoXMLDecl()) {
                if (null == this.document.getVersion())
                    format0(tok, "W_STRUCT4");
            }
            if (isWarningNoDoctypeDecl()) {
                if (null == this.document.getDTD())
                    format0(tok, "W_STRUCT5");
            }

            // Error if an element isn't processed
            if (null == this.document.getDocumentElement()) {
                format0(tok, "E_STRUCT2");
            }

            tok.close();
            //tok.report();

            checkIdref();
        } catch (TerminateSignal ts) {
                                                // Nothing to do.
        } catch (StopSignal ss) {
                                                // Also nothing to do.
                                                //   Should there be some notification available here?
        } catch (Exception e) {
            this.errorCount ++;
            String mes = null == e.getMessage() ? e.toString() : e.getMessage();
            if (null == tok) {
                this.errorListener.error(null, 0, 0, e, mes);
            } else {
                this.errorListener.error(tok.getFileName(), tok.getLineNumber(),
                                         tok.getCharLocation(), e, mes);
                try {
                    tok.close();
                } catch (Exception ex) {}
            }
            if (!(e instanceof IOException))
                throw new LibraryException("Parser#readStream(): Internal Error?: "+e);
        }
        TXDocument doc = this.document;
        doc.setCheckNodeLoop(isCheckNodeLoop);
        clean();
               
        return doc;
    }

    /**
     * Returns a parsed external DTD subset based on the specified <var>inputStream</var>.
     * The returned document can then be accessed by methods such as <code>DTD#externalElements</code>.
     * <p>Once this method is invoked, the parser instance is of no further use, and should NOT be reused.
     * @param   inputStream Byte-stream-oriented data source.
     * @return              Parsed DTD (should never be <var>null</var>).
     * @see com.ibm.xml.parser.Source
     * @see com.ibm.xml.parser.DTD
     * @see #readStream
     */
    public DTD readDTDStream(InputStream inputStream) throws IOException {
        return readDTDStream(new Source(inputStream));
    }
    
    /**
     * Returns a parsed external DTD subset based on the specified <var>reader</var>.
     * The returned document can then be accessed by methods such as <code>DTD#externalElements</code>.
     * <p>Once this method is invoked, the parser instance is of no further use, and should NOT be reused.
     * @param   reader      Character-stream-oriented data source.
     * @return              Parsed DTD (should never be <var>null</var>).
     * @see com.ibm.xml.parser.Source
     * @see com.ibm.xml.parser.DTD
     * @see #readStream
     */
    public DTD readDTDStream(Reader reader) throws IOException {
        return readDTDStream(new Source(reader));
    }
    
    /**
     * Returns a parsed external DTD subset based on the specified <var>source</var>.
     * The returned document can then be accessed by methods such as <code>DTD#externalElements</code>.
     * <p>Once this method is invoked, the parser instance is of no further use, and should NOT be reused.
     * @param   source      Byte-stream-oriented or character-stream-oriented data source.
     * @return              Parsed DTD (should never be <var>null</var>).
     * @see com.ibm.xml.parser.Source
     * @see com.ibm.xml.parser.DTD
     * @see #readStream
     */
    public DTD readDTDStream(Source source) throws IOException {
        if (this.document == null)
            this.document = new TXDocument();
        init2();
        Token extoken = null;
        this.dtd = this.document.createDTD();
        this.dtd.setEntityPool(this.entityPool);
        try {
            extoken = new Token(this, null, source);
            this.dtd.setParsingExternal(true);
            extoken.setState(Token.ST_EXTERNALDTD);
            parseMarkupDecl(this.dtd, extoken, true);
            this.dtd.setParsingExternal(false);
            extoken.close();
        } catch (TerminateSignal ts) {          // Added by defect #282.
                                                // Nothing to do.
        } catch (StopSignal ss) {               // Added by defect #282.
                                                // Also nothing to do.
                                                //   Should there be some notification available here?
        } catch (Exception e) {
            if (null == extoken)
                this.errorListener.error(null, 0, 0, e, e.toString());
            else
                error(extoken, e);
        }
        DTD d = this.dtd;
        clean();
        return d;
    }
    
    /**
     * Returns the size of this parser's character-stream-oriented input source buffer.  
     * Characters are buffered during I/O in order to provide efficient reading.  
     * This method has no meaning for byte-stream-oriented input sources.
     * @return      Size of this parser's character-stream-oriented input buffer.
     * @see #setReaderBufferSize
     */
    public int getReaderBufferSize() {
        return this.readerBufferSize;
    }

    /**
     * Sets the size of this parser's character-stream-oriented input source buffer.  
     * Characters are buffered during I/O in order to provide efficient reading.  
     * This method has no meaning for byte-stream-oriented input sources.
     * <p>The default buffer size is large enough for most purposes.
     * @param   readerBufferSize    Size of this parser's character-stream-oriented input buffer.
     * @see #getReaderBufferSize
     */
    public void setReaderBufferSize(int readerBufferSize) {
        this.readerBufferSize = readerBufferSize;
    }

    String getString(String key) {
        String s = this.resourceBundle.getString(key);
        return null == s ? "**** Internal Error: No message ***" : s;
    }
    
    void format0(FileScanner fileScanner, String key) {
        error(fileScanner, key, getString(key));
    }
    
    String format1(String key, String param1) {
        Object[] ao = new Object[1];
        ao[0] = param1;
        return java.text.MessageFormat.format(getString(key), ao);
    }
    
    void format1(FileScanner fileScanner, String key, String param1) {
        error(fileScanner, key, format1(key, param1));
    }
    
    String format2(String key, String param1, String param2) {
        Object[] ao = new Object[2];
        ao[0] = param1;
        ao[1] = param2;
        return java.text.MessageFormat.format(getString(key), ao);
    }
    
    void format2(FileScanner fileScanner, String key, String param1, String param2) {
        error(fileScanner, key, format2(key, param1, param2));
    }
    
    String format3(String key, String param1, String param2, String param3) {
        Object[] ao = new Object[3];
        ao[0] = param1;
        ao[1] = param2;
        ao[2] = param3;
        return java.text.MessageFormat.format(getString(key), ao);
    }
    
    void format3(FileScanner fileScanner, String key, String param1, String param2, String param3) {
        error(fileScanner, key, format3(key, param1, param2, param3));
    }
    
    String format4(String key, String param1, String param2, String param3, String param4) {
        Object[] ao = new Object[4];
        ao[0] = param1;
        ao[1] = param2;
        ao[2] = param3;
        ao[3] = param4;
        return java.text.MessageFormat.format(getString(key), ao);
    }
    void format4(FileScanner fileScanner, String key, String param1, String param2,
                 String param3, String param4) {
        error(fileScanner, key, format4(key, param1, param2, param3, param4));
    }
    

    private static class TerminateSignal extends RuntimeException {
    }

    void error(FileScanner fileScanner, Object key, String msg) {
        int d = this.errorListener.error(fileScanner.getFileName(), fileScanner.getLineNumber(), 
                                         fileScanner.getCharLocation(), key, msg);
        boolean fatal = key instanceof String && ((String)key).startsWith("E_");
        boolean validityFailure = key instanceof String && ((String)key).startsWith("V_");
        if (fatal)
            this.errorCount += d;
        else if (validityFailure)
            this.validityFailureCount += d;
        else
            this.warnCount += d;
        if (this.isEndBy1stError && fatal)  throw new TerminateSignal();
    }
    
    void error(FileScanner fileScanner, Exception e) {
        this.errorCount ++;
        this.errorListener.error(fileScanner.getFileName(), fileScanner.getLineNumber(), 
                                 fileScanner.getCharLocation(), e, e.toString());
        if (this.isEndBy1stError)  throw new TerminateSignal();
    }
    
    boolean isErrorNoByteMark() {
        return this.isErrorNoByteMark;
    }
    
    boolean isWarningRedefinedEntity() {
        return this.isWarningRedefinedEntity;
    }
    
    boolean isWarningNoXMLDecl() {
        return this.isWarningNoXMLDecl;
    }

    ProcessingInstruction parsePI(Token tok, boolean acceptxml) throws IOException {
        return parsePI(tok, acceptxml, false, null);
    }
    
    ProcessingInstruction parsePI(Token tok, boolean acceptxml, boolean allowtextdecl, DTD dtd)
        throws IOException {
        tok.getChar();              // '?'
        String n = tok.getName();
        if (null == n) {
            format0(tok, "E_PI0");
            tok.skipTagEnd();
            return null;
        }

        if ("xml".equals(n)) {                  // <?xml
            if (allowtextdecl) {
                TXAttribute[] aa = tok.getPIAttributes(true);
                String enc = null;
                for (int i = 0;  i < aa.length;  i ++) {
                    String aname = aa[i].getName();
                    if ("encoding".equals(aname)) {
                        enc = aa[i].getValue();
                    } else if ("version".equals(aname)) {
                                                // ignore
                    } else {
                        format1(tok, "E_PI6", aname);
                    }
                }
                if (null == enc)
                    format0(tok, "E_PI7");
                else if (dtd != null)
                    dtd.setEncoding(enc);

                tok.getChar();
                return null;
            }            // Process <?xml ...?>
            if (!acceptxml) {
                format0(tok, "E_STRUCT1");
                tok.skipTagEnd();
                return null;
            }
            TXAttribute[] aa = tok.getPIAttributes(true);
            tok.getChar();
            if (0 == aa.length) {
                format0(tok, "E_XML1");
                return null;
            }
            if (!"version".equals(aa[0].getName())) {
                format0(tok, "E_XML2");
                return null;
            }
            if (!Util.checkVersionNum(aa[0].getValue())) {
                format1(tok, "E_XML6", aa[0].getValue());
                return null;
            }
            if (!"1.0".equals(aa[0].getValue())) {
                format0(tok, "E_XML4");
                return null;
            }
            this.document.setVersion(aa[0].getValue());

            int i = 1;
            boolean encoding = false;
            if (i < aa.length && "encoding".equals(aa[i].getName())) {
                this.document.setEncoding(aa[i].getValue());
                i ++;
                encoding = true;
            }
            if (i < aa.length && "standalone".equals(aa[i].getName())) {
                String rv = aa[i].getValue();
                if ("yes".equals(rv)) {
                    this.document.setStandalone(rv);
                    this.doExternalDTD = false;
                } else if ("no".equals(rv)) {
                    this.document.setStandalone(rv);
                } else {
                    format1(tok, "E_XML3", rv);
                }
                i ++;
            }
            if (!encoding)
                tok.getReading().setEncoding("UTF-8", false);
            if (i < aa.length) {
                format1(tok, "E_XML5", aa[i].getName());
            }
            return null;

/*
        } else if ("xml:namespace".equals(n)) {
            TXAttribute[] aa = tok.getPIAttributes(false);
            tok.getChar();
            String url = null, as = null, src = null;
            StringBuffer sb = new StringBuffer(256);
            sb.append(" ");
            for (int i = 0;  i < aa.length;  i ++) {
                TXAttribute a = aa[i];
                if ("ns".equals(a.getName())) {
                    url = a.getValue();
                } else if ("prefix".equals(a.getName())) {
                    as = a.getValue();
                } else if ("src".equals(a.getName())) {
                    src = a.getValue();
                } else {
                    format1(tok, "E_XMLNS4", a.getName());
                }
                if (i != 0)  sb.append((char)' ');
                sb.append(a.toXMLString());
            }
            if (null == url) {
                format0(tok, "E_XMLNS0");
                return (TXPI)this.document.createProcessingInstruction(n, sb.toString());
            }
            int iv = Util.getInvalidURIChar(url);
            if (0 <= iv) {
                format2(tok, "E_EXT4", intstring(url.charAt(iv)), url);
            }
            if (null == as) {
                format0(tok, "E_XMLNS1");
                return (TXPI)this.document.createProcessingInstruction(n, sb.toString());
            }
            if (!Util.checkName(as) || 0 <= as.indexOf(':') || "xml".equals(as)) {
                format1(tok, "E_XMLNS2", as);
            }
            if (null != src) {
                iv = Util.getInvalidURIChar(src);
                if (0 <= iv)
                    format2(tok, "E_EXT4", intstring(src.charAt(iv)), src);
            }
            
            NamespacePI npi = this.document.createNamespacePI(n, sb.toString(), url, as, src);
            if (null != this.document.getNamespaceFor(as)) {
                format1(tok, "E_XMLNS3", as);
            }
            return npi;
*/
        } else if (StylesheetPI.S_XMLSTYLESHEET.equals(n)
                   || StylesheetPI.S_XMLALTSTYLESHEET.equals(n)) {
            TXAttribute[] aa = tok.getPIAttributes(false);
            tok.getChar();
            String type = null, href = null, title = null;
            StringBuffer sb = new StringBuffer(256);
            sb.append(" ");
            for (int i = 0;  i < aa.length;  i ++) {
                TXAttribute a = aa[i];
                if ("type".equals(a.getName())) {
                    type = a.getValue();
                } else if ("href".equals(a.getName())) {
                    href = a.getValue();
                } else if ("title".equals(a.getName())) {
                    title = a.getValue();
                } else {
                    format1(tok, "E_XMLSS0", a.getName());
                }
                if (i != 0)  sb.append((char)' ');
                sb.append(a.toXMLString());
            }
            if (null == type) {
                format0(tok, "E_XMLSS1");
                return this.document.createProcessingInstruction(n, sb.toString());
            }
            if (null == href) {
                format0(tok, "E_XMLSS2");
                return this.document.createProcessingInstruction(n, sb.toString());
            }
            int iv = Util.getInvalidURIChar(href);
            if (0 <= iv) {
                format2(tok, "E_EXT4", intstring(href.charAt(iv)), href);
            }
            
            StylesheetPI spi = this.document.createStylesheetPI(n, sb.toString(),
                                                                   type, href, title);
            return spi;

        } else if (n.length() >= 3 && "xml".equalsIgnoreCase(n.substring(0, 3))) {
            format0(tok, "E_PI5");
            return null;

        } else {                // PI except ?xml / ?xml:namespace
            String pidata = tok.getPIData();
            if (null == pidata) {
                format0(tok, "E_PI2");
                tok.nextTagStart();
                return null;
            }
            return this.document.createProcessingInstruction(n, pidata);
        }
    }

    String checkNamespace(Token tok, TXElement element, String qname) {
        int index = qname.indexOf(':');
        if (index < 0)  return null;
        if (null == element.getNamespaceForPrefix(qname.substring(0, index))) {
            return "E_TAGf";
        }
        if (0 <= qname.indexOf(':', index+1)) {
            return "E_TAGe";
        }
        return null;
    }

    /*
     * tok.m_next is next character of '<'.
     *
     * ispreservespace: t:preserve, nil:default(check this.isPreserveSpace)
     */
    TXElement parseElement(Token tok, Parent par, boolean ispreservespace)
        throws IOException {
        return parseElement(tok, par, ispreservespace, false, false);
    }
    
    TXElement parseElement(Token tok, Parent par, boolean ispreservespace,
                         boolean checktagname, boolean dohandler) throws IOException {
        String name = tok.getName();
        if (null == name) {
            format0(tok, "E_TAG0");
            tok.nextTagStart();
            return null;
        }

        if (dohandler)
            applyPreRootHandlers(name);

                                                // check DOCTYPE
        if (checktagname && null != this.document.getDTD()
            && !name.equals(this.document.getDTD().getName())) {
            format2(tok, "V_TAG5", this.document.getDTD().getName(), name);
        }
                                                // 
        TXAttribute[] aa = tok.getAttributes();
        int ch = tok.getChar();
        if (0 > ch)  throw new EOFException("com.ibm.xml.parser.Parser#parseElement(): "+getString("E_EOF"));
        if ('/' != ch && '>' != ch) {
            format0(tok, "E_TAG1");
            tok.nextTagStart();
        }
        boolean isempty = false;
        if ('/' == ch) {
            isempty = true;
            ch = tok.getChar();
            if (0 > ch)  throw new EOFException("com.ibm.xml.parser.Parser#parseElement(): "+getString("E_EOF"));
            if ('>' != ch) {
                format0(tok, "E_TAG1");
                tok.nextTagStart();
            }
        }

        TXElement tag = (TXElement)this.document.createElement(name);
                                                // Namespace
        tag.setParentNode(par);
        for (int i = 0;  i < aa.length;  i ++)
            tag.setAttributeNode(aa[i]);
        if (this.isProcessNamespace) {
            String[] localpartcache = new String[aa.length];
            for (int i = 0;  i < aa.length;  i ++) {
                String attrname = aa[i].getNodeName();
                String ret = this.checkNamespace(tok, tag, attrname);
                if (null != ret)  this.format1(tok, ret, attrname);
                if (aa[i].getValue().length() == 0 && attrname.startsWith("xmlns:"))
                    this.format1(tok, "E_TAGl", attrname);
                                                // The following code is O(n^2).
                localpartcache[i] = aa[i].getNSLocalName();
                String thisns = aa[i].getNSName();
                if (thisns != null) {
                    for (int j = 0;  j < i;  j ++) {
                        if (localpartcache[j].equals(localpartcache[i])) {
                            String ns = aa[j].getNSName();
                            if (ns != null && ns.equals(thisns))
                                this.format4(tok, "E_TAGm", aa[i].getNodeName(), thisns,
                                             localpartcache[i], aa[j].getNodeName());
                        }
                    }
                }
            }
        }

        if (this.isProcessNamespace) {
            String ret = this.checkNamespace(tok, tag, name);
            if (null != ret)  format1(tok, ret, name);
        }
        tag.setPreserveSpace(ispreservespace);

        if (this.document.isCheckValidity())  checkElementValidity(tok, tag, isempty);

        if (null != this.tagHandler)
            this.tagHandler.handleStartTag(tag, isempty);
                                                //
        String xmls = tag.getAttributeOrNull("xml:space");
        if (null != xmls) {
            if ("preserve".equals(xmls)) {
                ispreservespace = true;
            } else if ("default".equals(xmls)) {
                ispreservespace = this.isPreserveSpace;
            }
            tag.setPreserveSpace(ispreservespace);
        }

        if (isempty)  {
            pollForStop();
            if (this.tagHandler != null)
                this.tagHandler.handleEndTag(tag, !tag.hasChildNodes());
            tag = applyHandlers(tag);
            return tag;
        }
        /*
        if (null != this.document.getDTD()) {
            ContentModel cm = this.document.getDTD().getContentModel(tag.getName());
            if (null != cm && ElementDecl.EMPTY == cm.getType()) {
                format1(tok, "E_TAGk", tag.getName());
                return tag;
            }
        }
        */

        /*
         * content ::= (element | PCData | Reference | CDSect | PI | Commnet)*
         * ETag ::= '</' Name S? '>'
         */
        Node chi = null;
        boolean addchild = this.saxdriver == null || this.document.isCheckValidity();
        while (0 <= (ch = tok.m_next)) {
            chi = parseSingleContent(tok, name, tag, ispreservespace);
            if (chi == tag)  break;
            else if (null != chi) {
                switch (chi.getNodeType()) {
                  case Node.PROCESSING_INSTRUCTION_NODE:
                    tag.appendChild(chi);
                    applyPIHandlers((TXPI)chi);
                    break;

                  case Node.ELEMENT_NODE:
                    ((Child)chi).setParentNode(null);
                    if (addchild)
                        tag.appendChild(chi);
                    break;

                  case Node.TEXT_NODE:
                    tag.addTextElement((TXText)chi);
                    break;

                  case Node.ENTITY_REFERENCE_NODE:
                    if (this.isExpandEntityReferences) {
                        expandEntityReferences(tag, chi);
                    } else
                        tag.appendChild(chi);
                    break;

                  case Node.CDATA_SECTION_NODE:
                  default:
                    tag.appendChild(chi);
                }
            }
        }
        if (tag != chi)
            format1(tok, "E_TAG3", tag.getNodeName());
        /*
        if (!ispreservespace && 0 < tag.size()) {
            Child chi = tag.elementAt(tag.size()-1);
            if (chi instanceof TextElement
                && Util.checkAllSpace(chi.getText()))
                tag.removeElementAt(tag.size()-1);
        }
        */
        //if (0 > ch)  throw new EOFException(getString("E_EOF"));

        if (this.document.isCheckValidity())
            checkContentModel(tok, tag);

        pollForStop();
        if (this.tagHandler != null)
            this.tagHandler.handleEndTag(tag, !tag.hasChildNodes());
        tag = applyHandlers(tag);

        if (this.saxdriver != null) {
            while (tag.getFirstChild() != null)
                tag.removeChild(tag.getFirstChild());
        }

        return tag;
    }

    void expandEntityReferences(TXElement el, Node gr) {
        Node ch;
        while ((ch = gr.getFirstChild()) != null) {
            gr.removeChild(ch);
            if (ch.getNodeType() == Node.ENTITY_REFERENCE_NODE)
                expandEntityReferences(el, ch);
            else if (ch.getNodeType() == Node.TEXT_NODE)
                el.addTextElement((TXText)ch);
            else
                el.appendChild(ch);
        }
    }

    /*
     * content ::= (element | PCData | Reference | CDSect | PI | Commnet)*
     * ETag ::= '</' Name S? '>'
     */
    Node parseSingleContent(Token tok, String name, TXElement parent,
                            boolean ispreservespace) throws IOException {
        return parseSingleContent(tok, name, parent, ispreservespace, false);
    }
    
    Node parseSingleContent(Token tok, String name, TXElement parent,
                            boolean ispreservespace, boolean allowtextdecl) throws IOException {
        int ch;
        if (0 > (ch = tok.m_next))  return null;
        if ('<' == ch) {
            tok.getChar();                      // '<'
            ch = tok.m_next;
            if ('?' == ch) {                // PI
                ProcessingInstruction p = parsePI(tok, false, allowtextdecl, null);
                if (allowtextdecl && null == p)
                    return parseSingleContent(tok, name, parent, ispreservespace, false);
                /*
                if (null != p) {
                    tag.appendChild(p);
                    applyPIHandlers(p);
                }*/
                return p;

            } else if ('!' == ch) {             // comment | CDSect
                                                // <!-- | <![CDATA[
                tok.getChar();                  // '!'
                ch = tok.m_next;
                if ('-' == ch) {
                    tok.getChar();              // 1st '-'
                    if ('-' == tok.m_next) {    // comment
                        tok.getChar();          // 2nd '-'
                        if (this.isKeepComment)
                            return this.document.createComment(tok.getComment());
                        else
                            tok.getComment();
                    } else {
                        format0(tok, "E_COM0");
                        tok.skipTagEnd();
                    }

                } else if ('[' == ch) {         // CDATA
                    tok.getChar();              // 1st '['
                    String n = tok.getName();
                    if (null == n || !n.equals("CDATA") || '[' != tok.m_next) {
                        format0(tok, "E_CDATA0");
                        tok.skipTagEnd();
                    } else {
                        tok.getChar();          // 2nd '['
                        String d = tok.getCDATA();
                        if (null != d)
                            return this.document.createCDATASection(d);
                    }
                } else {
                    format0(tok, "E_!0");
                    tok.skipTagEnd();
                }

            } else if ('/' == ch) {             // ETag  '</' Name S? '>'
                tok.getChar();                  // '/'
                if (null == parent) {
                    format0(tok, "E_TAGg");
                    return null;
                }
                String n = tok.getName();
                boolean nomatch = false;
                if (null == n || !n.equals(name)) {
                    format1(tok, "E_TAG3", name);
                    nomatch = true;
                }
                tok.skipSpaces();
                if ('>' != tok.m_next) {
                    format0(tok, "E_TAG4");
                } else
                    ch = tok.getChar();         // '>'

                                                // Recovery
                if (nomatch && null != n) {
                    Node par = parent.getParentNode();
                    while (par.getNodeType() == Node.ELEMENT_NODE) {
                        if (par.getNodeName().equals(n)) {
                            pushPE(null);
                            tok.addReading(new StringReading("</"+n+">"));
                            break;
                        }
                        par = par.getParentNode();
                    }
                }

                return parent;                  // a special value which means an endtag

            } else {                            // element
                return parseElement(tok, parent, ispreservespace);
            }

        } else if ('&' == ch) {                 // Reference
            tok.getChar();                      // '&'
            boolean ischarref = '#' == tok.m_next;
            tok.doReference(true);
            if (ischarref) {
                return this.document.createTextNode(tok.m_tstrbuf, 0,
                                                            tok.m_tbufpos, false);
            } else {                            // &name;
                String entname = tok.m_lastreference;
                if (null == entname)  return null;
                Node gr = this.document.createEntityReference(entname);
                startGeneralReference(entname);
                if (!pushRef(entname)) {
                    format2(tok, "E_TAGi", entname, makeReferencePath(this.generalReferences, entname));
                    return null;
                }

                EntityDecl ev = tok.m_lastentity;
                if (null == ev)  return null;
                if (ev.getNotationName() != null) {
                    return null;
                }
                Token toks = ev.createToken(this, tok);
                boolean td = ev.isExternal();
                Node chi;
                do {
                    chi = parseSingleContent(toks, name, null, ispreservespace, td);
                    td = false;
                    if (null != chi)  gr.appendChild(chi);
                } while (toks.m_next >= 0);
                toks.close();
                endGeneralReference(entname);
                popRef(entname);
                if (!ev.getParsed()) {
                    if (this.saxdriver != null)
                        this.saxdriver.pauseEvent(true);
                    NodeList nl = gr.getChildNodes();
                    for (int i = 0;  i < nl.getLength();  i ++)
                        ev.appendChild(nl.item(i).cloneNode(true));
                    ev.setParsed(true);
                    if (this.saxdriver != null)
                        this.saxdriver.pauseEvent(false);
                }
                if (entname.equals("amp") || entname.equals("lt") || entname.equals("gt")
                    || entname.equals("quot") || entname.equals("apos")) {
                    gr = gr.getFirstChild();
                }
                return gr;
            }

        } else if (XMLChar.isChar((char)ch)) {  // PCData
            tok.getPCData();
            if (0 < tok.m_tbufpos) {
                if (tok.includebrabragt)
                    format0(tok, "E_TAGj");
                boolean ignorable;
                /*                           Comment out for defect #290
                if (parent == null)
                    ignorable = false;
                else*/
                    ignorable = !ispreservespace && tok.allspacepcdata;
                return (TXText)this.document.createTextNode(tok.m_tstrbuf, 0,
                                                            tok.m_tbufpos, ignorable);
            }
        } else {
            tok.getChar();
            format1(tok, "E_INVCHAR0", Integer.toString(ch, 16));
        }
        return null;
    }

    boolean pushRef(String n) {
        if (0 <= this.generalReferences.indexOf(n))
            return false;
        this.generalReferences.addElement(n);
        return true;
    }
    
    void popRef(String n) {
        this.generalReferences.removeElement(n);
    }
    
    boolean pushPE(String n) {
        if (n == null) {
            this.parameterEntityReferences.addElement(n);
            return true;
        }
        if (0 <= this.parameterEntityReferences.indexOf(n))
            return false;
        this.parameterEntityReferences.addElement(n);
        return true;
    }
    
    String popPE() {
        String ret = null;
        if (this.parameterEntityReferences.size() > 0) {
            int ind = this.parameterEntityReferences.size()-1;
            ret = (String)this.parameterEntityReferences.elementAt(ind);
            this.parameterEntityReferences.removeElementAt(ind);
        }
        return ret;
    }

    /*
     * attributecontent ::= (PCData | Reference)*
     */
    Node parseAttributeContent(Token tok, int startchar) throws IOException {
        int ch;
        if (0 > (ch = tok.m_next))  return null;
        if ('&' == ch) {                        // Reference
            tok.getChar();                      // '&'
            boolean ischarref = '#' == tok.m_next;
            String refv = tok.doReference(false);
            if (ischarref) {
                /*
                // Normalize
                if (tok.m_tbufpos == 1 && XMLChar.isSpace(tok.m_tstrbuf[0]))
                    tok.m_tstrbuf[0] = ' ';
                */
                return this.document.createTextNode(new String(tok.m_tstrbuf, 0, tok.m_tbufpos));
            } else {
                String entname = tok.m_lastreference;
                if (null == entname)  return null;
                Node gr = this.document.createEntityReference(entname);
                startGeneralReference(entname);
                if (!pushRef(entname)) {
                    format2(tok, "E_TAGi", entname, makeReferencePath(this.generalReferences, entname));
                    return null;
                }

                Token toks = new Token(this, refv);
                toks.getReading().setNext(tok.getReading());
                toks.setFollow(false);
                Node chi;
                do {
                    chi = parseAttributeContent(toks, -1);
                    if (null != chi)  gr.appendChild(chi);
                } while (null != chi);
                if (0 <= toks.m_next) {
                    format0(toks, "E_TAGh");
                }
                toks.close();
                endGeneralReference(entname);
                popRef(entname);
                if (entname.equals("amp") || entname.equals("lt") || entname.equals("gt")
                    || entname.equals("quot") || entname.equals("apos")) {
                    gr = gr.getFirstChild();
                }
                return gr;
            }

        } else if (XMLChar.isChar((char)ch)) {  // PCData
            String pcdata = tok.getPCDataForAttribute(startchar);
            if (0 < pcdata.length())
                // This factory method isn't overridden by SAXDriver
                return this.document.createTextNode(pcdata);

        } else {
            tok.getChar();
            format1(tok, "E_INVCHAR0", Integer.toString(ch, 16));
        }
        return null;
    }
    
    /*
     * m_next is S after `<!DOCTYPE'.
     * doctypedecl ::= '<!DOCTYPE' S Name (S ExternalID)? S? ('[' %markupdecl* ']' S?)? '>'
     */
    void parseDOCTYPE(Token tok) throws Exception {
        if (!XMLChar.isSpace((char)tok.m_next)) {
            format0(tok, "E_SPACE");
            tok.skipTagEnd();
            return;
        }
        tok.skipSpaces();
        String docname = tok.getName();
        if (null == docname) {
            format0(tok, "E_DOCTYPE1");
            tok.skipTagEnd();
            return;
        }

        tok.skipSpaces();

                                                // ExternalID | '[' | '>'
        ExternalID eid = tok.parseExternalID();
        this.dtd = this.document.createDTD(docname, eid);
        this.dtd.setEntityPool(this.entityPool);
        this.document.appendChild(this.dtd);

        tok.skipSpaces();                       // Internal subset
        if ('[' == tok.m_next) {
            tok.setState(Token.ST_INTERNALDTD);
            tok.getChar();                      // '['
            parseMarkupDecl(this.dtd, tok, false);
            tok.setState(Token.ST_NORMAL);
            if (']' == tok.m_next)  tok.getChar();
            tok.skipSpaces();
        }
        int ch = tok.m_next;
        if (0 > ch)  throw new EOFException("com.ibm.xml.parser.Parser#parseDOCTYPE(): "+getString("E_EOF"));
        if ('>' != ch) {
            format0(tok, "E_TAG4");
        } else {
            tok.getChar();                      // '>'
        }
                                                // load external DTD
        if (this.isProcessExternalDTD
            && null != eid
            && null != eid.getSystemLiteral()
            && this.doExternalDTD) {
            Token extoken = null;
            try {
                String name = eid.getSystemLiteral();
                if (null == name)  name = eid.getPubidLiteral();
                if (null == name)  name = "[DTD]";
                extoken = new Token(this, eid.getSystemLiteral(),
                                          this.streamProducer.getInputStream(name, eid.getPubidLiteral(),
                                                              eid.getSystemLiteral()));
                this.dtd.setParsingExternal(true);
                extoken.setState(Token.ST_EXTERNALDTD);
                parseMarkupDecl(this.dtd, extoken, true);
                this.dtd.setParsingExternal(false);
                extoken.close();
            } catch (Exception e) {
                error(null != extoken ? extoken : tok, e);
            }
        }
        this.checkNotationNames();
        checkUnparsedEntities();
        this.document.resetCheckValidity();

        //if (0 == dtd.getInternalSize()+dtd.getExternalSize())
        //    format0(tok, "E_DOCTYPE3");
    }

    void parseMarkupDecl(DTD par, Token tok, boolean allowencodingpi) throws Exception {
        while (parseSingleMarkupDecl(par, tok, allowencodingpi, true))
            allowencodingpi = false;
    }


    /*  '[' == tok.m_next */
    void parseConditionalSect(DTD par, Token tok) throws Exception {
        tok.getChar();                          // '['
        tok.skipSpaces();

        String n = tok.getName();
        tok.skipSpaces();
        if ('[' != tok.m_next || null == n) {
            format0(tok, "E_COND0");
            tok.skipTagEnd();
            return;
        }
        tok.getChar();                          // '['
        if ("INCLUDE".equals(n)) {
            int prevst = tok.setState(Token.ST_EXTERNALDTD);
            this.condSectLevel ++;
            parseMarkupDecl(par, tok, true);
            this.condSectLevel --;
            tok.setState(prevst);
            if (']' != tok.m_next) {
                format0(tok, "E_COND1");
                return;
            }
            tok.getChar();                      // ']'
            if (']' != tok.m_next) {
                format0(tok, "E_COND1");
                return;
            }
            tok.getChar();                      // ']'
            if ('>' != tok.m_next) {
                format0(tok, "E_COND1");
                return;
            }
            tok.getChar();                      // '>'

        } else if ("IGNORE".equals(n)) {
            int prevst = tok.setState(Token.ST_NORMAL);
            //ignoreSectContents(par, tok);
            ignoreSectContents(tok);
            tok.setState(prevst);

        } else {
            format0(tok, "E_COND0");
            tok.skipTagEnd();
        }
    }
    
    /*
     * [64]  ignoreSectContents ::=  Ignore ('<![' ignoreSectContents ']]>' Ignore)* 
     * [65]  Ignore ::=  Char* - (Char* ('<![' | ']]>') Char*)  
     */
    private void ignoreSectContents(Token tok) throws Exception {
        int ret = ignoreIgnore(tok);
        while (ret == 1) {
            ignoreSectContents(tok);
            ret = ignoreIgnore(tok);
        }
        return;
    }
    /**
     * @return 1= End by '<![',   2= End by ']]>'
     */
    private int ignoreIgnore(Token tok) throws Exception {
        int ch;
        int state = 0;
        while (state != 3 && state != 6) {
            ch = tok.getChar();
            if (ch < 0)
                throw new EOFException("Parser#ignoreIgnore(): "+getString("E_EOF"));
            switch (state) {
              case 0:
                if (ch == '<') {
                    state = 1;
                } else if (ch == ']') {
                    state = 4;
                }
                break;
              case 1:
                if (ch == '!')
                    state = 2;
                else if (ch == '<')
                    state = 1;
                else if (ch == ']')
                    state = 4;
                else
                    state = 0;
                break;
              case 2:
                if (ch == '[')
                    state = 3;
                else if (ch == '<')
                    state = 1;
                else if (ch == ']')
                    state = 4;
                else
                    state = 0;
                break;
              case 4:
                if (ch == ']')
                    state = 5;
                else if (ch == '<')
                    state = 1;
                else
                    state = 0;
                break;
              case 5:
                if (ch == '>')
                    state = 6;
                else if (ch == ']')
                    state = 5;
                else if (ch == '<')
                    state = 1;
                else
                    state = 0;
                break;
              default:
                throw new LibraryException("Parser#ignoreIgnore(): Internal Error: Invalid state: "+state);
            }
        }
        return state == 3 ? 1 : 2;
    }
    /*
    private void ignoreSectContents(Parent par, Token tok) throws Exception {
        int ch;
        while (0 <= (ch = tok.m_next) && ']' != ch) {
            if ('"' == ch || '\'' == ch) {      // SkipLit
                tok.getChar();
                int start = ch;
                while (0 <= (ch = tok.m_next)) {
                    if (start == ch)  break;
                    if (!XMLChar.isChar((char)ch)) {
                        format1(tok, "E_INVCHAR0", Integer.toString(ch, 16));
                        break;
                    }
                    tok.getChar();
                }
                if (0 >= ch)  throw new EOFException("com.ibm.xml.parser.Parser#ignoreSectContents(): "+getString("E_EOF"));
                tok.getChar();

            } else if ('<' == ch) {
                tok.getChar();                  // '<'
                ch = tok.m_next;
                if ('?' == ch) {                // PI
                    TXPI p = parsePI(tok, false);
                    if (null != p) {
                        par.appendChild(p);
                        applyPIHandlers(p);
                    }

                } else if ('!' == ch) {
                    tok.getChar();              // '!'
                    ch = tok.m_next;
                    if ('[' == ch) {
                        tok.getChar();          // '['
                        ignoreSectContents(par, tok);
                    } else if ('-' == ch) {
                        tok.getChar();          // '-'
                        if ('-' == tok.m_next) { // comment
                            tok.getChar();      // '-'
                            tok.getComment();
                        } else {
                            tok.getChar();
                            format0(tok, "E_COM0");
                        }
                    } else if (XMLChar.isChar((char)ch)) {
                        tok.getChar();
                    } else {
                        format1(tok, "E_INVCHAR0", Integer.toString(ch, 16));
                    }

                } else {
                    format0(tok, "E_COND3");
                }

            } else if (XMLChar.isChar((char)ch)) {
                tok.getChar();

            } else {
                tok.getChar();
                format1(tok, "E_INVCHAR0", Integer.toString(ch, 16));
            }
        }
        if (0 > ch)  throw new EOFException("com.ibm.xml.parser.Parser#ignoreSectContents(): "+getString("E_EOF"));
        tok.getChar();                          // ']'
        if (']' != tok.m_next) {
            format0(tok, "E_COND1");
            return;
        }
        tok.getChar();                          // ']'
        if ('>' != tok.m_next) {
            format0(tok, "E_COND1");
            return;
        }
        tok.getChar();                          // '>'
    }*/

    void parseNotation(Parent par, Token tok) throws Exception {
        if (!XMLChar.isSpace((char)tok.m_next)) {
            format0(tok, "E_SPACE");
            tok.skipTagEnd();
            return;
        }
        tok.skipSpaces();

        String n = tok.getName();
        if (null == n) {
            format0(tok, "E_NOT0");
            tok.skipTagEnd();
            return;
        }
        if (!XMLChar.isSpace((char)tok.m_next)) {
            format0(tok, "E_SPACE");
            tok.skipTagEnd();
            return;
        }
        tok.skipSpaces();

        ExternalID eid = tok.parseExternalID(true);
        if (null != eid)
            par.appendChild(this.document.createNotation(n, eid));
        tok.skipSpaces();
        if ('>' == tok.m_next)  tok.getChar();
    }
    
    static String makeReferencePath(Vector v, String refn) {
        StringBuffer sb = new StringBuffer(256);
        sb.append("(top-level)");
        synchronized (v) {
            for (int i = 0;  i < v.size();  i ++) {
                sb.append("-&");
                sb.append((String)v.elementAt(i));
                sb.append((char)';');
            }
        }
        sb.append("-&");
        sb.append(refn);
        sb.append((char)';');
        return sb.toString();
    }

    static String intstring(int ch) {
        return new Character((char)ch).toString();
    }
    
    
    boolean isWarningNoDoctypeDecl() {
        return this.isWarningNoDoctypeDecl;
    }
                                                // This method is called in constructors.
    private void init() {
                                /* Error messages:
                                 *   In Japanese environment, load parser.r.Message_ja
                                 *   Otherwise, load parser.r.Message (English messages)
                                 */
        this.resourceBundle = ListResourceBundle.getBundle("com.ibm.xml.parser.r.Message");
        this.entityPool = new EntityPool();
        this.stringPool = new StringPool();
    }

    private void init2() {
        this.entityPool.add(this.document.createEntityDecl("lt", "&#60;", false));
        this.entityPool.add(this.document.createEntityDecl("gt", "&#62;", false));
        this.entityPool.add(this.document.createEntityDecl("amp", "&#38;", false));
        this.entityPool.add(this.document.createEntityDecl("apos", "&#39;", false));
        this.entityPool.add(this.document.createEntityDecl("quot", "&#34;", false));
    }
    
    private void applyPreRootHandlers(String rootn) {
        if (null == this.preRootHandlers)  return;
        for (int i = 0;  i < this.preRootHandlers.size();  i ++) {
            ((PreRootHandler)this.preRootHandlers.elementAt(i)).handlePreRoot(this.document, rootn);
        }
    }
    
    private void applyPIHandlers(ProcessingInstruction p) {
        if (null == this.piHandlers)  return;
        String target = p.getNodeName();
        String data = p.getNodeValue();
        synchronized (this.piHandlers) {
            for (int i = 0;  i < this.piHandlers.size();  i ++) {
                PIHandler ph = (PIHandler)this.piHandlers.elementAt(i);
                ph.handlePI(target, data);
            }
        }
    }
    
    private TXElement applyHandlers(TXElement e) {
        if (null == this.elementHandlerHash)  return e;
        String u = e.getTagName();
        Vector v = (Vector)this.elementHandlerHash.get(u);
        if (null != v) {
            for (int i = 0;  i < v.size();  i ++) {
                e = ((ElementHandler)v.elementAt(i)).handleElement(e);
                if (null == e)  return null;
            }
        }
        v = (Vector)this.elementHandlerHash.get(s_enameall);
        if (null != v) {
            for (int i = 0;  i < v.size();  i ++) {
                e = ((ElementHandler)v.elementAt(i)).handleElement(e);
                if (null == e)  return null;
            }
        }
        return e;
    }
    
    private TXElement applyNoRequiredAttributeHandlers(TXElement el, String attname) {
        if (null == this.requiredAttributeHandlers)  return el;
        for (int i = 0;  i < this.requiredAttributeHandlers.size();  i ++) {
            NoRequiredAttributeHandler nrah
                = (NoRequiredAttributeHandler)this.requiredAttributeHandlers.elementAt(i);
            el = nrah.handleNoRequiredAttribute(el, attname);
            if (null != el.getAttributeOrNull(attname))
                break;
        }
        return el;
    }
    
    private void clean() {
        unparsedEntities.removeAllElements();
        this.resourceBundle = null;
        this.errorListener = null;
        this.streamProducer = null;
        this.entityPool = null;
        this.tagHandler = null;
        this.document = null;
        this.dtd = null;
        this.generalReferences = null;
        this.parameterEntityReferences = null;
        this.preRootHandlers = null;
        this.piHandlers = null;
        this.elementHandlerHash = null;
        this.requiredAttributeHandlers = null;
        this.contentModelErrors = null;
        this.idRefPool = null;
        this.notationNames = null;
        this.attributeErrors = null;
        this.referenceHandler = null;
        this.stringPool = null;
        this.saxdriver = null;
    }

    private void addIdrefPool(Token tok, String id) {
        if (null == this.idRefPool)  this.idRefPool = new Vector();
        this.idRefPool.addElement(new ParsePoint(tok, id));
    }
    
    private void checkIdref() {
        if (this.document.isCheckValidity()) {
            DTD d = this.document.getDTD();
            if (null != d && null != this.idRefPool) {
                for (int i = 0;  i < this.idRefPool.size();  i ++) {
                    ParsePoint pp = (ParsePoint)this.idRefPool.elementAt(i);
                    String idref = (String)pp.m_target;
                    if (null == d.checkID(idref))
                        format1(pp, "V_IDREF0", idref);
                }
            }
        }
        this.idRefPool = null;
    }

    private void addNotationName(Token tok, String notn) {
        if (null == this.notationNames)  this.notationNames = new Vector();
        this.notationNames.addElement(new ParsePoint(tok, notn));
    }
    
    private void checkNotationNames() {
        DTD d = this.document.getDTD();
        if (null != this.notationNames) {
            for (int i = 0;  i < this.notationNames.size();  i ++) {
                ParsePoint pp = (ParsePoint)this.notationNames.elementAt(i);
                String nn = (String)pp.m_target;
                if (null == d || null == d.getNotation(nn))
                    format1(pp, "E_ENUM5", nn);
            }
        }
        this.notationNames = null;
    }

    /**
     * This method checks to make sure that all of the unparsed entities
     * declared actually have matching declared notations. Unparsed
     * entities are those entities that declare that they have <i>NDATA</i>
     * of a specific notation type. An unparsed entity can specify that
     * it uses a specific notation type <em>before</em> the corresponding
     * notation is declared. For example:
     * <pre>
     * &lt;!ENTITY image SYSTEM "image.gif" NDATA gif&gt;
     * &lt;!NOTATION gif "<i>helper_application</i>"&gt;
     * </pre>
     */
    private void checkUnparsedEntities() {

        DTD dtd = document.getDTD();
        if (dtd != null) {
            Enumeration entities = unparsedEntities.elements();
            while (entities.hasMoreElements()) {
                // get parse location and notation name used at that point
                ParsePoint loc  = (ParsePoint)entities.nextElement();
                String     name = (String)loc.m_target;
    
                // Make sure that this the corresponding notation
                // was declared. -Ac
                if (name != null) {
                    if (dtd.getNotation(name) == null) {
                        format1(loc, "V_ENTITY8", name);
                        }
                    }
                }
            }

        } // checkUnparsedEntities()

    /*
     * @see com.ibm.xml.parser.TXDocument#isAddFixedAttributes()
     */
    private void checkElementValidity(Token tok, TXElement tag, boolean isempty) {
        DTD d = this.document.getDTD();
        if (null == d)   return;
        String name = tag.getTagName();
        TXAttribute[] aa = tag.getAttributeArray();
                                                // Check if there is #REQUIRED attribute.
                                                // Add default attribute
        for (Enumeration en = d.getAttributeDeclarations(name);
             en.hasMoreElements(); ) {
            AttDef ad = (AttDef)en.nextElement();
            String aname = ad.getName();
            switch (ad.getDefaultType()) {
              case AttDef.REQUIRED:
                if (null == tag.getAttributeNode(aname))
                    tag = applyNoRequiredAttributeHandlers(tag, aname);
                Attr a = tag.getAttributeNode(aname);
                if (null == a)
                    format1(tok, "V_TAGc", aname);
                break;
              case AttDef.IMPLIED:
                break;
              case AttDef.FIXED:
                String fixs = tag.getAttributeOrNull(aname);
                if (null == fixs) {
                    if (this.document.isAddFixedAttributes()) {
                        Attr attr = this.document.createAttribute(aname);
                        attr.setNodeValue(ad.getDefaultStringValue());
                        ((TXAttribute)attr).setSpecified(false);
                        tag.setAttributeNode(attr);
                    }
                } else if (!TXAttribute.normalize(ad.getDeclaredType(), fixs).equals(ad.getDefaultStringValue()))
                    format3(tok, "V_TAGd", aname, ad.getDefaultStringValue(), fixs);
                break;
              case AttDef.NOFIXED:
                if (null == tag.getAttributeNode(aname)) {
                    Attr attr = this.document.createAttribute(aname);
                    attr.setNodeValue(ad.getDefaultStringValue());
                    ((TXAttribute)attr).setSpecified(false);
                    tag.setAttributeNode(attr);
                }
                break;
            }
        }
                                                // Check attribute type
        for (int i = 0;  i < aa.length;  i ++) {
            TXAttribute a = aa[i];
            AttDef ad = d.getAttributeDeclaration(name, a.getName());
            if (null == ad) {
                if (null == this.attributeErrors)  this.attributeErrors = new Hashtable();
                String key = name+" "+a.getName();
                if (!this.attributeErrors.containsKey(key)) {
                    this.attributeErrors.put(key, key);
                    format2(tok, "V_TAG6", a.getName(), name);
                }
                continue;
            }
            this.checkAttributeDefaultValue(tok, ad, a.getValue().trim(), a, d, tag);
        }

        if (isempty)
            checkContentModel(tok, tag);
    }

    private void checkContentModel(Token tok, TXElement tag) {
        DTD d = this.document.getDTD();
        if (null != d) {
            String name = tag.getTagName();
            ContentModel cs = d.getContentModel(name);
            if (null == cs) {
                if (null == this.contentModelErrors)  this.contentModelErrors = new Hashtable();
                String key = name;
                if (!this.contentModelErrors.containsKey(key)) {
                    this.contentModelErrors.put(key, name);
                    format1(tok, "V_CONT0", name);
                }
            } else {
                if (!cs.check(tag))
                    format2(tok, "V_CONT1", name, cs.toString());
            }
        }
    }

    /**
     *
     * @param a  non-null=for real attribute values, null=for default attribute values
     */
    private void checkAttributeDefaultValue(Token tok, AttDef ad, String v,
                                            TXAttribute a, DTD d, TXElement tag) {
        int attt = ad.getDeclaredType();
        String[] as = null;
        if (AttDef.CDATA == attt) {
            as = new String[1];
            if (a != null) {
                as[0] = a.getValue();
                a.setType(AttDef.CDATA, as);
            }
        } else {
            switch (attt) {
              case AttDef.NAME_TOKEN_GROUP:
                if (!Util.checkNmtoken(v)) {
                    format1(tok, "V_TAG7", v);
                } else if (ad.contains(v)) {
                    as = new String[1];
                    as[0] = v;
                } else {
                    format1(tok, "V_TAG9", v);
                }
                if (a != null)
                    a.setType(AttDef.NAME_TOKEN_GROUP, as);
                break;
              case AttDef.NOTATION:
                if (!Util.checkName(v)) {
                    format1(tok, "V_TAG7", v);
                } else if (ad.contains(v)) {
                    as = new String[1];
                    as[0] = v;
                } else {
                    format1(tok, "V_TAGb", v);
                }
                if (a != null)
                    a.setType(AttDef.NOTATION, as);
                break;
              case AttDef.ID:
                if (!Util.checkName(v)) {
                    format1(tok, "V_TAG7", v);
                } else if (a != null) {
                    if (!d.registID(tag, v))
                        format1(tok, "V_TAG8", v);
                    else {
                        as = new String[1];
                        as[0] = v;
                    }
                    a.setType(AttDef.ID, as);
                }
                break;
              case AttDef.IDREF:
                if (!Util.checkName(v)) {
                    format1(tok, "V_TAG7", v);
                } else if (a != null) {
                    as = new String[1];
                    as[0] = v;
                    addIdrefPool(tok, v);
                    a.setType(AttDef.IDREF, as);
                }
                break;
              case AttDef.IDREFS:
                as = tok.getNames(v);
                if (a != null) {
                    for (int ai = 0;  ai < as.length;  ai ++)
                        addIdrefPool(tok, as[ai]);
                    a.setType(AttDef.IDREFS, as);
                }
                break;
              case AttDef.ENTITY:
                if (!Util.checkName(v)) {
                    format1(tok, "V_TAG7", v);
                } if (a != null) {
                    EntityDecl ev = this.entityPool.refer(v);
                    if (null == ev || !ev.isExternal() || ev.getNotationName() == null) {
                        format1(tok, "V_TAGa", v);
                    } else {
                        as = new String[1];
                        as[0] = v;
                    }
                    a.setType(AttDef.ENTITY, as);
                }
                break;
              case AttDef.ENTITIES:
                as = tok.getNames(v);
                if (a != null) {
                    for (int ai = 0;  ai < as.length;  ai ++) {
                        EntityDecl ev = this.entityPool.refer(as[ai]);
                        if (null == ev || !ev.isExternal() || ev.getNotationName() == null)
                            format1(tok, "V_TAGa", v);
                    }
                    a.setType(AttDef.ENTITIES, as);
                }
                break;
              case AttDef.NMTOKEN:
                if (!Util.checkNmtoken(v)) {
                    format1(tok, "V_TAG7", v);
                } else if (a != null) {
                    as = new String[1];
                    as[0] = v;
                    a.setType(AttDef.NMTOKEN, as);
                }
                break;
              case AttDef.NMTOKENS:
                as = tok.getNmtokens(v);
                if (a != null)
                    a.setType(AttDef.NMTOKENS, as);
            }
        }
    }
    
    private boolean parseSingleMarkupDecl(DTD par, Token tok, boolean allowencodingpi,
                                          boolean toplevel) throws Exception {
        int ch = tok.m_next;
        if (0 > ch || ((!par.isParsingExternal()||this.condSectLevel > 0) && ']' == ch))
            return false;

        // markupdecl
        //  ::= (elementdecl | AttlistDecl | EntityDecl | NotationDecl
        //       | PI | S | Comment | InternalPERef)*
        // <!ELEMENT <!ATTLIST <!ENTITY <!NOTATION <?Name S <!-- %Name; <![ 
            
        if (XMLChar.isSpace((char)ch)) {
            tok.getWhitespaces();
            par.appendChild(this.document.createTextNode(tok.m_tstrbuf, 0, tok.m_tbufpos, true));

        } else if ('<' == ch) {
            tok.getChar();                      // '<'
            ch = tok.m_next;
            if ('!' == ch) {                    // ELEMENT | ATTLIST | ENTITY | NOTATION | [ | --
                tok.getChar();                  // '!'
                ch = tok.m_next;
                if ('[' == ch) {                // check '<![' %'INCLUDE' | '<![' %'IGNORE'
                    if (!par.isParsingExternal())
                        format0(tok, "E_DTD4");
                    int prevst = tok.setState(Token.ST_TAG);
                    parseConditionalSect(par, tok);
                    tok.setState(prevst);

                } else if ('-' == ch) {         // check '<!--'
                    tok.getChar();              // 1st '-'
                    if ('-' == tok.m_next) {
                        tok.getChar();          // 2nd '-'
                        if (this.isKeepComment) {
                            Comment com = this.document.createComment(tok.getComment());
                            if (com != null)  par.appendChild(com);
                        } else
                            tok.getComment();
                    } else {
                        format0(tok, "E_COM0");
                        tok.skipTagEnd();
                    }

                } else {
                    int prevst = tok.setState(Token.ST_TAG);
                    String name = tok.getName();
                    if (null == name) {
                        format0(tok, "E_DTD3");
                        tok.skipTagEnd();
                    } else if ("ELEMENT".equals(name)) {
                        parseElementDecl(par, tok);
                    } else if ("ATTLIST".equals(name)) {
                        parseAttlist(par, tok);
                    } else if ("ENTITY".equals(name)) {
                        parseEntityDecl(par, tok);
                    } else if ("NOTATION".equals(name)) {
                        parseNotation(par, tok);
                    } else {
                        format1(tok, "E_DTD2", name);
                        tok.skipTagEnd();
                    }
                    tok.setState(prevst);
                }
            } else if ('?' == ch) {             // PI
                ProcessingInstruction p = parsePI(tok, false, allowencodingpi, toplevel ? dtd : null);
                if (null != p) {
                    par.appendChild(p);
                    applyPIHandlers(p);
                }
                        
            } else {
                format0(tok, "E_DTD1");
                tok.nextTagStart();
            }
                    
        } else if ('%' == ch) {                 // PEReference
                                                // This PE must include markup declarations.
            tok.getChar();                      // '%'
            String pename = tok.getName();
            if (null == pename) {
                format0(tok, "E_PEREF0");
            } else {
                if (tok.m_next != ';')
                    format1(tok, "E_PEREF1", pename);
                else
                    tok.getChar();
                if (!pushPE(pename)) {
                    format2(tok, "E_PEREF5", pename,
                            makeReferencePath(this.parameterEntityReferences, pename));
                } else {
                    EntityDecl per = this.entityPool.referPara(pename);
                    if (null == per) {
                        format1(tok, "V_PEREF2", pename);
                    } else {
                        Token toks = per.createToken(this, tok);
                        boolean textdecl = per.isExternal();
                        while (parseSingleMarkupDecl(par, toks, textdecl, false))
                            textdecl = false;

                        if (']' == toks.m_next)
                            format2(toks, "E_DTD0", intstring(toks.m_next), Integer.toHexString(toks.m_next));
                        toks.close();
                    }
                    popPE();
                }
            }

        } else {
            tok.getChar();
            format2(tok, "E_DTD0", intstring(ch), Integer.toHexString(ch));
            while (0 <= (ch = tok.m_next) && '<' != ch && ']' != ch)
                tok.getChar();
        }
        return true;
    }

    /*
     * elementdecl  ::= '<!ELEMENT' S Name S contentspec S? '>'
     *  contentspec ::= 'EMPTY' | 'ANY' | Mixed | elements
     */
    private void parseElementDecl(DTD par, Token tok) throws Exception {
        if (!XMLChar.isSpace((char)tok.m_next)) {
            format0(tok, "E_SPACE");
            tok.skipTagEnd();
            return;
        }
        tok.skipSpaces();

        String ename = tok.getName();
        if (null == ename) {
            format0(tok, "E_ELEM0");
            tok.skipTagEnd();
            return;
        }

        if (!XMLChar.isSpace((char)tok.m_next)) {
            format0(tok, "E_SPACE");
            tok.skipTagEnd();
            return;
        }
        int prevstate = tok.setState(Token.ST_CONTENTMODEL);
        tok.skipSpaces();

        ContentModel cs = null;
        if ('(' != tok.m_next) {
            String csn = tok.getName();
            if (null == csn) {
                format0(tok, "E_ELEM1");
                tok.skipTagEnd();
            } else if ("EMPTY".equals(csn)) {
                cs = this.document.createContentModel(ElementDecl.EMPTY);
            } else if ("ANY".equals(csn)) {
                cs = this.document.createContentModel(ElementDecl.ANY);
            } else {
                format0(tok, "E_ELEM1");
                tok.skipTagEnd();
            }
        } else {                                // regexp-like content model
            CMNode cmn = null;
            this.parenLevel = 0;
            cmn = parseContentModel0(tok);
            if (null != cmn) {
                if (!checkPCDATA(cmn)) {
                    format1(tok, "E_CS9", cmn.toString());
                }
                cs = this.document.createContentModel(cmn);
            } else
                tok.skipTagEnd();
            this.contentModelPCData = null;
        }
        if (cs != null) {
            if (par.isElementDeclared(ename)) {
                format1(tok, "V_ELEM3", ename);
            } else
                par.appendChild(this.document.createElementDecl(ename, cs));
        }
        tok.setState(prevstate);
        tok.skipSpaces();                       // S?
        int ch = tok.m_next;
        if ('>' == ch)
            tok.getChar();
        else
            format0(tok, "E_ELEM2");
    }

    /*
     *  contentspec ::= 'EMPTY' | 'ANY' | Mixed | children
     *   children   ::= (choice | seq) ('?' | '*' | '+')?
     *    choice    ::= '(' S? cp (S? '|' S? %cp)* S? ')'
     *       cp     ::= (Name | choice | seq) ('?' | '*' | '+')?
     *    seq       ::= '(' S? cp (S? ',' S? %cp)* S? ')'
     *   Mixed      ::= '(' S? '#PCDATA' (S? '|' S? Name)* S? ')*' |
     *                  '(' S? #PCDATA') S? ')'
     */
    private boolean checkPCDATA(CMNode cn) {
        CMNode pcdata = this.contentModelPCData;
        this.contentModelPCData = null;
        if (null == cn)  return false;
        if (null == pcdata)  return true;
        if (cn instanceof CM2op)  return false;
        if (cn instanceof CMLeaf)  return cn == pcdata;
        CM1op c1 = (CM1op)cn;
        if (c1.getType() != '*')  return false;
        CMNode c = c1.getNode();
        while (!(c instanceof CMLeaf)) {
            if (c instanceof CM1op)  return false;
            CM2op c2 = (CM2op)c;
            if (c2.getType() != '|')  return false;
            if (!(c2.getRight() instanceof CMLeaf))  return false;
            c = c2.getLeft();
        }
        return pcdata == c;
    }

    private CMNode parseContentModel0(Token tok) throws Exception {
        CMNode cs = null;
        if ('(' == tok.m_next) {
            tok.getChar();                      // '('
            this.parenLevel ++;
            cs = parseContentModelChoiceSeq(tok, null, 0);
            if (')' == tok.m_next) {
                tok.getChar();
                this.parenLevel --;
            } else
                format0(tok, "E_CS3");
        } else if ('#' == tok.m_next) {
            tok.getChar();                      // '#'
            String pcdata = tok.getName();
            if (null == pcdata || !"PCDATA".equals(pcdata)) {
                format0(tok, "E_CS2");
                while (0 <= tok.m_next && ')' != tok.m_next)
                    tok.getChar();
                if (')' == tok.m_next) {
                    tok.getChar();
                    this.parenLevel --;
                }
                return null;
            }
            if (this.parenLevel != 1) {
                format0(tok, "E_CSa");
            }
            cs = new CMLeaf("#PCDATA");
            this.contentModelPCData = cs;
        } else {
            String csn = tok.getName();
            if (null == csn) {
                format0(tok, "E_CS5");
                return cs;
            }
            cs = new CMLeaf(csn);
        }
        int ch = tok.m_next;
        if ('?' == ch || '*' == ch || '+' == ch) {
            tok.getChar();                      // ?*+
            if (null != cs) cs = new CM1op(ch, cs);
        }
        return cs;
    }

    /*
      ChoiceSeq := node (op node)*
     */
    private CMNode parseContentModelChoiceSeq(Token tok, CMNode prev, int op) throws Exception {
        while (true) {
            tok.skipSpaces();               // S?
            {
                CMNode c = parseContentModel0(tok);
                if (null == c)  return null;
                if (null == prev) {
                    prev = c;
                } else
                    prev = new CM2op(op, prev, c);
            }

            tok.skipSpaces();
            int ch = tok.m_next;
            if (')' == ch) {
                break;
            } else if ('|' == ch || ',' == ch) {
                tok.getChar();                  // '|' or ','
                if (0 != op && op != ch)
                    format2(tok, "E_CS6", intstring(ch), intstring(op));
                op = ch;
            } else if (0 > ch) {
                break;
            } else {
                format0(tok, "E_CS1");
                while (0 <= tok.m_next && ')' != tok.m_next)
                    tok.getChar();
            }
        }
        return prev;
    }

    /*
     * AttlistDecl ::= '<!ATTLIST' S Name S? AttDef+ S? '>'
     * AttDef ::= S Name S AttType S Default
     */
    private void parseAttlist(DTD par, Token tok) throws Exception {
        /*
        if (!XMLChar.isSpace((char)tok.m_next)) {
            format0(tok, "E_SPACE");
            tok.skipTagEnd();
            return;
            }*/
        tok.skipSpaces();

        String ename = tok.getName();
        if (null == ename) {
            format0(tok, "E_ATTL0");
            tok.skipTagEnd();
            return;
        }

        tok.skipSpaces();

        Attlist al = this.document.createAttlist(ename);
        while (parseAttDefs(par, tok, al));
        par.appendChild(al);
        if (0 > tok.m_next)  throw new EOFException("com.ibm.xml.parser.Parser#parseAttlist(): "+getString("E_EOF"));
        tok.getChar();                          // '>'
    }
    
    private boolean parseAttDefs(DTD dtd, Token tok, Attlist al) throws Exception {
        int ch = tok.m_next;
        if (0 > ch || '>' == ch)  return false;

        AttDef ad = parseAttDef(dtd, tok, al);
        if (null != ad) {
            if (dtd.isAttributeDeclared(al.getName(), ad.getName())
                ||null != al.getAttDef(ad.getName())) {
                format1(tok, "W_ATTL2", ad.getName());
            } else {
                if ("xml:space".equals(ad.getName())) {
                    if (AttDef.NAME_TOKEN_GROUP != ad.getDeclaredType()
                        || 2 != ad.size()) {
                        format0(tok, "W_ATTL1");
                    } else {
                        String s1 = ad.elementAt(0);
                        String s2 = ad.elementAt(1);
                        if (!("default".equals(s1) && "preserve".equals(s2))
                            && !("preserve".equals(s1) && "default".equals(s2))) {
                            format0(tok, "W_ATTL1");
                        }
                    }
                }
                if (AttDef.ID == ad.getDeclaredType()) {
                    Enumeration en = dtd.getAttributeDeclarations(al.getName());
                    while (en.hasMoreElements()) {
                        AttDef a = (AttDef)en.nextElement();
                        if (AttDef.ID == a.getDeclaredType()) {
                            format1(tok, "V_ATTL2", a.getName());
                            ad = null;
                            break;
                        }
                    }
                    en = al.elements();
                    while (en.hasMoreElements()) {
                        AttDef a = (AttDef)en.nextElement();
                        if (AttDef.ID == a.getDeclaredType()) {
                            format1(tok, "V_ATTL2", a.getName());
                            ad = null;
                            break;
                        }
                    }
                    // Check #REQUIRED or #IMPLIED
                    int defaultType = ad.getDefaultType();
                    if (defaultType != AttDef.REQUIRED && defaultType != AttDef.IMPLIED)
                        format1(tok, "V_ATTL3", ad.getName());
                }
                if (ad.getDefaultStringValue() != null)
                    this.checkAttributeDefaultValue(tok, ad, ad.getDefaultStringValue(),
                                                    null, null, null);
                al.addElement(ad);
            }
        }
        tok.skipSpaces();
        return true;
    }
    
    /*
     * AttDef ::= S %Name S %AttType S %Default
     */
    private AttDef parseAttDef(DTD dtd, Token tok, Attlist al) throws Exception {
        tok.skipSpaces();                       // S
        if ('>' == tok.m_next)  return null;
        String aname = tok.getName();           // %Name
        if (null == aname) {
            format0(tok, "E_ATTD0");
            tok.getChar();
            return null;
        }
                                                // S
        if (!XMLChar.isSpace((char)tok.m_next)) {
            format0(tok, "E_SPACE");
            tok.skipTagEnd();
            return null;
        }
        tok.skipSpaces();

        AttDef ret = this.document.createAttDef(aname);

                                                // %AttType
        ret = parseAttType(dtd, tok, ret);
        if (null == ret)  return null;
        if (!XMLChar.isSpace((char)tok.m_next))
            format0(tok, "E_SPACE");
        tok.skipSpaces();

                                                // %DefaultDecl
        ret = parseAttDefaultDecl(tok, ret);
        return ret;
    }
    
    /*
     * AttType ::= 'CDATA' | 'ID' | 'IDREF' | 'IDREFS' | 'ENTITY' | 'ENTITIES'
     *           | 'NMTOKEN' | 'NMTOKENS' | NotationType | Enumeration
     * NotationType ::= %'NOTATION' S '(' S? %Ntoks (S? '|' S? %Ntoks)* S? ')'
     * Enumeration ::= '(' S? %Etoks (S? '|' S? %Etoks)* S? ')'
     *  Ntoks      ::= %Name (S? '|' S? %Name)*
     *  Etoks      ::= %Nmtoken (S? '|' S? %Nmtoken)*
     *   Nmtoken   ::= NameChar+
     */
    private AttDef parseAttType(DTD dtd, Token tok, AttDef ret) throws Exception {
        String typen = null;
        int ch = tok.m_next;

        if ('(' == ch) {                 // Enumeration
            ret.setDeclaredType(AttDef.NAME_TOKEN_GROUP);
            parseEnumeration(dtd, tok, ret, true);
        } else {
            typen = tok.getName();
        }

        if (AttDef.NAME_TOKEN_GROUP == ret.getDeclaredType()) {
        } else if (null == typen) {
            format0(tok, "E_ATTD2");
            return null;
        } else if ("CDATA".equals(typen)) {
            ret.setDeclaredType(AttDef.CDATA);
        } else if ("ID".equals(typen)) {
            ret.setDeclaredType(AttDef.ID);
        } else if ("IDREF".equals(typen)) {
            ret.setDeclaredType(AttDef.IDREF);
        } else if ("IDREFS".equals(typen)) {
            ret.setDeclaredType(AttDef.IDREFS);
        } else if ("ENTITY".equals(typen)) {
            ret.setDeclaredType(AttDef.ENTITY);
        } else if ("ENTITIES".equals(typen)) {
            ret.setDeclaredType(AttDef.ENTITIES);
        } else if ("NMTOKEN".equals(typen)) {
            ret.setDeclaredType(AttDef.NMTOKEN);
        } else if ("NMTOKENS".equals(typen)) {
            ret.setDeclaredType(AttDef.NMTOKENS);
        } else if ("NOTATION".equals(typen)) {
            ret.setDeclaredType(AttDef.NOTATION);
            if (!XMLChar.isSpace((char)tok.m_next)) {
                format0(tok, "E_SPACE");
            }
            tok.skipSpaces();
            parseEnumeration(dtd, tok, ret, false);
        } else {
            ret.setDeclaredType(AttDef.CDATA);
            format0(tok, "E_ATTD2");
        }
                                                // S
        return ret;
    }

    /*
     * DefaultDecl ::= '#REQUIRED' | '#IMPLIED' | ((%'#FIXED' S)? %AttValue)
     */
    private AttDef parseAttDefaultDecl(Token tok, AttDef ret) throws Exception {
        int ch = tok.m_next;
        if ('#' == ch) {
            tok.getChar();                      // '#'
            String n = tok.getName();
            if (null == n) {
                format0(tok, "E_ATTD3");
                tok.getChar();
                return null;
            } else if ("REQUIRED".equals(n)) {
                ret.setDefaultType(AttDef.REQUIRED);
            } else if ("IMPLIED".equals(n)) {
                ret.setDefaultType(AttDef.IMPLIED);
            } else if ("FIXED".equals(n)) {
                ret.setDefaultType(AttDef.FIXED);
                if (!XMLChar.isSpace((char)tok.m_next)) {
                    format0(tok, "E_SPACE");
                }
                tok.skipSpaces();
                int prevstate = tok.setState(Token.ST_NORMAL);
                ret.setDefaultStringValue(tok.getAttValue());
                tok.setState(prevstate);
            } else {
                format0(tok, "E_ATTD3");
                return null;
            }
        } else if ('"' == ch || '\'' == ch) {
            int prevstate = tok.setState(Token.ST_NORMAL);
            ret.setDefaultStringValue(TXAttribute.normalize(ret.getDeclaredType(),
                                                            tok.getAttValue()));
            tok.setState(prevstate);

        } else {
            format0(tok, "E_ATTD4");
            return null;
        }
        return ret;
    }

    /*
     *  NotationType ::= 'NOTATION' S '(' S? Name (S? '|' S? Name)* S? ')'
     *  Enumeration  ::= '(' S? Name (S? '|' S? Name)* S? ')'
     */
    private void parseEnumeration(DTD dtd, Token tok, AttDef ad, boolean isnmtok) throws Exception {
        int ch = tok.m_next;
        if ('(' != ch) {
            format0(tok, "E_ENUM0");
            return;
        }
        tok.getChar();                          // '('
        tok.skipSpaces();                       // S?
        while (true) {
            String n = isnmtok ? tok.getNmtoken() : tok.getName();
            if (null == n) {
                format0(tok, isnmtok ? "E_ENUM2" : "E_ENUM1");
            } else {
                if (!isnmtok)                 // NOTATION
                    addNotationName(tok, n);
                ad.addElement(n);
                    /*
                if (!ad.addElement(n)) {
                    format1(tok, "E_ENUM4", n);
                    }*/
            }
            tok.skipSpaces();
            ch = tok.m_next;
            if (')' == ch)
                break;
            else if ('|' == ch) {
                tok.getChar();
                tok.skipSpaces();
            } else {
                format0(tok, "E_ENUM3");
                break;
            }
        }
        if (')' == ch)  tok.getChar();
    }

    /*
     * EntityDecl ::= '<!ENTITY' S (Name | '%' S Name) S EntityDef S? '>'
     * EntityDef ::= (EntityValue) | (ExternalID (S 'NDATA' S Name)? )
     * EntityValue ::=  '"' ([^%&"] | PEReference | Reference)* '"'  
     *                | "'" ([^%&'] | PEReference | Reference)* "'"
     */
    private void parseEntityDecl(DTD par, Token tok) throws Exception {
        if (!XMLChar.isSpace((char)tok.m_next)) {
            format0(tok, "E_SPACE");
            tok.skipTagEnd();
            return;
        }
        tok.skipSpaces();

        boolean ispara = false;
        String name = null;
        if ('%' == tok.m_next) {
            tok.getChar();                      // '%'
            if (XMLChar.isSpace((char)tok.m_next)) {
                tok.skipSpaces();
                ispara = true;                  // A declaration of Parameter Entity
                name = tok.getName();
            } else {
                //tok.expandPEReferenceWithoutGet();
                name = tok.getName();
            }
        } else
            name = tok.getName();
        if (null == name) {
            format0(tok, "E_ENTITY1");
            tok.skipTagEnd();
            return;
        }

        if (!XMLChar.isSpace((char)tok.m_next)) {
            format0(tok, "E_SPACE");
            tok.skipTagEnd();
            return;
        }
        tok.skipSpaces();

                                                // EntityDef
        int start = tok.m_next;
        if ('"' == start || '\'' == start) {    // EntityValue
            StringBuffer sb = new StringBuffer(256);
            int lv = tok.getLevel();
            int prevstate = tok.setState(Token.ST_VALUE);
            tok.getChar();
            while (true) {
                int n = tok.m_next;
                if ((lv == tok.getLevel() && n == start) || 0 > n)  break;
                tok.getChar();
                int ch = n;
                if (!XMLChar.isChar(ch)) {
                    format1(tok, "E_ENTITY3", Integer.toString(ch, 16));
                    sb.append((char)'?');
                } else if ('&' == ch) {
                    if ('#' == tok.m_next) {    // character referece is included.
                        tok.doReference();
                        sb.append(tok.m_tstrbuf, 0, tok.m_tbufpos);
                    } else {    // generic reference isn't included.
                        sb.append((char)'&');
                        String nm = tok.getName();
                        if (null == nm) {
                            format1(tok, "E_REFER0", intstring(tok.m_next));
                        } else
                            sb.append(nm);
                        if (';' != tok.m_next) {
                            format0(tok, "E_REFER1");
                        } else {
                            tok.getChar();
                            sb.append((char)';');
                        }
                    }
                } else if ('%' == ch) {
                    format0(tok, "E_ENTITYa");
                } else {
                    sb.append((char)ch);
                }
            }
            tok.setState(prevstate);
            if (0 > tok.m_next)  return;
            tok.getChar();                      // start
            tok.skipSpaces();
            if ('>' != tok.m_next) {
                format0(tok, "E_ENTITY4");
            } else
                tok.getChar();                  // '>'
            String cont = sb.toString();
            EntityDecl ent = this.document.createEntityDecl(name, cont, ispara);
            // Note: 
            // 1) entityPool.add will return false if preexisting, and only add 
            //    it if it is not preexisting.
            // 2) appendChild calls realInsert which also calls entityPool.add
            //    as well as appending the child.
            // So we can first check/add and report warning before appendChild.
            if (isWarningRedefinedEntity() && !this.entityPool.add(ent))
            {
                if ("lt".equals(name) || "gt".equals(name) || "amp".equals(name)
                    || "quot".equals(name) || "apos".equals(name)) {
                } else
                    format1(tok, "W_ENTITY6", name);
            } 
            par.appendChild(ent);

        } else {                                // ExternalID (S %'NDATA' S %Name)?
            ExternalID eid = tok.parseExternalID();
            if (null == eid)  return;
            boolean sp = false;
            if (XMLChar.isSpace((char)tok.m_next)) {
                tok.skipSpaces();
                sp = true;
            }
                                                // '%' or 'NDATA' or '>'
            String ndata = null;
            String ntype = null;
            int ch = tok.m_next;
            if ('>' != ch && ispara) {
                format0(tok, "E_ENTITY4");
                tok.skipTagEnd();

            } else if ('>' != ch) {
                if (!sp)
                    format0(tok, "E_SPACE");
                ndata = tok.getName();
                if (null == ndata || !"NDATA".equals(ndata)) {
                    format0(tok, "E_ENTITY5");
                    tok.skipTagEnd();
                    ndata = null;
                } else {                        // 'NDATA' exists
                    if (!XMLChar.isSpace((char)tok.m_next)) {
                        format0(tok, "E_SPACE");
                        tok.skipTagEnd();
                    } else
                        tok.skipSpaces();
                    ntype = tok.getName();
                    if (null == ntype) {
                        format0(tok, "E_ENTITY5");
                        ntype = null;
                        tok.skipTagEnd();
                    } else {
                        tok.skipSpaces();
                        if ('>' == tok.m_next) {
                            tok.getChar();      // '>'
                        } else {
                            format0(tok, "E_ENTITY4");
                            tok.skipTagEnd();
                        }
                        // BUG: This should be checked after the entire
                        //      doctype (internal and external) has been
                        //      parsed. -Ac
                        /**
                        if (null == par.getNotation(ntype))
                            format1(tok, "V_ENTITY8", ntype);
                        /**/
                    }
                }
            } else
                tok.getChar();                  // '>'

            EntityDecl ent = this.document.createEntityDecl(name, eid, ispara, ntype);
            // Note: 
            // 1) entityPool.add will return false if preexisting, and only add 
            //    it if it is not preexisting.
            // 2) appendChild calls realInsert which also calls entityPool.add
            //    as well as appending the child.
            // So we can first check/add and report warning before appendChild.
            if (isWarningRedefinedEntity() && !this.entityPool.add(ent))
            {
                format1(tok, "W_ENTITY6", name);
            } 
            par.appendChild(ent);

            // Save unparsed entity decl location for checking
            // once the DTD is fully parsed. -Ac
            String notationName = ent.getNotationName();
            if (notationName != null) {
                unparsedEntities.addElement(new ParsePoint(tok, notationName));
                }
        }
    }
    
    private void startGeneralReference(String entityname) {
        if (null != this.referenceHandler)
            this.referenceHandler.startReference(entityname);
    }
    
    private void endGeneralReference(String entityname) {
        if (null != this.referenceHandler)
            this.referenceHandler.endReference(entityname);
    }

}
