/*
 * (C) Copyright IBM Corp. 1997-1998  All rights reserved.
 *
 * US Government Users Restricted Rights Use, duplication or
 * disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * IBM will not be liable for any damages suffered by you as a result
 * of using the Program. In no event will IBM be liable for any
 * special, indirect or consequential damages or lost profits even if
 * IBM has been advised of the possibility of their occurrence. IBM
 * will not be liable for any third party claims against you.
 */

package com.ibm.xml.parser;

import org.w3c.dom.Node;

/**
 * PseudoNode provides support, in place of actual DOM Nodes, for generated document objects
 * when document validity is not an issue.  
 * PseudoNodes are useful when an application does not care about the validity of a document's
 * structure, and wishes to quickly specify a literal string to represent the Node.  
 * <p>PseudoNodes can be added to TXDocument, TXElement, or DTD objects in order to
 * expediently print text.
 * <p>For example, instead of specifying an Element hierarchy as:
 * <pre>
 *
 * TXElement root = factory.createElement(&quot;PERSON&quot;);
 * TXElement item = factory.createElement(&quot;NAME&quot;);      //  Make &lt;NAME&gt;John Doe&lt;/NAME&gt;
 * item.addElement(factory.createText(&quot;John Doe&quot;));
 * root.addElement(item);
 * item = factory.createElement(&quot;AGE&quot;);                 // Make &lt;AGE&gt;35&lt;/AGE&gt;
 * item.addElement(factory.createText(&quot;35&quot;));
 * root.addElement(item);
 * item = factory.createElement(&quot;EMAIL&quot;);               // Make &lt;EMAIL&gt;John.Doe@foo.com&lt;/EMAIL&gt;
 * item.addElement(factory.createText(&quot;John.Doe@foo.com&quot;));
 * root.addElement(item);                               // Now root has NAME, AGE, and EMAIL.
 * doc.addElement(root);
 *
 * </pre>
 * <p>...using PsuedoNode you can instead specify:
 * <pre>
 * 
 * doc.addElement(new PseudoNode(&quot;&lt;PERSON&gt;&lt;NAME&gt;John Doe&lt;/NAME&gt;&lt;AGE&gt;35&lt;/AGE&gt;&lt;EMAIL&gt;John.Doe@foo.com&lt;/EMAIL&gt;&lt;/PERSON&gt;&quot;));
 *
 * </pre>
 * <p>Note that because of the requirement for validation, the object tree produced by the 
 * XML4J parser doesn't include PseudoNodes.
 *
 * @version Revision: 20 1.10 src/com/ibm/xml/parser/PseudoNode.java, xml4jsrc, xml4j-jtcsv, xml4j_1_1_16 
 * @author TAMURA Kent &lt;kent@trl.ibm.co.jp&gt;
 * @see com.ibm.xml.parser.Child
 */
public class PseudoNode extends TXCharacterData {

    static final long serialVersionUID = 7400309181601885133L;
    /**
     * Constructor.
     * @param data      The actual content of the Pseudo Node.
     */
    public PseudoNode(String data) {
        this.data = data;
    }

    /**
     * Clone this Pseudo Node using the appropriate factory.  
     * <p>This method is defined by Child.
     * @return          Cloned Pseudo Node.
     * @see com.ibm.xml.parser.Child#clone
     */
    public synchronized Object clone() {        
        PseudoNode pn = new PseudoNode(this.data);
        pn.setFactory(getFactory());
        return pn;
    }
    
    /**
     * Returns that this object is a Pseudo Node. 
     * <p>This method is defined by DOM.
     * @return          Pseudo Node indicator.
     */
    public short getNodeType() {
        return Child.PSEUDONODE;
    }

    public String getNodeName() {
        return Child.NAME_PSEUDONODE;
    }

    public boolean equals(Node arg, boolean deep) {
        if (arg == null)  return false;
        if (!(arg instanceof PseudoNode))  return false;
        return this.data.equals(((PseudoNode)arg).getData());
    }

    /**
     * Returns all text associated with this Node without considering entities.                                           
     * <p>This method is defined by Child.
     * @return          Text associated with this object, or <var>""</var> if no Text.
     * @see com.ibm.xml.parser.Child#toXMLString
     * @see #getData
     */
    public String getText() {                   
        return this.data;
    }
    
    /**
     * Implements the accept operation of the visitor design pattern when the start of
     * a PseudoNode is recognized when traversing the document object tree. 
     * @param   visitor The implemention of the Visitor operation (toXMLString, digest, ...)
     * @exception Exception Thrown if this Node can not be visitted, or traversal modification is requested.
     * @see com.ibm.xml.parser.Visitor
     * @see com.ibm.xml.parser.TreeTraversal
     * @see com.ibm.xml.parser.NonRecursivePreorderTreeTraversal
     * @see com.ibm.xml.parser.TreeTraversalException
     */
    public void acceptPre(Visitor visitor) throws Exception {
        visitor.visitPseudoNodePre(this);
    }

    /**
     * Implements the accept operation of the visitor design pattern when the end of
     * a PseudoNode is recognized when traversing the document object tree. 
     * @param   visitor The implemention of the Visitor operation (toXMLString, digest, ...)
     * @exception Exception Thrown if this Node can not be visitted, or traversal modification is requested.
     * @see com.ibm.xml.parser.Visitor
     * @see com.ibm.xml.parser.TreeTraversal
     * @see com.ibm.xml.parser.NonRecursivePreorderTreeTraversal
     * @see com.ibm.xml.parser.TreeTraversalException
     */
    public void acceptPost(Visitor visitor) throws Exception {
        visitor.visitPseudoNodePost(this);
    }
}
