/*
 * (C) Copyright IBM Corp. 1997-1998  All rights reserved.
 *
 * US Government Users Restricted Rights Use, duplication or
 * disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * IBM will not be liable for any damages suffered by you as a result
 * of using the Program. In no event will IBM be liable for any
 * special, indirect or consequential damages or lost profits even if
 * IBM has been advised of the possibility of their occurrence. IBM
 * will not be liable for any third party claims against you.
 */

package com.ibm.xml.parser;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.Reader;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Hashtable;
import java.util.Stack;
import java.util.StringTokenizer;

/**
 * Stderr provides the XML4J parser's default implementations of the ErrorListener and
 * StreamProducer interfaces.
 *
 * @version Revision: 25 1.10 src/com/ibm/xml/parser/Stderr.java, xml4jsrc, xml4j-jtcsv, xml4j_1_1_16 
 * @author TAMURA Kent &lt;kent@trl.ibm.co.jp&gt;
 * @see com.ibm.xml.parser.ErrorListener
 * @see com.ibm.xml.parser.StreamProducer
 */
public class Stderr implements ErrorListener, StreamProducer {
    
    public  static  PrintWriter printer =   new PrintWriter(System.err);
    protected       String      name    =   null;
    protected       URL         url     =   null;
    protected       Stack       stack   =   new Stack();
    protected       Hashtable   catalog =   null;
    protected       boolean     isPrintWarning = true;

    /**
     * Constructor.
     * This class supports only URL that <CODE>java.net.URL</CODE> can handles.
     * @param   name    URL or filespec to use as the default input stream; if a filespec is
     *                  provided, this value can include a drive and directory spec.
     *                  This value is also used to associate a name with errors reported to the
     *                  default error listener (i.e. <code>error()</code> method) that 
     *                  have a <var>file</var> parameter <code>=null</code>.
     * @see #error
     */
    public Stderr(String name) {
        this.name = name;
        try {
            this.url = new URL(this.name);
        } catch (MalformedURLException e) {
            try {
                this.url = file2URL(name);
            } catch (MalformedURLException ex) {
                throw new LibraryException("Stderr#Stderr(): Internal Error: "+e);
            } catch (SecurityException ex2) {
                printer.println("Specify a complete URL");
            }
        }
    }

    /**
     * Sets whether prints warnings.
     * <p>By default, prints warnings.</p>
     * @param isPrintWarning <code>=true</code> means prints warnings;
     *                       <code>=false</code> means prints no warnings.
     */
    public void setPrintWarning(boolean isPrintWarning) {
        this.isPrintWarning = isPrintWarning;
    }

    /**
     * Returns a file-protocol URL constructed from a context of the specified 
     * <var>file</var> and the JVM's system property &quot;user.dir&quot;.
     * @param   file    The file to use in constructing the URL; this value can include a 
     *                  drive and directory spec.
     * @exception  MalformedURLException    Thrown if unable to construct a URL based on
     *                                      the specified <var>file</var>.
     */
    public static URL file2URL(String file) throws MalformedURLException {
        file = file.replace(File.separatorChar, '/');
        String basedir = System.getProperty("user.dir").replace(File.separatorChar, '/')+"/";
        if (basedir.charAt(0) != '/')  basedir = "/"+basedir;
        URL u = new URL("file", "", basedir);
        if (file.length() >= 2 && file.charAt(1) == ':') {
            char drive = Character.toUpperCase(file.charAt(0));
            if ('A' <= drive && drive <= 'Z')
                file = "file:///"+file;
        }
        return new URL(u, file);
    }

    /**
     * <p>Listen for XML4J parser errors, and reports these errors through the standard
     * error stream (System.err).
     * <p>This method is defined by ErrorListener.
     * @param   file        Processing file, or <var>null</var> if this constructor's 
     *                      <var>name</var> parameter is to be used.
     * @param   lineNo      The line number where the current document event ends, or 
     *                      <var>-1</var> if not available.  Note that this is the line 
     *                      position of the first character after the text associated with 
     *                      the document event.  Do NOT trust this value when the input 
     *                      stream includes invalid octet as its encoding.
     * @param   charOffset  The column number where the current document event ends, or
     *                      <var>-1</var> if not available.  Note that this is the column 
     *                      number of the first character after the text associated with the 
     *                      document event.  The first column in a line is position 1.  Do NOT
     *                      trust this value when the input stream includes invalid octet as its encoding.
     * @param   key         The object may be an instance of <code>String</code> or <code>Exception</code>.
     *                      When this object is <code>String</code>, this value may help 
     *                      classify <var>msg</var> as an error (begins with "E_") or warning
     *                      (begins with "W_") or a validation error (begins with "V_").
     * @param   msg         The error or warning message.
     * @see com.ibm.xml.parser.ErrorListener
     */
    public int error(String file, int lineNo, int charOffset, Object key, String msg) {
        if (null == file)  file = this.name;
        if (0 < charOffset)  charOffset --;
        if (!this.isPrintWarning && key instanceof String && ((String)key).startsWith("W_")) {
        } else
            printer.println(file+": "+lineNo+", "+charOffset+": "+msg);
        printer.flush();
        return 1;
    }

    private String URI2URL(String uri) throws IOException {
        if (Util.isURN(uri)) {
            String normalized = Util.normalizeURN(uri);
            if (this.catalog != null && this.catalog.containsKey(normalized)) {
                uri = (String)this.catalog.get(normalized);
            } else {
                throw new IOException("Can't resolve URN: "+uri);
            }
        }
        return uri;
    }
    /**
     * <p>Returns the source of the input stream (could be a character stream or a byte stream) 
     * based on this constructor's <var>name</var> parameter and the URL specification 
     * given by <var>systemID</var> and <var>publicID</var>.
     * <p>This method is defined by StreamProducer.
     * @param   name        CURRENTLY NOT IMPLEMENTED. 
     * @param   publicID    Entity's public ID which is to be used in preference to <var>systemID</var>, or
     *                      <var>null</var> if no catalog is available.    
     * @param   systemID    Entity's URI (URL or URN).
     * @return              The resolved source of the input stream, or <var>null</var> if unable to resolve.
     * @exception IOException   Thrown if unable to open the source defined by the specified IDs.
     * @see #loadCatalog
     * @see #closeInputStream
     * @see com.ibm.xml.parser.ExternalID
     * @see com.ibm.xml.parser.StreamProducer
     */
    public Source getInputStream(String name, String publicID, String systemID) throws IOException {
        if (publicID != null && this.catalog != null && this.catalog.containsKey(publicID)) {
            String resolvedURI = (String)this.catalog.get(publicID);
            // resolvedURI is a URL or a URN.
            resolvedURI = this.URI2URL(resolvedURI);
            // resolvedURI is a URL.
            URL u = new URL(this.url, resolvedURI);
            try {
                Source ret = new Source(u.openStream());
                this.stack.push(this.url);
                this.url = u;
                return ret;
            } catch (IOException io) {
            }
            // Can't find the resource pointed by a URL mapped from the publicID.
        }
        systemID = this.URI2URL(systemID);
        URL u = new URL(this.url, systemID);
        this.stack.push(this.url);
        this.url = u;
        return new Source(u.openStream());
        /*
        URL u = null;
        if (this.catalog != null && publicID != null) {
            if (this.catalog.containsKey(publicID))
                systemID = (String)this.catalog.get(publicID);
        }
        try {
            u = new URL(this.url, systemID);
        } catch (MalformedURLException e) {
            printer.println("Stderr: Malformed URL: "+this.url+", "+systemID);
        }
        this.stack.push(this.url);
        this.url = u;
        if (null == u) {// || "file".equalsIgnoreCase(u.getProtocol())) {
            return new Source(new FileInputStream(systemID));
        } else {
            return new Source(u.openStream());
        }
        */
    }

   /**
     * <p>Removes the input stream currently in use. 
     * <p>This method is defined by StreamProducer.
     * @param   source      CURRENTLY NOT IMPLEMENTED.
     * @see #getInputStream
     * @see com.ibm.xml.parser.StreamProducer
     */
    public void closeInputStream(Source source) {
        if (!this.stack.empty())
            this.url = (URL)this.stack.pop();
    }

    /**
     * Loads a catalog which provides mapping between public IDs to system IDs or URNs to URLs.
     * <p>This method reads data until it reaches the end of the stream, but it does not 
     * close the stream.
     * <DL>
     *   <DT>Catalog file format:</DT>
     *   <DD><PRE>
     * <I>public ID</I>&lt;TAB><I>URI<I>
     * <I>public ID</I>&lt;TAB><I>URI<I>
     * <I>URN</I>&lt;TAB><I>URL</I>
     *  :
     *  :
     * </PRE>
     * </DL>
     * <P>Note: Public IDs can not contain <code>#xD</code> or <code>#xA</code>.</P>
     * @param       reader      Character input stream reader.
     * @exception   IOException Thrown if <var>reader</var> is invalid.
     * @see #getInputStream
     * @see com.ibm.xml.parser.ExternalID
     * @see com.ibm.xml.parser.Parser#loadCatalog
     * @see com.ibm.xml.parser.StreamProducer#loadCatalog
     */
    public void loadCatalog(Reader reader) throws IOException {
        if (this.catalog == null)
            this.catalog = new Hashtable();
        BufferedReader br = new BufferedReader(reader);
        String line;
        while (null != (line = br.readLine())) {
            StringTokenizer st = new StringTokenizer(line, "\t");
            if (!st.hasMoreTokens()) continue;
            String fpi = st.nextToken();
            if (!st.hasMoreTokens()) continue;
            String sysid = st.nextToken();
            if (Util.isURN(fpi)) {
                this.catalog.put(Util.normalizeURN(fpi), sysid);
            } else {
                this.catalog.put(fpi, sysid);
            }
        }
    }
}

