/*
 * (C) Copyright IBM Corp. 1997-1998  All rights reserved.
 *
 * US Government Users Restricted Rights Use, duplication or
 * disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * IBM will not be liable for any damages suffered by you as a result
 * of using the Program. In no event will IBM be liable for any
 * special, indirect or consequential damages or lost profits even if
 * IBM has been advised of the possibility of their occurrence. IBM
 * will not be liable for any third party claims against you.
 */

package com.ibm.xml.parser;

import java.io.*;

/**
 * <P>The StyleSheetPI class provides specific support for the stylesheet processing 
 * instruction (PI).
 * <p>A stylesheet PI is formatted as:
 * <pre>
 * 
 * <code>&lt;?xml:stylesheet type="..." href="..."?&gt;</code> or
 * <code>&lt;?xml:alternate-stylesheet type="..." href="..."?&gt;</code>
 *
 </pre>
 * <P>Refer to <A href="http://www.w3.org/TR/NOTE-xml-stylesheet">Associating stylesheets
 * with XML documents</A>
 * for details.</P>
 *
 * @version Revision: 28 1.9 src/com/ibm/xml/parser/StylesheetPI.java, xml4jsrc, xml4j-jtcsv, xml4j_1_1_16 
 * @author TAMURA Kent &lt;kent@trl.ibm.co.jp&gt;
 * @see com.ibm.xml.parser.TXPI
 */
public class StylesheetPI extends TXPI {

    static final long serialVersionUID = -5551516051977603120L;
    /**
     * <CODE>"xml:stylesheet"</CODE>.
     */
    public static final String S_XMLSTYLESHEET = "xml:stylesheet";
    /**
     * <CODE>"xml:alternate-stylesheet"</CODE>.
     */
    public static final String S_XMLALTSTYLESHEET = "xml:alternate-stylesheet";
    /**
     * <CODE>"stylesheet"</CODE>.
     */
    public static final String S_STYLESHEET = "stylesheet";
    /**
     * <CODE>"alternate-stylesheet"</CODE>.
     */
    public static final String S_ALTSTYLESHEET = "alternate-stylesheet";
    /**
     * <CODE>"text/css"</CODE>.
     */
    public static final String S_TEXTCSS = "text/css";

    
            String type     =   null;
            String hrefURI  =   null;
            String title    =   null;

    /**
     * Cascading Style Sheet (CSS) stylesheet PI constructor.
     *
     * @param hrefURI       The value of the <code>href=</code> attribute.
     */
    public StylesheetPI(String hrefURI) {
        this(S_XMLSTYLESHEET,
             " type=\""+S_TEXTCSS+"\" href=\""+hrefURI+"\"", 
             S_TEXTCSS, 
             hrefURI, null);
    }
    /**
     * XML and XML-alternate stylesheet PI constructor.
     *
     * @param type          The value of the <code>type=</code> attribute.
     * @param hrefURI       The value of the <code>href=</code> attribute.
     * @param title         The value of the <CODE>title=</CODE> attribute, or <var>null</var>.
     */
    public StylesheetPI(String type, String hrefURI, String title) {
        this(S_XMLSTYLESHEET, 
             null == title
             ? " type=\""+type+"\" href=\""+hrefURI+"\""
             : " type=\""+type+"\" href=\""+hrefURI+"\" title=\""+title+"\"", 
             type, 
             hrefURI,
             title);
    }
    
    /**
     * Generic constructor.
     *
     * @param name          The first token following the markup (e.g. <CODE>"xml:stylesheet"</CODE>).
     * @param type          The value of the <code>type=</code> attribute.
     * @param hrefURI       The value of the <code>href=</code> attribute.
     * @param title         The value of the <CODE>title=</CODE> attribute, or <var>null</var>.
     */
    public StylesheetPI(String name, String type, String hrefURI, String title) {
        this(name,
             null == title
             ? " type=\""+type+"\" href=\""+hrefURI+"\""
             : " type=\""+type+"\" href=\""+hrefURI+"\" title=\""+title+"\"", 
             type, 
             hrefURI,
             title);
    }

    /**
     * Generic constructor.
     *
     * @param name          The first token following the markup (e.g. <CODE>"xml:stylesheet"</CODE>).
     * @param data          From the character immediately after <var>name</var> to the character immediately preceding the <code>?&gt;</code>.
     * @param type          The value of the <code>type=</code> attribute.
     * @param hrefURI       The value of the <code>href=</code> attribute.
     * @param title         The value of the <CODE>title=</CODE> attribute, or <var>null</var>.
     * @exception LibraryException Thrown if an invalid stylesheet declaration is detected.
     */
    public StylesheetPI(String name, String data, String type, String hrefURI, String title) 
        throws LibraryException {
        super(name, data);
        if (!name.equals(S_XMLSTYLESHEET) && !name.equals(S_XMLALTSTYLESHEET)) {
            throw new LibraryException("Invalid stylesheet declaration detected.");
        }
        this.type       = type;
        this.hrefURI    = hrefURI;
        this.title      = title;
    }
    
    /**
     * Clone this stylesheet PI Node using the appropriate factory.
     * <p>This method is defined by Child.
     * @return          Cloned stylesheet PI Node.
     * @see com.ibm.xml.parser.Child#clone
     */
    public synchronized Object clone() {
        checkFactory();
        StylesheetPI pi = factory.createStylesheetPI(this.name, this.data,
                                                     this.type, this.hrefURI, this.title);
        pi.setFactory(getFactory());
        return pi;
    }

    /**
     * Returns the stylesheet type (the value of the <code>type=</code> attribute) which
     * defines the stylesheet as <CODE>xml:stylesheet</CODE>, <CODE>xml:alternate-stylesheet</CODE>,
     * or <CODE>text/css</CODE>.
     * @return          The value of the <code>type=</code> attribute.
     */
    public String getType() {
        return this.type;
    }

    /**
     * Returns the stylesheet URI (the value of the <code>href=</code> attribute) which
     * defines where to retrieve the stylesheet.
     * @return          The value of the <code>href=</code> attribute.
     */
    public String getHref() {
        return this.hrefURI;
    }

    /**
     * Returns the stylesheet title (the value of the <code>title=</code> attribute).
     * @return          The value of the <code>title=</code> attribute, or <var>null</var> if no title.
     */
    public String getTitle() {
        return this.title;
    }

}
