/*
 * (C) Copyright IBM Corp. 1997-1998  All rights reserved.
 *
 * US Government Users Restricted Rights Use, duplication or
 * disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * IBM will not be liable for any damages suffered by you as a result
 * of using the Program. In no event will IBM be liable for any
 * special, indirect or consequential damages or lost profits even if
 * IBM has been advised of the possibility of their occurrence. IBM
 * will not be liable for any third party claims against you.
 */

package com.ibm.xml.parser;

import java.io.StringWriter;
import org.w3c.dom.Attr;
import org.w3c.dom.DOMException;
import org.w3c.dom.EntityReference;
import org.w3c.dom.Node;
import org.w3c.dom.Text;

/**
 * The <CODE>TXAttribute</CODE> class implements the <CODE>Attr</CODE> interface
 *  as defined by the Document Object Model (DOM),
 * and implements the namespace interface as defined by the W3C.
 * <p><CODE>Attr</CODE> Nodes represent an attribute in an <CODE>Element</CODE> object;
 * in other words, an
 * attribute name and an attribute object.  Typically the allowable values for the 
 * attribute are defined in a document type definition.  
 * <p>The attribute's effective value is determined as follows: if the attribute has been 
 * explicitly assigned any value, that value is the attribute's effective value.  Otherwise, 
 * if there is a declaration for this attribute, and that declaration includes a default 
 * value, then that default value is the attribute's effective value.  Otherwise, the 
 * attribute has no effective value.  
 * <p> In XML, the value of an attribute is represented by a list because
 * the value can be an arbitrarily complex list of entity references. 
 *
 * @version Revision: 31 1.25 src/com/ibm/xml/parser/TXAttribute.java, xml4jsrc, xml4j-jtcsv, xml4j_1_1_16 
 * @author TAMURA Kent &lt;kent@trl.ibm.co.jp&gt;
 * @see org.w3c.dom.Attr
 * @see com.ibm.xml.parser.Namespace
 * @see com.ibm.xml.parser.Parent
 */
public class TXAttribute extends Parent implements Attr, Namespace {

            static final long serialVersionUID = 710791704404680525L;
            String      name            = null;
            String      value           = null;
            int         type            = AttDef.UNKNOWN; 
            String[]    typedValue      = null;
            boolean     specified       = false;

    /**
     * Constructor.
     * @param name      The name of this attribute.
     * @param value     The string value of this attribute.
     */
    public TXAttribute(String name, String value) {
        this.name = name;
        this.value = value;
        this.specified = true;
    }

    /**
     * Clone this <CODE>TXAttribute</CODE> Node using the appropriate factory.  Note that this
     * method does not clone Parent or sibling Nodes.
     * <p>This method is defined by <CODE>Child</CODE>.
     * @return          Cloned <CODE>TXAttribute</CODE> Node.
     * @see com.ibm.xml.parser.Child#clone
     */
    public Object clone() {
        return cloneNode(true);
    }

    /**
     *
     */
    public synchronized Node cloneNode(boolean deep) {
        checkFactory();
        TXAttribute a = (TXAttribute)factory.createAttribute(this.name);
        a.setFactory(getFactory());
        if (deep) {
            a.children.ensureCapacity(children.getLength());
            for (int i = 0;  i < children.getLength();  i ++)
                a.insertBefore(children.item(i).cloneNode(true), null);
        }
        // setType must be called after children clone/insertion 
        // because it sets/normalizes the value.
        a.setType(getType(), getTypedValue());
        a.setSpecified(getSpecified());
        return a;
    }
    /**
     *
     */
    public synchronized boolean equals(Node arg, boolean deep) {
        if (arg == null)  return false;
        if (!(arg instanceof TXAttribute))  return false;
        TXAttribute a = (TXAttribute)arg;
        if (!a.getName().equals(this.getName()))
            return false;
        if (!a.getValue().equals(this.getValue()))
            return false;
        if (deep) {
            if (!a.children.equals(this.children, deep))
                return false;
        }
        return true;
    }

    /**
     * Returns that this object is an Attribute Node.
     * <p>This method is defined by DOM.
     * @return          <CODE>Attr</CODE> Node indicator.
     */
    public short getNodeType() {
        return Node.ATTRIBUTE_NODE;
    }

    /**
     * <p>This method is defined by DOM.
     * @see #getName
     */
    public String getNodeName() {
        return this.name;
    }

    /**
     * Returns the qualified name of this attribute.  If the attribute name has a namespace prefix, 
     * the prefix will still be attached.
     * <p>This method is defined by DOM.
     * @return          The name of this attribute (should never be null)
     */
    public String getName() {
        return this.name;
    }

    /**
     * Returns the value of this attribute.
     * <p>This method is defined by DOM.
     * @return          The value of this attribute, or "" if no value.
     * @see #getTypedValue
     * @see #setValue
     * @see #toString
     */
    public String getValue() {
        if (this.value != null)
            return this.value;
        this.value = getText();
        return this.value;
    }

    /**
     * Returns the value of this attribute.
     * @return          The value of this attribute, or "" if no value.
     * <p>This method is defined by DOM.
     */
    public String getNodeValue() {
        return getValue();
    }

    /**
     * Returns the value of this attribute.
     * @return          The value of this attribute, or "" if no value.
     * @see #getValue
     * @see #toXMLString
     */
    public String toString() {
        return getValue(); 
    }

    /**
     * Sets the value of this attribute.
     * @param value     The value of this attribute.
     * @see #getValue
     * @see #setNodeValue
     * @see #getTypedValue
     */
    public void setValue(String value) {
        setNodeValue(value);
    }

    /**
     * Sets the value of this attribute.
     * <p>This method is defined by DOM.
     * @param value     The value of this attribute.
     * @see #getNodeValue
     * @see #getValue
     * @see #setValue
     * @see #getTypedValue
     */
    public void setNodeValue(String value) {
        this.value = value;
        if (value != null) {
            synchronized (this.children) {
                while (this.getFirstChild() != null)
                    this.removeChild(this.getFirstChild());
                checkFactory();
                this.appendChild(this.factory.createTextNode(value));
                //this.appendChild(new TXText(value));
            }
        }
        clearDigest();
    }

    /**
     * Insert a Node into the specified position.
     * @param child     The Node being inserted. <CODE>Text</CODE> or <CODE>EntityReference</CODE>.
     * @param index     0-based index into the list of children.
     * @exception LibraryException Thrown if the document's root element is set twice.
     */     
    protected void realInsert(Node child, int index) throws LibraryException {
        if (!(child instanceof Text || child instanceof EntityReference)) {
            throw new TXDOMException(DOMException.HIERARCHY_REQUEST_ERR,
                                     "com.ibm.xml.parser.TXAttribute#realInsert(): Nodes except Text/EntityReference are not allowed as attribute children.");
        }
        super.realInsert(child, index);
        this.value = null;
    }

    /**
     * Replaces the child Node <var>oldChild</var> with <var>newChild</var> in this Node's list of children, and return
     * the <var>oldChild</var> Node. If <var>oldChild</var> was not already a Child of this Node,
     * a <CODE>DOMException</CODE> is thrown.
     * <p>This method is defined by DOM.
     *
     * @param newChild  The child Node to replace with. <CODE>Text</CODE> or <CODE>EntityReference</CODE>.
     * @param oldChild  The child Node being replaced.
     * @return          The child Node being replaced.
     * @exception org.w3c.dom.DOMException Thrown if <var>oldChild</var> is not a child of this object.
     * @see #removeChild  
     */
    public Node replaceChild(Node newChild, Node oldChild) throws DOMException {
        if (!(newChild instanceof Text || newChild instanceof EntityReference)) {
            throw new TXDOMException(DOMException.HIERARCHY_REQUEST_ERR,
                                     "com.ibm.xml.parser.TXAttribute#realInsert(): Nodes except Text/EntityReference are not allowed as attribute children.");
        }
        this.value = null;
        return super.replaceChild(newChild, oldChild);
    }

    /**
     * Removes the child Node indicated by <var>oldChild</var> from this Nodes list of children,
     * and returns it. If <var>oldChild</var>
     * was not a Child of this Node, a <CODE>DOMException</CODE> is thrown.
     * <p>This method is defined by DOM.
     * @param oldChild  The child Node being removed.
     * @return          The child Node being removed.
     * @exception org.w3c.dom.DOMException Thrown if <var>oldChild</var> is not a child of this object.
     * @see #replaceChild 
     */
    public Node removeChild(Node oldChild) throws DOMException {
        this.value = null;
        return super.removeChild(oldChild);
    }

    /**
     * Returns whether this attribute was explicitly given a value in the original 
     * document (=true), or was allowed to default according to the DTD specification.
     * @return          =true if value explicitly defined; otherwise, false.
     * @see #setSpecified
     */
    public boolean getSpecified() {
        return this.specified;
    }

    /**
     * Sets whether this attribute was explicitly given a value in the original 
     * document (=true), or was allowed to default according to the DTD specification.
     * @param   specified   =true if value explicitly defined; otherwise, false.
     * @see #getSpecified
     */
    public void setSpecified(boolean specified) {
        this.specified = specified;
    }

    /**
     * Returns the local name of the Namespace.
     * <p>This method is defined by Namespace.
     * @return          The Namespace local name, or <var>null</var> if no local name. 
     * @see com.ibm.xml.parser.Namespace
     */
    public String getNSLocalName() {
        return TXElement.getLocalNameForQName(this.getNodeName());
    }
    
    /**
     * Returns the Namespace URI.  
     * <p>This method is defined by Namespace.
     * @return          The Namespace URI, or <var>null</var> if the attribute name does not
     *                  contain a prefix or XML4J parser namespace processing is disabled.     
     * @see com.ibm.xml.parser.Parser#setProcessNamespace
     * @see com.ibm.xml.parser.Namespace
     */
    public String getNSName() {
        Node n = this.parent;
        if (n == null)  new LibraryException("This attribute isn't assigned to any Element.");
        return 0 > this.getNodeName().indexOf(':') ? null : ((TXElement)n).getNamespaceForQName(this.getNodeName());
    }

    /**
     * If namespace support is enabled and this Attribute's name contains a prefix name,
     * return a string 
     * of <CODE>getNSName()+":"+getNSLocalName()</CODE>;
     * otherwise, return <CODE>getNSLocalName()</CODE> .
     * @return          The universal name, or <var>null</var> if this attribute's name does not
     *                  contain a prefix name or XML4J parser namespace processing is disabled.
     * @see #getNSName
     * @see #getNSLocalName
     * @see com.ibm.xml.parser.Parser#setProcessNamespace
     * @deprecated Use createExpandedName().
     */
    public String getUniversalName() {
        return getNSName() == null ? getNSLocalName() : getNSName()+":"+getNSLocalName();
    }

    /**
     * Returns an expanded name of this attribute.
     * The expanded name is a set of <SPAN style="background-color: #d0d0d0;">[a namespace URI, an attribute localPart] (if this attribute has a prefix)</SPAN> or
     * <SPAN style="background-color: #d0d0d0;">[a namespace URI, an element localPart, an attribute localPart] (if this attribute has no prefix)</SPAN> connected by <CODE>TXDocument#expandedNameSeparator</CODE>.
     *<P>This method is defined by Namespace.
     * @exception com.ibm.xml.parser.LibraryException Thrown if this attribute doesn't belong to any element.
     * @see com.ibm.xml.parser.TXDocument#expandedNameSeparator
     * @see com.ibm.xml.parser.Parser#setProcessNamespace
     */
    public String createExpandedName() {
        if ("xmlns".equals(getNodeName()))
            return "xmlns";
        String ns = this.getNSName();
        return ns == null || ns.length() == 0
            ? ((TXElement)this.parent).createExpandedName() + getFactory().expandedNameSeparator + this.getNodeName()
            : (ns + getFactory().expandedNameSeparator + getNSLocalName());
    }

    /**
     * Returns the type of the attribute.  Attribute types are only set when
     * a grammar definition (DTD) is available.
     * @return          The attribute type as defined by <CODE>AttDef</CODE>.
     * @see com.ibm.xml.parser.AttDef
     * @see #setType
     * @see #getTypedValue
     */ 
    public int getType() {
        return this.type;
    }

    /**
     * Returns the type values of the attribute.  Attribute type values are only set when
     * a grammar definition (DTD) is available.
     * @return          <CODE>getTypeValue()[0] = getValue()</CODE> when <CODE>getType() = CDATA, NOTATION, ID, IDREF, ENTITY, NMTOKEN</CODE>;
     *                    otherwise, multiple values are returned.
     * @see com.ibm.xml.parser.AttDef
     * @see #setType
     * @see #getType
     */
    public String[] getTypedValue() {
        return this.typedValue;
    }

    /**
     * Sets the type and type values of the Attribute.  Attribute types are only set when
     * a grammar definition (DTD) is available.
     * @param   type        The Attribute type as defined by AttDef.Java.
     * @param   typedValue  <CODE>getTypeValue()[0] = getValue()</CODE> when <CODE>getType() = CDATA, NOTATION, ID, IDREF, ENTITY, NMTOKEN</CODE>;
     *                        otherwise, multiple values are returned.
     * @see com.ibm.xml.parser.AttDef
     * @see #setType
     * @see #getType
     * @see #getTypedValue
     */
    public void setType(int type, String[] typedValue) {
        this.type       = type;
        this.typedValue = typedValue;
        this.value      = normalize(type, getValue());
    }


    /**
     * Returns this Node in XML format, in <code>name="value"</code> format,
     * and using the specified character encoding.
     * @param encoding  Java character encoding to use.
     * @return          XML formatted string expressed as <code>name="value"</code>.
     * @see #getText
     */
    public String toXMLString(String encoding) {
        String ret = null;
        try {
            StringWriter stringWriter = new StringWriter();
            Visitor visitor = new ToXMLStringVisitor(stringWriter, encoding);
            new NonRecursivePreorderTreeTraversal(visitor).traverse(this);
            ret = stringWriter.toString();
        } catch (Exception e) {
            // Maybe exceptions are never thrown.
        }
        return ret;
    }

    /**
     * Returns this Node in XML format, in <code>name="value"</code> format,
     * and using the default character encoding.
     * @return          XML formatted string expressed as <code>name="value"</code>.
     * @see #getText
     */
    public String toXMLString() {
        return toXMLString((String) null);
    }

    /**
     * Returns whether the specified <var>object</var> matches this attribute Node.
     * A match is defined when the specified <var>object</var>'s name matches the name
     * of this attribute.
     * @return          =true if the names match; otherwise, false.
     */
    public boolean equals(Object object) {
        if (null == object)  return false;
        if (object instanceof Attr) {
            return getName().equals(((Attr)object).getName());
        } else if (object instanceof String) {
            return getName().equals((String)object);
        } else
            return false;
    }
    
    /**
     *
     */
    public int hashCode() {
        return getName().hashCode();
    }

    
    /**
     * Implements the accept operation of the visitor design pattern when the start of
     * a TXAttribute Node is recognized when traversing the document object tree. 
     * @param   visitor The implemention of the Visitor operation (toXMLString, digest, ...)
     * @exception Exception Thrown if this Node can not be visitted, or traversal modification is requested.
     * @see com.ibm.xml.parser.Visitor
     * @see com.ibm.xml.parser.TreeTraversal
     * @see com.ibm.xml.parser.NonRecursivePreorderTreeTraversal
     * @see com.ibm.xml.parser.TreeTraversalException
     */
    public void acceptPre(Visitor visitor) throws Exception {
        visitor.visitAttributePre(this);
    }

    /**
     * Implements the accept operation of the visitor design pattern when the end of
     * a TXAttribute Node is recognized when traversing the document object tree. 
     * @param   visitor The implemention of the Visitor operation (toXMLString, digest, ...)
     * @exception Exception Thrown if this Node can not be visitted, or traversal modification is requested.
     * @see com.ibm.xml.parser.Visitor
     * @see com.ibm.xml.parser.TreeTraversal
     * @see com.ibm.xml.parser.NonRecursivePreorderTreeTraversal
     * @see com.ibm.xml.parser.TreeTraversalException
     */
    public void acceptPost(Visitor visitor) throws Exception {
        visitor.visitAttributePost(this);
    }

    /**
     * Reuurns owner element node.
     * <P>In this DOM implementation, any attribute node is not shared in more than one element.
     */
    public Node getOwnerElement() {
        return this.parent;
    }

    /**
     * Returns <VAR>null</VAR>.
     */
    public Node getParentNode() {
        return null;
    }

    /**
     * Returns <VAR>null</VAR>.
     */
    public Node getPreviousSibling() {
        return null;
    }

    /**
     * Returns <VAR>null</VAR>.
     */
    public Node getNextSibling() {
        return null;
    }

    /**
     * Normalize the specified String by trimming leading and trailing space, and
     * removing superfluous embedded spaces.
     * @return          Normalized string.
     * @see #toXMLString
     */
    static String normalize(int type, String src) {   // S->&#x20 conversion is done.
        if (AttDef.UNKNOWN == type || AttDef.CDATA == type)  return src;
        src = src.trim();
        StringBuffer sb = new StringBuffer(src.length());
        boolean previousSpace = false;
        for (int i = 0;  i < src.length();  i ++) {
            char ch = src.charAt(i);
            if (' ' == ch) {
                if (!previousSpace)  sb.append(ch);
                previousSpace = true;
            } else {
                previousSpace = false;
                sb.append(ch);
            }
        }
        return sb.toString();
    }

    /**
     * Check whether <VAR>child</VAR> is allowed to be insered in this node or not.
     * When not allowed, a DOMException with HIERARCHY_REQUEST_ERR is thrown.
     */
    protected void checkChildType(Node child) throws DOMException {
        switch (child.getNodeType()) {
          case Node.TEXT_NODE:
          case Node.ENTITY_REFERENCE_NODE:
          case Child.PSEUDONODE:
            break;
          default:
            throw new TXDOMException(DOMException.HIERARCHY_REQUEST_ERR,
                                     "Specified node type ("+child.getNodeType()
                                     +") can't be a child of Attribute.");
        }
    }
}
