/*
 * (C) Copyright IBM Corp. 1997-1998  All rights reserved.
 *
 * US Government Users Restricted Rights Use, duplication or
 * disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * IBM will not be liable for any damages suffered by you as a result
 * of using the Program. In no event will IBM be liable for any
 * special, indirect or consequential damages or lost profits even if
 * IBM has been advised of the possibility of their occurrence. IBM
 * will not be liable for any third party claims against you.
 */

package com.ibm.xml.parser;

import org.w3c.dom.Attr;
import org.w3c.dom.DOMException;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;

/**
 * The <CODE>TXAttributeList</CODE> class represents collections of <CODE>Attr</CODE>
 * which can be accessed by name or by ordinal.
 *
 * @version Revision: 32 1.14 src/com/ibm/xml/parser/TXAttributeList.java, xml4jsrc, xml4j-jtcsv, xml4j_1_1_16 
 * @author TAMURA Kent &lt;kent@trl.ibm.co.jp&gt;
 * @see org.w3c.dom.NamedNodeMap
 */
class TXAttributeList extends java.util.Vector
    implements NamedNodeMap, Cloneable, java.io.Serializable {
    
            static final long serialVersionUID = 8198968721269694477L;
            Element         parentElement   =   null;
    
    /**
     * Constructor.
     */
    public TXAttributeList() {
        super();
    }
    
    /**
     * Clone this TXAttributeList using the appropriate factory.  Note that this
     * method does clone Child Attributes.
     * @return          Cloned TXAttributeList and children.
     */
    public synchronized Object clone() {
        TXAttributeList attl = new TXAttributeList();
        attl.parentElement = this.parentElement;
        for (int i = 0;  i < getLength();  i ++) {
            attl.setNamedItem((Attr)((TXAttribute)item(i)).clone());
        }
        return attl;
    }
    
    /**
     * Returns the index of the Attribute whose name matches the specified <var>name</var>.
     * @param name      Name to match against in the AttributeList.
     * @return          0-based index of matching Attribute, or <var>-1</var> if no matches.
     */
    public synchronized int indexOf(String name) {
        for (int i = 0;  i < size();  i ++) {
            TXAttribute a = (TXAttribute)elementAt(i);
            if (a.getName().equals(name))  return i;
        }
        return -1;
    }

    /**
     * Returns an Attribute instance whose name matches the specified <var>name</var>.  
     * <p>This method is defined by DOM.
     * @param name      Name to match against in the AttributeList.
     * @return          The matching Attribute instance, or <var>null</var> if no matches.
     * @see #indexOf
     */
    public Node getNamedItem(String name) {
        int index = indexOf(name);
        return index < 0 ? null : (Node)elementAt(index);
    }
    
    /**
     * Adds or replaces an Attribute. If the Attribute name already exists in this list, the
     * previous Attribute is replaced, and returned. If no Attribute of the same name 
     * exists <var>null</var> is returned, and the specified <var>attribute</var> is added to the 
     * end of this AttributeList.
     * <p>This method is defined by DOM.
     * @param   attribute   The Attribute to add or replace.
     * @return              If replaced, the Attribute that was replaced; otherwise, <var>null</var>.
     * @see #remove
     */
    public Node setNamedItem(Node arg) throws DOMException {
        if (arg.getNodeType() != Node.ATTRIBUTE_NODE)
            throw new TXDOMException(DOMException.HIERARCHY_REQUEST_ERR, "Not Attribute node");
        if (this.parentElement.getOwnerDocument() != arg.getOwnerDocument())
            throw new TXDOMException(DOMException.WRONG_DOCUMENT_ERR, "Specified child was created from a different document.");
        TXAttribute txAttribute = (TXAttribute)arg;

        Attr replaced = null;
        int index = indexOf(txAttribute.getNodeName());
        if (0 > index) {
            /*
            TXAttribute tail = size() > 0 ? (TXAttribute)elementAt(size()-1) : null;
            txAttribute.setNextSibling(null);
            txAttribute.setPreviousSibling(tail);
            if (null != tail)
                tail.setNextSibling(txAttribute);
            */
            txAttribute.setParentNode(this.parentElement);
            addElement(txAttribute);
        } else {
            replaced = (Attr)elementAt(index);
            setElementAt(txAttribute, index);
            /*
            txAttribute.setPreviousSibling(replaced.getPreviousSibling());
            txAttribute.setNextSibling(replaced.getNextSibling());
            */
            txAttribute.setParentNode(this.parentElement);
        }
        if (null != this.parentElement)
            ((Child)this.parentElement).clearDigest();
        return replaced;
    }

    /**
     * Removes the Attribute instance specified by <var>name</var> from this list and returns 
     * it. If the name provided does not exist, the NoSuchAttributeException is thrown. 
     * <p>This method is defined by DOM.
     * @param   name    Name to match against in the AttributeList.
     * @return          The matching Attribute instance that was removed.
     * @see #setAttribute
     * @see #clear
     */
    public Node removeNamedItem(String name) throws DOMException {
        int ind = indexOf(name);
        if (0 > ind)
            throw new TXDOMException(DOMException.NOT_FOUND_ERR, "Attribute `"+name+"' is not found.");
        TXAttribute ret = (TXAttribute)elementAt(ind);
        /*
        TXAttribute prev = (TXAttribute)ret.getPreviousSibling();
        TXAttribute next = (TXAttribute)ret.getNextSibling();
        */
        removeElementAt(ind);
        /*
        if (null != prev)  prev.setNextSibling(next);
        if (null != next)  next.setPreviousSibling(prev);

        ret.setPreviousSibling(null);
        ret.setNextSibling(null);
        */
        ret.setParentNode(null);

        if (null != this.parentElement)
            ((Child)this.parentElement).clearDigest();
        return ret;
    }

    /**
     * Removes all Attributes from this AttributeList. 
     * @see #remove
     */
    public void clear() {
        removeAllElements();
    }

    /**
     * Returns the Attribute at the specified <var>index</var>.
     * If the specified <var>index</var> is greater than or equal to the number of Nodes 
     * in this list, a NoSuchAttributeException exception is thrown. 
     * <p>This method is defined by DOM.
     * @param   index   0-based index which identifies the Attribute to return.
     * @return          The Attribute at the specified index.
     */
    public Node item(int index) {          
        if (index < 0 || size() <= index)  return null;
        return (Node)elementAt(index);
    }

    /**
     * Returns the number of Attributes in this AttributeList. 
     * <p>This method is defined by DOM.
     * @return          The number of Attributes in this list.
     */
    public int getLength() {                   
        return size();
    }

    /**
     * Sets this AttributeList and all its Child Attributes to have the   
     * specified <var>parent</var> Node. 
     * @param   parent      The Parent for this AttributeList and its child Attributes.
     */
    public void setParent(Element parent) {
        for (int i = 0;  i < getLength();  i ++) {
            TXAttribute attr = (TXAttribute)item(i);
            attr.setParentNode(parent);
        }
        this.parentElement = parent;
    }

    /**
     * Returns all Attributes in this list as an array of <var>TXAttributes</var>. 
     * @return          All Attributes in this list.
     */
    public TXAttribute[] makeArray() {
        TXAttribute[] taa = new TXAttribute[size()];
        copyInto(taa);
        return taa;
    }
}
