/*
 * (C) Copyright IBM Corp. 1998  All rights reserved.
 *
 * US Government Users Restricted Rights Use, duplication or
 * disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * IBM will not be liable for any damages suffered by you as a result
 * of using the Program. In no event will IBM be liable for any
 * special, indirect or consequential damages or lost profits even if
 * IBM has been advised of the possibility of their occurrence. IBM
 * will not be liable for any third party claims against you.
 */

package com.ibm.xml.parser;

import org.w3c.dom.CharacterData;
import org.w3c.dom.DOMException;

/**
 * The TXCharacterData class implements the CharacterData interface as defined
 * by the Document Object Model (DOM).
 *
 * @version Revision: 38 1.3 src/com/ibm/xml/parser/TXCharacterData.java, xml4jsrc, xml4j-jtcsv, xml4j_1_1_16 
 * @author TAMURA Kent &lt;kent@trl.ibm.co.jp&gt;
 * @see org.w3c.dom.CharacterData
 */
public abstract class TXCharacterData extends Child implements CharacterData {
    
           static final long serialVersionUID = 8142858563528507880L;
           String  data                     = null;

    /**
     * <p>This method is defined by DOM.
     */
    public String getNodeValue() {
        return getData();
    }

    /**
     * Returns the actual content.
     * <p>This method is defined by DOM.
     * @return          The actual content.
     * @see #setData         
     */
    public String getData() {
        return this.data;
    }

    /**
     * Sets the actual content.
     * <p>This method is defined by DOM.
     * @param data      The actual content.
     * @see #getNodeValue 
     */
    public void setNodeValue(String data) {
        setData(data);
    }

    /**
     * Sets the actual content.
     * <p>This method is defined by DOM.
     * @param data      The actual content.
     * @see #getData         
     */
    public void setData(String data) {
        this.data = data;
        clearDigest();
    }

    /**
     * <p>This method is defined by DOM.
     *
     */
    public int getLength() {
        return this.data.length();
    }

    /**
     * Returns a substring in specified range.
     * <p>This method is defined by DOM.
     *
     * @param start Start offset to extract.
     * @param count The number of characters to extract.
     */
    public synchronized String substringData(int start, int count) throws DOMException {
        if (start < 0 || start >= this.data.length())
            throw new TXDOMException(DOMException.INDEX_SIZE_ERR, "Out of bounds: "+start
                                     +", the length of data is "+this.data.length());
        if (count < 0)
            throw new TXDOMException(DOMException.INDEX_SIZE_ERR, "Invalid count: "+count);
        int end = start+count;
        if (end > this.data.length())  end = this.data.length();
        return this.data.substring(start, end);
    }

    /**
     * Append <var>data</var> to the end of the character data in this Text Node.
     * <p>This method is defined by DOM.
     * @param data      Data to append to existing character data in this Text Node.
     * @see #insertData
     * @see #replaceData
     */
    public synchronized void appendData(String data) throws DOMException {
        StringBuffer sb = new StringBuffer(this.data.length()+data.length());
        sb.append(this.data);
        sb.append(data);
        this.data = sb.toString();
        clearDigest();
    }
    
    /**
     * Insert <var>data</var> at the specified character offset. 
     * <p>This method is defined by DOM.
     * @param offset    0-based offset into existing character data in this Text Node.
     * @param data      Data to insert into existing character data in this Text Node.
     * @see #appendData
     * @see #replaceData
    */
    public synchronized void insertData(int offset, String data) throws DOMException {
        if (offset < 0 || offset > this.data.length())
            throw new TXDOMException(DOMException.INDEX_SIZE_ERR, "offset is out of bounds: "+offset);
        StringBuffer sb = new StringBuffer(this.data.length()+data.length());
        sb.append(this.data);
        sb.insert(offset, data);
        this.data = sb.toString();
        clearDigest();
    }
    
    /**
     * Delete characters at the specified character offset. 
     * <p>This method is defined by DOM.
     * @param offset    0-based offset into existing character data in this Text Node.
     * @param count     Number of characters to delete.
     * @see #appendData
     * @see #replaceData
     */
    public synchronized void deleteData(int offset, int count) throws DOMException {
        if (offset < 0 || offset >= this.data.length())
            throw new TXDOMException(DOMException.INDEX_SIZE_ERR, "Out of bounds: "+offset
                                     +", the length of data is "+this.data.length());
        if (count < 0)
            throw new TXDOMException(DOMException.INDEX_SIZE_ERR, "Invalid count: "+count);
        int end = offset+count;
        if (end >= this.data.length()) {
            this.data = this.data.substring(0, offset);
        } else {
            this.data = this.data.substring(0, offset)+this.data.substring(offset+count);
        }
        clearDigest();
    }
    
    /**
     * Replace the characters starting at the specified character offset with the specified string.     
     * <p>This method is defined by DOM.
     * @param offset    0-based offset into existing character data in this Text Node.
     * @param count     Number of existing characters to replace.
     * @param data      Data to insert into existing character data in this Text Node.
     * @see #appendData
     * @see #insertData
     * @see #deleteData
     */
    public synchronized void replaceData(int offset, int count, String data) throws DOMException {
        if (offset < 0 || offset >= this.data.length())
            throw new TXDOMException(DOMException.INDEX_SIZE_ERR, "Out of bounds: "
                                     +offset+", the length of data is "+this.data.length());
        if (count < 0)
            throw new TXDOMException(DOMException.INDEX_SIZE_ERR, "Invalid count: "+count);
        int end = offset+count;
        if (end >= this.data.length()) {
            this.data = this.data.substring(0, offset)+data;
        } else {
            this.data = this.data.substring(0, offset)+data+this.data.substring(end);
        }
        clearDigest();
    }
}
