/*
 * (C) Copyright IBM Corp. 1997-1998  All rights reserved.
 *
 * US Government Users Restricted Rights Use, duplication or
 * disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * IBM will not be liable for any damages suffered by you as a result
 * of using the Program. In no event will IBM be liable for any
 * special, indirect or consequential damages or lost profits even if
 * IBM has been advised of the possibility of their occurrence. IBM
 * will not be liable for any third party claims against you.
 */

package com.ibm.xml.parser;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Enumeration;
import java.util.Vector;
import org.w3c.dom.Attr;
import org.w3c.dom.CDATASection;
import org.w3c.dom.Comment;
import org.w3c.dom.DOMException;
import org.w3c.dom.DOMImplementation;
import org.w3c.dom.Document;
import org.w3c.dom.DocumentFragment;
import org.w3c.dom.DocumentType;
import org.w3c.dom.Element;
import org.w3c.dom.Entity;
import org.w3c.dom.EntityReference;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.w3c.dom.ProcessingInstruction;
import org.w3c.dom.Text;

/**
 * The TXDocument class implements the Document interface as defined by the Document Object Model (DOM).
 * <p>The Document object represents the entire XML document. Conceptually, it is the root 
 * of the document tree, and provides the primary access to the document's data.
 *
 * @version Revision: 35 1.31 src/com/ibm/xml/parser/TXDocument.java, xml4jsrc, xml4j-jtcsv, xml4j_1_1_16 
 * @author TAMURA Kent &lt;kent@trl.ibm.co.jp&gt;
 * @see org.w3c.dom.Document
 * @see com.ibm.xml.parser.Parent
 * @see com.ibm.xml.parser.Child
 * @see com.ibm.xml.parser.Parser#setElementFactory
 */
public class TXDocument extends Parent implements Document {
    
    static final long serialVersionUID = -7547414605402949958L;
    static transient TXDocument        s_instance   = null;
    static transient DOMImplementation s_domImpl    = null;
    
    transient   MessageDigest   messageDigest       = null;
                String          defDigestAlgorithm  = "MD5";

                TXElement       rootElement         = null;
                DTD             doctype             = null;
                String          standalone          = null;
                String          xmlVersion          = null;
                String          xmlEncoding         = null;
                boolean         isCheckDocument     = false;
                boolean         checkValidity       = false;
                boolean         isProcessNamespace  = false;
                boolean         isCheckNodeLoop     = true;
                boolean         isAddFixedAttributes= true;
    /**
     *
     * @see com.ibm.xml.parser.Namespace#createExpandedName
     */
    public      String          expandedNameSeparator= ":";

    /**
     * Constructor.  Adds the <CODE>xml</CODE> namespace to this document.
     */
    public TXDocument() {
    }

    /**
     * Clone this Document Node and its children using the appropriate factories.
     * <p>This method is defined by Child.
     * @return          Cloned Document Node and all children.
     * @see com.ibm.xml.parser.Child#clone
     */
    public Object clone() {
        return cloneNode(true);
    }

    /**
     *
     */
    public synchronized Node cloneNode(boolean deep) {
        checkFactory();
        TXDocument d;
        try {
            d = (TXDocument)this.getClass().newInstance();
        } catch (InstantiationException ie) {
            throw new LibraryException("TXDocument#cloneNode(): "+ie);
        } catch (IllegalAccessException iae) {
            throw new LibraryException("TXDocument#cloneNode(): "+iae);
        }
        d.setFactory(getFactory());
        d.defDigestAlgorithm  = "MD5";
        d.standalone          = this.standalone;
        d.xmlVersion          = this.xmlVersion;
        d.xmlEncoding         = this.xmlEncoding;
        d.isCheckDocument     = this.isCheckDocument;
        d.checkValidity       = this.checkValidity;
        d.isProcessNamespace  = this.isProcessNamespace;
        d.isCheckNodeLoop     = this.isCheckNodeLoop;
        d.isAddFixedAttributes= this.isAddFixedAttributes;
        
        if (deep) {
            d.children.ensureCapacity(children.getLength());
            for (int i = 0;  i < children.getLength();  i ++)
                d.appendChild(children.item(i).cloneNode(true));
        }
        d.rootElement         = (TXElement)d.getDocumentElement();
        d.doctype             = d.getDTD();
        return d;
    }

    /**
     *
     */
    public synchronized boolean equals(Node arg, boolean deep) {
        if (!(arg instanceof TXDocument))  return false;
        TXDocument doc = (TXDocument)arg;
        if (!((doc.standalone == null && this.standalone == null)
              || doc.standalone != null && doc.standalone.equals(this.standalone)))
            return false;
        if (!((doc.xmlVersion == null && this.xmlVersion == null)
              || doc.xmlVersion != null && doc.xmlVersion.equals(this.xmlVersion)))
            return false;
        if (!((doc.xmlEncoding == null && this.xmlEncoding == null)
              || doc.xmlEncoding != null && doc.xmlEncoding.equals(this.xmlEncoding)))
            return false;
        if (deep) {
            if (!doc.children.equals(this.children, deep))
                return false;
        }
        return true;
    }

    /**
     * Returns that this object is a Document Node.
     * <p>This method is defined by DOM.
     * @return          Document Node indicator.
     */
    public short getNodeType() {
        return Node.DOCUMENT_NODE;
    }

    /**
     *
     */
    public String getNodeName() {
        return Child.NAME_DOCUMENT;
    }

    /**
     * Returns this Document's root Element.
     * @return          Document root Element, or <var>null</var> if no root Element.
     * @see #getRootName
     */
    public Element getDocumentElement() {
        return this.rootElement;
    }

    /**
     * Sets this Document's root Element.
     * @return          Document root Element.
     * @see #getDocumentElement
     */
    void setDocumentElement(Element rootElement) {
        this.rootElement = (TXElement)rootElement;
    }

    /**
     * Returns the name of this Document's root Element (not DTD).
     * @return          Name of Document root Element, or <var>null</var> if no root element.
     * @see #getDocumentElement
     */
    public String getRootName() {
        return null == this.rootElement ? null : this.rootElement.getTagName();
    }

    /**
     * Returns a <var>NodeList</var> of matches through all child Element Nodes.
     * Searching is done recursively, not just for immediate Child Nodes.
     * The returned NodeList is not "live".
     * The specified qualified name refers to the Element tag name (see Namespace for details).
     * <p>This method is defined by DOM.
     * @param   qName   Qualified name to match against in all subordinate Elements.
     * @return          A <var>NodeList</var> of matched <var>Element</var> Nodes (<var>TXElement</var>).
     *                  If no matches, an empty NodeList is returned.
     * @see com.ibm.xml.parser.TXElement#getElementsByTagName
     * @see com.ibm.xml.parser.Namespace
     */
    public NodeList getElementsByTagName(String qName) {
        return null != rootElement ? rootElement.getElementsByTagName(qName)
            : TXNodeList.emptyNodeList;
    }
    
    /**
     * Returns this Document's Type Definition (DTD) as a Node.
     * @return          Document Type Definition (DTD), or <var>null</var> if no type.
     * @see #getDTD
     */
    public DocumentType getDoctype() {
        return this.doctype;
    }

    /**
     * Returns this Document's Type Definition (DTD) as a DTD.
     * @return          Document Type Definition (DTD), or <var>null</var> if no type.
     * @see #getDoctype
     * @see #isCheckValidity
     */
    public DTD getDTD() {
        return this.doctype;
    }

    /**
     * Sets this Document's Type Definition (DTD).
     * @param documentType  Document Type Definition (DTD) in the form of a Node.
     * @see #getDoctype
     * @see #getDTD
     * @see #isCheckValidity
    void setDoctype(Node documentType) {
        this.doctype = (DTD)documentType;
    }
     */

    /**
     * Returns whether validity constraints are checked in this doument.  In other words,
     * return whether this document has a Document Type Definition (DTD) and the DTD has
     * one or more element declarations (<KBD>&lt;!ELEMENT ...&gt;</KBD>).
     * <P>This flag doesn't affect all validity constraints in XML 1.0.
     * it affects only validity constraints about content model matching of elements
     * and attribute types.
     * @return          =true if validity costraints are checked in this document; otherwise =false.
     */
    public boolean isCheckValidity() {
        return this.checkValidity;
    }

    /**
     * Reconfigure checkValidity flag.
     */
    protected void resetCheckValidity() {
        this.checkValidity = false;
        if (this.doctype != null) {
            Enumeration en = this.doctype.getElementDeclarations();
            if (en.hasMoreElements()) {
                this.checkValidity = true;
            }
        }
    }

    /**
     * Returns if this Document is standalone. A document that contains a standalone value
     * of <var>yes</var> indicates that there are no markup declarations external to the document 
     * entity either in the DTD external subset, or in an external parameter entity 
     * referenced from the internal subset. The value <var>no</var> indicates that there 
     * are or may be such external markup declarations. 
     * @return          =<var>yes</var> if standalone; =<var>no</var> if not standalone; 
     *                    =<var>null</var> if not known.
     * @see #setStandalone
     * @see #isStandalone
     */
    public String getStandalone() {
        return this.standalone;
    }

    /**
     * Returns if this Document is standalone. A document that contains a standalone value
     * of <var>true</var> indicates that there are no markup declarations external to the document 
     * entity either in the DTD external subset, or in an external parameter entity 
     * referenced from the internal subset. The value <var>false</var> indicates that there 
     * are or may be such external markup declarations. 
     * @return          =<var>true</var> if standalone; otherwise, =<var>false</var>. 
     * @see #getStandalone
     * @see #setStandalone
     */
    public boolean isStandalone() {
        return null == this.standalone ? false : "yes".equals(this.standalone);
    }

    /**
     * Sets if this Document is standalone. A document that contains a standalone value
     * of <var>yes</var> indicates that there are no markup declarations external to the document 
     * entity either in the DTD external subset, or in an external parameter entity 
     * referenced from the internal subset. The value <var>no</var> indicates that there 
     * are or may be such external markup declarations. 
     * @return          =<var>yes</var> if standalone; =<var>no</var> if not standalone; 
     *                    =<var>null</var> if not known.
     * @see #isStandalone
     * @see #getStandalone
     */
    public void setStandalone(String standalone) {
        this.standalone = standalone;
        if (null == this.xmlVersion)
            this.xmlVersion = "1.0";
    }

    /**
     * Returns the XML version of this Document from the XML prolog declaration (e.g. <code>&lt;?xml ...></code>).
     * @return          XML version of this Document (e.g. "1.0"), or <var>null</var> if version not specified.
     * @see #setVersion
     */
    public String getVersion() {
        return this.xmlVersion;
    }

    /**
     * Sets the XML version of this Document from the XML prolog declaration (e.g. <code>&lt;?xml ...></code>).
     * @param   xmlVersion  XML version of this Document (e.g. "1.0").
     * @see #getVersion
     */
    public void setVersion(String xmlVersion) {
        this.xmlVersion = xmlVersion;
    }

    /**
     * Returns the value of the XML encoding parameter from the XML prolog declaration (e.g. <code>&lt;?xml encoding="..."&gt;</code>).
     * @return          Value of the XML encoding parameter, or <var>null</var> if encoding not specified.
     * @see #setEncoding
     */
    public String getEncoding() {
        return this.xmlEncoding;
    }

    /**
     * Sets the value of the XML encoding parameter from the XML prolog declaration (e.g. <code>&lt;?xml encoding="..."&gt;</code>).
     * <p>The supported XML encodings are the intersection of XML-supported code sets and 
     * those supported in JDK 1.1:
     * <ul>
     * <li>UTF-16
     * <li>ISO-10646-UCS-2
     * <li>ISO-10646-UCS-4
     * <li>UTF-8
     * <li>US-ASCII
     * <li>ISO-8859-1 ... ISO-8859-9
     * <li>ISO-2022-JP
     * <li>Shift_JIS
     * <li>EUC-JP
     * <li>GB2312
     * <li>Big5
     * <li>EUC-KR
     * <li>ISO-2022-KR
     * <li>KOI8-R
     * </ul>
     * @param   xmlEncoding Value of the XML encoding parameter. 
     * @see com.ibm.xml.parser.MIME2Java#convert
     * @see #getEncoding
     * @see com.ibm.xml.parser.DTD#setEncoding
     */
    public void setEncoding(String xmlEncoding) {
        this.xmlEncoding = xmlEncoding;
        if (null == this.xmlVersion)
            this.xmlVersion = "1.0";
    }

    /**
     * Sets whether namespaces are considered in some operations to nodes created by this document.
     * This flag is automatically set when <CODE>Parser#setProcessNamespace(true)</CODE> is called and
     * this document is created by the parser.
     * @see com.ibm.xml.parser.Parser#setProcessNamespace
     */
    public void setProcessNamespace(boolean isProcessNamespace) {
        this.isProcessNamespace = isProcessNamespace;
    }

    /**
     * Returns whether namespaces are considered.
     */
    public boolean isProcessNamespace() {
        return this.isProcessNamespace;
    }

    /**
     * Returns the specified <var>namespace</var> with all aspects of its namespace set
     * according to NamespacePIs defined in this document, and the raw tag name.
     * @param namespace Namespace to be initialized. Note: <code>namespace.getName()</code>
     *                  must contain the tag's qualified name.
     * @return          Error message, or <code>null</code> if successful.
     * @see #getNamespaces
     * @see #getNamespaceFor
     * @see #addNamespace
     * @see com.ibm.xml.parser.Namespace
     * @see com.ibm.xml.parser.NamespacePI
    public String setNamespaceParameters(Namespace namespace) {
        String ret = null;
        String name = namespace.getName();
        int nssep = name.indexOf(':');
        if (0 < nssep) {
            String prefix = name.substring(0, nssep);
            String local = name.substring(nssep+1);
            namespace.setNSLocalName(local);
            NamespacePI npi = getNamespaceFor(prefix);
            if (null != npi)
                namespace.setNSName(npi.getNs());
            else
                ret = "E_TAGf";
            if (0 <= local.indexOf(':'))  ret = "E_TAGe";
        } else {
            namespace.setNSLocalName(name);
            namespace.setNSName(null);
        }
        return ret;
    }
     */

    /**
     * Insert a Child Node into the specified position.  Special consideration is given
     * to DTDs, the root document Element, and namspace PI children.
     * @param child     The Node being inserted.
     * @param index     0-based index into the list of children.
     * @exception LibraryException Thrown if the document's root element is set twice.
     */
    protected void realInsert(Node child, int index) throws LibraryException {     
        super.realInsert(child, index);
        if (child instanceof DTD) {
            if (null != this.doctype)
                this.removeChild(this.doctype);
            this.doctype = (DTD)child;
        } else if (child instanceof TXElement) {
            if (null != this.rootElement)
                throw new LibraryException("com.ibm.xml.parser.TXDocument#insert(): Document root Element was set twice.");
            else
                this.setDocumentElement((Element)child);
        }
    }

    /**
     * Replaces the Child Node <var>oldChild</var> with <var>newChild</var> in this Node's list of children, and return
     * the <var>oldChild</var> Node. If <var>oldChild</var> was not already a Child of this Node,
     * a <var>DOMException</var> is thrown.
     * <P>A DTD can't be replaced by this method.
     * <p>This method is defined by DOM.
     *
     * @param newChild  The Child Node to replace with.
     * @param oldChild  The Child Node being replaced.
     * @return          The Child Node being replaced.
     * @exception org.w3c.dom.DOMException Thrown if <var>oldChild</var> is not a Child of this object.
     * @see #removeChild  
     */
    public Node replaceChild(Node newChild, Node oldChild) throws DOMException {
        if (newChild.getNodeType() == Node.ELEMENT_NODE) {
            if (this.getDocumentElement() != null)
                if (oldChild.getNodeType() != Node.ELEMENT_NODE)
                    throw new TXDOMException(DOMException.HIERARCHY_REQUEST_ERR,
                                             "Root element already exists.");
        }
            
        super.replaceChild(newChild, oldChild);
        if (newChild.getNodeType() != Node.ELEMENT_NODE
            && oldChild.getNodeType() == Node.ELEMENT_NODE)
            this.setDocumentElement(null);
        if (newChild.getNodeType() == Node.ELEMENT_NODE)
            this.setDocumentElement((Element)newChild);
        return oldChild;
    }

    /**
     * Removes the Child Node indicated by <var>oldChild</var> from this Nodes list of children,
     * and returns it. If <var>oldChild</var>
     * was not a Child of this Node, a <var>DOMException</var> is thrown.
     * <p>This method is defined by DOM.
     * @param oldChild  The Child Node being removed.
     * @return          The Child Node being removed.
     * @exception org.w3c.dom.DOMException Thrown if <var>oldChild</var> is not a Child of this object.
     * @see #replaceChild 
     */
    public Node removeChild(Node oldChild) throws DOMException {
        super.removeChild(oldChild);
        if (oldChild == this.getDocumentElement())
            this.setDocumentElement(null);
		else if (oldChild == this.getDoctype()) 
			this.doctype = null;
        return oldChild;
    }

    /**
     * Format and print this Node and any children in XML format using the default character 
     * encoding.
     * @param pw        The character output stream to use.
     * @see com.ibm.xml.parser.Child#print 
     * @exception IOException       Thrown if a Node can not be visitted because of an invalid <var>pw</var>.
     * @exception LibraryException  Thrown if the document object hierarchy is in an unknown state.
     */
    public void printWithFormat(java.io.Writer pw) throws IOException, LibraryException {
        printWithFormat(pw, null, 2);
    }  
    
    /**
     * Format and print this Node and any children in XML format using the specified character
     * <var>encoding</var>.
     * @param pw        The character output stream to use.
     * @param encoding  Java character encoding in use by <VAR>pw</VAR>.
     * @see com.ibm.xml.parser.Child#print 
     * @exception IOException       Thrown if a Node can not be visitted because of an invalid <var>pw</var>.
     * @exception LibraryException  Thrown if the document object hierarchy is in an unknown state.
     */
    public void printWithFormat(java.io.Writer pw, String encoding) throws IOException, LibraryException {
        printWithFormat(pw, encoding, 2);
    }
    
    /**
     * Format and print this Node and any children in XML format, using the specified 
     * character <var>encoding</var>, and indented by <var>indent</var> spaces.
     * @param pw        The character output stream to use.
     * @param encoding  Java character encoding in use by <VAR>pw</VAR>.
     * @param indent    Number of spaces to indent at each nesting level.
     * @see com.ibm.xml.parser.Child#print 
     * @exception IOException       Thrown if a Node can not be visitted because of an invalid <var>pw</var>.
     * @exception LibraryException  Thrown if the document object hierarchy is in an unknown state.
     */
    public void printWithFormat(java.io.Writer pw, String encoding, int indent)
        throws IOException, LibraryException {
        try {
            Visitor v = new FormatPrintVisitor(pw, encoding, indent);
            new NonRecursivePreorderTreeTraversal(v).traverse(this);
        } catch (IOException ioe) {
            throw ioe;
        } catch (Exception e) {
            throw new LibraryException("com.ibm.xml.parser.TXDocument#printWithFormat(): Unexpected Exception: "+e.toString());
        }
    }

    /**
     * Sets whether the internal DTD is printed by the <code>print()</code> methods.
     * @param   printInternalDTD    =true to print the internal DTD; =false to not print.
     */
    public void setPrintInternalDTD(boolean printInternalDTD) {
        if (null != this.doctype)
            this.doctype.setPrintInternalDTD(printInternalDTD);
    }

    /**
     * Return all text associated with this Node and its children without considering entities.                                           
     * <p>This method is defined by Child.
     * @return          Text associated with all children, or <var>""</var> if no children.
     * @see com.ibm.xml.parser.Child#toXMLString
     */
    public String getText() {                   
        return null != this.rootElement ? this.rootElement.getText() : "";
    }

    /**
     * Return an instance of <KBD>DOMImplementation</KBD> handling this document.
     * <p>This method is defined by DOM.
     */
    public DOMImplementation getImplementation() {
        if (s_domImpl == null) {
            s_domImpl = new DOMImplementation () {
                public boolean hasFeature(String feature, String version) {
                    if (!feature.equalsIgnoreCase("XML"))
                        return false;
                    if (version == null)  return true;
                                                // Check version...
                    return version.equals("1.0");
                }
            };
        }
        return s_domImpl;
    }

    /**
     * Implements the accept operation of the visitor design pattern when the start of
     * a TXDocument Node is recognized when traversing the document object tree. 
     * @param   visitor The implemention of the Visitor operation (toXMLString, digest, ...)
     * @exception Exception Thrown if this Node can not be visitted, or traversal modification is requested.
     * @see com.ibm.xml.parser.Visitor
     * @see com.ibm.xml.parser.TreeTraversal
     * @see com.ibm.xml.parser.NonRecursivePreorderTreeTraversal
     * @see com.ibm.xml.parser.TreeTraversalException
     */
    public void acceptPre(Visitor visitor) throws Exception {
        visitor.visitDocumentPre(this);
    }

    /**
     * Implements the accept operation of the visitor design pattern when the end of
     * a TXDocument Node is recognized when traversing the document object tree. 
     * @param   visitor The implemention of the Visitor operation (toXMLString, digest, ...)
     * @exception Exception Thrown if this Node can not be visitted, or traversal modification is requested.
     * @see com.ibm.xml.parser.Visitor
     * @see com.ibm.xml.parser.TreeTraversal
     * @see com.ibm.xml.parser.NonRecursivePreorderTreeTraversal
     * @see com.ibm.xml.parser.TreeTraversalException
     */
    public void acceptPost(Visitor visitor) throws Exception {
        visitor.visitDocumentPost(this);
    }

    /**
     * Returns <VAR>null</VAR>.
     * <p>This method is defined by DOM.
     * @return Always return <VAR>null</VAR>
     */
    public Document getOwnerDocument() {
        return null;
    }

    // ================================================================
    // Factory methods
    // ================================================================

    /**
     *
     */
    public TXDocument getFactory() {
        return this;
    }


    /**
     * Returns the current <CODE>TXDocument</CODE> instance; if an instance
     * does not currently exist, it will be created.
     * @return      The current default factory instance.
     */
    public static TXDocument getInstance() {
        if (null == s_instance)  s_instance = new TXDocument();
        return s_instance;
    }

    /**
     * Create and initialize a <code>TXElement</code> instance using the supplied parameters.
     * <p>This method is defined by DOM.
     * @param   name    This Element's tag name (qualified name).
     * @return          Newly created TXElement.
     * @exception org.w3c.dom.DOMException INVALID_NAME_ERR: <VAR>name</var> is invalid.
     */
    public Element createElement(String name) throws DOMException {
        if (!Util.checkName(name))
            throw new TXDOMException(DOMException.INVALID_CHARACTER_ERR, "Invalid name: "+name);
        TXElement el = new TXElement(name);
        el.setFactory(this);
        return el;
    }

    /**
     * Create and return a new <CODE>Attr</CODE>.
     * <p>This method is defined by DOM.
     * @param name      The name of this attribute. 
     * @return          New <var>TXAttribute</var>.
     * @exception org.w3c.dom.DOMException INVALID_NAME_ERR: <VAR>name</var> is invalid.
     */
    public Attr createAttribute(String name) throws DOMException {
        if (!Util.checkName(name))
            throw new TXDOMException(DOMException.INVALID_CHARACTER_ERR, "Invalid name: "+name);
        TXAttribute attr = new TXAttribute(name, null);
        attr.setFactory(this);
        return attr;
    }

    /**
     * Create and initialize a <code>TXText</code> instance using the supplied parameters.
     * <p>This method is defined by DOM.
     * @param data      The actual content of the Text Node.
     * @return          Newly created TXText.
     */
    public Text createTextNode(String data) {
        return createTextNode(data, false);
    }

    /**
     * Create and initialize a <code>TXText</code> instance using the supplied parameters.
     * @param   data                    The actual content of the Text Node.
     * @param  isIgnorableWhitespace    <code>=true</code> space is to be preserved; 
     *                                  <code>=false</code> space is to be ignored.
     * @return      Newly created TXText.
     * @see com.ibm.xml.parser.TXText#setIsIgnorableWhitespace
     */
    public TXText createTextNode(String data, boolean isIgnorableWhitespace) {
        TXText te = new TXText(data);
        te.setFactory(this);
        te.setIsIgnorableWhitespace(isIgnorableWhitespace);
        return te;
    }

    /**
     * Create and initialize a <code>TXText</code> instance using the supplied parameters.
     * @param   charArray               Existing character array to use as actual content of the Text Node.
     * @param   offset                  Offset into <var>charArray</var>.
     * @param   length                  Number of <var>charArray</var> characters to use.
     * @param   isIgnorableWhitespace   <code>=true</code> space is to be preserved; 
     *                                  <code>=false</code> space is to be ignored.
     * @return      Newly created TXText.
     * @see com.ibm.xml.parser.TXText#setIsIgnorableWhitespace
     */
    public TXText createTextNode(char[] charArray, int offset,
                                 int length, boolean isIgnorableWhitespace) {
        TXText te = new TXText(new String(charArray, offset, length));
        te.setFactory(this);
        te.setIsIgnorableWhitespace(isIgnorableWhitespace);
        return te;
    }

    /**
     * Create and initialize a <code>TXCDATASection</code> instance using the supplied parameters.
     * <p>This method is defined by DOM.
     * @param data      The actual content of the CDATASection Node.
     * @return          Newly created TXCDATASection.
     * @exception org.w3c.dom.DOMException Never thrown.
     */
    public CDATASection createCDATASection(String data) throws DOMException {
        TXCDATASection cdata = new TXCDATASection(data);
        cdata.setFactory(this);
        return cdata;
    }

    /**
     * Create and initialize a <code>TXComment</code> instance using the supplied parameters.
     * <p>This method is defined by DOM.
     * @param data      The actual content of the Comment Node.
     * @return          Newly created TXComment.
     */
    public Comment createComment(String data) { 
        TXComment comm = new TXComment(data);
        comm.setFactory(this);
        return comm;
    }

    /**
     * Create and initialize a <code>TXPI</code> instance using the supplied parameters.
     * <p>This method is defined by DOM.
     * @param name      The first token following the markup.
     * @param data      From the character immediately after <var>name</var> to the 
     *                    character immediately preceding the <code>?&gt;</code>.
     * @return          Newly created TXPI.
     * @exception org.w3c.dom.DOMException INVALID_NAME_ERR: <VAR>name</var> is invalid.
     */
    public ProcessingInstruction createProcessingInstruction(String name, String data)
        throws DOMException {
        if (!Util.checkName(name))
            throw new TXDOMException(DOMException.INVALID_CHARACTER_ERR, "Invalid PI target name: "+name);
        TXPI pi = new TXPI(name, data);
        pi.setFactory(this);
        return pi;
    }

    /**
     * Create and initialize a <code>StylesheetPI</code> instance using the supplied parameters.
     * @param name          The first token following the markup (e.g. <CODE>"xml:stylesheet"</CODE>).
     * @param data          From the character immediately after <var>name</var> to the character immediately preceding the <code>?&gt;</code>.
     * @param type          The value of the <code>type=</code> attribute.
     * @param hrefURI       The value of the <code>href=</code> attribute.
     * @param title         The value of the <CODE>title=</CODE> attribute, or <var>null</var>.
     * @return              Newly created StylesheetPI.
     */
    public StylesheetPI createStylesheetPI(String name, String data,
                                           String type, String hrefURI, String title) {
        StylesheetPI pi = new StylesheetPI(name, data, type, hrefURI, title);
        pi.setFactory(this);
        return pi;
    }

    /**
     * Create and initialize a <code>DTD</code> instance.
     * @return      Newly created DTD.
     */
    public DTD createDTD() {
        DTD d = new DTD();
        d.setFactory(this);
        return d;
    }

    /**
     * Create and initialize a <code>DTD</code> instance using the supplied parameters.
     * @param name          The name of this DTD.  This value is also known as the <code>DOCTYPE</code>
     *                      and the root Element Name.
     * @param externalID    The external ID associated with this DTD.
     * @return              Newly created DTD.
     * @see com.ibm.xml.parser.ExternalID
     */
    public DTD createDTD(String name, ExternalID externalID) {
        DTD d = new DTD(name, externalID);
        d.setFactory(this);
        return d;
    }
    
    protected void initDTD() {
    }

    /**
     * Create and initialize a <code>ElementDecl</code> instance using the supplied parameters.
     * @param name          This element definition's name. 
     * @param contentModel  The content model to associate with this element definition,
     *                      or <var>null</var> if the content model is to be set later.
     * @return              Newly created ElementDecl.
     */
    public ElementDecl createElementDecl(String name, ContentModel contentModel) {
        ElementDecl ed = new ElementDecl(name, contentModel);
        ed.setFactory(this);
        return ed;
    }
    
    /**
     * Create and initialize a <code>ContentModel</code> instance using the supplied parameters.
     * This factory is for content models NOT of type <var>MODEL_GROUP</var>.
     * @param   type    The type for this content model. 
     *                  Must be one of org.w3c.dom.ElementDefinition#ContentType.
     *                  Note that the XML4J parser will never set <code>#PCDATA</code> as the
     *                  content type; <code>MODEL_GROUP</code> will be set instead.
     * @return          Newly created ContentModel.
     */
    public ContentModel createContentModel(int type) {
        ContentModel cm = new ContentModel(type);
        cm.setFactory(this);
        return cm;
    }

    /**
     * Create and initialize a <code>ContentModel</code> instance using the supplied parameters.
     * This factory is for content models of <var>type</var> <code>MODEL_GROUP</code>.
     * @param   modelGroupNode    The content model associated with the model group.
     * @return      Newly created ContentModel.
     */
    public ContentModel createContentModel(CMNode modelGroupNode) {
        ContentModel cm = new ContentModel(modelGroupNode);
        cm.setFactory(this);
        return cm;
    }

    /**
     * Create and initialize a <code>Attlist</code> instance using the supplied parameters.
     * @param name      This attribute list's name; this value is also known as the Element type.
     * @return          Newly created Attlist.
     */
    public Attlist createAttlist(String name) {
        Attlist al = new Attlist(name);
        al.setFactory(this);
        return al;
    }
    
    /**
     * Create and initialize a <code>AttDef</code> instance using the supplied parameters.
     * @param   name        Name of this attribute as defined by the DTD.
     * @return              Newly created AttDef.
     */
    public AttDef createAttDef(String name) {
        AttDef ad = new AttDef(name);
        ad.setFactory(this);
        return ad;
    }


    /**
     * Create and initialize a <code>EntityDecl</code> instance using the supplied parameters.
     * This factory is for internal entities.
     * @param   name        Name of this entity.
     * @param   value       The XML-encoded value that was directly assigned to the EntityDecl.
     * @param   isParameter =true if a parameter entity; otherwise =false.
     * @return              Newly created EntityDecl.
     */
    public EntityDecl createEntityDecl(String name, String value, boolean isParameter) {
        EntityDecl en = new EntityDecl(name, value, isParameter);
        en.setFactory(this);
        return en;
    }
    
    /**
     * Create and initialize a <code>EntityDecl</code> instance using the supplied parameters.
     * Constructor for external entities.
     * @param   name        Name of the entity.
     * @param   externalID  The reference(s) to the external entity to retrieve. 
     * @param   isParameter =true if a parameter entity; otherwise =false.
     * @param   ndata       The notation associated with the binary entity, or <var>null</var> if
     *                      the Entity is a text entity.
     * @return              Newly created EntityDecl.
     * @see com.ibm.xml.parser.ExternalID
     */
    public EntityDecl createEntityDecl(String name, ExternalID externalID,
                                       boolean isParameter, String ndata) {
        EntityDecl en = new EntityDecl(name, externalID, isParameter, ndata);
        en.setFactory(this);
        return en;
    }
    
    /**
     * Create and initialize a <code>TXNotation</code> instance using the supplied parameters.
     * @param name          The name of the Notation.
     * @param externalID    The public or system identifier which defines the DTD's notation.
     * @return              Newly created TXNotation.
     */
    public TXNotation createNotation(String name, ExternalID externalID) {
        TXNotation no = new TXNotation(name, externalID);
        no.setFactory(this);
        return no;
    }

    /**
     * Create and initialize a <code>EntityReference</code> instance using the supplied parameters.
     * @param name  This reference's name.  This is also the name of the entity
     *              being referred to by the general entity reference.
     * @return      Newly created <CODE>EntityReference</CODE>.
     */
    public EntityReference createEntityReference(String name) throws DOMException {
        if (!Util.checkName(name))
            throw new TXDOMException(DOMException.INVALID_CHARACTER_ERR, "Invalid entity name: "+name);
        GeneralReference gr = new GeneralReference(name);
        gr.setFactory(this);
        return gr;
    }

    /**
     * <p>This method is defined by DOM.
     *
     */
    public DocumentFragment createDocumentFragment() {
        TXDocumentFragment docfrag = new TXDocumentFragment();
        docfrag.setFactory(this);
        return docfrag;
    }

    /**
     * <p>This method is defined by DOM.
     * NOT IMPLEMENTED.
     * @return Always return <VAR>null</VAR>.
     */
    public Entity createEntity() {
        return null;
    }
    
    /**
     * Returns a newly created <code>MessageDigest</code> instance if none currently exists, 
     * or resets the existing digest.  The digest will be created based on the existing
     * digest algorithm (see <code>setDigestAlgorithm</code>).
     * @return      Newly created or reset digest.
     * @exception   NoSuchAlgorithmException    Thrown if unable to create a new message
     *                                          digest based on the default digest algorithm.
     * @see #setDigestAlgorithm
     */
    public MessageDigest createMessageDigest()
        throws java.security.NoSuchAlgorithmException {
        if (null == this.messageDigest) {
            this.messageDigest = java.security.MessageDigest.getInstance(this.defDigestAlgorithm);
        } else {
            this.messageDigest.reset();
        }
        return this.messageDigest;
    }
    
    /**
     * Sets the default message digest algorithm for use by <CODE>createMessageDigest()</CODE>.
     * <p>By default, the message digest algorithm is MD5.
     * @param   defDigestAlgorithm  Refer to java.security.MessageDigest for acceptable values.
     * @see #createMessageDigest
     */
    public void setDigestAlgorithm(String defDigestAlgorithm) {
        if (!this.defDigestAlgorithm.equals(defDigestAlgorithm)) {
            this.defDigestAlgorithm = defDigestAlgorithm;
            this.messageDigest = null;
        }
    }

    /**
     * Check whether <VAR>child</VAR> is allowed to be insered in this node or not.
     * When not allowed, a DOMException with HIERARCHY_REQUEST_ERR is thrown.
     */
    protected void checkChildType(Node child) throws DOMException {
        switch (child.getNodeType()) {
          case Node.ELEMENT_NODE:
          case Node.PROCESSING_INSTRUCTION_NODE:
          case Node.COMMENT_NODE:
          case Node.DOCUMENT_TYPE_NODE:
          case Node.TEXT_NODE:                  // ****
          case Child.PSEUDONODE:                // ****
            break;
          default:
            throw new TXDOMException(DOMException.HIERARCHY_REQUEST_ERR,
                                     "Specified node type ("+child.getNodeType()
                                     +") can't be a child of Document.");
        }
    }

    /**
     * Returns a flag whether new child was created from another document.
     * Default is <VAR>false</VAR>.
     */
    public boolean isCheckOwnerDocument() {
        return this.isCheckDocument;
    }

    /**
     * Sets a flag whether new child was created from another document.
     * Default is <VAR>false</VAR>.
     */
    public void setCheckOwnerDocument(boolean doCheck) {
        this.isCheckDocument = doCheck;
    }

    /**
     * Returns a flag whether new child is an ancestor of a parent.
     * Default is <VAR>true</VAR>.
     */
    public boolean isCheckNodeLoop() {
        return this.isCheckNodeLoop;
    }

    /**
     * Sets a flag whether new child is an ancestor of a parent.
     * Default is <VAR>true</VAR>.
     */
    public void setCheckNodeLoop(boolean doCheck) {
        this.isCheckNodeLoop = doCheck;
    }

    /**
     * Returns a flag whether <KBD>#FIXED</KBD> attributes are automatically added.
     * Default is <VAR>true</VAR>.
     * @see com.ibm.xml.parser.Parser
     * @see org.w3c.dom.Element#removeAttribute(java.lang.String)
     * @see org.w3c.dom.Element#removeAttributeNode(org.w3c.dom.Attr)
     * @see com.ibm.xml.parser.TXElement#removeAttribute(java.lang.String)
     * @see com.ibm.xml.parser.TXElement#removeAttributeNode(org.w3c.dom.Attr)
     * @see com.ibm.xml.parser.TXElement#resetDefaultAttribute(java.lang.String)
     */
    public boolean isAddFixedAttributes() {
        return this.isAddFixedAttributes;
    }
    /**
     * Sets a flag whether <KBD>#FIXED</KBD> attributes are automatically added.
     * Default is <VAR>true</VAR>.
     * To stop <CODE>Parser</CODE> adding <KBD>#FIXED</KBD> attributes,
     * You must call this method before <CODE>Parser#readStream()</CODE>.
     * <PRE>
     *   TXDocument doc = new TXDocument();
     *   doc.setAddFixedAttributes(false);
     *   parser.setElementFactory(doc);
     *   parser.readStream(....);  // This method returns doc.
     * </PRE>
     *
     * @see com.ibm.xml.parser.Parser
     * @see org.w3c.dom.Element#removeAttribute(java.lang.String)
     * @see org.w3c.dom.Element#removeAttributeNode(org.w3c.dom.Attr)
     * @see com.ibm.xml.parser.TXElement#removeAttribute(java.lang.String)
     * @see com.ibm.xml.parser.TXElement#removeAttributeNode(org.w3c.dom.Attr)
     * @see com.ibm.xml.parser.TXElement#resetDefaultAttribute(java.lang.String)
     */
    public void setAddFixedAttributes(boolean addFixedAttributes) {
        this.isAddFixedAttributes = addFixedAttributes;
    }
}
