/*
 * (C) Copyright IBM Corp. 1997-1998  All rights reserved.
 *
 * US Government Users Restricted Rights Use, duplication or
 * disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * IBM will not be liable for any damages suffered by you as a result
 * of using the Program. In no event will IBM be liable for any
 * special, indirect or consequential damages or lost profits even if
 * IBM has been advised of the possibility of their occurrence. IBM
 * will not be liable for any third party claims against you.
 */

package com.ibm.xml.parser;

import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.Vector;
import org.w3c.dom.Attr;
import org.w3c.dom.CDATASection;
import org.w3c.dom.DOMException;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.w3c.dom.Text;


/**
 * The TXElement class implements the Element interface as defined by the Document Object Model (DOM),
 * and implements the namespace interface as defined by the W3C.
 * <p>By far the vast majority (apart from text) of Node types that authors will
 * generally encounter when traversing a document will be Element Nodes. An Element
 * consists of its start tag, any attributes, any children, and its end tag.
 *
 * @version Revision: 37 1.26 src/com/ibm/xml/parser/TXElement.java, xml4jsrc, xml4j-jtcsv, xml4j_1_1_16 
 * @author TAMURA Kent &lt;kent@trl.ibm.co.jp&gt;
 * @see org.w3c.dom.Element
 * @see com.ibm.xml.parser.Namespace
 * @see com.ibm.xml.parser.Parent
 * @see com.ibm.xml.parser.Child
 */
public class TXElement extends Parent implements Element, Namespace {

    static final long serialVersionUID = 1851699621748517673L;
    public static final String S_XMLNS         = "xmlns";
    public static final String S_XMLNAMESPACEURI = "http://www.w3.org/XML/1998/namespace";

            String          name            = null;
            TXAttributeList attributes      = null;
            boolean         isPreserveSpace = false;

    /**
     * Constructor.
     * @param tagName   this <CODE>Element</CODE>'s tag name (qualified name).
     *                  In the example <code>&lt;elementExample id="demo"&gt; ... &lt;/elementExample&gt;</code>,
     *                  the tag name is <code>elementExample</code>.
     */
    public TXElement(String tagName) {
        setTagName(tagName);
    }

    /**
     * Constructor for namespace support.
     * @param document      The TXDocument providing a namespace to resolve <var>prefix</var>.
     * @param prefixName    A value of the <code>prefix=</code> attribute of a namespace PI
     *                      in <var>document</var>.
     * @param nsLocalName   The Namespace local name.
     * @see com.ibm.xml.parser.NamespacePI
     * @see com.ibm.xml.parser.Namespace
    public TXElement(TXDocument document, String prefixName, String nsLocalName) {
        this.nsLocalName = nsLocalName;
        // translate prefix:localpart to nsURL+localpart for the tagName
        if (prefixName != null) {
            NamespacePI npi = document.getNamespaceFor(prefixName);
            if (npi == null)
                throw new LibraryException("com.ibm.xml.parser.TXElement: no namespace for prefix: "+prefixName);
            this.nsURI = npi.getNs();
            this.name = prefixName+":"+nsLocalName;
        }
    }
     */

    /**
     * Clone this <CODE>Element</CODE> Node and its children using the appropriate factories.
     * <p>This method is defined by Child.
     * @return          Cloned Element Node.
     * @see com.ibm.xml.parser.Child#clone
     */
    public synchronized Object clone() {
        TXElement el = cloneWithoutChildren();
        el.children.ensureCapacity(children.getLength());
        for (int i = 0;  i < children.getLength();  i ++)
            el.insertBefore(children.item(i).cloneNode(true), null);
        return el;
    }

    /**
     * Clone this <CODE>Element</CODE> Node using the appropriate factory.
     * @return          Cloned Element Node.
     */
    public synchronized TXElement cloneWithoutChildren() {
        checkFactory();
        TXElement el = (TXElement)factory.createElement(getTagName());
        el.setFactory(getFactory());
        el.setPreserveSpace(this.isPreserveSpace);
        if (null != this.attributes) {
            for (int i = 0;  i < this.attributes.getLength();  i ++)
                el.setAttributeNode((TXAttribute)this.attributes.item(i).cloneNode(true));
        }
        // el.collectNamespaceAttributes(this.getParentNode());
        return el;
    }

    /**
     *
     */
    public Node cloneNode(boolean deep) {
        return deep ? (Node)clone() : (Node)cloneWithoutChildren();
    }

    /**
     *
     */
    public boolean equals(Node arg, boolean deep) {
        if (arg == null)  return false;
        if (!(arg instanceof TXElement))  return false;
        TXElement el = (TXElement)arg;
        if (!el.getTagName().equals(this.getTagName()))
            return false;
        NamedNodeMap nnm1 = el.getAttributes();
        NamedNodeMap nnm2 = this.getAttributes();
        if (nnm1.getLength() != nnm2.getLength())  return false;
        for (int i = 0;  i < nnm1.getLength();  i ++) {
            Attr a1 = (Attr)nnm1.item(i);
            Attr a2 = (Attr)nnm2.getNamedItem(a1.getName());
            if (a2 == null || !((Child)a1).equals(a2, true))
                return false;
        }
        if (deep) {
            if (!el.children.equals(this.children, deep))
                return false;
        }
        return true;
    }

    /**
     * Returns that this object is an Element Node.
     * <p>This method is defined by DOM.
     * @return          Element Node indicator.
     */
    public short getNodeType() {
        return Node.ELEMENT_NODE;
    }

    /**
     *
     * @see #getTagName
     * @see #getName
     */
    public String getNodeName() {
        return getTagName();
    }

    /**
     * Returns this <CODE>Element</CODE>'s name.
     * In the example <code>&lt;elementExample id="demo"&gt; ... &lt;/elementExample&gt;</code>,
     * the tag name is <code>elementExample</code>.  If the Element's name has a namespace
     * prefix, the prefix will still be attached.
     * <p>This method is defined by DOM.
     * @return          The string that is this <CODE>Element</CODE>'s name, or <var>null</var> if no name.
     * @see #setTagName
     * @see #getName
     * @see #getNodeName
     */
    public String getTagName() {
        return this.name;
    }

    /**
     * Returns this <CODE>Element</CODE>'s name.
     * This method is equivalent to <CODE>getTagName()</CODE>.
     * @return          The string that is this <CODE>Element</CODE>'s name, or <var>null</var> if no name.
     * @see #getTagName
     * @see #setTagName
     * @deprecated Use getNodeName() or getTagName()
     */
    public String getName() {
        return getTagName();
    }

    /**
     * Sets this <CODE>Element</CODE>'s name.
     * In the example <code>&lt;elementExample id="demo"&gt; ... &lt;/elementExample&gt;</code>,
     * the tag name is <code>elementExample</code>.
     * @param   tagName The string that is this <CODE>Element</CODE>'s name.
     * @see #getTagName
     */
    public void setTagName(String tagName) {
        this.name = tagName;
        clearDigest();
    }

    /**
     * Returns an attribute's value from this <CODE>Element</CODE> that matches the specified
     * attribute name.
     * Both implicitly-defined (DTD) and explicitly-defined attributes are considered.
     * <p>This method is defined by DOM.
     * @param   name    The name to match in this <CODE>Element</CODE>'s list of attributes.
     * @return          The string value of the matching attribute; otherwise, <var>empty string</var>.
     * @see #getAttributeNode
     * @see #setAttribute
     * @see #setAttributeNode
     * @see #getLanguage
     */
    public String getAttribute(String name) {
        if (null != this.attributes) {
            TXAttribute a = (TXAttribute)this.attributes.getNamedItem(name);
            return null == a ? "" : a.getValue();
        }
        return "";
    }

    /**
     * <b>Internal use only<b>
     * <p>
     * Returns an attribute's value from this <CODE>Element</CODE> that matches the specified
     * attribute name.
     * Both implicitly-defined (DTD) and explicitly-defined attributes are considered.
     * <p>This method is defined by DOM.
     * @param   name    The name to match in this <CODE>Element</CODE>'s list of attributes.
     * @return          The string value of the matching attribute; otherwise, <var>null</var>.
     * @see #getAttributeNode
     * @see #setAttribute
     * @see #setAttributeNode
     * @see #getLanguage
     */
    String getAttributeOrNull(String name) {
        if (null != this.attributes) {
            TXAttribute a = (TXAttribute)this.attributes.getNamedItem(name);
            return null == a ? null : a.getValue();
        }
        return null;
    }
    /**
     * Returns an attribute's value from this <CODE>Element</CODE> that matches the specified
     * attribute name.
     * Both implicitly-defined (DTD) and explicitly-defined attributes are considered.
     * <p>This method is defined by DOM.
     * @param   name    The name to match in this <CODE>Element</CODE>'s list of attributes.
     * @return          The matching Attribute Node; otherwise, <var>null</var>.
     * @see #getAttribute
     * @see #setAttribute
     * @see #setAttributeNode
     */
    public Attr getAttributeNode(String name) {
        if (null != this.attributes)
            return (Attr)this.attributes.getNamedItem(name);
        return null;
    }

    /**
     * Adds a new attribute name/value pair to this <CODE>Element</CODE> using the appropriate
     * factory. If an attribute by that name is already present in this <CODE>Element</CODE>,
     * it's value is changed to be that of <var>value</var>.
     * Both implicitly-defined (DTD) and explicitly-defined attributes are considered.
     * <p>This method is defined by DOM.
     * @param   name    The name of the attribute to create or update.
     * @param   value   The value of the created or updated attribute.
     * @see #getAttribute
     * @see #getAttributeNode
     * @see #setAttributeNode
     */
    public void setAttribute(String name, String value) throws DOMException {
        checkFactory();
        Attr attr = this.getAttributeNode(name);
        if (attr == null) {
            attr = factory.createAttribute(name);
            setAttributeNode(attr);
            attr.setValue(value);
        } else {
            attr.setValue(value);
            clearDigest();
        }
    }

    /**
     * Adds a new attribute to this <CODE>Element</CODE> using the appropriate
     * factory. If an attribute by that name is already present in this <CODE>Element</CODE>,
     * it's value is changed to be that of the specified <var>attribute</var>'s value.
     * Both implicitly-defined (DTD) and explicitly-defined attributes are considered.
     * @param   attribute   The attribute to create or update.
     * @see #getAttribute
     * @see #getAttributeNode
     * @see #setAttributeNode
     * @deprecated Use setAttributeNode()
     */
    public void setAttribute(TXAttribute attribute) {
        makeAttributeList();
        this.attributes.setNamedItem(attribute);
    }

    /**
     * Adds a new attribute to this <CODE>Element</CODE> using the appropriate
     * factory. If an attribute by that name is already present in this <CODE>Element</CODE>,
     * it's value is changed to be that of the specified <var>attribute</var>'s value.
     * Both implicitly-defined (DTD) and explicitly-defined attributes are considered.
     * <p>This method is defined by DOM.
     * @param   attribute   The attribute to create or update.
     * @return              The attribute that was created or updated.
     * @see #getAttribute
     * @see #getAttributeNode
     * @see #setAttribute
     */
    public Attr setAttributeNode(Attr attribute) {
        makeAttributeList();
        return (Attr)this.attributes.setNamedItem(attribute);
    }

    /**
     * Removes the attribute that matches the specified <var>name</var> from this <CODE>Element</CODE>.
     * Both implicitly-defined (DTD) and explicitly-defined attributes are considered.
     * <p>This method is defined by DOM.
     * @param   name    Attribute name to match against.
     * @see #removeAttributeNode
     */
    public void removeAttribute(String name) {
        if (null != this.attributes)
            this.attributes.removeNamedItem(name);
        this.resetDefaultAttribute(name);
    }

    /**
     * Removes the specified <var>attribute</var> from this <CODE>Element</CODE>.
     * Both implicitly-defined (DTD) and explicitly-defined attributes are considered.
     * <p>This method is defined by DOM.
     * @param   attribute   Attribute to remove.
     * @return Removed attribute node.
     * @see #removeAttribute
     */
    public Attr removeAttributeNode(Attr attribute) {
        Attr attr = null;
        if (null != this.attributes) {
            String value = getAttributeOrNull(attribute.getName());
            if (null != value && value.equals(attribute.getValue()))
                attr = (Attr)this.attributes.removeNamedItem(attribute.getName());
        }
        this.resetDefaultAttribute(attribute.getName());
        return attr;
    }

    /**
     * Sets a default attribute if <VAR>name</VAR> has default value.
     * This method is called by <CODE>removeAttribute(), removeAttributeNode()</CODE>.
     *
     * @param name An attribute name
     * @see #removeAttribute(java.lang.String)
     * @see #removeAttributeNode(org.w3c.dom.Attr)
     * @see com.ibm.xml.parser.TXDocument#isAddFixedAttributes()
     */
    protected void resetDefaultAttribute(String name) {
        TXDocument doc = (TXDocument)this.getOwnerDocument();
        DTD dtd = (DTD)doc.getDoctype();
        if (dtd == null)  return;
        AttDef ad = dtd.getAttributeDeclaration(this.getNodeName(), name);
        if (ad == null)  return;
        switch (ad.getDefaultType()) {
          case AttDef.REQUIRED:
          case AttDef.IMPLIED:
            break;

          case AttDef.FIXED:
            if (!doc.isAddFixedAttributes())
                break;
            // Throw down.
          case AttDef.NOFIXED:
            Attr attr = doc.createAttribute(name);
            attr.setNodeValue(ad.getDefaultStringValue());
            ((TXAttribute)attr).setSpecified(false);
            this.setAttributeNode(attr);
            break;
        }
    }

    /**
     * Returns a <var>NamedNodeMap</var> of the attributes of this <CODE>Element</CODE>.
     * Both implicitly-defined (DTD) and explicitly-defined attributes are considered.
     * <p>This method is defined by DOM.
     * @return          A NamedNodeMap of attributes, or an empty NamedNodeMap is returned 
     *                  if no attributes.
     * @see #getAttributeArray
     * @see #attributeElements
     * @see com.ibm.xml.parser.Parent#getChildNodes
     */
    public NamedNodeMap getAttributes() {
        makeAttributeList();                    // NamedNodeMap is editable
        return this.attributes;
    }

    /**
     * Returns a <CODE>TXAttribute</CODE> array of the attributes of this <CODE>Element</CODE>.
     * Both implicitly-defined (DTD) and explicitly-defined attributes are considered.
     * @return          A <CODE>TXAttribute</CODE> array of attributes, or a 0-length array if no Attributes.
     * @see #getAttributes
     * @see #attributeElements
     */
    public TXAttribute[] getAttributeArray() {
        return null == this.attributes ? new TXAttribute[0] : this.attributes.makeArray();
    }

    /**
     * Returns an enumeration of <CODE>TXAttribute</CODE> of the attributes of this <CODE>Element</CODE>.
     * Both implicitly-defined (DTD) and explicitly-defined attributes are considered.
     * @return          An enumeration of TXAttribute Attributes, or <var>null</var> if no Attributes.
     * @see #getAttributes
     * @see #attributeElements
     */
    public java.util.Enumeration attributeElements() {
        makeAttributeList();
        return this.attributes.elements();
    }

    /**
     * Returns the local name of the Namespace.
     * <p>This method is defined by Namespace.
     * @return          The Namespace local name, or <var>null</var> if no local name.
     * @see com.ibm.xml.parser.Namespace
     */
    public String getNSLocalName() {
        return this.getLocalNameForQName(getNodeName());
    }

    /**
     * Returns the Namespace URI.
     * <p>This method is defined by Namespace.
     * @return          The Namespace URI, or <var>null</var> if the tag name does not
     *                  belong to any namespace  or XML4J parser namespace processing is disabled.
     * @see #getNamespaceForPrefix
     * @see com.ibm.xml.parser.Parser#setProcessNamespace
     */
    public String getNSName() {
        return this.getNamespaceForQName(getNodeName());
    }

    /**
     * If namespace support is enabled and the Element's name contains a prefix name, return a string 
     * of <CODE>getNSName()+":"+getNSLocalName()</CODE>; otherwise, return <CODE>getNSLocalName()</CODE> .
     * @return          The universal name, or <var>null</var> if this <CODE>Element</CODE>'s name does not
     *                  contain a prefix name or XML4J parser namespace processing is disabled.
     * @see #getNSName
     * @see #getNSLocalName
     * @see com.ibm.xml.parser.Parser#setProcessNamespace
     * @deprecated Use createExpandedName().
     */
    public String getUniversalName() {
        String ns = getNSName();
        return ns == null ? getNSLocalName() : ns+":"+getNSLocalName();
    }

    /**
     * Returns an expanded name of this <CODE>Element</CODE>.
     * The expanded name is <SPAN style="background-color: #d0d0d0;">a set of [a namespace URI, localPart]</SPAN> connected by <CODE>TXDocument#expandedNameSeparator</CODE> or
     * <SPAN style="background-color: #d0d0d0;">localPart if this <CODE>Element</CODE> has no namespace</SPAN>.
     *<P>This method is defined by Namespace.
     *
     * @see com.ibm.xml.parser.TXDocument#expandedNameSeparator
     * @see com.ibm.xml.parser.Parser#setProcessNamespace
     */
    public String createExpandedName() {
        String ns = getNSName();
        return ns == null || ns.length() == 0 ? getNSLocalName()
            : (ns + getFactory().expandedNameSeparator + getNSLocalName());
    }

    /**
     * Returns whether this <CODE>Element</CODE> has any children.
     * @return          =true if this <CODE>Element</CODE> has no children; otherwise, =false.
     * @deprecated Use hasChildNodes().
     */
    public boolean isEmpty() {
        return 0 == children.getLength();
    }

    /**
     * Returns, at the Element level, whether space is to be preserved.  
     * This value is used, for example, to determine if space is to be preserved
     * in Text Nodes during printWithFormat() operations.
     * @return                  <code>=true</code> space is to be preserved; 
     *                          <code>=false</code> space is to be ignored.
     * @see #setPreserveSpace
     * @see com.ibm.xml.parser.Parser#setPreserveSpace
     * @see com.ibm.xml.parser.TXText#setIsIgnorableWhitespace
     * @see com.ibm.xml.parser.TXDocument#printWithFormat
     */
    public boolean isPreserveSpace() {
        return this.isPreserveSpace;
    }

    /**
     * Sets, at the Element level, whether space is to be preserved.  
     * This value is used, for example, to determine if space is to be preserved
     * in children of this Node during printWithFormat() operations.  
     * <p>By default, space is not preserved.  The XML4J parser will override this value
     * if it detects an attribute of <code>xml:namespace=&quot;preserve&quot;</code>.
     * @param   isPreserveSpace <code>=true</code> space is to be preserved; 
     *                          <code>=false</code> space is to be ignored.
     * @see #isPreserveSpace
     * @see com.ibm.xml.parser.Parser#setPreserveSpace
     * @see com.ibm.xml.parser.TXText#setIsIgnorableWhitespace
     * @see com.ibm.xml.parser.TXDocument#printWithFormat
     */
    public void setPreserveSpace(boolean isPreserveSpace) {
        this.isPreserveSpace = isPreserveSpace;
    }

    /**
     * Returns the XML language ID (the value of the <code>xml:lang</code> Attribute) in
     * use for this <CODE>Element</CODE> Node.  If this <CODE>Element</CODE> Node has no explicitly defined language Attribute,
     * Parent Nodes will be recursively checked.
     * @return          The XML language ID, or <var>null</var> if all Parents have no language ID.
     * @see #getAttribute
     */
    public String getLanguage() {
        String xmllang = this.getAttributeOrNull("xml:lang");
        if (null != xmllang) {
            return xmllang;
        } else if (null != parent) {
            if (parent instanceof TXElement) {
                return ((TXElement)parent).getLanguage();
            } else if (parent instanceof GeneralReference) {
                return ((GeneralReference)parent).getLanguage();
            }
        }
        return null;
    }

    /**
     * Puts all Text Nodes in the sub-tree underneath this <CODE>Element</CODE> into a "normal"
     * form where only markup (e.g., tags, comments, PIs, CDATASections, and entity
     * references) separates Text Nodes.  This has the effect of combining Text Nodes
     * that have been separated due to document manipulation.
     * <p>This method is defined by DOM.
     */
    public void normalize() {
        TXElement.normalize(this, this.isPreserveSpace());
    }

    private static void normalize(Node par, boolean preservespace) {
        Node prev = par.getFirstChild();
        if (prev == null)  return;
        if (prev.getNodeType() == Node.ELEMENT_NODE) {
            TXElement.normalize(prev, ((TXElement)prev).isPreserveSpace());
        } else if (prev.getNodeType() == Node.ENTITY_REFERENCE_NODE)
            TXElement.normalize(prev, preservespace);
        Node current;
        while ((current = prev.getNextSibling()) != null) {
            int type = current.getNodeType();
            if (type == Node.TEXT_NODE && prev.getNodeType() == Node.TEXT_NODE) {
                ((Text)current).setData(prev.getNodeValue()+current.getNodeValue());
                par.removeChild(prev);
                if (!preservespace && Util.checkAllSpace(current.getNodeValue())) {
                    Node p = current.getPreviousSibling();
                    Node n = current.getNextSibling();
                    if ((p == null || p.getNodeType() != Node.TEXT_NODE)
                        && (n == null || n.getNodeType() != Node.TEXT_NODE))
                        ((TXText)current).setIsIgnorableWhitespace(true);
                }
            } else if (type == Node.ELEMENT_NODE) {
                TXElement.normalize(current, ((TXElement)current).isPreserveSpace());
            } else if (type == Node.ENTITY_REFERENCE_NODE) {
                TXElement.normalize(current, preservespace);
            }
            prev = current;
        }
    }

    /**
     * Adds the specified <var>text</var> Node to the end of this <CODE>Element</CODE>. If the last
     * child of this <CODE>Element</CODE> is a Text Node and not a CDATA section, <var>text</var> is
     * appended to the last child.  Otherwise, a new TXText Node is inserted at the end of this
     * Element.
     * @param   text        Text Node to add to this <CODE>Element</CODE>.
     * @see com.ibm.xml.parser.Parent#addElement(com.ibm.xml.parser.Child)
     */
    public void addTextElement(TXText text) {
        if (text == null)
            return;
        if (text instanceof CDATASection) {
            appendChild(text);
            return;
        }
        Node n = getLastChild();
        if (n == null) {
            appendChild(text);
            return;
        }
        if (n.getNodeType() == Node.TEXT_NODE) {
            ((Text)n).appendData(text.getData());
            if (!text.getIsIgnorableWhitespace() || !((TXText)n).getIsIgnorableWhitespace())
                ((TXText)n).setIsIgnorableWhitespace(false);
        } else
            appendChild(text);
    }

    /**
     * Returns the first <var>TXElement</var> match through all child Element Nodes.
     * Searching is done recursively, not just for immediate Child Nodes.
     * The specified qualified name refers to the Element tag name (see Namespace for details).
     * @param   qName   Qualified name to match against in all subordinate Elements.
     * @return          The first matching <var>TXElement</var> Node, or <var>null</var> if no matches.
     * @see #getElementNamed
     * @see #getNthElementNamed
     * @see com.ibm.xml.parser.Namespace
     */
    public TXElement searchDescendants(String qName) {
        return searchDescendants(Match.QNAME, null, qName);
    }

    /**
     * Returns the first <var>TXElement</var> match through all child Element Nodes.
     * All specified input parameters must match for an element to be judged as matched (see Namespace for details).
     * Various namespace matching algorithms are supported: qualified name, local name & URI, or URI.
     * Searching is done recursively, not just for immediate Child Nodes.
     * @param   matchType           Namespace match type: <code>Match.QNAME, Match.NSLOCAL, Match.NS</code>
     * @param   uri                 When matching a URI, specify the value to match; otherwise, specify <var>null</var>.
     * @param   qNameOrLocalName    When matching a qualified name or local name, specify the value to match; otherwise, specify <var>null</var>.
     * @return                      An array of matched <var>TXElement</var> Nodes, or <var>null</var> if no matches.
     * @see #getElementNamed
     * @see #getNthElementNamed
     * @see com.ibm.xml.parser.Namespace
     */
    public TXElement searchDescendants(int matchType, String uri, String qNameOrLocalName) {
        for (Child child = (Child)getFirstChild();
             child != null;
             child = (Child)child.getNextSibling()) {
            if (child instanceof TXElement) {
                if (Match.matchName((TXElement)child, matchType, uri, qNameOrLocalName))
                    return (TXElement)child;
                else {
                    TXElement t = ((TXElement)child).searchDescendants(matchType, uri, qNameOrLocalName);
                    if (null != t)  return t;
                }
            }
        }
        return null;
    }

    /**
     * Returns the first <var>TXElement</var> match through the immediate Child Element Nodes.
     * Only immediate children are searched (i.e. no grandchildren).
     * The specified qualified name refers to the Element tag name (see Namespace for details).
     * @param   qName   Qualified name to match against in immediate children.
     * @return          The first matching <var>TXElement</var> Node, or <var>null</var> if no matches.
     * @see #searchDescendants
     * @see #getNthElementNamed
     * @see com.ibm.xml.parser.Namespace
     */
    public TXElement getElementNamed(String qName) {
        return getNthElementNamed(0, Match.QNAME, null, qName);
    }

    /**
     * Returns the first <var>TXElement</var> match through the immediate child Element Nodes.
     * All specified input parameters must match for an Element to be judged as matched (see Namespace for details).
     * Only immediate children are searched (i.e. no grandchildren).
     * @param   uri         The Namespace URI to match against in immediate children.
     * @param   localName   The Namespace local name to match against in immediate children.
     * @return              The first matching <var>TXElement</var> Node, or <var>null</var> if no matches.
     * @see #searchDescendants
     * @see #getNthElementNamed
     * @see com.ibm.xml.parser.Namespace
     */
    public TXElement getElementNamed(String uri, String localName) {
        return getNthElementNamed(0, Match.NSLOCAL, uri, localName);
    }

    /**
     * Returns the first <var>TXElement</var> match through the immediate child Elemen Nodes.
     * All specified input parameters must match for an Element to be judged as matched (see Namespace for details).
     * Various namespace matching algorithms are supported: qualified name, local name & URI, or URI.
     * Only immediate children are searched (i.e. no grandchildren).
     * @param   matchType           Namespace match type: <code>Match.QNAME, Match.NSLOCAL, Match.NS</code>
     * @param   uri                 When matching a URI, specify the value to match; otherwise, specify <var>null</var>.
     * @param   qNameOrLocalName    When matching a qualified name or local name, specify the value to match; otherwise, specify <var>null</var>.
     * @return                      An array of matched <var>TXElement</var> Nodes, or <var>null</var> if no matches.
     * @see #searchDescendants
     * @see #getNthElementNamed
     * @see com.ibm.xml.parser.Namespace
     */
    public TXElement getElementNamed(int matchType, String uri, String qNameOrLocalName) {
        return getNthElementNamed(0, matchType, uri, qNameOrLocalName);
    }

    /**
     * Returns the Nth <var>TXElement</var> match through the immediate child Element Nodes.
     * Only immediate children are searched (i.e. no grandchildren).
     * The specified qualified name refers to the Element tag name (see Namespace for details).
     * <dl>
     *   <dt>Example:</dt>
     *   <dd><pre>
     *      &lt;BODY&gt;            --- this
     *        &lt;H1&gt;...&lt;/H1&gt;    --- getNthElementNamed(0, "H1")
     *        &lt;P&gt;...&lt;/P&gt;      --- getNthElementNamed(0, "P")
     *        &lt;P&gt;...&lt;/P&gt;      --- getNthElementNamed(1, "P")
     *        &lt;HR/&gt;           --- getNthElementNamed(0, "HR");
     *        &lt;P&gt;...&lt;/P&gt;      --- getNthElementNamed(2, "P")
     *        &lt;HR/&gt;           --- getNthElementNamed(1, "HR");
     *      &lt;/BODY&gt;
     *    </pre></dd>
     *  </dl>
     * @param   nth     0-based match occurrence.
     * @param   qName   Qualified name to match against in immediate children.
     * @return          The Nth matching <var>TXElement</var> Node, or <var>null</var> if no matches.
     * @see #searchDescendants
     * @see #getElementNamed
     * @see com.ibm.xml.parser.Namespace
     */
    public TXElement getNthElementNamed(int nth, String qName) {
        return getNthElementNamed(nth, Match.QNAME, null, qName);
    }

    /**
     * Returns the Nth <var>TXElement</var> match through the immediate child Element Nodes.
     * All specified input parameters must match for an Element to be judged as matched (see Namespace for details).
     * Only immediate children are searched (i.e. no grandchildren).
     * @param   nth         0-based match occurrence.
     * @param   uri         The Namespace URI to match against in immediate children.
     * @param   localName   The Namespace local name to match against in immediate children.
     * @return              The Nth matching <var>TXElement</var> Node, or <var>null</var> if no matches.
     * @see #searchDescendants
     * @see #getElementNamed
     * @see com.ibm.xml.parser.Namespace
     */
    public TXElement getNthElementNamed(int nth, String uri, String localName) {
        return getNthElementNamed(nth, Match.NSLOCAL, uri, localName);
    }

    /**
     * Returns the Nth <var>TXElement</var> match through the immediate child Element Nodes.
     * All specified input parameters must match for an Element to be judged as matched (see Namespace for details).
     * Various namespace matching algorithms are supported: qualified name, local name & URI, or URI.
     * Only immediate children are searched (i.e. no grandchildren).
     * @param   matchType           Namespace match type: <code>Match.QNAME, Match.NSLOCAL, Match.NS</code>
     * @param   uri                 When matching a URI, specify the value to match; otherwise, specify <var>null</var>.
     * @param   qNameOrLocalName    When matching a qualified name or local name, specify the value to match; otherwise, specify <var>null</var>.
     * @return                      The Nth matching <var>TXElement</var> Node, or <var>null</var> if no matches.
     * @see #searchDescendants
     * @see #getElementNamed
     * @see com.ibm.xml.parser.Namespace
     */
    public TXElement getNthElementNamed(int nth, int matchType, String uri, String qNameOrLocalName) {
        for (Node child = getFirstChild();
             child != null;
             child = child.getNextSibling()) {
            if (child instanceof TXElement) {
                TXElement te = (TXElement)child;
                if (Match.matchName(te, matchType, uri, qNameOrLocalName)) {
                    if (0 == nth)
                        return te;
                    else
                        nth --;
                }
            }
        }
        return null;                            // Not found
    }

    /**
     * Returns an array of <var>TXElement</var> of matches through the immediate child Element Nodes.
     * Only immediate children are searched (i.e. no grandchildren).
     * The specified qualified name refers to the Element tag name (see Namespace for details).
     * @param   qName   Qualified name to match against in immediate children.
     * @return          An array of matched <var>TXElement</var> Nodes, or <var>null</var> if no matches.
     * @see #getElementsByTagName
     * @see #searchDescendantsAll
     * @see #getElementsNamed
     * @see com.ibm.xml.parser.Namespace
     */
    public TXElement[] searchChildrenAll(String qName) {
        Vector v = new Vector();
        for (Node child = getFirstChild();
             child != null;
             child = child.getNextSibling()) {
            if (child instanceof Element) {
                if (Match.matchName((Namespace)child, Match.QNAME, null, qName))
                    v.addElement(child);
            }
        }
        TXElement[] ae = new TXElement[v.size()];
        v.copyInto(ae);
        return ae;
    }

    /**
     * Returns a <var>NodeList</var> of matches through all child Element Nodes.
     * Searching is done recursively, not just for immediate Child Nodes.
     * The specified qualified name refers to the Element tag name (see Namespace for details).
     * <p>This method is defined by DOM.
     * @param   qName   Qualified name to match against in all subordinate Elements. If it is "*", this method return all subordinate Elements.
     * @return          A <var>NodeList</var> of matched <var>Element</var> Nodes (<var>TXElement</var>).
     *                  If no matches, an empty NodeList is returned.
     * @see #searchChildrenAll
     * @see #searchDescendantsAll
     * @see #getElementsNamed
     * @see com.ibm.xml.parser.Parent#getChildNodes
     * @see com.ibm.xml.parser.Namespace
     */
    public NodeList getElementsByTagName(String qName) {
        return new TXNodeList.VectorNodeList(realSearchDescendantsAll(Match.QNAME, null, qName));
    }

    /**
     * Returns an array of <var>TXElement</var> matches through all child Element Nodes.
     * Searching is done recursively, not just for immediate Child Nodes.
     * The specified qualified name refers to the Element tag name (see Namespace for details).
     * @param   qName   Qualified name to match against in all subordinate Elements.
     * @return          An array of matched <var>TXElement</var> Nodes
     * @see #searchChildrenAll
     * @see #getElementsByTagName
     * @see #getElementsNamed
     * @see com.ibm.xml.parser.Namespace
     */
    public TXElement[] searchDescendantsAll(String qName) {
        return searchDescendantsAll(Match.QNAME, null, qName);
    }

    /**
     * Returns an array of <var>TXElement</var> matches through all child Element Nodes.
     * All specified input parameters must match for an element to be judged as matched (see Namespace for details).
     * Various namespace matching algorithms are supported: qualified name, local name & URI, or URI.
     * Searching is done recursively, not just for immediate Child Nodes.
     * @param   matchType           Namespace match type: <code>Match.QNAME, Match.NSLOCAL, Match.NS</code>
     * @param   uri                 When matching a URI, specify the value to match; otherwise, specify <var>null</var>.
     * @param   qNameOrLocalName    When matching a qualified name or local name, specify the value to match; otherwise, specify <var>null</var>.
     * @return                      An array of matched <var>TXElement</var> Nodes, or <var>null</var> if no matches.
     * @see #searchChildrenAll
     * @see #getElementsByTagName
     * @see #getElementsNamed
     * @see com.ibm.xml.parser.Namespace
     */
    public TXElement[] searchDescendantsAll(int matchType, String uri, String qNameOrLocalName) {
        Vector v = realSearchDescendantsAll(matchType, uri, qNameOrLocalName);
        TXElement[] ae = new TXElement[v.size()];
        v.copyInto(ae);
        return ae;
    }

    /**
     * Returns a <var>NodeList</var> of all matches through the immediate child Element Nodes.
     * Only immediate children are searched (i.e. no grandchildren).
     * The specified qualified name refers to the Element tag name (see Namespace for details).
     * @param   qName   Qualified name to match against in immediate children.
     * @return          A <var>NodeList</var> of matches, or an empty NodeList if no matches.
     * @see #searchChildrenAll
     * @see #getElementsByTagName
     * @see #searchDescendantsAll
     * @see com.ibm.xml.parser.Parent#getChildNodes
     * @see com.ibm.xml.parser.Namespace
     */
    public NodeList getElementsNamed(String qName) {
        return getElementsNamed(Match.QNAME, null, qName);
    }

    /**
     * Returns a <var>NodeList</var> of matches through the immediate child Element Nodes.
     * All specified input parameters must match for an Element to be judged as matched (see Namespace for details).
     * Only immediate children are searched (i.e. no grandchildren).
     * @param   uri         The Namespace URI to match against in immediate children.
     * @param   localName   The Namespace local name to match against in immediate children.
     * @return              A <var>TXElement</var> of matches, or an empty NodeList if no matches.
     * @see #searchChildrenAll
     * @see #getElementsByTagName
     * @see #searchDescendantsAll
     * @see com.ibm.xml.parser.Parent#getChildNodes
     * @see com.ibm.xml.parser.Namespace
     */
    public NodeList getElementsNamed(String uri, String localName) {
        return getElementsNamed(Match.NSLOCAL, uri, localName);
    }

    /**
     * Returns a <var>NodeList</var> of all matches through the immediate child Element Nodes.
     * All specified input parameters must match for an Element to be judged as matched (see Namespace for details).
     * Various namespace matching algorithms are supported: qualified name, local name & URI, or URI.
     * Only immediate children are searched (i.e. no grandchildren).
     * @param   matchType           Namespace match type: <code>Match.QNAME, Match.NSLOCAL, Match.NS</code>
     * @param   uri                 When matching a URI, specify the value to match; otherwise, specify <var>null</var>.
     * @param   qNameOrLocalName    When matching a qualified name or local name, specify the value to match; otherwise, specify <var>null</var>.
     * @return                      A <var>NodeList</var> of matches, or an empty NodeList if no matches.
     * @see #searchChildrenAll
     * @see #getElementsByTagName
     * @see #searchDescendantsAll
     * @see com.ibm.xml.parser.Parent#getChildNodes
     * @see com.ibm.xml.parser.Namespace
     */
    public NodeList getElementsNamed(int matchType, String uri, String qNameOrLocalName) {
        Vector v = new Vector();
        for (Node child = getFirstChild();
             child != null;
             child = child.getNextSibling()) {
            if (child instanceof Element) {
                if (Match.matchName((Namespace)child, matchType, uri, qNameOrLocalName))
                    v.addElement(child);
            }
        }
        return new TXNodeList.VectorNodeList(v);
    }
    
    /**
     * Implements the accept operation of the visitor design pattern when the start of
     * a TXElement Node is recognized when traversing the document object tree. 
     * @param   visitor The implemention of the Visitor operation (toXMLString, digest, ...)
     * @exception Exception Thrown if this Node can not be visitted, or traversal modification is requested.
     * @see com.ibm.xml.parser.Visitor
     * @see com.ibm.xml.parser.TreeTraversal
     * @see com.ibm.xml.parser.NonRecursivePreorderTreeTraversal
     * @see com.ibm.xml.parser.TreeTraversalException
     */
    public void acceptPre(Visitor visitor) throws Exception {
        visitor.visitElementPre(this);
    }

    /**
     * Implements the accept operation of the visitor design pattern when the end of
     * a TXElement Node is recognized when traversing the document object tree.
     * @param   visitor The implemention of the Visitor operation (toXMLString, digest, ...)
     * @exception Exception Thrown if this Node can not be visitted, or traversal modification is requested.
     * @see com.ibm.xml.parser.Visitor
     * @see com.ibm.xml.parser.TreeTraversal
     * @see com.ibm.xml.parser.NonRecursivePreorderTreeTraversal
     * @see com.ibm.xml.parser.TreeTraversalException
     */
    public void acceptPost(Visitor visitor) throws Exception {
        visitor.visitElementPost(this);
    }


    private Vector realSearchDescendantsAll(int matchtype, String nsorpre, String ename) {
        Vector result = new Vector();
        Visitor visitor = new SearchElementVisitor(result, matchtype, nsorpre, ename);
        try {
            new NonRecursivePreorderTreeTraversal(visitor).traverse(this);
        } catch (Exception ex) {
                                                // Non-Runtime Exceptions are never thrown.
        }
        return result;
        /*
        for (Node p = getFirstChild();  p != null;  p = p.getNextSibling()) {
            if (p instanceof Element) {
                if (Match.matchName((TXElement)p, matchtype, nsorpre, ename))
                    v.addElement(p);
                ((TXElement)p).searchDescendantsAll(v, matchtype, nsorpre, ename);
            }
        }
        */
    }

    static class SearchElementVisitor extends NOOPVisitor {
        Vector result;
        int matchtype;
        String nsorpre;
        String ename;
        SearchElementVisitor(Vector result, int matchtype, String nsorpre, String ename) {
            this.result = result;
            this.matchtype = matchtype;
            this.nsorpre = nsorpre;
            this.ename = ename;
        }
        public void visitElementPre(TXElement element) throws Exception {
            if (Match.matchName(element, this.matchtype, this.nsorpre, this.ename))
                this.result.addElement(element);
        }
    }

    private void makeAttributeList() {
        if (null == this.attributes) {
            checkFactory();
            this.attributes = new TXAttributeList();
            this.attributes.setParent(this);
        }
    }

    /**
     * Check whether <VAR>child</VAR> is allowed to be insered in this node or not.
     * When not allowed, a DOMException with HIERARCHY_REQUEST_ERR is thrown.
     */
    protected void checkChildType(Node child) throws DOMException {
        switch (child.getNodeType()) {
          case Node.ELEMENT_NODE:
          case Node.PROCESSING_INSTRUCTION_NODE:
          case Node.COMMENT_NODE:
          case Node.TEXT_NODE:
          case Node.CDATA_SECTION_NODE:
          case Node.ENTITY_REFERENCE_NODE:
          case Child.PSEUDONODE:
            break;
          default:
            throw new TXDOMException(DOMException.HIERARCHY_REQUEST_ERR,
                                     "Specified node type ("+child.getNodeType()
                                     +") can't be a child of Element.");
        }
    }

    /**
     * Returns a namespace URI for specified prefix.
     * <TABLE border="1">
     *   <TR><TD>namespace URI String<TD>for valid prefix or no prefix with default namespace</TR>
     *   <TR><TD>""<TD>for no prefix and no default namespace</TR>
     *   <TR><TD>null<TD>for undefined prefix</TR>
     * </TABLE>
     * @param prefix Namespace prefix; null isn't allowed.
     */
    public String getNamespaceForPrefix(String prefix) {
        if (prefix.equals("xml"))
            return S_XMLNAMESPACEURI;
        if (prefix.equals(S_XMLNS))
            return prefix;                      // ???
        String attributeName = prefix.length() == 0 ? S_XMLNS : ("xmlns:"+prefix);
        Node element = this;
        do {
            String nsname = ((TXElement)element).getAttributeOrNull(attributeName);
            if (nsname != null)
                return nsname;
            while (true) {
                element = element.getParentNode();
                if (element == null)  return attributeName == S_XMLNS ? "" : null;
                int type = element.getNodeType();
                if (type == Node.ENTITY_REFERENCE_NODE) {
                } else if (type == Node.ELEMENT_NODE) {
                    break;
                } else
                    return attributeName == S_XMLNS ? "" : null;
            }
        } while (element != null);              // This condition has no sense.
        return null;                            // Not reached
    }

    /**
     * Returns a namespace URI for specified qualified name.
     * @return The same as getNamespaceForPrefix().
     * @see #getNamespaceForPrefix
     */
    public String getNamespaceForQName(String qname) {
        int index = qname.indexOf(':');
        String prefix = index < 0 ? "" : qname.substring(0, index).intern();
        return this.getNamespaceForPrefix(prefix);
    }

    /**
     * Returns localPart(a name after a colon) of specified qualified name.
     * If <VAR>qname</VAR> has no prefix, this method returns <VAR>qname</VAR> itself.
     */
    public static String getLocalNameForQName(String qname) {
        int index = qname.indexOf(':');
        if (index < 0)  return qname;
        return qname.substring(index+1).intern();
    }

    /**
     * Collect all <KBD>xmlns:</KBD> attributes in anchestors and set to this elemnent.
     * <PRE>
     *   &lt;parent xmlns:foo="http://foo/">
     *     &lt;foo:child/>
     *   &lt;/parent>
     * </PRE>
     * <P><CODE>collectNamespaceAttributes()</CODE> of <KBD>foo:child</KBD> element adds
     * <KBD>xmlns:foo="http://foo/"</KBD> attribute to <KBD>foo:child</KBD> element.</P>
     * <P>This method is automatically called in <CODE>TXElement#clone()</CODE> and
     * <CODE>Parent#repalceChild()</CODE> and <CODE>Parent#removeChild()</CODE>.</P>
     * @see com.ibm.xml.parser.TXDocument#isProcessNamespace
     */
    public void collectNamespaceAttributes() {
        this.collectNamespaceAttributes(this.getParentNode());
    }

    /**
     * Collect all <KBD>xmlns:</KBD> attributes in anchestors and set to this elemnent.
     * @see com.ibm.xml.parser.TXDocument#isProcessNamespace
     */
    public void collectNamespaceAttributes(Node parent) {
        if (!getFactory().isProcessNamespace())  return;
        int type;
        while (parent != null
               && ((type = parent.getNodeType()) == Node.ELEMENT_NODE
                   || type == Node.ENTITY_REFERENCE_NODE)) {
            if (type == Node.ELEMENT_NODE) {
                NamedNodeMap nnm = parent.getAttributes();
                for (int i = 0;  i < nnm.getLength();  i ++) {
                    Node attr = nnm.item(i);
                    String aname = attr.getNodeName();
                    if (aname.equals(S_XMLNS) || aname.startsWith("xmlns:")) {
                        if (this.getAttributeNode(aname) == null) {
                            this.setAttribute(aname, attr.getNodeValue());
                        }
                    }
                }
            }
            parent = parent.getParentNode();
        }
    }

    /**
     * Remove overlapped <KBD>xmlns:</KBD> attributes.
     * For example,
     * <PRE>
     *   &lt;parent xmlns:foo="http://foo/">
     *     &lt;child xmlns:bar="http://bar/" xmlns:foo="http://foo/"/>
     *   &lt;/parent>
     * </PRE>
     * <P>By calling <CODE>removeOverlappedNamespaceAttributes()</CODE> of <KBD>child</KBD> element,
     * <KBD>xmlns:foo</KBD> attribute of <KBD>child</KBD> element is removed.</P>
     * <P>This method isn't automatically called by all XML4J classes.</P>
     * @see com.ibm.xml.parser.TXDocument#isProcessNamespace
     */
    public void removeOverlappedNamespaceAttributes() {
        if (!getFactory().isProcessNamespace())  return;
        Vector remove = new Vector();
        NamedNodeMap nnm = this.getAttributes();
        for (int i = 0;  i < nnm.getLength();  i ++) {
            Node attr = nnm.item(i);
            String aname = attr.getNodeName();
            if (aname.equals(S_XMLNS) || aname.startsWith("xmlns:")) {
                Node parent = this.getParentNode();
                int type;
                while (parent != null
                       && ((type = parent.getNodeType()) == Node.ELEMENT_NODE
                           || type == Node.ENTITY_REFERENCE_NODE)) {
                    if (type == Node.ELEMENT_NODE) {
                        TXElement elem = (TXElement)parent;
                        String value = elem.getAttributeOrNull(aname);
                        if (value != null && value.equals(attr.getNodeValue()))
                            remove.addElement(aname);
                    }
                    parent = parent.getParentNode();
                }
            }
        }
        for (int i = 0;  i < remove.size();  i ++)
            this.removeAttribute((String)remove.elementAt(i));
    }
}
