/*
 * (C) Copyright IBM Corp. 1997-1998  All rights reserved.
 *
 * US Government Users Restricted Rights Use, duplication or
 * disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * IBM will not be liable for any damages suffered by you as a result
 * of using the Program. In no event will IBM be liable for any
 * special, indirect or consequential damages or lost profits even if
 * IBM has been advised of the possibility of their occurrence. IBM
 * will not be liable for any third party claims against you.
 */

package com.ibm.xml.parser;

import org.w3c.dom.DOMException;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import java.util.Vector;

/**
 * The TXNodeList class is used to represent collections of Node 
 * objects which need to be accessed by name, ordinal, etc. 
 * For example, the Parent class uses TXNodeList to organize and access 
 * its Child Nodes.
 *
 * @version Revision: 38 1.9 src/com/ibm/xml/parser/TXNodeList.java, xml4jsrc, xml4j-jtcsv, xml4j_1_1_16 
 * @author TAMURA Kent &lt;kent@trl.ibm.co.jp&gt;
 * @see com.ibm.xml.parser.Parent
 */
class TXNodeList implements NodeList, java.io.Serializable {

            static final long serialVersionUID = 3392852660870983354L;
            Vector      nodes   =   new Vector();

    /**
     * Returns the number of Nodes in this TXNodeList.
     * <p>This method is defined by DOM.
     * @return      The number of Nodes in this TXNodeList.
     */
    public int getLength() {
        return this.nodes.size();
    }

    /**
     * Returns an enumeration of the Nodes of this TXNodeList.
     * @return      An enumeration of the Nodes of this list.
     * @see java.util.Enumeration
     * @see #getIterator
     */
    public java.util.Enumeration elements() {
        return this.nodes.elements();
    }

    /**
     * Returns the Node at the specified index.
     * <p>This method is defined by DOM.
     * @param      index   0-based index into this list.
     * @return     The Node at the specified index.
     * @see #indexOf
     */
    public Node item(int index) {
        return (Node)this.nodes.elementAt(index);
    }

    /**
     * Searches for the first occurence of the specified Node, testing 
     * for equality using the Vector <code>equals</code> method. 
     * @param   node   Node to search for in this list.
     * @return  The 0-based index of the first occurrence of a matching Node in this list, or -1 if no matching Node.
     * @see     java.lang.Object#equals(java.lang.Object)
     * @see     #item
     */
    public int indexOf(Node node) {
        return this.nodes.indexOf(node);
    }

    /**
     * Replaces the Node at the specified index with <var>newNode</var>. 
     * @param   index       0-based index of the Node to replace.
     * @param   newNode     The new Node to replace at the specified index.
     * @see #insert      
     * @see #remove      
     * @exception  org.w3c.dom.DOMException  Thrown if an invalid index is specified.
     */
    public void replace(int index, Node newNode) throws DOMException {
        if (index < 0 || this.nodes.size() <= index) {
            throw new TXDOMException(DOMException.INDEX_SIZE_ERR, "com.ibm.xml.parser.TXNodeList#replace(): Wrong index: "+index);
        }
        Child oldChild = (Child)item(index);
        Child prev = (Child)oldChild.getPreviousSibling();
        Child next = (Child)oldChild.getNextSibling();
        this.nodes.setElementAt(newNode, index);
        Child newChild = (Child)newNode;
        newChild.setPreviousSibling(prev);
        if (null != prev)  prev.setNextSibling(newChild);
        newChild.setNextSibling(next);
        if (null != next)  next.setPreviousSibling(newChild);
        oldChild.setPreviousSibling(null);
        oldChild.setNextSibling(null);
        oldChild.setParentNode(null);
    }

    /**
     * Inserts a new Node at the specified index.
     * @param   index       0-based index of where to insert <var>newNode</var>.
     * @param   newNode     The new Node to insert at the specified index.
     * @see #replace     
     * @see #remove
     * @see #addElement
     * @exception  org.w3c.dom.DOMException  Thrown if an invalid index is specified.
     */
    public void insert(int index, Node newNode) throws DOMException {
        if (index < 0 || index > this.nodes.size()) {
            throw new TXDOMException(DOMException.INDEX_SIZE_ERR, "com.ibm.xml.parser.TXNodeList#insert(): Wrong index: "+index);
        }
        Child prev = 0 < index ? (Child)this.nodes.elementAt(index-1) : null;
        Child next = this.nodes.size() > index ? (Child)this.nodes.elementAt(index) : null;
        this.nodes.insertElementAt(newNode, index);
        Child child = (Child)newNode;
        child.setPreviousSibling(prev);
        if (null != prev)  prev.setNextSibling(child);
        child.setNextSibling(next);
        if (null != next)  next.setPreviousSibling(child);
    }

    /**
     * Removes a Node at the specified index.
     * @param   index       0-based index of where to remove a Node.
     * @return              The Node removed at the specified index.
     * @see #replace     
     * @see #insert      
     * @exception  org.w3c.dom.DOMException  Thrown if an invalid index is specified.
     */
    public Node remove(int index) throws DOMException {
        if (index < 0 || this.nodes.size() <= index) {
            throw new TXDOMException(DOMException.INDEX_SIZE_ERR, "com.ibm.xml.parser.TXNodeList#remove(): Wrong index: "+index);
        }
        Child old = (Child)this.nodes.elementAt(index);
        Child prev = (Child)old.getPreviousSibling();
        Child next = (Child)old.getNextSibling();
        this.nodes.removeElementAt(index);
        old.setParentNode(null);
        old.setPreviousSibling(null);
        old.setNextSibling(null);
        if (null != prev)  prev.setNextSibling(next);
        if (null != next)  next.setPreviousSibling(prev);
        return old;
    }

    /**
     * Inserts a new Node at the end of this NodeList.
     * @param   newNode     The new Node to insert at the end.
     * @see #insert
     * @see #replace     
     * @see #remove
     */
    public void append(Node newNode) {
        insert(this.nodes.size(), newNode);
    }

    /**
     * Increases the capacity of this TXNodeList, if necessary, to ensure 
     * that it can hold at least the number of Nodes specified by 
     * the minimum capacity argument. 
     * @param   minCapacity   The desired minimum capacity.
     */
    public void ensureCapacity(int minCapacity) {
        this.nodes.ensureCapacity(minCapacity);
    }

    /**
     *
     */
    public boolean equals(Object obj) {
        return equals(obj, true);
    }

    /**
     *
     */
    public synchronized boolean equals(Object obj, boolean deep) {
        if (obj == null)  return false;
        if (!(obj instanceof NodeList))  return false;
        NodeList nl = (NodeList)obj;
        if (nl.getLength() != this.getLength())  return false;
        for (int i = 0;  i < nl.getLength();  i ++) {
            Node n = nl.item(i);
            if (!((Child)n).equals(this.item(i), deep))
                return false;
        }
        return true;
    }

    /**
     *
     */
    public int hashCode() {
        return 0; // hash value should not change over the life time of
                  // object. We do not anticpiate anyone using this method.
                  // 0 is logically correct as the hash value, just not
                  // efficient.
    }

    
    static class EmptyNodeListImpl implements NodeList {
        public Node item(int index) {
            throw new ArrayIndexOutOfBoundsException();
        }
        public int getLength() {
            return 0;
        }
    }
    static EmptyNodeListImpl emptyNodeList = new EmptyNodeListImpl();

    static class VectorNodeList implements NodeList {
        Vector data;

        VectorNodeList(Vector v) {
            this.data = v;
        }

        public Node item(int index) {
            return (Node)this.data.elementAt(index);
        }

        public int getLength() {
            return this.data.size();
        }
    }

    static class ArrayNodeList implements NodeList {
        Node[] data;

        ArrayNodeList(Node[] nodes) {
            this.data = nodes;
        }

        public Node item(int index) {
            return this.data[index];
        }

        public int getLength() {
            return this.data.length;
        }
    }
}

