/*
 * (C) Copyright IBM Corp. 1997-1998  All rights reserved.
 *
 * US Government Users Restricted Rights Use, duplication or
 * disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * IBM will not be liable for any damages suffered by you as a result
 * of using the Program. In no event will IBM be liable for any
 * special, indirect or consequential damages or lost profits even if
 * IBM has been advised of the possibility of their occurrence. IBM
 * will not be liable for any third party claims against you.
 */

package com.ibm.xml.parser;

import org.w3c.dom.Node;
import org.w3c.dom.Notation;

/**
 * The <CODE>TXNotation</CODE> class represents the definition of a notation within a DTD.
 * Notations identify by name the format of external binary entities, or the application to 
 * which a processing instruction is addressed.
 * <P><CODE>TXNotation</CODE> isn't an implement class of <CODE>org.w3c.dom.Notation</CODE>.
 *
 * @version Revision: 39 1.14 src/com/ibm/xml/parser/TXNotation.java, xml4jsrc, xml4j-jtcsv, xml4j_1_1_16 
 * @author TAMURA Kent &lt;kent@trl.ibm.co.jp&gt;
 * @see org.w3c.dom.Notation
 */
public class TXNotation extends Child {
    
                static final long serialVersionUID = -1546534879299424756L;
                String      name        =   null;
                ExternalID  externalID  =   null;

    /**
     * Constructor.
     * @param name          The name of the Notation.
     * @param externalID    The public or system identifier which defines the DTD's notation.
     */
    public TXNotation(String name, ExternalID externalID) {
        this.name       = name;
        this.externalID = externalID;
    }
    
    /**
     * Clone this Notation Node using the appropriate factory.  
     * <p>This method is defined by Child.
     * @return          Cloned Notation Node.
     * @see com.ibm.xml.parser.Child#clone
     */
    public synchronized Object clone() {
        checkFactory();
        TXNotation no = factory.createNotation(this.name, this.externalID);
        no.setFactory(getFactory());
        return no;
    }

    /**
     *
     * @param deep Ignored.
     */
    public synchronized boolean equals(Node arg, boolean deep) {
        if (arg == null)  return false;
        if (!(arg instanceof TXNotation))  return false;
        TXNotation no = (TXNotation)arg;
        if (!this.getName().equals(no.getName()))  return false;
        return this.getExternalID().equals(no.getExternalID());
    }
    
    /**
     * Returns that this object is a Notation Node.
     * <!--p>This method is defined by DOM.-->
     *
     * @return          Node indicator.
     */
    public short getNodeType() { 
        return Node.NOTATION_NODE;
    }

    /**
     * Returns the name of the Notation.
     * <!--p>This method is defined by DOM.-->
     */
    public String getNodeName() {
        return this.name;
    }

    /**
     * Returns the name of the Notation.
     * @return          The name of the Notation, or <var>null</var> if no name.
     * @see #setName         
     * @deprecated Use getNodeName().
     */
    public String getName() {
        return this.name;
    }

    /**
     * Sets the name of the Notation.
     * @param name      The name of the Notation.
     * @see #getName
     * @deprecated
     */
    public void setName(String name) {
        this.name = name;
    }
    
    /**
     * Returns the system identifier of the Notation.  
     * A system identifier is a URI, which may be used to retrieve an external entity's content. 
     * <!--p>This method is defined by DOM.-->
     * @return          The system identifier, or <var>null</var> if the identifier is not defined.
     * @see com.ibm.xml.parser.ExternalID#getSystemLiteral
     */
    public String getSystemId() {
        return this.externalID.getSystemLiteral();
    }
    
    /**
     * Returns the public identifier of the Notation.  This value is only valid if the
     * identifier is defined as <var>public</var> (as opposed to <var>system</var>).
     * Public identifiers may be used to try to generate an alternative URI in order to
     * retrieve the an external entities content.  If retrieval fails using
     * the public identifier, an attempt must be made to retrieve content using the system
     * identifier.
     * <!--p>This method is defined by DOM.-->
     * @return          The public identifier, or <var>null</var> if the identifier is not defined.
     * @see com.ibm.xml.parser.ExternalID
     */
    public String getPublicId() {
        return this.externalID.getPubidLiteral();
    }

    /**
     * Returns the external ID associated with this Notation Node.  An external ID contains system,
     * and optionally, public identifiers.
     * @return          The external ID associated with this Notation Node.
     * @see com.ibm.xml.parser.ExternalID
     */
    public ExternalID getExternalID() {
        return this.externalID;
    }
    
    /**
     * Implements the accept operation of the visitor design pattern when the start of
     * a TXNotation Node is recognized when traversing the document object tree. 
     * @param   visitor The implemention of the Visitor operation (toXMLString, digest, ...)
     * @exception Exception Thrown if this Node can not be visitted, or traversal modification is requested.
     * @see com.ibm.xml.parser.Visitor
     * @see com.ibm.xml.parser.TreeTraversal
     * @see com.ibm.xml.parser.NonRecursivePreorderTreeTraversal
     * @see com.ibm.xml.parser.TreeTraversalException
     */
    public void acceptPre(Visitor visitor) throws Exception {
        visitor.visitNotationPre(this);
    }

    /**
     * Implements the accept operation of the visitor design pattern when the end of
     * a TXNotation Node is recognized when traversing the document object tree. 
     * @param   visitor The implemention of the Visitor operation (toXMLString, digest, ...)
     * @exception Exception Thrown if this Node can not be visitted, or traversal modification is requested.
     * @see com.ibm.xml.parser.Visitor
     * @see com.ibm.xml.parser.TreeTraversal
     * @see com.ibm.xml.parser.NonRecursivePreorderTreeTraversal
     * @see com.ibm.xml.parser.TreeTraversalException
     */
    public void acceptPost(Visitor visitor) throws Exception {
        visitor.visitNotationPost(this);
    }

    /**
     * Wrapper for DOM.
     */
    protected Notation getNotationImpl() {
        return new NotationImpl(this.name, this.externalID);
    }

    static class NotationImpl extends TXNotation implements Notation {
        NotationImpl(String name, ExternalID exid) {
            super(name, exid);
        }
        public Node getParentNode() {
            return null;
        }
        public Node cloneNode(boolean deep) {
            checkFactory();
            NotationImpl no = (NotationImpl)factory.createNotation(this.name, this.externalID).getNotationImpl();
            no.setFactory(getFactory());
            return no;
        }
    }
}
