/*
 * (C) Copyright IBM Corp. 1997-1998  All rights reserved.
 *
 * US Government Users Restricted Rights Use, duplication or
 * disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * IBM will not be liable for any damages suffered by you as a result
 * of using the Program. In no event will IBM be liable for any
 * special, indirect or consequential damages or lost profits even if
 * IBM has been advised of the possibility of their occurrence. IBM
 * will not be liable for any third party claims against you.
 */

package com.ibm.xml.parser;

import org.w3c.dom.Node;
import org.w3c.dom.ProcessingInstruction;

/**
 * The TXPI class implements the PI interface as defined by the Document Object Model (DOM),
 * and implements the Namespace interface as defined by the W3C.
 * <p>A PI Node is a "processing instruction" for use by XML4J or an application. 
 * The content of the PI Node is the entire content between the delimiters of the processing 
 * instruction.
 *
 * @version Revision: 40 1.15 src/com/ibm/xml/parser/TXPI.java, xml4jsrc, xml4j-jtcsv, xml4j_1_1_16 
 * @author TAMURA Kent &lt;kent@trl.ibm.co.jp&gt;
 * @see org.w3c.dom.ProcessingInstruction
 * @see com.ibm.xml.parser.Child
 */
public class TXPI extends Child implements ProcessingInstruction {

            static final long serialVersionUID = 5857404877602965574L;
            String  name            = null;
            String  data            = null;

    /**
     * Constructor.
     * @param name      The first token following the markup.
     * @param data      From the first non white space character after PI target (<var>name</var>)
     *                  to the character immediately preceding the <code>?&gt;</code>.
     */
    public TXPI(String name, String data) {
        this.name = name;
        if (data.length() > 0) {
            int start = 0;
            while (start < data.length() && XMLChar.isSpace(data.charAt(start)))
                start ++;
            this.data = data.substring(start);
        } else
            this.data = data;
    }

    /**
     * Clone this PI Node using the appropriate factory.
     * <p>This method is defined by Child.
     * @return          Cloned PI Node.
     * @see com.ibm.xml.parser.Child#clone
     */
    public synchronized Object clone() {
        checkFactory();
        TXPI pi = (TXPI)factory.createProcessingInstruction(this.name, this.data);
        pi.setFactory(getFactory());
        return pi;
    }

    /**
     *
     * @param deep Ignored.
     */
    public synchronized boolean equals(Node arg, boolean deep) {
        if (arg == null)  return false;
        if (!(arg instanceof ProcessingInstruction))
            return false;
        ProcessingInstruction pi = (ProcessingInstruction)arg;
        return pi.getTarget().equals(this.getTarget()) && pi.getData().equals(this.getData());
    }

    /**
     * Returns that this object is a PI Node.
     * <p>This method is defined by DOM.
     * @return          PI Node indicator.
     */
    public short getNodeType() {
        return Node.PROCESSING_INSTRUCTION_NODE;
    }

    /**
     * <p>This method is defined by DOM.
     *
     */
    public String getNodeName() {
        return getTarget();
    }

    /**
     * Returns the qualified name of this PI.  The name is the first token following the markup.
     * If this PI's name has a namespace prefix, the prefix will still be attached.
     * @return          The PI name, or <var>null</var> if no name.
     * @deprecated Use getNodeName()
     */
    public String getName() {
        return this.name;
    }

    /**
     * <p>This method is defined by DOM.
     *
     * @see #getName
     */
    public String getTarget() {
        return this.name;
    }
    /**
     * <p>This method is defined by DOM.
     */
    public void setTarget(String arg) {
        this.name = arg;
        clearDigest();
    }

    /**
     * Returns the data of the PI.  The PI data is from the character immediately after the 
     * PI name to the character immediately preceding the <code>?&gt;</code>.
     * <p>This method is defined by DOM.
     * @return          The PI data.
     * @see #setData
     */
    public String getData() {
        return this.data;
    }

    /**
     * Returns the data of the PI.  The PI data is from the character immediately after the 
     * PI name to the character immediately preceding the <code>?&gt;</code>.
     * <p>This method is defined by DOM.
     */
    public String getNodeValue() {
        return getData();
    }

    /**
     * Sets the data of the PI.  The PI data is from the character immediately after the 
     * PI name to the character immediately preceding the <code>?&gt;</code>.
     * <p>This method is defined by DOM.
     * @param data      The PI data.
     * @see #getData
     */
    public void setData(String data) {
        this.data = data;
        clearDigest();
    }

    /**
     * Sets the data of the PI.  The PI data is from the character immediately after the 
     * PI name to the character immediately preceding the <code>?&gt;</code>.
     * <p>This method is defined by DOM.
     * @param data      The PI data.
     * @see #getNodeValue
     */
    public void setNodeValue(String data) {
        this.setData(data);
    }

    /**
     * Return all text associated with this Node without considering entities.                                           
     * <p>This method is defined by Child.
     * @return          Always returns <var>""</var>.
     * @see com.ibm.xml.parser.Child#toXMLString
     */
    public String getText() {
        return "";
    }

    /**
     * Implements the accept operation of the visitor design pattern when the start of
     * a TXPI Node is recognized when traversing the document object tree. 
     * @param   visitor The implemention of the Visitor operation (toXMLString, digest, ...)
     * @exception Exception Thrown if this Node can not be visitted, or traversal modification is requested.
     * @see com.ibm.xml.parser.Visitor
     * @see com.ibm.xml.parser.TreeTraversal
     * @see com.ibm.xml.parser.NonRecursivePreorderTreeTraversal
     * @see com.ibm.xml.parser.TreeTraversalException
     */
    public void acceptPre(Visitor visitor) throws Exception {
        visitor.visitPIPre(this);
    }

    /**
     * Implements the accept operation of the visitor design pattern when the end of
     * a TXPI Node is recognized when traversing the document object tree. 
     * @param   visitor The implemention of the Visitor operation (toXMLString, digest, ...)
     * @exception Exception Thrown if this Node can not be visitted, or traversal modification is requested.
     * @see com.ibm.xml.parser.Visitor
     * @see com.ibm.xml.parser.TreeTraversal
     * @see com.ibm.xml.parser.NonRecursivePreorderTreeTraversal
     * @see com.ibm.xml.parser.TreeTraversalException
     */
    public void acceptPost(Visitor visitor) throws Exception {
        visitor.visitPIPost(this);
    }
}
