/*
 * (C) Copyright IBM Corp. 1997-1998  All rights reserved.
 *
 * US Government Users Restricted Rights Use, duplication or
 * disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * IBM will not be liable for any damages suffered by you as a result
 * of using the Program. In no event will IBM be liable for any
 * special, indirect or consequential damages or lost profits even if
 * IBM has been advised of the possibility of their occurrence. IBM
 * will not be liable for any third party claims against you.
 */

package com.ibm.xml.parser;

import org.w3c.dom.DOMException;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.Text;

/**
 * The TXText class implements the Text interface as defined by the Document Object Model (DOM).
 * <p>The Text object contains the non-markup content of an Element. If there is no markup
 * inside an Element's content, the text will be contained in a single Text object that is the child
 * of the Element. Any markup will parse into Child Elements that are siblings of the Text Nodes
 * on either side of it, and whose content will be represented as Text Node children of the
 * markup Element. 
 *
 * @version Revision: 41 1.12 src/com/ibm/xml/parser/TXText.java, xml4jsrc, xml4j-jtcsv, xml4j_1_1_16 
 * @author TAMURA Kent &lt;kent@trl.ibm.co.jp&gt;
 * @see org.w3c.dom.Text 
 * @see com.ibm.xml.parser.TXCDATASection
 */
public class TXText extends TXCharacterData implements Text {
    
           static final long serialVersionUID = -8266596319046593304L;
           boolean isIgnorableWhitespace    = false;

    /**
     * Constructor.
     * @param data      The actual content of the Text Node.
     */
    public TXText(String data) {
        this.data = data;
    }

    /**
     * Clone this Text Node using the appropriate factory.  
     * <p>This method is defined by Child.
     * @return          Cloned Text Node.
     * @see com.ibm.xml.parser.Child#clone
     */
    public synchronized Object clone() {
        checkFactory();
        TXText t = (TXText)factory.createTextNode(this.data);
        t.setFactory(getFactory());
        t.setIsIgnorableWhitespace(getIsIgnorableWhitespace());
        return t;
    }

    /**
     *
     * @param deep ignored.
     */
    public boolean equals(Node arg, boolean deep) {
        if (arg == null)  return false;
        if (!(arg instanceof Text))  return false;
        return ((Text)arg).getData().equals(this.getData());
    }

    /**
     * Returns that this object is a Text Node. 
     * <p>This method is defined by DOM.
     * @return          Text Node indicator.
     */
    public short getNodeType() {
        return Node.TEXT_NODE;
    }

    /**
     *
     */
    public String getNodeName() {
        return Child.NAME_TEXT;
    }

    /**
     * Returns all text associated with this Node without considering entities.                                           
     * <p>This method is defined by Child.
     * @return          Text associated with this object, or <var>""</var> if no Text.
     * @see com.ibm.xml.parser.Child#toXMLString
     * @see #getData
     */
    public String getText() {                   
        return getData();
    }

    /**
     * Returns, at the Text level, whether space is to be preserved.  
     * This value is used, for example, to determine if space is to be preserved
     * in Text Nodes during printWithFormat() operations.
     * @return                  <code>=true</code> space is to be preserved; 
     *                          <code>=false</code> space is to be ignored.
     * @see #setIsIgnorableWhitespace
     * @see com.ibm.xml.parser.Parser#setPreserveSpace
     * @see com.ibm.xml.parser.TXElement#setPreserveSpace
     * @see com.ibm.xml.parser.TXDocument#printWithFormat
     */
    public boolean getIsIgnorableWhitespace() {
        return this.isIgnorableWhitespace;
    }

    /**
     * Sets, at the Text level, whether space is to be preserved.  
     * This value is used, for example, to determine if space is to be preserved
     * in Text Nodes during printWithFormat() operations.
     * <p>By default, this Text Node is not ignorable whitespace.  The XML4J parser
     * may, depending on the value if its <var>isPreserveSpace</var>, override this default
     * setting if no significant text is detected for this Text Node.
     * @param  isIgnorableWhitespace    <code>=true</code> space is to be preserved; 
     *                                  <code>=false</code> space is to be ignored.
     * @see #getIsIgnorableWhitespace
     * @see com.ibm.xml.parser.Parser#setPreserveSpace
     * @see com.ibm.xml.parser.TXElement#setPreserveSpace
     * @see com.ibm.xml.parser.TXDocument#printWithFormat
     */
    public void setIsIgnorableWhitespace(boolean isIgnorableWhitespace) {
        this.isIgnorableWhitespace = isIgnorableWhitespace;
    }

    /**
     * Returns the XML language ID (the value of the <code>xml:lang</code> attribute) in 
     * use for this Text Node.  If this Text Node has no explicitly defined language attribute,
     * Parent Nodes will be recursively checked.
     * @return          The XML language ID, or <var>null</var> if all Parents have no language ID.
     */
    public String getLanguage() {
        if (null == parent)  return null;
        if (parent instanceof TXElement)
            return ((TXElement)parent).getLanguage();
        if (parent instanceof GeneralReference)
            return ((GeneralReference)parent).getLanguage();
        return null;
    }
    
    /**
     * Insert the specified Element as a sibling of this Text Node. The result
     * of this operation may be the creation of up to 2 new Text Nodes: the character
     * data specified by the offset and count will form one Text Node that will
     * become the Child of the inserted Element, and the remainder of the character
     * data (after the offset and count) will form another Text Node, becoming a sibling
     * of this Text Node.  

     * @deprecated This method will be removed in next release.
     * @param element   The Element to insert in the tree; this instance must be an instance of TXElement.
     * @param offset    0-based offset into existing character data in this Text Node.
     * @param count     Number of characters to copy to Child Text Node of Element.
     * @exception IllegalArgumentException Thrown if <var>element</var> is not a TXElement.
     * @exception RuntimeException Thrown if this Text Node has no Parent.
     */
    public synchronized void splice(Element element, int offset, int count) throws IllegalArgumentException, RuntimeException {
        if (!(element instanceof TXElement))
            throw new IllegalArgumentException("com.ibm.xml.parser.TXText#splice(): An instance of TXElement is required.");
        String second = this.data.substring(offset, count);
        String third = this.data.substring(offset+count);
        this.data = this.data.substring(0, offset);
        clearDigest();
        Parent parent = (Parent)getParentNode();
        if (null == parent)
            throw new RuntimeException("com.ibm.xml.parser.TXText#splice(): This Text Node has no parent and can not be spliced.");
        checkFactory();
        element.insertBefore(factory.createTextNode(second), null);
        parent.insertAfter(element, this);
        if (0 < third.length())
            parent.insertAfter(factory.createTextNode(third), this);
    }

    /**
     * Split this text node into two text nodes at the specified offset,
     * keeping both in the tree as siblings.
     * <p>This method is defined by DOM.
     *
     */
    public Text splitText(int offset) throws DOMException {
        if (offset < 0 || offset > this.data.length())
            throw new TXDOMException(DOMException.INDEX_SIZE_ERR, "Out of bounds: "+offset
                                     +", the length of data is "+this.data.length());
        checkFactory();
        Text text = this.factory.createTextNode(this.data.substring(offset));
        setData(substringData(0, offset));
        Parent parent = (Parent)getParentNode();
        if (null == parent)
            throw new RuntimeException("com.ibm.xml.parser.TXText#splitText(): This Text Node has no parent and can not be split.");
        parent.insertAfter(text, this);
        return text;
    }

    /**
     * Trim all leading and trailing whitespace in the specified String.  All strings of white space are 
     * replaced by a single space character (#x20).
     * @param   string  String to be trimmed.
     * @return          The trimmed string.
    */
    public static String trim(String string) {
        return trim(string, true, true);
    }
    
    /**
     * Conditionally trim all leading and trailing whitespace in the specified String.  All strings of white space are 
     * replaced by a single space character (#x20).
     * @param   string      String to be trimmed.
     * @param   trimHead    Trim leading whitespace?
     * @param   trimTail    Trim trailing whitespace?
     * @return              The trimmed string.
    */
    public static String trim(String string, boolean trimHead, boolean trimTail) {
        char[] buf = string.toCharArray();
        int len = buf.length;
        boolean edit = false;
        int s;
        for (s = 0;  s < len;  s ++) {
            if (com.ibm.xml.parser.XMLChar.isSpace(buf[s]))  break;
        }
        /* replace S to ' '. and ' '+ -> single ' '. */
        int d = s;
        boolean pres = false;
        for ( ;  s < len;  s ++) {
            char c = buf[s];
            if (com.ibm.xml.parser.XMLChar.isSpace(c)) {
                if (!pres) {
                    if (' ' != c)  edit = true;
                    buf[d++] = ' ';
                } else
                    edit = true;
                pres = true;
            } else {
                buf[d++] = c;
                pres = false;
            }
        }
        if (trimTail && 1 <= d && ' ' == buf[d-1]) {
            edit = true;
            d --;
        }
        int start = 0;
        if (trimHead && 0 < d && ' ' == buf[0]) {
            edit = true;
            start ++;
        }
        return edit ? new String(buf, start, d-start) : string;
    }

    /**
     * Translate escape sequences in the specified string into printable characters.  
     * For example, <var>\r</var> is translated into <var>\\r</var>.
     * @param   string      String to translate.
     * @return              The translated string.
     */
    public static String makePrintable(String string) {
        StringBuffer sb = new StringBuffer(string.length()*2);
        for (int i = 0;  i < string.length();  i ++) {
            int ch = string.charAt(i);
            if ('\r' == ch) {
                sb.append("\\r");
            } else if ('\n' == ch) {
                sb.append("\\n");
            } else if ('\t' == ch) {
                sb.append("\\t");
            } else if ('\b' == ch) {
                sb.append("\\b");
            } else if ('\f' == ch) {
                sb.append("\\f");
            } else if ('"' == ch) {
                sb.append("\\\"");
            } else if ('\\' == ch) {
                sb.append("\\\\");
            } else {
                sb.append((char)ch);
            }
        }
        return sb.toString();
    }
    
    /**
     * Implements the accept operation of the visitor design pattern when the start of
     * a TXText Node is recognized when traversing the document object tree. 
     * @param   visitor The implemention of the Visitor operation (toXMLString, digest, ...)
     * @exception Exception Thrown if this Node can not be visitted, or traversal modification is requested.
     * @see com.ibm.xml.parser.Visitor
     * @see com.ibm.xml.parser.TreeTraversal
     * @see com.ibm.xml.parser.NonRecursivePreorderTreeTraversal
     * @see com.ibm.xml.parser.TreeTraversalException
     */
    public void acceptPre(Visitor visitor) throws Exception {
        visitor.visitTextPre(this);
    }

    /**
     * Implements the accept operation of the visitor design pattern when the end of
     * a TXText Node is recognized when traversing the document object tree. 
     * @param   visitor The implemention of the Visitor operation (toXMLString, digest, ...)
     * @exception Exception Thrown if this Node can not be visitted, or traversal modification is requested.
     * @see com.ibm.xml.parser.Visitor
     * @see com.ibm.xml.parser.TreeTraversal
     * @see com.ibm.xml.parser.NonRecursivePreorderTreeTraversal
     * @see com.ibm.xml.parser.TreeTraversalException
     */
    public void acceptPost(Visitor visitor) throws Exception {
        visitor.visitTextPost(this);
    }

    /**
     * @deprecated Use o.getNodeType() == Node.TEXT_NODE
     */
    static boolean isText(Object obj) {
        return !(obj instanceof TXCDATASection) && obj instanceof TXText;
    }
}
