/*
 * (C) Copyright IBM Corp. 1998  All rights reserved.
 *
 * US Government Users Restricted Rights Use, duplication or
 * disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * IBM will not be liable for any damages suffered by you as a result
 * of using the Program. In no event will IBM be liable for any
 * special, indirect or consequential damages or lost profits even if
 * IBM has been advised of the possibility of their occurrence. IBM
 * will not be liable for any third party claims against you.
 */

package com.ibm.xml.parser;

import java.io.Writer;
import java.util.Enumeration;
import org.w3c.dom.Node;
import org.w3c.dom.CDATASection;

/**
 * ToXMLStringVisitor implements the Visitor interface in the visitor design pattern for the
 * purpose of a toString operation on the various DOM- and XML4J-defined Nodes as the document
 * object tree is traversed.  The returned string will be in XML format.
 * <p>The following sample code uses the ToXMLStringVisitor on a hierarchy of nodes:
 * <pre>
 *
 * Visitor visitor = new ToXMLStringVisitor(stringWriter, encoding);
 * new NonRecursivePreorderTreeTraversal(visitor).traverse(this);
 *
 * </pre>
 *
 * @version Revision: %M% %I% %W% %Q%
 * @author ROWE Tom &lt;trowe@us.ibm.com&gt;
 * @see com.ibm.xml.parser.Visitor
 * @see com.ibm.xml.parser.NOOPVisitor
 * @see com.ibm.xml.parser.TreeTraversal
 * @see com.ibm.xml.parser.NonRecursivePreorderTreeTraversal
 */
public class ToXMLStringVisitor extends NOOPVisitor implements Visitor {

    protected  Writer writer      =   null;
    protected  String encoding    =   null;
    protected boolean isPrintNonSpecifiedAttributes = true;

    /**
     * Constructor for customized encoding.
     * @param writer        The character output stream to use.
     * @param encoding      Java character encoding in use by <VAR>writer</VAR>.
     */
    public ToXMLStringVisitor(Writer writer, String encoding) {
        this.writer   =   writer;
        this.encoding =   encoding;
    }

    /**
     * Constructor for default encoding.
     * @param writer  The character output stream to use.
     */
    public ToXMLStringVisitor(Writer writer) {
        this(writer, null);
    }

    /**
     * Sets whether this prints non-specified default attributes appended by DTD.
     * <P>By default, non-specified attributes are printed.
     * @see com.ibm.xml.parser.TXAttribute#getSpecified
     */
    public void setPrintNonSpecifiedAttributes(boolean print) {
        this.isPrintNonSpecifiedAttributes = print;
    }

    /**
     * Returns whether this prints non-specified default attributes appended by DTD.
     */
    public boolean getPrintNonSpecifiedAttributes() {
        return this.isPrintNonSpecifiedAttributes;
    }

    /**
     * Creates a string representation of the specified <var>document</var> Node in XML format.
     * @param   document    Node toString in XML format.
     * @exception Exception   Thrown if this Node can not be visitted because of an invalid character output stream.
     * @see com.ibm.xml.parser.TXDocument
     */
    public void visitDocumentPre(TXDocument document) throws Exception {
        if (null != document.getVersion()) {
            this.writer.write("<?xml version=\"" + document.getVersion() + "\"");
            if (null != document.getEncoding()) {
                this.writer.write(" encoding=\"" + document.getEncoding() + "\"");
            }
            if (null != document.getStandalone()) {
                this.writer.write(" standalone=\"" + document.getStandalone() + "\"");
            }
            this.writer.write("?>");
        }
    }

    /**
     * Flush the writer.
     * @param   document    CURRENTLY NOT IMPLEMENTED.
     * @exception Exception   Thrown if this Node can not be visitted because of an invalid character output stream.
     * @see com.ibm.xml.parser.TXDocument
     */
    public void visitDocumentPost(TXDocument document) throws Exception {
        this.writer.flush();
    }

    /**
     * Creates a string representation of the start of the specified <var>element</var> Node
     * and its associated attributes in XML format.
     * @param   element     Node toString in XML format.
     * @exception Exception   Thrown if this Node can not be visitted because of an invalid character output stream.
     * @see com.ibm.xml.parser.TXElement
     */
    public void visitElementPre(TXElement element) throws Exception {
        this.writer.write("<" + element.getTagName());
        TXAttribute[] attributes = element.getAttributeArray();
        for (int i = 0;  i < attributes.length;  i ++) {
            try {
                TXAttribute attr = attributes[i];
                if (this.isPrintNonSpecifiedAttributes || attr.getSpecified())
                    visitAttributePre(attr);
            } catch (ToNextSiblingTraversalException e) {
                                                // Ignore
            }
        }
        if (element.hasChildNodes()) {
            this.writer.write(">");
        } else {
            this.writer.write("/>");
        }
    }

    /**
     * Creates a string representation of the end of the specified <var>element</var> Node
     * in XML format.
     * @param   element     Node toString in XML format.
     * @exception Exception   Thrown if this Node can not be visitted because of an invalid character output stream.
     * @see com.ibm.xml.parser.TXElement
     */
    public void visitElementPost(TXElement element) throws Exception {
        if (element.hasChildNodes()) {
            this.writer.write("</" + element.getTagName() + ">");
        }
    }

    /**
     * Creates a string representation of the specified <var>attribute</var> Node in XML
     * format.
     * <p>Note that TXAttribute Nodes are not parsed into the document object hierarchy by the
     * XML4J parser; attributes exist as part of a TXElement Node.
     * @param   attribute   Node toString in XML format.
     * @exception Exception   Thrown if this Node can not be visitted because of an invalid character output stream.
     * @exception com.ibm.xml.parser.ToNextSiblingTraversalException Always thrown.
     * @see com.ibm.xml.parser.TXAttribute
     */
    public void visitAttributePre(TXAttribute attribute) throws Exception {
        this.writer.write(" ");
        this.writer.write(attribute.getName());
        this.writer.write("=\"");
        for (Node child = attribute.getFirstChild();  child != null;
             child = child.getNextSibling()) {
            int type = child.getNodeType();
            if (type == Node.TEXT_NODE) {
                this.writer.write(Util.backReference(child.getNodeValue(), "<&\"", this.encoding));
            } else if (type == Node.ENTITY_REFERENCE_NODE) {
                this.writer.write("&");
                this.writer.write(child.getNodeName());
                this.writer.write(";");
            } else {
                //???
            }
        }
        this.writer.write("\"");
    
        throw new ToNextSiblingTraversalException();
    }

    /**
     * Creates a string representation of the specified <var>pi</var> Node in XML format.
     * @param   pi          Node toString in XML format.
     * @exception Exception   Thrown if this Node can not be visitted because of an invalid character output stream.
     * @see com.ibm.xml.parser.TXPI
     */
    public void visitPIPre(TXPI pi) throws Exception {
        this.writer.write("<?" + pi.getNodeName());
        String data = pi.getData();
        if (pi.getData().length() > 0 && !XMLChar.isSpace(data.charAt(0)))
           this.writer.write(" ");
        this.writer.write(pi.getData() + "?>");
    }

    /**
     * Creates a string representation of the specified <var>comment</var> Node in XML format.
     * @param   comment     Node toString in XML format.
     * @exception Exception   Thrown if this Node can not be visitted because of an invalid character output stream.
     * @see com.ibm.xml.parser.TXComment
     */
    public void visitCommentPre(TXComment comment) throws Exception {
        this.writer.write("<!--" + comment.getData() + "-->");
    }

    /**
     * Creates a string representation of the specified <var>text</var> Node in XML format.
     * CDATASections are respected.
     * @param   text        Node toString in XML format.
     * @exception Exception   Thrown if this Node can not be visitted because of an invalid character output stream.
     * @see com.ibm.xml.parser.TXText
     */
    public void visitTextPre(TXText text) throws Exception {
        if (text instanceof CDATASection) {
            this.writer.write("<![CDATA[" + text.getData() + "]]>");
        } else {
            this.writer.write(Util.backReference(text.getData(), this.encoding));
        }
    }

    /**
     * Creates a string representation of the specified <var>dtd</var> Node, and optionally
     * its internal DTD subset, in XML format.  The internal DTD subset will be
     * included based on the value of <code>isPrintInternalDTD()</code>.
     * @param   dtd         Node toString in XML format.
     * @exception Exception   Thrown if this Node can not be visitted because of an invalid character output stream.
     * @exception com.ibm.xml.parser.ToNextSiblingTraversalException Thrown if isPrintInternalDTD() is false.
     * @see com.ibm.xml.parser.DTD#isPrintInternalDTD
     * @see com.ibm.xml.parser.DTD
     */
    public void visitDTDPre(DTD dtd) throws Exception {
        this.writer.write("<!DOCTYPE " + dtd.getName());
        if (null != dtd.getExternalID()) {
            this.writer.write(" "+dtd.getExternalID());
        }
        if (dtd.isPrintInternalDTD() && 0 < dtd.getInternalSize()) {
            this.writer.write(" [");
        } else
            throw new ToNextSiblingTraversalException();
    }

    /**
     * Creates a string representation of the specified <var>dtd</var> Node, and optionally
     * its internal DTD subset, in XML format.  The internal DTD subset will be
     * included based on the value of <code>isPrintInternalDTD()</code>.
     * @param   dtd         Node toString in XML format.
     * @exception Exception   Thrown if this Node can not be visitted because of an invalid character output stream.
     * @see com.ibm.xml.parser.DTD#isPrintInternalDTD
     * @see com.ibm.xml.parser.DTD
     */
    public void visitDTDPost(DTD dtd) throws Exception {
        if (dtd.isPrintInternalDTD() && 0 < dtd.getInternalSize()) {
            this.writer.write("]");
        }
        this.writer.write(">");
    }

    /**
     * Creates a string representation of the specified <var>elementDecl</var> Node in XML format.
     * @param   elementDecl Node toString in XML format.
     * @exception Exception   Thrown if this Node can not be visitted because of an invalid character output stream.
     * @see com.ibm.xml.parser.ElementDecl
     */
    public void visitElementDeclPre(ElementDecl elementDecl) throws Exception {
        this.writer.write("<!ELEMENT " + elementDecl.getName() + " " +
                                elementDecl.getXML4JContentModel() + ">");
    }

    /**
     * Creates a string representation of the specified <var>attlist</var> Node and any
     * associated AttDefs in XML format.
     * @param   attlist     Node toString in XML format.
     * @exception Exception   Thrown if this Node can not be visitted because of an invalid character output stream.
     * @see com.ibm.xml.parser.Attlist
     */
    public void visitAttlistPre(Attlist attlist) throws Exception {
        this.writer.write("<!ATTLIST " + attlist.getName());
        int asize = attlist.size();
        if (0 >= asize) {
            this.writer.write(" >");
        } else if (1 == asize) {
            visitAttDefPre(attlist.elementAt(0));
            this.writer.write((char)'>');
        } else {
            this.writer.write("\n");
            for (int i = 0;  i < asize;  i ++) {    // Can't use TreeTraversal
                Util.printSpace(this.writer, 4);
                visitAttDefPre(attlist.elementAt(i));
                if (i == asize-1) {
                    this.writer.write(">");
                } else {
                    this.writer.write("\n");
                }
            }
        }
    }

    /**
     * Creates a string representation of the specified <var>attDef</var> Node in XML format.
     * @param   attDef      Node toString in XML format.
     * @exception Exception   Thrown if this Node can not be visitted because of an invalid character output stream.
     * @see com.ibm.xml.parser.AttDef
     */
    public void visitAttDefPre(AttDef attDef) throws Exception {
        this.writer.write((char)' ' + attDef.getName() + (char)' ');
        int declType = attDef.getDeclaredType();
        if (AttDef.NAME_TOKEN_GROUP != declType) {
            this.writer.write(AttDef.S_TYPESTR[attDef.getDeclaredType()] + (char)' ');
        }
        if (AttDef.NOTATION == declType
            || AttDef.NAME_TOKEN_GROUP == declType) {
            this.writer.write((char)'(' + attDef.elementAt(0));
            for (int i = 1;  i < attDef.size();  i ++) {
                this.writer.write((char)'|' + attDef.elementAt(i));
            }
            this.writer.write(") ");
        }

        int defType = attDef.getDefaultType();
        if (AttDef.REQUIRED == defType) {
            this.writer.write("#REQUIRED");
        } else if (AttDef.IMPLIED == defType) {
            this.writer.write("#IMPLIED");
        } else if (AttDef.FIXED == defType) {
            this.writer.write("#FIXED \"");
            this.writer.write(null == attDef.getDefaultStringValue()
                     ? "(null)" : Util.backReferenceForEntity(attDef.getDefaultStringValue(),
                                                              this.encoding));
            this.writer.write((char)'"');
        } else if (AttDef.NOFIXED == defType) {
            this.writer.write((char)'"');
            this.writer.write(null == attDef.getDefaultStringValue()
                     ? "(null)" : Util.backReferenceForEntity(attDef.getDefaultStringValue(),
                                                              this.encoding));
            this.writer.write((char)'"');
        }
    }

    /**
     * Creates a string representation of the specified <var>entityDecl</var> Node in XML format.
     * @param   entity      Node toString in XML format.
     * @exception Exception   Thrown if this Node can not be visitted because of an invalid character output stream.
     * @see com.ibm.xml.parser.EntityDecl
     */
    public void visitEntityDeclPre(EntityDecl entityDecl) throws Exception {
        this.writer.write("<!ENTITY ");
        if (entityDecl.isParameter()) {
            this.writer.write("% ");
        }
        this.writer.write(entityDecl.getNodeName()+" ");
        if (null != entityDecl.getValue()) {
            this.writer.write("\"" + Util.backReferenceForEntity(entityDecl.getValue(), this.encoding)
                              +"\">");
        } else {
            this.writer.write(entityDecl.getExternalID().toString());
            if (null != entityDecl.getNotationName()) {
                this.writer.write(" NDATA " + entityDecl.getNotationName());
            }
            this.writer.write(">");
        }
        throw new ToNextSiblingTraversalException();
    }

    /**
     * Creates a string representation of the specified <var>notation</var> Node in XML format.
     * @param   notation    Node toString in XML format.
     * @exception Exception   Thrown if this Node can not be visitted because of an invalid character output stream.
     * @see com.ibm.xml.parser.TXNotation
     */
    public void visitNotationPre(TXNotation notation) throws Exception {
        this.writer.write("<!NOTATION " + notation.getNodeName()
                          + " " + notation.getExternalID() + ">");
    }

    /**
     * Creates a string representation of the specified <var>generalReference</var> Node in XML format.
     * @param   generalReference    Node toString in XML format.
     * @exception Exception   Thrown if this Node can not be visitted because of an invalid character output stream.
     * @exception com.ibm.xml.parser.ToNextSiblingTraversalException Alway thrown.
     * @see com.ibm.xml.parser.GeneralReference
     */
    public void visitGeneralReferencePre(GeneralReference generalReference) throws Exception {
        this.writer.write((char)'&' + generalReference.getName() + (char)';');
        throw new ToNextSiblingTraversalException();
    }

    /**
     * Creates a string representation of the specified <var>pseudoNode</var> Node in XML format.
     * @param   pseudoNode  Node toString in XML format.
     * @exception Exception   Thrown if this Node can not be visitted because of an invalid character output stream.
     * @see com.ibm.xml.parser.PseudoNode
     */
    public void visitPseudoNodePre(PseudoNode pseudoNode) throws Exception {
        this.writer.write(pseudoNode.getData());
    }

}  //ToXMLStringVisitor
