/*
 * (C) Copyright IBM Corp. 1998  All rights reserved.
 *
 * US Government Users Restricted Rights Use, duplication or
 * disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * IBM will not be liable for any damages suffered by you as a result
 * of using the Program. In no event will IBM be liable for any
 * special, indirect or consequential damages or lost profits even if
 * IBM has been advised of the possibility of their occurrence. IBM
 * will not be liable for any third party claims against you.
 */
 
package com.ibm.xml.parser;

/**
 * An interface of the visitor design pattern which defines the visit operations for
 * each concrete visitor (toXMLString, digest, ...) that can be performed on the DOM- and XML4J-defined
 * Nodes as the document object tree is traversed.
 * <p>If implementing a visitor, consider subclassing NOOPVisitor in order to avoid
 * implementing all methods defined by this interface.
 *
 * @version Revision: %M% %I% %W% %Q%
 * @author ROWE Tom &lt;trowe@us.ibm.com&gt;
 * @see com.ibm.xml.parser.NOOPVisitor
 * @see com.ibm.xml.parser.ToXMLStringVisitor
 * @see com.ibm.xml.parser.FormatPrintVisitor
 * @see com.ibm.xml.parser.TreeTraversal
 * @see com.ibm.xml.parser.NonRecursivePreorderTreeTraversal
 */

public interface Visitor {
    
    /**
     * Interface to be implemented for receiving control when the start of
     * a TXDocumentFragment Node is recognized when traversing the document object tree.
     * @param   documentFrag    The Node currently being traversed.
     * @exception Exception Thrown if this Node can not be visitted, or traversal modification is requested.
     * @see #visitDocumentFragmentPost
     * @see com.ibm.xml.parser.TXDocumentFragment
     * @see com.ibm.xml.parser.TreeTraversalException
     */
    public void visitDocumentFragmentPre(TXDocumentFragment documentFrag) throws Exception;
    
    /**
     * Interface to be implemented for receiving control when the end of
     * a TXDocumentFragment Node is recognized when traversing the document object tree.
     * @param   documentFrag    The Node currently being traversed.
     * @exception Exception Thrown if this Node can not be visitted, or traversal modification is requested.
     * @see #visitDocumentFragmentPre
     * @see com.ibm.xml.parser.TXDocumentFragment
     * @see com.ibm.xml.parser.TreeTraversalException
     */
    public void visitDocumentFragmentPost(TXDocumentFragment documentFrag) throws Exception;
    
    /**
     * Interface to be implemented for receiving control when the start of
     * a TXDocument Node is recognized when traversing the document object tree.
     * @param   document    The Node currently being traversed.
     * @exception Exception Thrown if this Node can not be visitted, or traversal modification is requested.
     * @see #visitDocumentPost
     * @see com.ibm.xml.parser.TXDocument
     * @see com.ibm.xml.parser.TreeTraversalException
     */
    public void visitDocumentPre(TXDocument document) throws Exception;
    
    /**
     * Interface to be implemented for receiving control when the end of
     * a TXDocument Node is recognized when traversing the document object tree.
     * @param   document    The Node currently being traversed.
     * @exception Exception Thrown if this Node can not be visitted, or traversal modification is requested.
     * @see #visitDocumentPre
     * @see com.ibm.xml.parser.TXDocument
     * @see com.ibm.xml.parser.TreeTraversalException
     */
    public void visitDocumentPost(TXDocument document) throws Exception;
    
    /**
     * Interface to be implemented for receiving control when the start of
     * a TXElement Node is recognized when traversing the document object tree.
     * @param   element The Node currently being traversed.
     * @exception Exception Thrown if this Node can not be visitted, or traversal modification is requested.
     * @see #visitElementPost
     * @see com.ibm.xml.parser.TXElement
     * @see com.ibm.xml.parser.TreeTraversalException
     */
    public void visitElementPre(TXElement element) throws Exception;
    
    /**
     * Interface to be implemented for receiving control when the end of
     * a TXElement Node is recognized when traversing the document object tree.
     * @param   element The Node currently being traversed.
     * @exception Exception Thrown if this Node can not be visitted, or traversal modification is requested.
     * @see #visitElementPre
     * @see com.ibm.xml.parser.TXElement
     * @see com.ibm.xml.parser.TreeTraversalException
     */
    public void visitElementPost(TXElement element) throws Exception;
    
    /**
     * Interface to be implemented for receiving control when the start of
     * a TXAttribute Node is recognized when traversing the document object tree.
     * @param   attribute   The Node currently being traversed.
     * @exception Exception Thrown if this Node can not be visitted, or traversal modification is requested.
     * @see #visitAttributePost
     * @see com.ibm.xml.parser.TXAttribute
     * @see com.ibm.xml.parser.TreeTraversalException
     */
    public void visitAttributePre(TXAttribute attribute) throws Exception;
    
    /**
     * Interface to be implemented for receiving control when the endt of
     * a TXAttribute Node is recognized when traversing the document object tree.
     * @param   attribute   The Node currently being traversed.
     * @exception Exception Thrown if this Node can not be visitted, or traversal modification is requested.
     * @see #visitAttributePre
     * @see com.ibm.xml.parser.TXAttribute
     * @see com.ibm.xml.parser.TreeTraversalException
     */
    public void visitAttributePost(TXAttribute attribute) throws Exception;
    
    /**
     * Interface to be implemented for receiving control when the start of
     * a TXPI Node is recognized when traversing the document object tree.
     * @param   pi      The Node currently being traversed.
     * @exception Exception Thrown if this Node can not be visitted, or traversal modification is requested.
     * @see #visitPIPost
     * @see com.ibm.xml.parser.TXPI
     * @see com.ibm.xml.parser.TreeTraversalException
     */
    public void visitPIPre(TXPI pi) throws Exception;
    
    /**
     * Interface to be implemented for receiving control when the end of
     * a TXPI Node is recognized when traversing the document object tree.
     * @param   pi      The Node currently being traversed.
     * @exception Exception Thrown if this Node can not be visitted, or traversal modification is requested.
     * @see #visitPIPre
     * @see com.ibm.xml.parser.TXPI
     * @see com.ibm.xml.parser.TreeTraversalException
     */
    public void visitPIPost(TXPI pi) throws Exception;
    
    /**
     * Interface to be implemented for receiving control when the start of
     * a TXComment Node is recognized when traversing the document object tree.
     * @param   comment The Node currently being traversed.
     * @exception Exception Thrown if this Node can not be visitted, or traversal modification is requested.
     * @see #visitCommentPost
     * @see com.ibm.xml.parser.TXComment
     * @see com.ibm.xml.parser.TreeTraversalException
     */
    public void visitCommentPre(TXComment comment) throws Exception;
    
    /**
     * Interface to be implemented for receiving control when the end of
     * a TXComment Node is recognized when traversing the document object tree.
     * @param   comment The Node currently being traversed.
     * @exception Exception Thrown if this Node can not be visitted, or traversal modification is requested.
     * @see #visitCommentPre
     * @see com.ibm.xml.parser.TXComment
     * @see com.ibm.xml.parser.TreeTraversalException
     */
    public void visitCommentPost(TXComment comment) throws Exception;

    /**
     * Interface to be implemented for receiving control when the start of
     * a TXText Node is recognized when traversing the document object tree.
     * @param   text    The Node currently being traversed.
     * @exception Exception Thrown if this Node can not be visitted, or traversal modification is requested.
     * @see #visitTextPost
     * @see com.ibm.xml.parser.TXText
     * @see com.ibm.xml.parser.TreeTraversalException
     */
    public void visitTextPre(TXText text) throws Exception;
    
    /**
     * Interface to be implemented for receiving control when the end of
     * a TXText Node is recognized when traversing the document object tree.
     * @param   text    The Node currently being traversed.
     * @exception Exception Thrown if this Node can not be visitted, or traversal modification is requested.
     * @see #visitTextPre
     * @see com.ibm.xml.parser.TXText
     * @see com.ibm.xml.parser.TreeTraversalException
     */
    public void visitTextPost(TXText text) throws Exception;
    
    /**
     * Interface to be implemented for receiving control when the start of
     * a DTD Node is recognized when traversing the document object tree.
     * @param   dtd     The Node currently being traversed.
     * @exception Exception Thrown if this Node can not be visitted, or traversal modification is requested.
     * @see #visitDTDPost
     * @see com.ibm.xml.parser.DTD
     * @see com.ibm.xml.parser.TreeTraversalException
     */
    public void visitDTDPre(DTD dtd) throws Exception;
    
    /**
     * Interface to be implemented for receiving control when the end of
     * a DTD Node is recognized when traversing the document object tree.
     * @param   dtd     The Node currently being traversed.
     * @exception Exception Thrown if this Node can not be visitted, or traversal modification is requested.
     * @see #visitDTDPre
     * @see com.ibm.xml.parser.DTD
     * @see com.ibm.xml.parser.TreeTraversalException
     */
    public void visitDTDPost(DTD dtd) throws Exception;
    
    /**
     * Interface to be implemented for receiving control when the start of
     * an ElementDecl Node is recognized when traversing the document object tree.
     * @param   elementDecl The Node currently being traversed.
     * @exception Exception Thrown if this Node can not be visitted, or traversal modification is requested.
     * @see #visitElementDeclPost
     * @see com.ibm.xml.parser.ElementDecl
     * @see com.ibm.xml.parser.TreeTraversalException
     */
    public void visitElementDeclPre(ElementDecl elementDecl) throws Exception;
    
    /**
     * Interface to be implemented for receiving control when the end of
     * an ElementDecl Node is recognized when traversing the document object tree.
     * @param   elementDecl The Node currently being traversed.
     * @exception Exception Thrown if this Node can not be visitted, or traversal modification is requested.
     * @see #visitElementDeclPre
     * @see com.ibm.xml.parser.ElementDecl
     * @see com.ibm.xml.parser.TreeTraversalException
     */
    public void visitElementDeclPost(ElementDecl elementDecl) throws Exception;
    
    /**
     * Interface to be implemented for receiving control when the start of
     * an Attlist Node is recognized when traversing the document object tree.
     * @param   attlist The Node currently being traversed.
     * @exception Exception Thrown if this Node can not be visitted, or traversal modification is requested.
     * @see #visitAttlistPost
     * @see com.ibm.xml.parser.Attlist
     * @see com.ibm.xml.parser.TreeTraversalException
     */
    public void visitAttlistPre(Attlist attlist) throws Exception;
    
    /**
     * Interface to be implemented for receiving control when the end of
     * an Attlist Node is recognized when traversing the document object tree.
     * @param   attlist The Node currently being traversed.
     * @exception Exception Thrown if this Node can not be visitted, or traversal modification is requested.
     * @see #visitAttlistPre
     * @see com.ibm.xml.parser.Attlist
     * @see com.ibm.xml.parser.TreeTraversalException
     */
    public void visitAttlistPost(Attlist attlist) throws Exception;
    
    /**
     * Interface to be implemented for receiving control when the start of
     * an AttDef Node is recognized when traversing the document object tree.
     * @param   attDef  The Node currently being traversed.
     * @exception Exception Thrown if this Node can not be visitted, or traversal modification is requested.
     * @see #visitAttDefPost
     * @see com.ibm.xml.parser.AttDef
     * @see com.ibm.xml.parser.TreeTraversalException
     */
    public void visitAttDefPre(AttDef attDef) throws Exception;
    
    /**
     * Interface to be implemented for receiving control when the end of
     * an AttDef Node is recognized when traversing the document object tree.
     * @param   attDef   The Node currently being traversed.
     * @exception Exception Thrown if this Node can not be visitted, or traversal modification is requested.
     * @see #visitAttDefPre
     * @see com.ibm.xml.parser.AttDef
     * @see com.ibm.xml.parser.TreeTraversalException
     */
    public void visitAttDefPost(AttDef attDef) throws Exception;
    
    /**
     * Interface to be implemented for receiving control when the start of
     * an EntityDecl Node is recognized when traversing the document object tree.
     * @param   entityDecl  The Node currently being traversed.
     * @exception Exception Thrown if this Node can not be visitted, or traversal modification is requested.
     * @see #visitEntityDeclPost
     * @see com.ibm.xml.parser.EntityDecl
     * @see com.ibm.xml.parser.TreeTraversalException
     */
    public void visitEntityDeclPre(EntityDecl entityDecl) throws Exception; 
    
    /**
     * Interface to be implemented for receiving control when the end of
     * an EntityDecl Node is recognized when traversing the document object tree.
     * @param   entityDecl  The Node currently being traversed.
     * @exception Exception Thrown if this Node can not be visitted, or traversal modification is requested.
     * @see #visitEntityDeclPre
     * @see com.ibm.xml.parser.EntityDecl
     * @see com.ibm.xml.parser.TreeTraversalException
     */
    public void visitEntityDeclPost(EntityDecl entityDecl) throws Exception;
    
    /**
     * Interface to be implemented for receiving control when the start of
     * a TXNotation Node is recognized when traversing the document object tree.
     * @param   notation    The Node currently being traversed.
     * @exception Exception Thrown if this Node can not be visitted, or traversal modification is requested.
     * @see #visitNotationPost
     * @see com.ibm.xml.parser.TXNotation
     * @see com.ibm.xml.parser.TreeTraversalException
     */
    public void visitNotationPre(TXNotation notation) throws Exception;
    
    /**
     * Interface to be implemented for receiving control when the end of
     * a TXNotation Node is recognized when traversing the document object tree.
     * @param   notation    The Node currently being traversed.
     * @exception Exception Thrown if this Node can not be visitted, or traversal modification is requested.
     * @see #visitNotationPre
     * @see com.ibm.xml.parser.TXNotation
     * @see com.ibm.xml.parser.TreeTraversalException
     */
    public void visitNotationPost(TXNotation notation) throws Exception;
    
    /**
     * Interface to be implemented for receiving control when the start of
     * a GeneralReference Node is recognized when traversing the document object tree.
     * @param   generalReference    The Node currently being traversed.
     * @exception Exception Thrown if this Node can not be visitted, or traversal modification is requested.
     * @see #visitGeneralReferencePost
     * @see com.ibm.xml.parser.GeneralReference
     * @see com.ibm.xml.parser.TreeTraversalException
     */
    public void visitGeneralReferencePre(GeneralReference generalReference) throws Exception;
    
    /**
     * Interface to be implemented for receiving control when the end of
     * a GeneralReference Node is recognized when traversing the document object tree.
     * @param   generalReference    The Node currently being traversed.
     * @exception Exception Thrown if this Node can not be visitted, or traversal modification is requested.
     * @see #visitGeneralReferencePre
     * @see com.ibm.xml.parser.GeneralReference
     * @see com.ibm.xml.parser.TreeTraversalException
     */
    public void visitGeneralReferencePost(GeneralReference generalReference) throws Exception;
    
    /**
     * Interface to be implemented for receiving control when the start of
     * a PseudoNode is recognized when traversing the document object tree.
     * @param   pseudoNode  The Node currently being traversed.
     * @exception Exception Thrown if this Node can not be visitted, or traversal modification is requested.
     * @see #visitPseudoNodePost
     * @see com.ibm.xml.parser.PseudoNode
     * @see com.ibm.xml.parser.TreeTraversalException
     */
    public void visitPseudoNodePre(PseudoNode pseudoNode) throws Exception;
    
    /**
     * Interface to be implemented for receiving control when the end of
     * a PseudoNode is recognized when traversing the document object tree.
     * @param   pseudoNode  The Node currently being traversed.
     * @exception Exception Thrown if this Node can not be visitted, or traversal modification is requested.
     * @see #visitPseudoNodePre
     * @see com.ibm.xml.parser.PseudoNode
     * @see com.ibm.xml.parser.TreeTraversalException
     */
    public void visitPseudoNodePost(PseudoNode pseudoNode) throws Exception;
    

}
