/*
 * (C) Copyright IBM Corp. 1997-1998  All rights reserved.
 *
 * US Government Users Restricted Rights Use, duplication or
 * disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * IBM will not be liable for any damages suffered by you as a result
 * of using the Program. In no event will IBM be liable for any
 * special, indirect or consequential damages or lost profits even if
 * IBM has been advised of the possibility of their occurrence. IBM
 * will not be liable for any third party claims against you.
 */

package com.ibm.xml.parser.util;

import com.ibm.xml.parser.AttDef;
import com.ibm.xml.parser.Attlist;
import com.ibm.xml.parser.DTD;
import com.ibm.xml.parser.ElementDecl;
import com.ibm.xml.parser.EntityDecl;
import com.ibm.xml.parser.GeneralReference;
import com.ibm.xml.parser.TXAttribute;
import com.ibm.xml.parser.TXCDATASection;
import com.ibm.xml.parser.TXDocument;
import com.ibm.xml.parser.TXElement;
import com.ibm.xml.parser.TXNotation;
import com.ibm.xml.parser.TXPI;
import com.ibm.xml.parser.TXText;
import com.ibm.xml.parser.ToNextSiblingTraversalException;
import com.ibm.xml.parser.ToXMLStringVisitor;
import com.ibm.xml.parser.Util;
import java.io.Writer;
import java.util.Hashtable;
import org.w3c.dom.Node;

/**
 * HTMLPrintVisitor implements the Visitor interface in the visitor design pattern for the
 * purpose of printing in HTML-like format the various DOM- and XML4J-defined Nodes.
 * <p>In HTML-like printing, only the following Nodes are printed:
 * <DL>
 *  <DT>TXDocument</DT>
 *  <DD>Only the doctype provided on this constructor is written (i.e. no XML declaration, &lt;!DOCTYPE&gt;, or internal DTD).</DD>
 *  <DT>TXElement</DT>
 *  <DD>All element names are uppercased.</DD>
 *  <DD>Empty elements are written as <code>&lt;BR&gt;</code> instead of <code>&lt;BR/&gt;</code>.</DD>
 *  <DT>TXAttribute</DT>
 *  <DD>All attribute names are lowercased.</DD>
 *  <DT>TXText</DT>
 *  <DD>CDATASections are respected.</DD>
 *  <DT>TXComment</DT>
 *  <DD></DD>
 *  <DT>GeneralReference</DT>
 *  <DD></DD>
 *  <DT>PseudoNode</DT>
 *  <DD></DD>
 * </DL>
 *
 * <p>The following sample code uses the HTMLPrintVisitor on a hierarchy of nodes:
 * <pre>
 *
 * PrintWriter printWriter = new PrintWriter();
 * Visitor htmlPrintVisitor = new HTMLPrintVisitor(printWriter);
 * TreeTraversal treeTraversal = new NonRecursivePreorderTreeTraversal(htmlPrintVisitor);
 * treeTraversal.traverse(document);
 * printWriter.close();
 *
 * </pre>
 *
 * <P>By default, this doesn't print non-specified attributes.</P>
 *
 * @version Revision: %M% %I% %W% %Q%
 * @author TAMURA Kent &lt;kent@trl.ibm.co.jp&gt;
 * @see com.ibm.xml.parser.Visitor
 * @see com.ibm.xml.parser.TreeTraversal
 * @see com.ibm.xml.parser.NonRecursivePreorderTreeTraversal
 * @see com.ibm.xml.parser.ToXMLStringVisitor
 * @see com.ibm.xml.parser.FormatPrintVisitor
 */
public class HTMLPrintVisitor extends ToXMLStringVisitor {
    
    protected   int     level   =   0;
    protected   String  doctype =   null;
    static   Hashtable  s_empties;
    static   Hashtable  s_entities;
    static {
        s_empties = new Hashtable();
                                // HTML 4.0 strict DTD
        s_empties.put("AREA", "AREA");
        s_empties.put("BASE", "BASE");
        s_empties.put("BR", "BR");
        s_empties.put("COL", "COL");
        s_empties.put("HR", "HR");
        s_empties.put("IMG", "IMG");
        s_empties.put("INPUT", "INPUT");
        s_empties.put("LINK", "LINK");
        s_empties.put("META", "META");
        s_empties.put("PARAM", "PARAM");
                                // HTML 4.0 loose DTD
        s_empties.put("BASEFONT", "BASEFONT");
        s_empties.put("FRAME", "FRAME");
        s_empties.put("ISINDEX", "ISINDEX");

        s_entities = new Hashtable();
        s_entities.put("nbsp", "nbsp");
        s_entities.put("iexcl", "iexcl");
        s_entities.put("cent", "cent");
        s_entities.put("pound", "pound");
        s_entities.put("curren", "curren");
        s_entities.put("yen", "yen");
        s_entities.put("brvbar", "brvbar");
        s_entities.put("sect", "sect");
        s_entities.put("uml", "uml");
        s_entities.put("copy", "copy");
        s_entities.put("ordf", "ordf");
        s_entities.put("laquo", "laquo");
        s_entities.put("not", "not");
        s_entities.put("shy", "shy");
        s_entities.put("reg", "reg");
        s_entities.put("macr", "macr");
        s_entities.put("deg", "deg");
        s_entities.put("plusmn", "plusmn");
        s_entities.put("sup2", "sup2");
        s_entities.put("sup3", "sup3");
        s_entities.put("acute", "acute");
        s_entities.put("micro", "micro");
        s_entities.put("para", "para");
        s_entities.put("middot", "middot");
        s_entities.put("cedil", "cedil");
        s_entities.put("sup1", "sup1");
        s_entities.put("ordm", "ordm");
        s_entities.put("raquo", "raquo");
        s_entities.put("frac14", "frac14");
        s_entities.put("frac12", "frac12");
        s_entities.put("frac34", "frac34");
        s_entities.put("iquest", "iquest");
        s_entities.put("Agrave", "Agrave");
        s_entities.put("Aacute", "Aacute");
        s_entities.put("Acirc", "Acirc");
        s_entities.put("Atilde", "Atilde");
        s_entities.put("Auml", "Auml");
        s_entities.put("Aring", "Aring");
        s_entities.put("AElig", "AElig");
        s_entities.put("Ccedil", "Ccedil");
        s_entities.put("Egrave", "Egrave");
        s_entities.put("Eacute", "Eacute");
        s_entities.put("Ecirc", "Ecirc");
        s_entities.put("Euml", "Euml");
        s_entities.put("Igrave", "Igrave");
        s_entities.put("Iacute", "Iacute");
        s_entities.put("Icirc", "Icirc");
        s_entities.put("Iuml", "Iuml");
        s_entities.put("ETH", "ETH");
        s_entities.put("Ntilde", "Ntilde");
        s_entities.put("Ograve", "Ograve");
        s_entities.put("Oacute", "Oacute");
        s_entities.put("Ocirc", "Ocirc");
        s_entities.put("Otilde", "Otilde");
        s_entities.put("Ouml", "Ouml");
        s_entities.put("times", "times");
        s_entities.put("Oslash", "Oslash");
        s_entities.put("Ugrave", "Ugrave");
        s_entities.put("Uacute", "Uacute");
        s_entities.put("Ucirc", "Ucirc");
        s_entities.put("Uuml", "Uuml");
        s_entities.put("Yacute", "Yacute");
        s_entities.put("THORN", "THORN");
        s_entities.put("szlig", "szlig");
        s_entities.put("agrave", "agrave");
        s_entities.put("aacute", "aacute");
        s_entities.put("acirc", "acirc");
        s_entities.put("atilde", "atilde");
        s_entities.put("auml", "auml");
        s_entities.put("aring", "aring");
        s_entities.put("aelig", "aelig");
        s_entities.put("ccedil", "ccedil");
        s_entities.put("egrave", "egrave");
        s_entities.put("eacute", "eacute");
        s_entities.put("ecirc", "ecirc");
        s_entities.put("euml", "euml");
        s_entities.put("igrave", "igrave");
        s_entities.put("iacute", "iacute");
        s_entities.put("icirc", "icirc");
        s_entities.put("iuml", "iuml");
        s_entities.put("eth", "eth");
        s_entities.put("ntilde", "ntilde");
        s_entities.put("ograve", "ograve");
        s_entities.put("oacute", "oacute");
        s_entities.put("ocirc", "ocirc");
        s_entities.put("otilde", "otilde");
        s_entities.put("ouml", "ouml");
        s_entities.put("divide", "divide");
        s_entities.put("oslash", "oslash");
        s_entities.put("ugrave", "ugrave");
        s_entities.put("uacute", "uacute");
        s_entities.put("ucirc", "ucirc");
        s_entities.put("uuml", "uuml");
        s_entities.put("yacute", "yacute");
        s_entities.put("thorn", "thorn");
        s_entities.put("yuml", "yuml");
        s_entities.put("fnof", "fnof");
        s_entities.put("Alpha", "Alpha");
        s_entities.put("Beta", "Beta");
        s_entities.put("Gamma", "Gamma");
        s_entities.put("Delta", "Delta");
        s_entities.put("Epsilon", "Epsilon");
        s_entities.put("Zeta", "Zeta");
        s_entities.put("Eta", "Eta");
        s_entities.put("Theta", "Theta");
        s_entities.put("Iota", "Iota");
        s_entities.put("Kappa", "Kappa");
        s_entities.put("Lambda", "Lambda");
        s_entities.put("Mu", "Mu");
        s_entities.put("Nu", "Nu");
        s_entities.put("Xi", "Xi");
        s_entities.put("Omicron", "Omicron");
        s_entities.put("Pi", "Pi");
        s_entities.put("Rho", "Rho");
        s_entities.put("Sigma", "Sigma");
        s_entities.put("Tau", "Tau");
        s_entities.put("Upsilon", "Upsilon");
        s_entities.put("Phi", "Phi");
        s_entities.put("Chi", "Chi");
        s_entities.put("Psi", "Psi");
        s_entities.put("Omega", "Omega");
        s_entities.put("alpha", "alpha");
        s_entities.put("beta", "beta");
        s_entities.put("gamma", "gamma");
        s_entities.put("delta", "delta");
        s_entities.put("epsilon", "epsilon");
        s_entities.put("zeta", "zeta");
        s_entities.put("eta", "eta");
        s_entities.put("theta", "theta");
        s_entities.put("iota", "iota");
        s_entities.put("kappa", "kappa");
        s_entities.put("lambda", "lambda");
        s_entities.put("mu", "mu");
        s_entities.put("nu", "nu");
        s_entities.put("xi", "xi");
        s_entities.put("omicron", "omicron");
        s_entities.put("pi", "pi");
        s_entities.put("rho", "rho");
        s_entities.put("sigmaf", "sigmaf");
        s_entities.put("sigma", "sigma");
        s_entities.put("tau", "tau");
        s_entities.put("upsilon", "upsilon");
        s_entities.put("phi", "phi");
        s_entities.put("chi", "chi");
        s_entities.put("psi", "psi");
        s_entities.put("omega", "omega");
        s_entities.put("thetasym", "thetasym");
        s_entities.put("upsih", "upsih");
        s_entities.put("piv", "piv");
        s_entities.put("bull", "bull");
        s_entities.put("hellip", "hellip");
        s_entities.put("prime", "prime");
        s_entities.put("Prime", "Prime");
        s_entities.put("oline", "oline");
        s_entities.put("frasl", "frasl");
        s_entities.put("weierp", "weierp");
        s_entities.put("image", "image");
        s_entities.put("real", "real");
        s_entities.put("trade", "trade");
        s_entities.put("alefsym", "alefsym");
        s_entities.put("larr", "larr");
        s_entities.put("uarr", "uarr");
        s_entities.put("rarr", "rarr");
        s_entities.put("darr", "darr");
        s_entities.put("harr", "harr");
        s_entities.put("crarr", "crarr");
        s_entities.put("lArr", "lArr");
        s_entities.put("uArr", "uArr");
        s_entities.put("rArr", "rArr");
        s_entities.put("dArr", "dArr");
        s_entities.put("hArr", "hArr");
        s_entities.put("forall", "forall");
        s_entities.put("part", "part");
        s_entities.put("exist", "exist");
        s_entities.put("empty", "empty");
        s_entities.put("nabla", "nabla");
        s_entities.put("isin", "isin");
        s_entities.put("notin", "notin");
        s_entities.put("ni", "ni");
        s_entities.put("prod", "prod");
        s_entities.put("sum", "sum");
        s_entities.put("minus", "minus");
        s_entities.put("lowast", "lowast");
        s_entities.put("radic", "radic");
        s_entities.put("prop", "prop");
        s_entities.put("infin", "infin");
        s_entities.put("ang", "ang");
        s_entities.put("and", "and");
        s_entities.put("or", "or");
        s_entities.put("cap", "cap");
        s_entities.put("cup", "cup");
        s_entities.put("int", "int");
        s_entities.put("there4", "there4");
        s_entities.put("sim", "sim");
        s_entities.put("cong", "cong");
        s_entities.put("asymp", "asymp");
        s_entities.put("ne", "ne");
        s_entities.put("equiv", "equiv");
        s_entities.put("le", "le");
        s_entities.put("ge", "ge");
        s_entities.put("sub", "sub");
        s_entities.put("sup", "sup");
        s_entities.put("nsub", "nsub");
        s_entities.put("sube", "sube");
        s_entities.put("supe", "supe");
        s_entities.put("oplus", "oplus");
        s_entities.put("otimes", "otimes");
        s_entities.put("perp", "perp");
        s_entities.put("sdot", "sdot");
        s_entities.put("lceil", "lceil");
        s_entities.put("rceil", "rceil");
        s_entities.put("lfloor", "lfloor");
        s_entities.put("rfloor", "rfloor");
        s_entities.put("lang", "lang");
        s_entities.put("rang", "rang");
        s_entities.put("loz", "loz");
        s_entities.put("spades", "spades");
        s_entities.put("clubs", "clubs");
        s_entities.put("hearts", "hearts");
        s_entities.put("diams", "diams");
        s_entities.put("fnof", "fnof");
        s_entities.put("Alpha", "Alpha");
        s_entities.put("Beta", "Beta");
        s_entities.put("Gamma", "Gamma");
        s_entities.put("Delta", "Delta");
        s_entities.put("Epsilon", "Epsilon");
        s_entities.put("Zeta", "Zeta");
        s_entities.put("Eta", "Eta");
        s_entities.put("Theta", "Theta");
        s_entities.put("Iota", "Iota");
        s_entities.put("Kappa", "Kappa");
        s_entities.put("Lambda", "Lambda");
        s_entities.put("Mu", "Mu");
        s_entities.put("Nu", "Nu");
        s_entities.put("Xi", "Xi");
        s_entities.put("Omicron", "Omicron");
        s_entities.put("Pi", "Pi");
        s_entities.put("Rho", "Rho");
        s_entities.put("Sigma", "Sigma");
        s_entities.put("Tau", "Tau");
        s_entities.put("Upsilon", "Upsilon");
        s_entities.put("Phi", "Phi");
        s_entities.put("Chi", "Chi");
        s_entities.put("Psi", "Psi");
        s_entities.put("Omega", "Omega");
        s_entities.put("alpha", "alpha");
        s_entities.put("beta", "beta");
        s_entities.put("gamma", "gamma");
        s_entities.put("delta", "delta");
        s_entities.put("epsilon", "epsilon");
        s_entities.put("zeta", "zeta");
        s_entities.put("eta", "eta");
        s_entities.put("theta", "theta");
        s_entities.put("iota", "iota");
        s_entities.put("kappa", "kappa");
        s_entities.put("lambda", "lambda");
        s_entities.put("mu", "mu");
        s_entities.put("nu", "nu");
        s_entities.put("xi", "xi");
        s_entities.put("omicron", "omicron");
        s_entities.put("pi", "pi");
        s_entities.put("rho", "rho");
        s_entities.put("sigmaf", "sigmaf");
        s_entities.put("sigma", "sigma");
        s_entities.put("tau", "tau");
        s_entities.put("upsilon", "upsilon");
        s_entities.put("phi", "phi");
        s_entities.put("chi", "chi");
        s_entities.put("psi", "psi");
        s_entities.put("omega", "omega");
        s_entities.put("thetasym", "thetasym");
        s_entities.put("upsih", "upsih");
        s_entities.put("piv", "piv");
        s_entities.put("bull", "bull");
        s_entities.put("hellip", "hellip");
        s_entities.put("prime", "prime");
        s_entities.put("Prime", "Prime");
        s_entities.put("oline", "oline");
        s_entities.put("frasl", "frasl");
        s_entities.put("weierp", "weierp");
        s_entities.put("image", "image");
        s_entities.put("real", "real");
        s_entities.put("trade", "trade");
        s_entities.put("alefsym", "alefsym");
        s_entities.put("larr", "larr");
        s_entities.put("uarr", "uarr");
        s_entities.put("rarr", "rarr");
        s_entities.put("darr", "darr");
        s_entities.put("harr", "harr");
        s_entities.put("crarr", "crarr");
        s_entities.put("lArr", "lArr");
        s_entities.put("uArr", "uArr");
        s_entities.put("rArr", "rArr");
        s_entities.put("dArr", "dArr");
        s_entities.put("hArr", "hArr");
        s_entities.put("forall", "forall");
        s_entities.put("part", "part");
        s_entities.put("exist", "exist");
        s_entities.put("empty", "empty");
        s_entities.put("nabla", "nabla");
        s_entities.put("isin", "isin");
        s_entities.put("notin", "notin");
        s_entities.put("ni", "ni");
        s_entities.put("prod", "prod");
        s_entities.put("sum", "sum");
        s_entities.put("minus", "minus");
        s_entities.put("lowast", "lowast");
        s_entities.put("radic", "radic");
        s_entities.put("prop", "prop");
        s_entities.put("infin", "infin");
        s_entities.put("ang", "ang");
        s_entities.put("and", "and");
        s_entities.put("or", "or");
        s_entities.put("cap", "cap");
        s_entities.put("cup", "cup");
        s_entities.put("int", "int");
        s_entities.put("there4", "there4");
        s_entities.put("sim", "sim");
        s_entities.put("cong", "cong");
        s_entities.put("asymp", "asymp");
        s_entities.put("ne", "ne");
        s_entities.put("equiv", "equiv");
        s_entities.put("le", "le");
        s_entities.put("ge", "ge");
        s_entities.put("sub", "sub");
        s_entities.put("sup", "sup");
        s_entities.put("nsub", "nsub");
        s_entities.put("sube", "sube");
        s_entities.put("supe", "supe");
        s_entities.put("oplus", "oplus");
        s_entities.put("otimes", "otimes");
        s_entities.put("perp", "perp");
        s_entities.put("sdot", "sdot");
        s_entities.put("lceil", "lceil");
        s_entities.put("rceil", "rceil");
        s_entities.put("lfloor", "lfloor");
        s_entities.put("rfloor", "rfloor");
        s_entities.put("lang", "lang");
        s_entities.put("rang", "rang");
        s_entities.put("loz", "loz");
        s_entities.put("spades", "spades");
        s_entities.put("clubs", "clubs");
        s_entities.put("hearts", "hearts");
        s_entities.put("diams", "diams");
    }
    
    /**
     * Constructor for customized encoding and doctype.
     * @param writer        The character output stream to use.
     * @param encoding      Java character encoding in use by <VAR>writer</VAR>.
     * @param doctype       String to be printed at the top of the document.
     */
    public HTMLPrintVisitor(Writer writer, String encoding, String doctype) {
        super(writer, encoding);
        this.doctype = doctype;
        this.isPrintNonSpecifiedAttributes = false;
    }
    
    /**
     * Constructor for customized encoding.
     * @param writer        The character output stream to use.
     * @param encoding      Java character encoding in use by <VAR>writer</VAR>.
     */
    public HTMLPrintVisitor(Writer writer, String encoding) {
        this(writer, encoding, null);
    }
    
    /**
     * Constructor for default encoding.
     * @param writer  The character output stream to use.
     */
    public HTMLPrintVisitor(Writer writer) {
        this(writer, null, null);
    }

    /**
     * Writes the <var>doctype</var> from the constructor (if any).
     * @param   document        Node print as HTML.
     * @exception Exception     Thrown if this Node can not be visitted because of an invalid character output stream.
     * @see com.ibm.xml.parser.TXDocument
     */
    public void visitDocumentPre(TXDocument document) throws Exception {
        if (this.doctype != null)
            this.writer.write(doctype);
    }
    
    /**
     * Flush the writer.
     * @param   document        Node to print as HTML.
     * @exception Exception     Thrown if this Node can not be visitted because of an invalid character output stream.
     * @see com.ibm.xml.parser.TXDocument
     */
    public void visitDocumentPost(TXDocument document) throws Exception {
        this.writer.write("\n");
        this.writer.flush();
    }

    /**
     * Creates a formatted string representation of the start of the specified <var>element</var> Node
     * and its associated attributes, and directs it to the print writer.
     * @param   element         Node to print as HTML.
     * @exception Exception     Thrown if this Node can not be visitted because of an invalid character output stream.
     * @see com.ibm.xml.parser.TXElement
     */
    public void visitElementPre(TXElement element) throws Exception {
        this.level ++;
        this.writer.write("<" + element.getTagName().toUpperCase());
        TXAttribute[] attributes = element.getAttributeArray();
        for (int i = 0;  i < attributes.length;  i ++) {
            TXAttribute attr = attributes[i];
            if (this.isPrintNonSpecifiedAttributes || attr.getSpecified())
                visitAttributePre(attr);
        }
        this.writer.write(">");
    }
    
    /**
     * Creates a formatted string representation of the end of the specified <var>element</var> Node, 
     * and directs it to the print writer.
     * @param   element         Node to print as HTML.
     * @exception Exception     Thrown if this Node can not be visitted because of an invalid character output stream.
     * @see com.ibm.xml.parser.TXElement
     */
    public void visitElementPost(TXElement element) throws Exception {
        String tagName = element.getTagName().toUpperCase();
        if (element.hasChildNodes() || s_empties.get(tagName) == null) {
            this.writer.write("</" + tagName + ">");
        }
        this.level --;
    }
    
    /**
     * Creates a formatted string representation of the specified <var>attribute</var> Node
     * and its associated attributes, and directs it to the print writer.
     * <p>Note that TXAttribute Nodes are not parsed into the document object hierarchy by the
     * XML4J parser; attributes exist as part of a TXElement Node.
     * @param   element         Node to print as HTML.
     * @exception Exception     Thrown if this Node can not be visitted because of an invalid character output stream.
     * @see com.ibm.xml.parser.TXElement
     */
    public void visitAttributePre(TXAttribute attribute) throws Exception {
        String aname = attribute.getName().toLowerCase();
        if (aname.equals("xml:lang"))  aname = "lang";
        this.writer.write(" " + aname + "=\"" +
                          Util.backReference(attribute.getValue(), "&<>\"", this.encoding) + "\"");
    }
    
    /**
     * Ignore TXPI Nodes.
     * @param   pi              CURRENTLY NOT IMPLEMENTED.
     * @exception Exception     CURRENTLY NEVER THROWN.
     * @see com.ibm.xml.parser.TXPI
     */
    public void visitPIPre(TXPI pi) throws Exception {
    }
    
    /**
     * Creates a formatted string representation of the specified <var>text</var> Node, 
     * and directs it to the print writer.  CDATASections are respected.
     * @param     text          Node to print with format.
     * @exception Exception     Thrown if this Node can not be visitted because of an invalid character output stream.
     * @see com.ibm.xml.parser.TXText
     */
    public void visitTextPre(TXText text) throws Exception {
        if (this.level > 0) {
            if (text instanceof TXCDATASection) {
                this.writer.write(text.getData());
            } else {
                this.writer.write(Util.backReference(text.getData(), this.encoding));
            }
        }
    }
    
    /**
     * Ignore DTD Nodes.
     * @param   dtd             CURRENTLY NOT IMPLEMENTED.
     * @exception ToNextSibligTraversalException Always thrown.
     * @see com.ibm.xml.parser.DTD 
     */
    public void visitDTDPre(DTD dtd) throws Exception {
        throw new ToNextSiblingTraversalException();
    }
    /**
     * Ignore DTD Nodes.
     * @param   dtd             CURRENTLY NOT IMPLEMENTED.
     * @exception Exception     CURRENTLY NEVER THROWN.
     * @see com.ibm.xml.parser.DTD 
     */
    public void visitDTDPost(DTD dtd) throws Exception {
    }
    
    /**
     * Ignore ElementDecl Nodes.
     * @param   elementDecl     CURRENTLY NOT IMPLEMENTED.
     * @exception Exception     CURRENTLY NEVER THROWN.
     * @see com.ibm.xml.parser.ElementDecl
     */
    public void visitElementDeclPre(ElementDecl elementDecl) throws Exception {
    }
    
    /**
     * Ignore Attlist Nodes.
     * @param   attlist         CURRENTLY NOT IMPLEMENTED.
     * @exception Exception     CURRENTLY NEVER THROWN.
     * @see com.ibm.xml.parser.Attlist
     */
    public void visitAttlistPre(Attlist attlist) throws Exception {
    }
    
    /**
     * Ignore AttDef Nodes.
     * @param   attDef          CURRENTLY NOT IMPLEMENTED.
     * @exception Exception     CURRENTLY NEVER THROWN.
     * @see com.ibm.xml.parser.AttDef
     */
    public void visitAttDefPre(AttDef attDef) throws Exception {
    }
    
    /**
     * Ignore Entity Nodes.
     * @param   entity          CURRENTLY NOT IMPLEMENTED.
     * @exception Exception     CURRENTLY NEVER THROWN.
     * @see com.ibm.xml.parser.EntityDecl
     */
    public void visitEntityDeclPre(EntityDecl entity) throws Exception {
    }
    
    /**
     * Ignore TXNotation Nodes.
     * @param   notation        CURRENTLY NOT IMPLEMENTED.
     * @exception Exception     CURRENTLY NEVER THROWN.
     * @see com.ibm.xml.parser.TXNotation
     */
    public void visitNotationPre(TXNotation notation) throws Exception {
    }
    
    /**
     * Ignore GeneralReference Nodes. Print contents of GeneralReferences.
     * @param   generalReference        CURRENTLY NOT IMPLEMENTED.
     * @exception Exception     Never thrown.
     * @see com.ibm.xml.parser.GeneralReference
     */
    public void visitGeneralReferencePre(GeneralReference generalReference) throws Exception {
        String name = generalReference.getName();
        if (s_entities.get(name) != null && generalReference.getChildNodes().getLength() == 1) {
            Node child = generalReference.getFirstChild();
            if (child.getNodeType() == Node.TEXT_NODE && child.getNodeValue().length() == 1) {
                this.writer.write((char)'&' + name + (char)';');
                throw new ToNextSiblingTraversalException();
            }
        }
    }
}

