/*
 * (C) Copyright IBM Corp. 1998  All rights reserved.
 *
 * US Government Users Restricted Rights Use, duplication or
 * disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * IBM will not be liable for any damages suffered by you as a result
 * of using the Program. In no event will IBM be liable for any
 * special, indirect or consequential damages or lost profits even if
 * IBM has been advised of the possibility of their occurrence. IBM
 * will not be liable for any third party claims against you.
 */
 
package com.ibm.xml.parser.util;

import org.w3c.dom.*;
import com.ibm.xml.parser.*;

/**
 * RecursivePreorderTreeTraversal defines a specific document object tree traversal algorithm
 * for use by the visitor design pattern. This algorithm visits the Parent before visiting
 * its children.
 *
 * @version Revision: 47 1.6 src/com/ibm/xml/parser/util/RecursivePreorderTreeTraversal.java, xml4jsrc, xml4j-jtcsv, xml4j_1_1_16 
 * @author KING Adam &lt;raking@us.ibm.com&gt;
 * @see com.ibm.xml.parser.TreeTraversal
 * @see com.ibm.xml.parser.NonRecursivePreorderTreeTraversal
 */
public class RecursivePreorderTreeTraversal extends TreeTraversal {
    
        boolean endTraversal    =   false;

    /**
     * Constructor.
     * @param   visitor The implemention of the Visitor operation (toXMLString, digest, ...)
     * @see com.ibm.xml.parser.Visitor
     */
    public RecursivePreorderTreeTraversal(Visitor visitor) {
        super(visitor);
        this.endTraversal = false;
    }

    /**
     * Traverses the document object tree at the specified <var>node</var>.
     * <p>The numbered nodes in the trees below indicate the order of traversal given
     * the specified <code>node</code> of &quot;1&quot;.
     * <pre>
     *
     *                 1              x              x
     *                / \            / \            / \
     *               2   6          1   x          x   x
     *              /|\   \        /|\   \        /|\   \
     *             3 4 5   7      2 3 4   x      x 1 x   x
     *
     * </pre>
     * <p>Once this method is invoked, the RecursivePreorderTreeTraversal instance is of no 
     * further use, and should NOT be reused.
     * @param   node        The starting point to begin traversing the document object tree.
     * @exception Exception Thrown if the document hierarchy can not be visitted.
     */
    public void traverse(Node node) throws Exception {
        try {
            if(node!=null && !endTraversal) {
                try {
                    ((Visitee) node).acceptPre(getVisitor());
                    for(Node i=node.getFirstChild(); i!=null && !endTraversal; i=i.getNextSibling()) {
                        this.traverse(i);
                    }
                } catch (ToNextSiblingTraversalException exception1) {
                    // Do nothing because we've already broken the loop to recurse to children.
                }
                if (!endTraversal)
                    ((Visitee) node).acceptPost(getVisitor());
            }
        } catch (EndTraversalException exception2) {
            this.endTraversal = true;
        }
    }

}  //RecursivePreorderTreeTraversal
