/*
 * (C) Copyright IBM Corp. 1997-1998  All rights reserved.
 *
 * US Government Users Restricted Rights Use, duplication or
 * disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * IBM will not be liable for any damages suffered by you as a result
 * of using the Program. In no event will IBM be liable for any
 * special, indirect or consequential damages or lost profits even if
 * IBM has been advised of the possibility of their occurrence. IBM
 * will not be liable for any third party claims against you.
 */

package com.ibm.xml.parser.util;

import com.ibm.xml.parser.AttDef;
import com.ibm.xml.parser.Attlist;
import com.ibm.xml.parser.Child;
import com.ibm.xml.parser.ContentModel;
import com.ibm.xml.parser.DTD;
import com.ibm.xml.parser.ElementDecl;
import com.ibm.xml.parser.EntityDecl;
import com.ibm.xml.parser.ExternalID;
import com.ibm.xml.parser.GeneralReference;
import com.ibm.xml.parser.PseudoNode;
import com.ibm.xml.parser.StylesheetPI;
import com.ibm.xml.parser.TXAttribute;
import com.ibm.xml.parser.TXCDATASection;
import com.ibm.xml.parser.TXComment;
import com.ibm.xml.parser.TXDocument;
import com.ibm.xml.parser.TXElement;
import com.ibm.xml.parser.TXNotation;
import com.ibm.xml.parser.TXPI;
import com.ibm.xml.parser.TXText;
import java.util.Enumeration;
import javax.swing.tree.TreeNode;
import org.w3c.dom.CDATASection;
import org.w3c.dom.Comment;
import org.w3c.dom.Element;
import org.w3c.dom.EntityReference;
import org.w3c.dom.Node;
import org.w3c.dom.ProcessingInstruction;
import org.w3c.dom.Text;

/**
 * <P>A factory for Swing <CODE>JTree</CODE> class.</P>
 * <PRE>
 *   Parser p = new Parser(...);
 *   TreeFactory doc = new TreeFactory();
 *   p.setElementFactory(doc);
 *   p.readStream(...);
 *   tree = new JTree(doc);
 * </PRE>
 *
 * @version Revision: 53 1.13 src/com/ibm/xml/parser/util/TreeFactory.java, xml4jsrc, xml4j-jtcsv, xml4j_1_1_16 
 * @author TAMURA Kent &lt;kent@trl.ibm.co.jp&gt;
 * @see com.ibm.xml.parser.Parser#setElementFactory
 * @see javax.swing.JTree
 * @see javax.swing.tree.TreeNode
 */
public class TreeFactory extends TXDocument implements TreeNode {
    public String convertValueToText(Object ch) {
        String ret = "";
        String encoding = null;
        switch (((Child)ch).getNodeType()) {
          case Node.ELEMENT_NODE:
            ret = toStringElement((TXElement)ch, encoding);
            break;

          case Node.ATTRIBUTE_NODE:
            ret = toStringAttribute((TXAttribute)ch, encoding);
            break;

          case Node.PROCESSING_INSTRUCTION_NODE: // TXPI and StylesheetPI
            ret = toStringPI((ProcessingInstruction)ch, encoding);
            break;

          case Node.COMMENT_NODE:
            ret = toStringComment((Comment)ch, encoding);
            break;

          case Node.TEXT_NODE:
            ret = toStringText((Text)ch, encoding);
            break;

          case Node.CDATA_SECTION_NODE:
            ret = toStringCDATA((CDATASection)ch, encoding);
            break;

          case Node.DOCUMENT_NODE:
            ret = toStringDocument((TXDocument)ch, encoding);
            break;

          case Node.DOCUMENT_TYPE_NODE:
            ret = toStringDTD((DTD)ch, encoding);
            break;

          case Child.ELEMENT_DECL:
            ret = toStringElementDecl((ElementDecl)ch, encoding);
            break;

          case Child.ATTLIST:
            ret = toStringAttlist((Attlist)ch, encoding);
            break;

          case Child.ATTDEF:
            ret = toStringAttDef((AttDef)ch, encoding);
            break;

          case Node.ENTITY_NODE:
            ret = toStringEntityDecl((EntityDecl)ch, encoding);
            break;

          case Node.NOTATION_NODE:
            ret = toStringNotation((TXNotation)ch, encoding);
            break;

          case Node.ENTITY_REFERENCE_NODE:
            ret = toStringGeneralReference((GeneralReference)ch, encoding);
            break;

          case Child.PSEUDONODE:
            ret = toStringPseudoNode((PseudoNode)ch, encoding);
            break;

          default:
        }
        return ret;
    }

    public String toStringAttDef(AttDef ad, String encoding) {
        return ad.getName();
    }
    public String toStringAttlist(Attlist al, String encoding) {
        return "!ATTLIST "+al.getName();
    }
    public String toStringDTD(DTD dtd, String encoding) {
        return "!DOCTYPE "+dtd.getName();
    }
    public String toStringElementDecl(ElementDecl ed, String encoding) {
        return "!ELEMENT "+ed.getName();
    }
    public String toStringEntityDecl(EntityDecl ent, String encoding) {
        return "!ENTITY "+(ent.isParameter() ? "% " : "")+ent.getNodeName();
    }
    public String toStringComment(Comment com, String encoding) {
        return "<!--"+com.getData()+"-->";
    }
    public String toStringDocument(TXDocument doc, String encoding) {
        return "[DOCUMENT]";
    }
    public String toStringElement(TXElement el, String encoding) {
        StringBuffer sb = new StringBuffer();
        sb.append("<");
        sb.append(el.getNodeName());
        java.util.Enumeration en = el.attributeElements();
        while (en.hasMoreElements()) {
            sb.append(" ");
            sb.append(((TXAttribute)en.nextElement()).toXMLString(encoding));
        }
        sb.append(">");
        return sb.toString();
    }
    public String toStringNotation(TXNotation no, String encoding) {
        return "!NOTATION "+no.getNodeName();
    }
    public String toStringPI(ProcessingInstruction pi, String encoding) {
        return "<?"+pi.getTarget()+pi.getData()+"?>";
    }
    public String toStringText(Text text, String encoding) {
        return "\""+TXText.makePrintable(text.getData())+"\"";
    }
    public String toStringCDATA(CDATASection text, String encoding) {
        return "<![CDATA["+text.getData()+"]]>";
    }

    public String toStringAttribute(TXAttribute attribute, String encoding) {
        return "";
    }
    public String toStringGeneralReference(GeneralReference gr, String encoding) {
        return "&"+gr.getName()+";";
    }
    public String toStringPseudoNode(PseudoNode pn, String encoding) {
        return pn.getData();
    }

    // ================================================================
    // Factory methods
    // ================================================================

    public Element createElement(String name) {
        TXElement el = new TreeElement(name);
        el.setFactory(this);
        return el;
    }

    public TXText createTextNode(String data, boolean ignorable) {
        TXText te = new TreeText(data);
        te.setFactory(this);
        te.setIsIgnorableWhitespace(ignorable);
        return te;
    }
    public TXText createTextNode(char[] ac, int offset, int length, boolean ignorable) {
        TXText te = new TreeText(new String(ac, offset, length));
        te.setFactory(this);
        te.setIsIgnorableWhitespace(ignorable);
        return te;
    }

    public CDATASection createCDATASection(String data) {
        TXCDATASection cd = new TreeCDATASection(data);
        cd.setFactory(this);
        return cd;
    }

    public Comment createComment(String data) {
        TXComment comm = new TreeComment(data);
        comm.setFactory(this);
        return comm;
    }

    public ProcessingInstruction createProcessingInstruction(String name, String data) {
        TXPI pi = new TreePI(name, data);
        pi.setFactory(this);
        return pi;
    }

    public StylesheetPI createStylesheetPI(String name, String data,
                                           String type, String href, String title) {
        StylesheetPI pi = new TreeStylesheetPI(name, data, type, href, title);
        pi.setFactory(this);
        return pi;
    }

    public DTD createDTD() {
        DTD d = new TreeDTD();
        d.setFactory(this);
        return d;
    }
    public DTD createDTD(String rootname, ExternalID eid) {
        DTD d = new TreeDTD(rootname, eid);
        d.setFactory(this);
        return d;
    }
    public ElementDecl createElementDecl(String elementname, ContentModel cs) {
        ElementDecl ed = new TreeElementDecl(elementname, cs);
        ed.setFactory(this);
        return ed;
    }

    public Attlist createAttlist(String elementname) {
        Attlist al = new TreeAttlist(elementname);
        al.setFactory(this);
        return al;
    }
    public AttDef createAttDef(String attributename) {
        AttDef ad = new TreeAttDef(attributename);
        ad.setFactory(this);
        return ad;
    }
    public EntityDecl createEntityDecl(String name, String value, boolean parameter) {
        EntityDecl en = new TreeEntity(name, value, parameter);
        en.setFactory(this);
        return en;
    }
    public EntityDecl createEntity(String name, ExternalID eid, boolean isParameter, String ndata) {
        EntityDecl en = new TreeEntity(name, eid, isParameter, ndata);
        en.setFactory(this);
        return en;
    }
    public TXNotation createNotation(String name, ExternalID eid) {
        TXNotation no = new TreeNotation(name, eid);
        no.setFactory(this);
        return no;
    }
    public EntityReference createEntityReference(String name) {
        GeneralReference gr = new TreeGeneralReference(name);
        gr.setFactory(this);
        return gr;
    }

    /**
     * This method isn't ElementFactory's method.
     */
    public PseudoNode createPseudoNode(String data) {
        PseudoNode pn = new TreePseudoNode(data);
        pn.setFactory(this);
        return pn;
    }


    // ================================================

    public Enumeration children() {
        return elements();
    }
    public boolean getAllowsChildren() {
        return true;
    }
    public TreeNode getChildAt(int childIndex) {
        return (TreeNode)getChildNodes().item(childIndex);
    }
    public int getChildCount() {
        return getChildNodes().getLength();
    }
    public int getIndex(TreeNode node) {
        int ret = 0;
        for (Node n = getFirstChild();
             n != null;
             n = n.getNextSibling()) {
            if (n == node)  return ret;
            ret ++;
        }
        return -1;
    }
    public TreeNode getParent() {
        return (TreeNode)getParentNode();
    }
    public boolean isLeaf() {
        return false;
    }
    public String toString() {
        return ((TreeFactory)getFactory()).convertValueToText(this);
    }

    // ================================================

    public static class TreeElement extends TXElement implements TreeNode {
        public TreeElement(String qname) {
            super(qname);
        }
        public Enumeration children() {
            return elements();
        }
        public boolean getAllowsChildren() {
            return true;
        }
        public TreeNode getChildAt(int childIndex) {
            return (TreeNode)getChildNodes().item(childIndex);
        }
        public int getChildCount() {
            return getChildNodes().getLength();
        }
        public int getIndex(TreeNode node) {
            int ret = 0;
            for (Node n = getFirstChild();
                 n != null;
                 n = n.getNextSibling()) {
                if (n == node)  return ret;
                ret ++;
            }
            return -1;
        }
        public TreeNode getParent() {
            return (TreeNode)getParentNode();
        }
        public boolean isLeaf() {
            return !hasChildNodes();
        }
        public String toString() {
            return ((TreeFactory)getFactory()).convertValueToText(this);
        }
    }

    // ================================================

    public static class TreeDTD extends DTD implements TreeNode {
        public TreeDTD() {
            super();
        }
        public TreeDTD(String root, ExternalID eid) {
            super(root, eid);
        }
        public Enumeration children() {
            return elements();
        }
        public boolean getAllowsChildren() {
            return true;
        }
        public TreeNode getChildAt(int childIndex) {
            return (TreeNode)getChildNodes().item(childIndex);
        }
        public int getChildCount() {
            return getChildNodes().getLength();
        }
        public int getIndex(TreeNode node) {
            int ret = 0;
            for (Node n = getFirstChild();
                 n != null;
                 n = n.getNextSibling()) {
                if (n == node)  return ret;
                ret ++;
            }
            return -1;
        }
        public TreeNode getParent() {
            return (TreeNode)getParentNode();
        }
        public boolean isLeaf() {
            return !hasChildNodes();
        }
        public String toString() {
            return ((TreeFactory)getFactory()).convertValueToText(this);
        }
    }
    
    // ================================================

    public static class TreeText extends TXText implements TreeNode {
        public TreeText(String data) {
            super(data);
        }
        public Enumeration children() {
            return null;
        }
        public boolean getAllowsChildren() {
            return false;
        }
        public TreeNode getChildAt(int childIndex) {
            return null;
        }
        public int getChildCount() {
            return 0;
        }
        public int getIndex(TreeNode node) {
            return -1;
        }
        public TreeNode getParent() {
            return (TreeNode)getParentNode();
        }
        public boolean isLeaf() {
            return true;
        }
        public String toString() {
            return ((TreeFactory)getFactory()).convertValueToText(this);
        }
    }

    // ================================================

    public static class TreeCDATASection extends TXCDATASection implements TreeNode {
        public TreeCDATASection(String data) {
            super(data);
        }
        public Enumeration children() {
            return null;
        }
        public boolean getAllowsChildren() {
            return false;
        }
        public TreeNode getChildAt(int childIndex) {
            return null;
        }
        public int getChildCount() {
            return 0;
        }
        public int getIndex(TreeNode node) {
            return -1;
        }
        public TreeNode getParent() {
            return (TreeNode)getParentNode();
        }
        public boolean isLeaf() {
            return true;
        }
        public String toString() {
            return ((TreeFactory)getFactory()).convertValueToText(this);
        }
    }

    // ================================================

    public static class TreeComment extends TXComment implements TreeNode {
        public TreeComment(String data) {
            super(data);
        }
        public Enumeration children() {
            return null;
        }
        public boolean getAllowsChildren() {
            return false;
        }
        public TreeNode getChildAt(int childIndex) {
            return null;
        }
        public int getChildCount() {
            return 0;
        }
        public int getIndex(TreeNode node) {
            return -1;
        }
        public TreeNode getParent() {
            return (TreeNode)getParentNode();
        }
        public boolean isLeaf() {
            return true;
        }
        public String toString() {
            return ((TreeFactory)getFactory()).convertValueToText(this);
        }
    }

    // ================================================

    public static class TreePI extends TXPI implements TreeNode {
        public TreePI(String name, String data) {
            super(name, data);
        }
        public Enumeration children() {
            return null;
        }
        public boolean getAllowsChildren() {
            return false;
        }
        public TreeNode getChildAt(int childIndex) {
            return null;
        }
        public int getChildCount() {
            return 0;
        }
        public int getIndex(TreeNode node) {
            return -1;
        }
        public TreeNode getParent() {
            return (TreeNode)getParentNode();
        }
        public boolean isLeaf() {
            return true;
        }
        public String toString() {
            return ((TreeFactory)getFactory()).convertValueToText(this);
        }
    }

    // ================================================

    public static class TreeStylesheetPI extends StylesheetPI implements TreeNode {
        public TreeStylesheetPI(String n, String d, String t, String h, String tit) {
            super(n, d, t, h, tit);
        }
        public Enumeration children() {
            return null;
        }
        public boolean getAllowsChildren() {
            return false;
        }
        public TreeNode getChildAt(int childIndex) {
            return null;
        }
        public int getChildCount() {
            return 0;
        }
        public int getIndex(TreeNode node) {
            return -1;
        }
        public TreeNode getParent() {
            return (TreeNode)getParentNode();
        }
        public boolean isLeaf() {
            return true;
        }
        public String toString() {
            return ((TreeFactory)getFactory()).convertValueToText(this);
        }
    }

    // ================================================

    public static class TreeElementDecl extends ElementDecl implements TreeNode {
        public TreeElementDecl(String n, ContentModel cm) {
            super(n, cm);
        }
        public Enumeration children() {
            return null;
        }
        public boolean getAllowsChildren() {
            return false;
        }
        public TreeNode getChildAt(int childIndex) {
            return null;
        }
        public int getChildCount() {
            return 0;
        }
        public int getIndex(TreeNode node) {
            return -1;
        }
        public TreeNode getParent() {
            return (TreeNode)getParentNode();
        }
        public boolean isLeaf() {
            return true;
        }
        public String toString() {
            return ((TreeFactory)getFactory()).convertValueToText(this);
        }
    }

    // ================================================

    public static class TreeAttlist extends Attlist implements TreeNode {
        public TreeAttlist(String n) {
            super(n);
        }
        public Enumeration children() {
            return elements();
        }
        public boolean getAllowsChildren() {
            return true;
        }
        public TreeNode getChildAt(int childIndex) {
            return (TreeNode)elementAt(childIndex);
        }
        public int getChildCount() {
            return size();
        }
        public int getIndex(TreeNode node) {
            Enumeration en = elements();
            int ret = 0;
            while (en.hasMoreElements()) {
                if (node == en.nextElement())  return ret;
                ret ++;
            }
            return -1;
        }
        public TreeNode getParent() {
            return (TreeNode)getParentNode();
        }
        public boolean isLeaf() {
            return 0 == size();
        }
        public String toString() {
            return ((TreeFactory)getFactory()).convertValueToText(this);
        }
    }

    // ================================================

    public static class TreeAttDef extends AttDef implements TreeNode {
        public TreeAttDef(String n) {
            super(n);
        }
        public Enumeration children() {
            return null;
        }
        public boolean getAllowsChildren() {
            return false;
        }
        public TreeNode getChildAt(int childIndex) {
            return null;
        }
        public int getChildCount() {
            return 0;
        }
        public int getIndex(TreeNode node) {
            return -1;
        }
        public TreeNode getParent() {
            return (TreeNode)getParentNode();
        }
        public boolean isLeaf() {
            return true;
        }
        public String toString() {
            return ((TreeFactory)getFactory()).convertValueToText(this);
        }
    }

    // ================================================

    public static class TreeEntity extends EntityDecl implements TreeNode {
        public TreeEntity(String n, String v, boolean p) {
            super(n, v, p);
        }
        public TreeEntity(String n, ExternalID eid, boolean p, String nd) {
            super(n, eid, p, nd);
        }
        public Enumeration children() {
            return null;
        }
        public boolean getAllowsChildren() {
            return false;
        }
        public TreeNode getChildAt(int childIndex) {
            return null;
        }
        public int getChildCount() {
            return 0;
        }
        public int getIndex(TreeNode node) {
            return -1;
        }
        public TreeNode getParent() {
            return (TreeNode)getParentNode();
        }
        public boolean isLeaf() {
            return true;
        }
        public String toString() {
            return ((TreeFactory)getFactory()).convertValueToText(this);
        }
    }

    // ================================================

    public static class TreeNotation extends TXNotation implements TreeNode {
        public TreeNotation(String n, ExternalID eid) {
            super(n, eid);
        }
        public Enumeration children() {
            return null;
        }
        public boolean getAllowsChildren() {
            return false;
        }
        public TreeNode getChildAt(int childIndex) {
            return null;
        }
        public int getChildCount() {
            return 0;
        }
        public int getIndex(TreeNode node) {
            return -1;
        }
        public TreeNode getParent() {
            return (TreeNode)getParentNode();
        }
        public boolean isLeaf() {
            return true;
        }
        public String toString() {
            return ((TreeFactory)getFactory()).convertValueToText(this);
        }
    }

    // ================================================

    public static class TreeGeneralReference extends GeneralReference implements TreeNode {
        public TreeGeneralReference(String name) {
            super(name);
        }
        public Enumeration children() {
            return elements();
        }
        public boolean getAllowsChildren() {
            return true;
        }
        public TreeNode getChildAt(int childIndex) {
            return (TreeNode)getChildNodes().item(childIndex);
        }
        public int getChildCount() {
            return getChildNodes().getLength();
        }
        public int getIndex(TreeNode node) {
            int ret = 0;
            for (Node n = getFirstChild();
                 n != null;
                 n = n.getNextSibling()) {
                if (n == node)  return ret;
                ret ++;
            }
            return -1;
        }
        public TreeNode getParent() {
            return (TreeNode)getParentNode();
        }
        public boolean isLeaf() {
            return !hasChildNodes();
        }
        public String toString() {
            return ((TreeFactory)getFactory()).convertValueToText(this);
        }
    }

    // ================================================

    public static class TreePseudoNode extends PseudoNode implements TreeNode {
        public TreePseudoNode(String n) {
            super(n);
        }
        public Enumeration children() {
            return null;
        }
        public boolean getAllowsChildren() {
            return false;
        }
        public TreeNode getChildAt(int childIndex) {
            return null;
        }
        public int getChildCount() {
            return 0;
        }
        public int getIndex(TreeNode node) {
            return -1;
        }
        public TreeNode getParent() {
            return (TreeNode)getParentNode();
        }
        public boolean isLeaf() {
            return true;
        }
        public String toString() {
            return ((TreeFactory)getFactory()).convertValueToText(this);
        }
    }
}
