/*
 * (C) Copyright IBM Corp. 1997-1998  All rights reserved.
 *
 * US Government Users Restricted Rights Use, duplication or
 * disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * IBM will not be liable for any damages suffered by you as a result
 * of using the Program. In no event will IBM be liable for any
 * special, indirect or consequential damages or lost profits even if
 * IBM has been advised of the possibility of their occurrence. IBM
 * will not be liable for any third party claims against you.
 */

package com.ibm.xml.xpointer;

/**
 * The AbsTerm class provides support for absolute location terms in XPointers.  An absolute term 
 * selects one or more elements or locations in an XML document without reference to any 
 * other sub-resource location.
 * <p>The keywords of an absolute term do not depend on the existence of a location source. 
 * They can be used to establish a location source or can serve as self-contained XPointers:
 * <dl>
 * <dt>root()</dt>
 * <dd>The location source is the root element of the containing resource. 
 * If an XPointer omits any leading absolute location term (that is, it consists only of 
 * OtherTerms), it is assumed to have a leading root() absolute location term.</dd>
 * <dt>origin()</dt>
 * <dd>The origin keyword produces a meaningful location source for any following location 
 * terms only if the XPointer is being processed by application software in response to a 
 * request for traversal such as defined in the XLink specification. If an XPointer begins 
 * with origin(), the location source is the sub-resource from which the user initiated 
 * traversal rather than the default root element. This allows XPointers to select abstract 
 * items such as &quot;the next chapter&quot;.</dd>
 * <dt>id(<var>Name</var>)</dt>
 * <dd>The location source is the element in the containing resource with an attribute having 
 * a declared type of ID and a case-insensitive value matching the specified <var>Name</var>.
 * <p>For example, the location term <code>id(a27)</code> chooses the necessarily unique 
 * element of the containing resource which has an attribute declared to be of type ID whose 
 * value is <code>a27</CODE>.</dd>
 * <dt>html(<var>&quot;nameValue&quot;</var>)</dt>
 * <dd>The location source is the first element whose type is <code>A</code> and which has an 
 * attribute called <code>NAME</code> whose value is the same as the supplied case-sensitive
 * <var>nameValue</var>. This is exactly the function performed by the &quot;#&quot; fragment 
 * identifier in the context of an HTML document.</dd>
 * </dl>
 *
 * @version Revision: 72 1.6 src/com/ibm/xml/xpointer/AbsTerm.java, xml4jsrc, xml4j-jtcsv, xml4j_1_1_16 
 * @author TAMURA Kent &lt;kent@trl.ibm.co.jp&gt;
 * @see com.ibm.xml.xpointer.XPointer
 */

public class AbsTerm implements java.io.Serializable {
    
        static final long serialVersionUID = -668186084615794705L;
        int     type    =   -1;     // ST_ROOT or ST_ORIGIN or ST_ID or ST_HTML
        String  param   =   null;   // Name for type=id() or NameValue (SkipLit) for type=html()

    /**
     * Constructor for <code>id(<var>Name</var>)</code> and <code>html(<var>nameValue</var>)</code>.
     * @param   type    An integer representing the keyword type of this absolute term.
     *                  Must be one of: <code>XPointer.ST_ID, or XPointer.ST_HTML</code>.
     * @param   param   If type=ST_ROOTID, <var>Name</var> in <code>id(<var>Name</var>)</code>.
     *                  If type=ST_HTML, <var>nameValue</var> (excluding quotes) in <code>html(<var>nameValue</var>)</code>.
     */
    public AbsTerm(int type, String param) {
        this.type   = type;
        this.param  = param;
    }

    /**
     * Constructor for <code>root()</code> and <code>origin()</code>.
     * @param   type    An integer representing the keyword type of this absolute term.
     *                  Must be one of: <code>XPointer.ST_ROOT, XPointer.ST_ORIGIN</code>.
     */
    public AbsTerm(int type) {
        this(type, null);
    }

    /**
     * Returns the integer representing the keyword type of this absolute term.
     * @return          One of: <code>XPointer.ST_ROOT, XPointer.ST_ORIGIN, XPointer.ST_ID, or XPointer.ST_HTML</code>.
     * @see #getTypeName
     */
    public int getType() {
        return this.type;
    }

    /**
     * Returns the string keyword type of this absolute term.
     * @return          One of: <code>root, origin, id, or html</code>.
     * @see #getType
     */
    public String getTypeName() {
        return XPointer.literals[this.type];
    }

    /**
     * Returns the argument to the keyword of this absolute term.
     * @return          If type=ST_ROOTID, <var>Name</var> in <code>id(<var>Name</var>)</code>.
     *                  If type=ST_HTML, <var>nameValue</var> (excluding quotes) in <code>html(<var>nameValue</var>)</code>.
     *                  Otherwise <code>=null</code>.
     */
    public String getParameter() {
        return this.param;
    }

    /**
     * Returns this absolute term in the form of either:
     * <ul>
     * <li><code>root()</code>
     * <li><code>origin()</code>
     * <li><code>id(<var>name</var>)</code>
     * <li><code>html(<var>&quot;nameValue&quot;</var>)</code>
     * </ul>
     * @return          A string represention of this absolute term.
     */
    public String toString() {
        String p = "";
        if (XPointer.ST_ID == this.type)
            p = this.param;
        else if (XPointer.ST_HTML == this.type)
            p = XPointerParser.makeSkipLit(this.param);
        return XPointer.literals[this.type]+"("+p+")";
    }
}

