/*
 * (C) Copyright IBM Corp. 1997-1998  All rights reserved.
 *
 * US Government Users Restricted Rights Use, duplication or
 * disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * IBM will not be liable for any damages suffered by you as a result
 * of using the Program. In no event will IBM be liable for any
 * special, indirect or consequential damages or lost profits even if
 * IBM has been advised of the possibility of their occurrence. IBM
 * will not be liable for any third party claims against you.
 */

package com.ibm.xml.xpointer;

import org.w3c.dom.Node;
import java.util.Vector;

/**
 * Pointed provides support for strings or Nodes being pointed to by an XPointer.
 *
 * @version Revision: 76 1.5 src/com/ibm/xml/xpointer/Pointed.java, xml4jsrc, xml4j-jtcsv, xml4j_1_1_16 
 * @author TAMURA Kent &lt;kent@trl.ibm.co.jp&gt;
 */
public class Pointed extends Vector {
    
    static final long serialVersionUID = -188837514608910653L;
    public static final int T_NODE          = 0;
    public static final int T_STRINGINNODE  = 1;
    public static final int T_STRING        = 2;

    /**
     * Item provides support for an individual string or Node being pointed to by an XPointer. 
     *
     * @version 1.1
     * @author TAMURA Kent &lt;kent@trl.ibm.co.jp&gt;
     */
    public static class Item {
        
        /**
         * Type of node pointed to.  One of T_NODE, T_STRINGINNODE, or T_STRING
         */
        public  int     type    =   -1;
        
        /**
         * The node being pointed to.
         */
        public  Node    node    =   null;
        
        /**
         * The string being pointed to.  Only valid when type=T_STRINGINNODE or T_STRING
         */
        public  String  string  =   null;
        
        /**
         * Only valid when type=T_STRINGINNODE.
         */
                int     offset  =   -1;
        
        /**
         * Only valid when type=T_STRINGINNODE.
         */
                int     length  =   -1;

        /**
         * Constructor.
         * @param   node    Node being pointed to.  Specify <var>null</var> when <code>type=T_STRING</code>.
         * @param   string  String being pointed to.  Specify <var>null</var> when <code>type=T_NODE</code>.
         * @param   offset  0-based offset into string being pointed to.  Specify 0 when <code>type!=T_STRINGINNODE</code>.
         * @param   length  Length of string being pointed to.  Specify 0 when <code>type!=T_STRINGINNODE</code>.
         */
        public Item(Node node, String string, int offset, int length) {
            this.node = node;
            this.string = string;
            if (node == null)
                this.type = T_STRING;
            else if (string == null)
                this.type = T_NODE;
            else {
                this.type = T_STRINGINNODE;
                this.offset = offset;
                this.length = length;
            }
        }

        /**
         * Returns whether <var>object</var> indicates the same location as this item.
         * @return      <code>=true</code> if <var>object</var> is an item that points
         *              to the same location as this item; otherwise, <code>=false</code>.
         */
        public boolean equals(Object object) {
            if (object == null || !(object instanceof Item))
                return false;
            boolean ret;
            Item it = (Item)object;
            if (this.type != it.type)  return false;
            switch (this.type) {
              case T_NODE:
                ret = this.node == it.node;
                break;
              case T_STRINGINNODE:
                ret = this.node == it.node && this.offset == it.offset && this.length == it.length;
                break;
              case T_STRING:
                ret = this.string.equals(it.string);
                break;
              default:
                ret = false;
            }
            return ret;
        }
    } // Item class

    /**
     * Add that a string in a Node is being pointed to by an XPointer.
     * @param   node    Node being pointed to.
     * @param   string  String being pointed to.
     * @param   offset  0-based offset into string being pointed to.
     * @param   length  Length of string being pointed to.
     * @return          <code>=true</code> if pointer added; <code>=false</code> if pointer already exists.
     */
    public boolean add(Node node, String string, int offset, int length) {
        Item it = new Item(node, string, offset, length);
        int ind = indexOf(it);
        if (ind >= 0) {
            return false;
        } else {
            addElement(it);
            return true;
        }
    }
    
    /**
     * Add that a Node is being pointed to by an XPointer.
     * @param   node    Node being pointed to.
     * @return          <code>=true</code> if pointer added; <code>=false</code> if pointer already exists.
     */
    public boolean add(Node node) {
        return add(node, null, 0, 0);
    }
    
    /**
     * Add that a string is being pointed to by an XPointer.
     * @param   string  String being pointed to.
     * @return          <code>=true</code> if pointer added; <code>=false</code> if pointer already exists.
     */
    public boolean add(String string) {
        return add(null, string, 0, 0);
    }

    /**
     * Returns an individual string or Node being pointed to by an XPointer. 
     * @param   index   Index into the Vector of pointers to strings and Nodes.
     * @return          The pointer to the string or Node at the specified <var>index</var>.
     * @exception       ArrayIndexOutOfBoundsException  Thrown if an invalid <var>index</var> was given.
     */
    public Item item(int index) throws ArrayIndexOutOfBoundsException {
        return (Item)elementAt(index);
    }
}
