/*
 * (C) Copyright IBM Corp. 1997-1998  All rights reserved.
 *
 * US Government Users Restricted Rights Use, duplication or
 * disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * IBM will not be liable for any damages suffered by you as a result
 * of using the Program. In no event will IBM be liable for any
 * special, indirect or consequential damages or lost profits even if
 * IBM has been advised of the possibility of their occurrence. IBM
 * will not be liable for any third party claims against you.
 */

package com.ibm.xml.xpointer;

import com.ibm.xml.parser.TXAttribute;
import com.ibm.xml.parser.TXElement;
import java.util.Enumeration;
import java.util.Vector;
import org.w3c.dom.Attr;
import org.w3c.dom.CDATASection;
import org.w3c.dom.Comment;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.ProcessingInstruction;
import org.w3c.dom.Text;


/**
 * The RelTerm class provides support for relative location terms in XPointers. 
 * A relative term specifies a location in terms of another location, called the location 
 * source.  The location source is the entire resource if there are no preceding location 
 * terms; otherwise it is the location specified by the preceding term (which might be 
 * relative to a location term before that).
 * <p>Each relative location term consists of a keyword, followed by arguments that define 
 * one or more steps. These location terms provide facilities for navigating forward, 
 * backward, up, and down through the element tree. The arguments passed to the keyword 
 * determine which node types from that sequence are chosen. 
 * <p>Each keyword is summarized:
 * <dl>
 * <dt>child</dt>
 * <dd>Identifies direct child nodes of the location source.</dd>
 * <dt>descendant</dt>
 * <dd>Identifies nodes appearing anywhere within the content of the location source.</dd>
 * <dt>ancestor</dt>
 * <dd>Identifies element nodes containing the location source.</dd>
 * <dt>preceding</dt>
 * <dd>Identifies nodes that appear before (preceding) the location source.</dd>
 * <dt>following</dt>
 * <dd>Identifies nodes that appear after (following) the location source.</dd>
 * <dt>psibling</dt>
 * <dd>Identifies sibling nodes (sharing their parent with the location source) that appear 
 * before (preceding) the location source.</dd>
 * <dt>fsibling</dt>
 * <dd>Identifies sibling nodes (sharing their parent with the location source) that appear 
 * after (following) the location source.</dd>
 * </dl>
 *
 * <p>Each of the keywords accepts the following arguments:
 * <ol>
 * <li>InstanceOrAll: For a positive instance number n, the nth of the candidate locations is 
 * identified. For a negative instance number, the candidate locations are counted from last 
 * to first (in a manner that is specific to each keyword). If the instance value <code>all</code> is 
 * given, then all the candidate locations are selected.</li>
 * <li>NodeType: The node type may be specified by one of the following values:
 *   <ul>
 *   <li><var>elementName</var>: Selects a particular case-insensitive XML element type; 
 *   only elements of the specified type will count as candidates. For example, the following 
 *   identifies the 29th paragraph of the fourth sub-division of the third major division of 
 *   the location source: <code>child(3,DIV1).child(4,DIV2).child(29,P)</code></li>
 *   <li>#element: Identifies XML elements. If no NodeType is specified, #element is the 
 *   default. The following example identifies the fifth child element: <code>child(5)</code></li>
 *   <li>#pi: Identifies XML processing instructions. This node type cannot satisfy any 
 *   attribute constraints. The only location term that can meaningfully be used with a PI 
 *   location source is StringTerm.</li>
 *   <li>#comment: Identifies XML comments. This node type cannot satisfy any attribute 
 *   constraints. The only location term that can meaningfully be used with a comment 
 *   location source is StringTerm.</li>
 *   <li>#text: Selects among text regions directly inside elements and CDATA sections. 
 *   This node type cannot satisfy any attribute constraints. The only location term that can 
 *   meaningfully be used with a text-region location source is StringTerm.</li>
 *   <li>#cdata: Selects among text regions found inside CDATA sections. This node type 
 *   cannot satisfy any attribute constraints. The only location term that can meaningfully 
 *   be used with a CDATA-region location source is StringTerm.</li>
 *   <li>#all: Selects among nodes of all the above types. No node but an element can satisfy 
 *   any attribute constraints, so if attribute constraints are provided, #all is 
 *   effectively equivalent to #element.</li>
 *   </ul></li>
 * <li>Attributes: Candidate elements can be selected based on their attribute names and 
 * values. Note that non-element node types have no attributes, and so can never satisfy 
 * selection criteria that include attribute name or value specifications.  See RelTermAttribute
 * for details.</li>
 * </ol>
 * @version Revision: 77 1.7 src/com/ibm/xml/xpointer/RelTerm.java, xml4jsrc, xml4j-jtcsv, xml4j_1_1_16 
 * @author TAMURA Kent &lt;kent@trl.ibm.co.jp&gt;
 * @see com.ibm.xml.xpointer.XPointer
 * @see com.ibm.xml.xpointer.RelTermAttribute
 * @see com.ibm.xml.xpointer.StringTerm
 * @see com.ibm.xml.xpointer.OtherTerm
 */

public class RelTerm implements OtherTerm, java.io.Serializable {
    
        static final long serialVersionUID = -742339128434233398L;
        int     keywordType =   -1;
                                                // Arguments
        boolean isAll       =   false;
        int     instance    =   -1;
        int     nodeType    =   -1;     // XPointer.NT_*;
        String  elementName =   null;   // An element name if nodeType is NT_NAME
                                         
        Vector  attributes  =   null;   // Attribute

    /**
     * Constructor for specifying <var>attributes</var> and for full customization.
     * @param   keywordType Must be one of: <code>XPointer.ST_NONE, ST_CHILD, ST_DESCENDANT,
     *                      ST_ANCESTOR, ST_PRECEDING, ST_PSIBLING, ST_FOLLOWING, ST_FSIBLING</code>.
     * @param   isAll       </code=true</code> means all the candidate locations are selected.
     *                      If <code>=false</code>, <var>instance</var> should be specified.
     * @param   instance    The nth of the candidate locations is selected.
     *                      Only relevant if <var>isAll</var><code>=false</code>.
     * @param   nodeType    Must be one of: XPointer.NT_NAME, NT_ELEMENT, NT_PI, NT_COMMENT, 
     *                      NT_TEXT, NT_CDATA, or NT_ALL. 
     *                      If <code>=NT_NAME</code>, <var>elementName</var> should be specified.
     * @param   elementName Selects a particular XML element type to count as candidates.
     *                      Only relevant if <var>nodeType</var><code>=NT_NAME</code>.
     * @param   attributes  A vector of attribute name and values to further qualify candidate
     *                      elements.
     */
    public RelTerm(int keywordType, boolean isAll, int instance,
                   int nodeType, String elementName, Vector attributes) {
        this.keywordType    = keywordType;
        this.isAll          = isAll;
        this.instance       = instance;
        this.nodeType       = nodeType;
        this.elementName    = elementName;
        this.attributes     = attributes;
    }
    
    /**
     * Constructor for <code><var>keywordType</var>(<var>instance</var>,<var>nodeType</var>)</code>.
     * @param   keywordType Must be one of: <code>XPointer.ST_NONE, ST_CHILD, ST_DESCENDANT,
     *                      ST_ANCESTOR, ST_PRECEDING, ST_PSIBLING, ST_FOLLOWING, ST_FSIBLING</code>.
     * @param   instance    The nth of the candidate locations is selected.
     * @param   nodeType    Must be one of: XPointer.NT_ELEMENT, NT_PI, NT_COMMENT, 
     *                      NT_TEXT, NT_CDATA, or NT_ALL. 
     */
    public RelTerm(int keywordType, int instance, int nodeType) {
        this(keywordType, false, instance, nodeType, null, new Vector());
    }
    
    /**
     * Constructor for <code><var>keywordType</var>(all,<var>nodeType</var>)</code>.
     * @param   keywordType Must be one of: <code>XPointer.ST_NONE, ST_CHILD, ST_DESCENDANT,
     *                      ST_ANCESTOR, ST_PRECEDING, ST_PSIBLING, ST_FOLLOWING, ST_FSIBLING</code>.
     * @param   isAll       Should be <code>=true</code> for this constructor.
     * @param   nodeType    Must be one of: XPointer.NT_ELEMENT, NT_PI, NT_COMMENT, 
     *                      NT_TEXT, NT_CDATA, or NT_ALL. 
     */
    public RelTerm(int keywordType, boolean isAll, int nodeType) {
        this(keywordType, true, -1, nodeType, null, new Vector());
    }
    
    /**
     * Constructor for <code><var>keywordType</var>(<var>instance</var>,<var>elementName</var>)</code>.
     * @param   keywordType Must be one of: <code>XPointer.ST_NONE, ST_CHILD, ST_DESCENDANT,
     *                      ST_ANCESTOR, ST_PRECEDING, ST_PSIBLING, ST_FOLLOWING, ST_FSIBLING</code>.
     * @param   instance    The nth of the candidate locations is selected.
     * @param   elementName Selects a particular XML element type to count as candidates.
     */
    public RelTerm(int keywordType, int instance, String elementName) {
        this(keywordType, false, instance, XPointer.NT_NAME, elementName, new Vector());
    }
    
    /**
     * Constructor for <code><var>keywordType</var>(all,<var>elementName</var>)</code>.
     * @param   keywordType Must be one of: <code>XPointer.ST_NONE, ST_CHILD, ST_DESCENDANT,
     *                      ST_ANCESTOR, ST_PRECEDING, ST_PSIBLING, ST_FOLLOWING, ST_FSIBLING</code>.
     * @param   elementName Selects a particular XML element type to count as candidates.
     */
    public RelTerm(int keywordType, String elementName) {
        this(keywordType, true, -1, XPointer.NT_NAME, elementName, new Vector());
    }
    
    /**
     * Constructor for <code><var>keywordType</var>(<var>instance</var>)</code>.
     * @param   keywordType Must be one of: <code>XPointer.ST_NONE, ST_CHILD, ST_DESCENDANT,
     *                      ST_ANCESTOR, ST_PRECEDING, ST_PSIBLING, ST_FOLLOWING, ST_FSIBLING</code>.
     * @param   instance    The nth of the candidate locations is selected.
     */
    public RelTerm(int keywordType, int instance) {
        this(keywordType, false, instance, -1, null, new Vector());
    }
    
    /**
     * Constructor for <code><var>keywordType</var>(all)</code>.
     * @param   keywordType Must be one of: <code>XPointer.ST_NONE, ST_CHILD, ST_DESCENDANT,
     *                      ST_ANCESTOR, ST_PRECEDING, ST_PSIBLING, ST_FOLLOWING, ST_FSIBLING</code>.
     */
    public RelTerm(int keywordType) {
        this(keywordType, true, -1, -1, null, new Vector());
    }

    /**
     * Returns the integer representing the keyword type of this relative term.
     * @return          One of: <code>XPointer.ST_NONE, ST_CHILD, ST_DESCENDANT, ST_ANCESTOR, 
     *                  ST_PRECEDING, ST_PSIBLING, ST_FOLLOWING, ST_FSIBLING</code>.
     * @see #getTypeName
     * @see #setType
     */
    public int getType() {
        return this.keywordType;
    }

    /**
     * Sets the integer representing the keyword type of this relative term.
     * @param   keywordType One of: <code>XPointer.ST_NONE, ST_CHILD, ST_DESCENDANT, ST_ANCESTOR, 
     *                      ST_PRECEDING, ST_PSIBLING, ST_FOLLOWING, ST_FSIBLING</code>.
     * @see #getType
     * @see #setType
     */
    public void setType(int keywordType) {
        this.keywordType = keywordType;
    }

    /**
     * Returns the string keyword type of this relative term.
     * @return          One of: <var>null</var>, <code>child, descendant, ancestor, preceding,
     *                  psibling, following, fsibling</code>.
     * @see #getType
     * @see #setType
     */
    public String getTypeName() {
        return this.keywordType == XPointer.ST_NONE ? null : XPointer.literals[this.keywordType];
    }

    
    /**
     * Returns whether all candidate locations are selected by this relative term.
     * @return              </code>=true</code> means all the candidate locations are selected.
     *                      If <code>=false</code>, <var>instance</var> is relevant.
     * @see #getInstance
     */
    public boolean isAll() {
        return this.isAll;
    }

    /**
     * Returns the nth of the candidate locations to be selected by this relative term.
     * Only relevant if <var>isAll</var><code>=false</code>.
     * @return              The nth of the candidate locations to select.
     * @see #isAll
     */
    public int getInstance() {
        return this.instance;
    }

    /**
     * Returns the node type to be selected by this relative term.
     * @return              One of: XPointer.NT_NONE, NT_NAME, NT_ELEMENT, NT_PI, NT_COMMENT, 
     *                      NT_TEXT, NT_CDATA, or NT_ALL. 
     *                      If <code>=NT_NAME</code>, <var>elementName</var> is relevant.
     * @see #getElementName
     */
    public int getNodeType() {
        return this.nodeType;
    }

    /**
     * Returns the name of the element to be selected by this relative term.
     * Only relevant if <var>nodeType</var><code>=NT_NAME</code>.
     * @return              The particular XML element type to count as a candidate.
     * @see #getNodeType
     */
    public String getElementName() {
        return this.elementName;
    }

    /**
     * Returns a <code>Vector</code> of attribute names and values to further qualify 
     * candidate elements.
     * @return              A <code>Vector</code> of <code>RelTermAttribute</code>.
     * @see com.ibm.xml.xpointer.RelTermAttribute
     */
    public Vector getAttributesVector() {
        return this.attributes;
    }

    /**
     * Returns this relative term in the form of either:
     * <ul>
     * <li><code><var>keywordType</var>(all)</code>
     * <li><code><var>keywordType</var>(<var>instance</var>)
     * <li><code><var>keywordType</var>(all,<var>elementName</var>)</code>
     * <li><code><var>keywordType</var>(<var>instance</var>,<var>elementName</var>)</code>
     * <li><code><var>keywordType</var>(all,<var>nodeType</var>)</code>
     * <li><code><var>keywordType</var>(<var>instance</var>,<var>nodeType</var>)</code>
     * <li><code><var>keywordType</var>(all,<var>elementName</var>,<var>attributes</var>)</code>
     * <li><code><var>keywordType</var>(<var>instance</var>,<var>elementName</var>,<var>attributes</var>)</code>
     * <li><code><var>keywordType</var>(all,<var>nodeType</var>,<var>attributes</var>)</code>
     * <li><code><var>keywordType</var>(<var>instance</var>,<var>nodeType</var>,<var>attributes</var>)</code>
     * <li>
     * </ul>
     * @return          A string represention of this relative term.
     */
    public String toString() {
        StringBuffer sb = new StringBuffer(48);
        if (this.keywordType != XPointer.ST_NONE)  sb.append(XPointer.literals[this.keywordType]);

        sb.append("(");
        if (this.isAll) {
            sb.append(XPointer.S_ALL);
        } else
            sb.append(this.instance);
        if (this.nodeType != XPointer.NT_NONE) {
            sb.append(",");
            sb.append(this.nodeType == XPointer.NT_NAME
                      ? this.elementName : XPointer.nodetypes[this.nodeType]);
            for (Enumeration en = this.attributes.elements();  en.hasMoreElements();  )
                sb.append(en.nextElement().toString());
        }
        sb.append(")");
        return sb.toString();
    }

    /**
     * Returns whether the specified <var>child</var> matches this relative term.
     * A match is defined if this relative term's:
     * <ul>
     * <li><var>nodeType</var> matches <var>child</var>'s DOM node type, or
     * <var>elementName</var> matches <var>child</var>'s element tag name.
     * <li><var>attributes</var> element qualifiers are satisfied by <var>child</var>'s
     * Attribute Nodes.
     * </ul>
     * @return          <code>=true</code> if all the above conditions are satisfied; 
     *                  otherwise, <code>=false</code>.
     * @see com.ibm.xml.parser.Child
     */
    public boolean match(Node child) {
        boolean ret = false;
        switch (this.nodeType) {
          case XPointer.NT_NAME:
            if (!(child instanceof Element)) break;
            ret = ((Element)child).getTagName().equals(this.elementName);
            break;
          case XPointer.NT_NONE:
          case XPointer.NT_ELEMENT:
            ret = child instanceof Element;
            break;
          case XPointer.NT_PI:
            ret = child instanceof ProcessingInstruction;
            break;
          case XPointer.NT_COMMENT:
            ret = child instanceof Comment;
            break;
          case XPointer.NT_TEXT:
            ret = child instanceof Text;
            break;
          case XPointer.NT_CDATA:
            ret = child instanceof CDATASection;
            break;
          case XPointer.NT_ALL:
            ret = true;
            break;
        }
        if (!ret)  return ret;

        if (this.attributes.size() == 0)
            return true;
        if (!(child instanceof Element))
            return false;

        Element el = (Element)child;
                                                // ret is true;
        for (Enumeration en = this.attributes.elements();  en.hasMoreElements();  ) {
            RelTermAttribute rta = (RelTermAttribute)en.nextElement();
            String name = rta.getName();
            if (name.equals("*")) {
                Enumeration aen = ((TXElement)el).attributeElements();
                switch (rta.getValueType()) {
                  case XPointer.T_IMPLIED:
                    if (aen.hasMoreElements())
                        return false;
                    break;
                  case XPointer.T_ANY:
                    if (!aen.hasMoreElements())
                        return false;
                    break;
                  case XPointer.T_NAME:
                    ret = false;
                    while (aen.hasMoreElements()) {
                        if (((Attr)aen.nextElement()).getValue().equalsIgnoreCase(rta.getValue())) {
                            ret = true;
                            break;
                        }
                    }
                    if (!ret)  return false;
                    break;

                  case XPointer.T_EXACT:
                    ret = false;
                    while (aen.hasMoreElements()) {
                        if (((Attr)aen.nextElement()).getValue().equals(rta.getValue())) {
                            ret = true;
                            break;
                        }
                    }
                    if (!ret)  return false;
                    break;
                }
            } else {
                Enumeration aen = ((TXElement)el).attributeElements();
                Attr attr = null;
                while (aen.hasMoreElements()) {
                    Attr a = (Attr)aen.nextElement();
                    if (a.getName().equals(name)) {
                        attr = a;
                        break;
                    }
                }
                switch (rta.getValueType()) {
                  case XPointer.T_IMPLIED:
                    ret = attr == null;
                    break;
                  case XPointer.T_ANY:
                    ret = attr != null;
                    break;
                  case XPointer.T_NAME:
                    ret = attr != null && attr.getValue().equalsIgnoreCase(rta.getValue());
                    break;
                  case XPointer.T_EXACT:
                    ret = attr != null && attr.getValue().equals(rta.getValue());
                    break;
                }
                if (!ret)  return false;
            }
        }
        return ret;
    }
}
