/*
 * (C) Copyright IBM Corp. 1997-1998  All rights reserved.
 *
 * US Government Users Restricted Rights Use, duplication or
 * disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * IBM will not be liable for any damages suffered by you as a result
 * of using the Program. In no event will IBM be liable for any
 * special, indirect or consequential damages or lost profits even if
 * IBM has been advised of the possibility of their occurrence. IBM
 * will not be liable for any third party claims against you.
 */

package com.ibm.xml.xpointer;

import java.util.Enumeration;
import java.util.Vector;

/**
 * The RelTermAttribute class provides support for attribute qualifiers to elements which
 * are candidates for relative location terms in XPointers.  Refer to RelTerm for details
 * on how relative location terms are specified.
 * <p>Attribute qualifiers consist of two parts:
 * <dl>
 * <dt>Attr</dt>
 * <dd>This is the name of the attribute qualifier.  It can be specified in two ways:
 *   <ul>
 *   <li>*: Matches any attribute name.  For example, the following location term 
 *       selects the first child of the location source for which the attribute <code>TARGET</code> 
 *       has a value: <code>child(1,#element,TARGET,*)</code></li>
 *   <li><var>Name: </var>Matches the specified case-insensitive attribute name.  
 *       For example, the following location term selects the first child with an <code>N</code> 
 *       attribute having the value 2 is chosen; then that element's first child element having
 *       the value 1 for the same attribute is chosen: <code>child(1,#element,N,2).(1,#element,N,1)</code></li>
 *   </ul></dd>
 * <dt>Value</dt>
 * <dd>This is the value of the attribute qualifier.  It can be specified in four ways:
 *   <ul>
 *   <li>#IMPLIED: Matches an unspecified, or defaulted, value.  For example,
 *       the following location term selects the first child of the location source that is 
 *       an <code>FS</code> element for which the <code>RESP</code> attribute has been left 
 *       unspecified: <code>child(1,FS,RESP,#IMPLIED)</code></li>
 *   <li>*: Matches any value, including defaulted.</li>
 *   <li><var>Name: </var>Matches the specified case-insensitive attribute value.</li>  
 *   <li><var>&quot;Name&quot;</var>: Matches the specified case-sensitive attribute value.</li>
 *   </ul></dd>
 * </dl>
 *
 * @version Revision: 78 1.6 src/com/ibm/xml/xpointer/RelTermAttribute.java, xml4jsrc, xml4j-jtcsv, xml4j_1_1_16 
 * @author TAMURA Kent &lt;kent@trl.ibm.co.jp&gt;
 * @see com.ibm.xml.xpointer.RelTerm
 * @see com.ibm.xml.xpointer.XPointer
 */

public class RelTermAttribute implements java.io.Serializable {
    
        static final long serialVersionUID = 1475177589880703773L;
        String  attr        =   null;   // "*" / Name
        int     valueType   =   -1;     // T_IMPLIED / T_ANY / T_NAME / T_EXACT
        String  value       =   null;   // "#IMPLIED" / "*" / Name / a content of SkipLit

    /**
     * Constructor.
     * @param   attr        Name of this attribute qualifier. Specify <code>*</code> to match
     *                      any name.
     * @param   valueType   One of: XPointer.T_IMPLIED, T_ANY, T_NAME, T_EXACT
     * @param   value       Value of this attribute qualifier. Specify <code>#IMPLIED</code>
     *                      to match defaulted values.  Specify <code>*</code> to match any
     *                      value.  Case-sensitive values should be specified without quotes.
     */
    public RelTermAttribute(String attr, int valueType, String value) {
        this.valueType  = valueType;
        this.attr       = attr;
        this.value      = value;
    }

    /**
     * Returns this attribute qualifier's name.
     * @return          Name of this attribute qualifier.
     */
    public String getName() {
        return this.attr;
    }

    /**
     * Returns this attribute qualifier's value type.
     * @return          One of: XPointer.T_IMPLIED, T_ANY, T_NAME, T_EXACT
     */
    public int getValueType() {
        return this.valueType;
    }

    /**
     * Returns this attribute qualifier's value.
     * @return          Value of this attribute qualifier.  Case sensitive values will not
     *                  contain surrounding quotes.
     */
    public String getValue() {
        return this.value;
    }

    /**
     * Returns this attribute qualifier in the form of either:
     * <ul>
     * <li><code>,<var>attr</var>,#IMPLIED</code></li>
     * <li><code>,<var>attr</var>,*</code></li>
     * <li><code>,<var>attr</var>,<var>name</var></code></li>
     * <li><code>,<var>attr</var>,&quot;</var>name</var>&quot;</code></li>
     * </ul>
     * @return          A string represention of this attribute qualifier.  Note: This is
     *                  not a standalone location term.
     * @see com.ibm.xml.xpointer.XPointer#toString
     */
    public String toString() {
        return ","+this.attr+","+(this.valueType == XPointer.T_EXACT ? XPointerParser.makeSkipLit(this.value) : this.value);
    }

}
