/*
 * (C) Copyright IBM Corp. 1997-1998  All rights reserved.
 *
 * US Government Users Restricted Rights Use, duplication or
 * disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * IBM will not be liable for any damages suffered by you as a result
 * of using the Program. In no event will IBM be liable for any
 * special, indirect or consequential damages or lost profits even if
 * IBM has been advised of the possibility of their occurrence. IBM
 * will not be liable for any third party claims against you.
 */

package com.ibm.xml.xpointer;

/**
 * The StringTerm class provides support for string location terms in XPointers.
 * Absolute and relative location terms have as their target one or more elements.  In order
 * to allow text, CDATASections, comments, and PIs to be the target of an XPointer, the string 
 * location term is provided.
 * <p>A string term specifies a location in terms of another location, called the location 
 * source.  The location source is the entire resource if there are no preceding location 
 * terms; otherwise it is the location specified by the preceding term (which might be 
 * relative to a location term before that). 
 * <p>The string term has the single keyword <code>string</code> which selects one or more 
 * strings or positions between strings in the location source.  The following arguments
 * may be passed on the <code>string</code> keyword:
 * <ol>
 * <li>InstanceOrAll: Identifies the nth occurrence of the specified string. For a positive 
 * instance number, it counts right from the beginning of the location source. For a negative 
 * instance number, it counts left from the end of the location source. For the value <code>all</code>, 
 * all occurrences of the string are used as candidates in forming the designated resource.</li>
 * <li>SkipLit: Identifies the candidate string to be found within the location source. A 
 * <code>null</code> SkipLit string is considered to identify the position immediately 
 * preceding each character in the location source. For example, assuming that the element 
 * with ID <code>x37</code> contains the character string &quot;Thomas&quot;, the following 
 * XPointer identifies the position before the third character (&quot;o&quot;):
 * <code>id(x37).string(3,&quot;&quot;)</code></li>
 * <li>Position: Identifies a character offset from the start of the candidate 
 * string(s) to the beginning of the desired final string match. The position number may not 
 * be zero; if omitted, it is assumed to be 1. A positive position number counts right from 
 * the beginning of the specified string. A negative position number counts left from the end 
 * of the string; for example, position -1 is the position immediately preceding the last 
 * character in the match. A position value of <code>end</code> selects the position immediately 
 * <i>following</i> the last character of the match.</li> 
 * <li>Length: Specifies the number of characters to be selected. A length of zero or an 
 * omitted length references a precise point preceding the character indicated by Position.</li>
 * </ol>
 *
 * @version Revision: 80 1.5 src/com/ibm/xml/xpointer/StringTerm.java, xml4jsrc, xml4j-jtcsv, xml4j_1_1_16 
 * @author TAMURA Kent &lt;kent@trl.ibm.co.jp&gt;
 * @see com.ibm.xml.xpointer.XPointer
 * @see com.ibm.xml.xpointer.OtherTerm
 */

public class StringTerm implements OtherTerm, java.io.Serializable {
    
        static final long serialVersionUID = -5213912898288372011L;
        boolean isAll       =   false;
        int     instance    =   -1;
        
        String  string      =   null;
        
        boolean isPosition  =   false;
        boolean isEnd       =   false;
        int     position    =   -1;
        
        int     length      =   -1;

    /**
     * Constructor for specifying <var>length</var> and for full customization.
     * @param       isAll       <code>=true</code> if all occurrences of <var>string</var> are used 
     *                          as candidates in forming the designated resource. If
     *                          <code>=false</code>, <var>instance</var> should be specified.
     * @param       instance    Select the nth occurrence of <var>string</var> in forming the
     *                          designated resource.  Only relevant if <var>isAll</var><code>=false</code>.
     * @param       string      The candidate string to be found within the location source. 
     *                          <code>=null</code> identifies the position immediately 
     *                          preceding each character in the location source.
     * @param       isPosition  <code>=true</code> if position is specified. If
     *                          <code>=false</code>, <var>isEnd</var>, <var>position</var>,
     *                          and <var>length</var> are not relevant.
     * @param       isEnd       <code>=true</code>, selects the position immediately 
     *                          following the last character of the match. If <code>=false</code>,
     *                          <var>position</var> should be specified.
     * @param       position    Character offset from the start of the candidate string(s) to 
     *                          the beginning of the desired final string match.  Only
     *                          relevant if <var>isEnd</var><code>=false</code>.
     * @param       length      Specifies the number of characters to be selected.
     */
    public StringTerm(boolean isAll, int instance, String string, boolean isPosition, boolean isEnd, int position, int length) {
        this.isAll      = isAll;
        this.instance   = instance;
        this.string     = string;
        this.isPosition = isPosition;
        this.isEnd      = isEnd;
        this.position   = position;
        this.length     = length;
    }

    /**
     * Constructor for <code>string(<var>instance</var>,&quot;<var>string</var>&quot;,<var>position</var>)</code>.
     * @param       instance    Select the nth occurrence of <var>string</var> in forming the
     *                          designated resource.  
     * @param       string      The candidate string to be found within the location source. 
     *                          <code>=null</code> identifies the position immediately 
     *                          preceding each character in the location source.
     * @position    position    Character offset from the start of the candidate string(s) to 
     *                          the beginning of the desired final string match.
     */
    public StringTerm(int instance, String string, int position) {
        this(false, instance, string, false, false, position, 0);
    }

    /**
     * Constructor for <code>string(all,&quot;<var>string</var>&quot;,<var>position</var>)</code>.
     * @param       string      The candidate string to be found within the location source. 
     *                          <code>=null</code> identifies the position immediately 
     *                          preceding each character in the location source.
     * @position    position    Character offset from the start of the candidate string(s) to 
     *                          the beginning of the desired final string match.
     */
    public StringTerm(String string, int position) {
        this(true, -1, string, false, false, position, 0);
    }

    /**
     * Constructor for <code>string(<var>instance</var>,&quot;<var>string</var>&quot;,end)</code>.
     * @param       instance    Select the nth occurrence of <var>string</var> in forming the
     *                          designated resource.  
     * @param       string      The candidate string to be found within the location source. 
     *                          <code>=null</code> identifies the position immediately 
     *                          preceding each character in the location source.
     * @param       isEnd       Should be <code>=true</code> for this constructor.
     */
    public StringTerm(int instance, String string, boolean isEnd) {
        this(false, instance, string, false, isEnd, 1, 0);
    }

    /**
     * Constructor for <code>string(all,&quot;<var>string</var>&quot;,end)</code>.
     * @param       string      The candidate string to be found within the location source. 
     *                          <code>=null</code> identifies the position immediately 
     *                          preceding each character in the location source.
     * @param       isEnd       Should be <code>=true</code> for this constructor.
     */
    public StringTerm(String string, boolean isEnd) {
        this(true, -1, string, false, isEnd, 1, 0);
    }

    /**
     * Constructor for <code>string(<var>instance</var>,&quot;<var>string</var>&quot;)</code>.
     * @param       instance    Select the nth occurrence of <var>string</var> in forming the
     *                          designated resource.  
     * @param       string      The candidate string to be found within the location source. 
     *                          <code>=null</code> identifies the position immediately 
     *                          preceding each character in the location source.
     */
    public StringTerm(int instance, String string) {
        this(false, instance, string, false, false, 1, 0);
    }

    /**
     * Constructor for <code>string(all,&quot;<var>string</var>&quot;)</code>.
     * @param       string      The candidate string to be found within the location source. 
     *                          <code>=null</code> identifies the position immediately 
     *                          preceding each character in the location source.
     */
    public StringTerm(String string) {
        this(true, -1, string, false, false, 1, 0);
    }

    /**
     * Returns whether all occurrences of <var>string</var> are used as candidates in forming
     * the designated resource.
     * @return                  <code>=true</code> if all occurrences of <var>string</var> are used 
     *                          as candidates in forming the designated resource. If
     *                          <code>=false</code>, <var>instance</var> has relevance.
     * @see #getInstance
     */
    public boolean isAll() {
        return this.isAll;
    }
    
    /**
     * Returns the nth occurrence of <var>string</var> to be used in forming the designated
     * resource.  Only relevant if <var>isAll</var><code>=false</code>.
     * @return                  The nth occurrence of <var>string</var> in forming the designated.
     *                          resource.  
     * @see #isAll
     */
    public int getInstance() {
        return this.instance;
    }
    
    /**
     * Returns the candidate string to be found within the location source.
     * @return                  The candidate string (excluding quotes) to be found within 
     *                          the location source.  <code>=null</code> identifies the 
     *                          position immediately preceding each character in the location source.
     */
    public String getString() {
        return this.string;
    }
    
    /**
     * Returns whether position is specified.
     * @return                  <code>=true</code> if position is specified. If
     *                          <code>=false</code>, <var>isEnd</var>, <var>position</var>,
     *                          and <var>length</var> are not relevant.
     * @see #isEnd
     * @see #getPosition
     * @see #getLength
     */
    public boolean hasPosition() {
        return this.isPosition;
    }
    
    /**
     * Returns whether to select the position immediately following the last character of
     * the match.  Only relevant if <var>hasPosition</var><code>=true</code>.
     * @return                  <code>=true</code>, selects the position immediately 
     *                          following the last character of the match. 
     *                          If <code>=false</code>, <var>position</var> is relevant.
     * @see #hasPosition
     * @see #getPosition
     * @see #getLength
     */
    public boolean isEnd() {
        return this.isEnd;
    }
    
    /**
     * Returns the character offset from the start of the candidate string(s) to 
     * the beginning of the desired final string match.  Only relevant if 
     * <var>hasPosition</var><code>=true</code> and <var>isEnd</var><code>=false</code>. 
     * @return                  Character offset from the start of the candidate string(s) to 
     *                          the beginning of the desired final string match.  
     * @see #hasPosition
     * @see #isEnd
     * @see #getLength
     */
    public int getPosition() {
        return this.position;
    }
    
    /**
     * Returns the number of characters to be selected.  Only relevant if 
     * <var>hasPosition</var><code>=true</code>.
     * @return                  The number of characters to be selected, or -1 when no length
     *                          is specified.
     * @see #hasPosition
     * @see #isEnd
     * @see #getPosition
     */
    public int getLength() {
        return this.length;
    }

    /**
     * Returns this string term in the form of either:
     * <ul>
     * <li><code>string(all,<var>&quot;string&quot;</var>)</code>
     * <li><code>string(<var>instance</var>,<var>&quot;string&quot;</var>)</code>
     * <li><code>string(all,<var>&quot;string&quot;</var>,end)</code>
     * <li><code>string(<var>instance</var>,<var>&quot;string&quot;</var>,end)</code>
     * <li><code>string(all,<var>&quot;string&quot;</var>,<var>position</var>)</code>
     * <li><code>string(<var>instance</var>,<var>&quot;string&quot;</var>,<var>position</var>)</code>
     * <li><code>string(all,<var>&quot;string&quot;</var>,end,<var>length</var>)</code>
     * <li><code>string(<var>instance</var>,<var>&quot;string&quot;</var>,end,<var>length</var>)</code>
     * <li><code>string(all,<var>&quot;string&quot;</var>,<var>position</var>,<var>length</var>)</code>
     * <li><code>string(<var>instance</var>,<var>&quot;string&quot;</var>,<var>position</var>,<var>length</var>)</code>
     * </ul>
     * @return          A string represention of this string term.
     */
    public String toString() {
        StringBuffer sb = new StringBuffer();
        sb.append(XPointer.literals[XPointer.ST_STRING]);
        sb.append("(");
        sb.append(this.isAll ? XPointer.S_ALL : Integer.toString(this.instance));
        sb.append(",");
        sb.append(XPointerParser.makeSkipLit(this.string));
        if (this.isPosition) {
            sb.append(",");
            sb.append(this.isEnd ? XPointer.S_END : Integer.toString(this.position));
            if (this.length >= 0) {
                sb.append(",");
                sb.append(Integer.toString(this.length));
            }
        }
        sb.append(")");
        return sb.toString();
    }

}
