/*
 * (C) Copyright IBM Corp. 1997-1998  All rights reserved.
 *
 * US Government Users Restricted Rights Use, duplication or
 * disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * IBM will not be liable for any damages suffered by you as a result
 * of using the Program. In no event will IBM be liable for any
 * special, indirect or consequential damages or lost profits even if
 * IBM has been advised of the possibility of their occurrence. IBM
 * will not be liable for any third party claims against you.
 */

package com.ibm.xml.xpointer;

import com.ibm.xml.parser.AttDef;
import com.ibm.xml.parser.Child;
import com.ibm.xml.parser.DTD;
import com.ibm.xml.parser.LibraryException;
import com.ibm.xml.parser.Parent;
import com.ibm.xml.parser.TXDocument;
import com.ibm.xml.parser.TXElement;
import com.ibm.xml.parser.XMLChar;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Vector;
import org.w3c.dom.Attr;
import org.w3c.dom.CDATASection;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * The XPointer class provides support for addressing into the internal structures of XML documents
 * as defined by <a href="http://www.w3.org/TR/1998/WD-xptr-19980303">WD-xptr-19980303</a>.
 * <p>An XPointer consists of a series of location terms, each of which specifies a location,
 * usually relative to the location specified by the prior location term. Each location term
 * has a keyword (such as id, child, ancestor, and so on) and can have arguments such as an
 * instance number, element type, or attribute. For example, the location term <code>child(2,CHAP)</code>
 * refers to the second child element whose type is <code>CHAP</code>.
 * <p>Location terms are classified into absolute terms, relative terms, span terms,
 * attribute terms, and string data terms. An absolute term selects one or more elements or
 * locations in an XML document without reference to any other sub-resource location. A
 * relative or string data term specifies a location in terms of another location, called the
 * location source . The location source is the entire resource if there are no preceding
 * location terms; otherwise it is the location specified by the preceding term (which might
 * be relative to a location term before that).
 *
 * @version Revision: 83 1.13 src/com/ibm/xml/xpointer/XPointer.java, xml4jsrc, xml4j-jtcsv, xml4j_1_1_16 
 * @author TAMURA Kent &lt;kent@trl.ibm.co.jp&gt;
 * @see com.ibm.xml.xpointer.AbsTerm
 * @see com.ibm.xml.xpointer.RelTerm
 * @see com.ibm.xml.xpointer.SpanTerm
 * @see com.ibm.xml.xpointer.AttrTerm
 * @see com.ibm.xml.xpointer.StringTerm
 */
public class XPointer implements java.io.Serializable {

    static final long serialVersionUID = 5186377505549567985L;
    public final static String  S_ALL       = "all";   // InstanceOrAll
    public final static String  S_END       = "end";   // Position

    public final static int     NT_NONE     = 0;
    public final static int     NT_NAME     = 1;
    public final static int     NT_ELEMENT  = 2;
    public final static int     NT_PI       = 3;
    public final static int     NT_COMMENT  = 4;
    public final static int     NT_TEXT     = 5;
    public final static int     NT_CDATA    = 6;
    public final static int     NT_ALL      = 7;
    public final static String  nodetypes[] = {null,
                                               null,
                                               "#element",
                                               "#pi",
                                               "#comment",
                                               "#text",
                                               "#cdata",
                                               "#all",     };

    public final static String  S_ELEMENT   = "element";
    public final static String  S_PI        = "pi";
    public final static String  S_COMMENT   = "comment";
    public final static String  S_TEXT      = "text";
    public final static String  S_CDATA     = "cdata";
    public final static String  S_NIMPLIED  = "#IMPLIED";
    public final static String  S_IMPLIED   = "IMPLIED";

    public final static String  literals[]  = {null,
                                               "root",
                                               "origin",
                                               "id",
                                               "html",
                                               "child",
                                               "descendant",
                                               "ancestor",
                                               "preceding",
                                               "psibling",
                                               "following",
                                               "fsibling",
                                               "span",
                                               "attr",
                                               "string",    };
    public final static int     ST_NONE     = -1;
    public final static int     ST_ROOT     = 1;
    public final static int     ST_ORIGIN   = ST_ROOT+1;
    public final static int     ST_ID       = ST_ORIGIN+1;
    public final static int     ST_HTML     = ST_ID+1;
    public final static int     ST_CHILD    = ST_HTML+1;
    public final static int     ST_DESCENDANT = ST_CHILD+1;
    public final static int     ST_ANCESTOR = ST_DESCENDANT+1;
    public final static int     ST_PRECEDING = ST_ANCESTOR+1;
    public final static int     ST_PSIBLING = ST_PRECEDING+1;
    public final static int     ST_FOLLOWING = ST_PSIBLING+1;
    public final static int     ST_FSIBLING = ST_FOLLOWING+1;
    public final static int     ST_SPAN     = ST_FSIBLING+1;
    public final static int     ST_ATTR     = ST_SPAN+1;
    public final static int     ST_STRING   = ST_ATTR+1;

    public static final int     T_IMPLIED   = 0;
    public static final int     T_ANY       = T_IMPLIED+1;
    public static final int     T_NAME      = T_ANY+1;
    public static final int     T_EXACT     = T_NAME+1;

                    AbsTerm     absTerm     = null;
                    Vector      otherTerms  = null;
                    String      string      = null;

    private transient int       previous    = -1;


    /**
     * Constructor.
     * @param   absTerm     Absolute term of XPointer. Specification of <code>null</code> is allowed.
     * @param   otherTerms  A vector of OtherTerm instances.  Specification of <code>null</code> is allowed.
     * @see com.ibm.xml.xpointer.AbsTerm
     * @see com.ibm.xml.xpointer.OtherTerm
     */
    public XPointer(AbsTerm absTerm, Vector otherTerms) {
        this.absTerm = absTerm;
        this.otherTerms = otherTerms == null ? new Vector() : otherTerms;
    }

    /**
     * Returns the absolute location term of this XPointer.
     * @return      This XPointer's absolute location term, or <code>null</code> if no absolute term.
     * @see #setAbsTerm
     * @see com.ibm.xml.xpointer.AbsTerm
     */
    public AbsTerm getAbsTerm() {
        return this.absTerm;
    }

    /**
     * Sets the absolute location term of this XPointer.
     * @param   absTerm This XPointer's absolute location term, or <code>null</code> if no absolute term.
     * @see #getAbsTerm
     * @see com.ibm.xml.xpointer.AbsTerm
     */
    public void setAbsTerm(AbsTerm absTerm) {
        this.absTerm = absTerm;
        this.string = null;
    }

    /**
     * Returns a Vector of Relterm, SpanTerm, AttrTerm, and StringTerm items which comprise
     * the other terms of this XPointer.
     * @return      This XPointer's other terms, or <var>null</var> if no other terms.
     * @see com.ibm.xml.xpointer.RelTerm
     * @see com.ibm.xml.xpointer.AttrTerm
     * @see com.ibm.xml.xpointer.StringTerm
     * @see com.ibm.xml.xpointer.SpanTerm
     * @see com.ibm.xml.xpointer.OtherTerm
     * @see #appendOtherTerm
     * @see #insertOtherTerm
     * @see #removeLastOtherTerm
     */
    public Vector getOtherTermsVector() {
        return this.otherTerms;
    }

    /**
     * Appends the specified <var>otherTerm</var> to the end of the other terms
     * for this XPointer.
     * @param   otherTerm   The RelTerm, AttrTerm, StringTerm, or SpanTerm to be appended.
     * @see com.ibm.xml.xpointer.RelTerm
     * @see com.ibm.xml.xpointer.AttrTerm
     * @see com.ibm.xml.xpointer.StringTerm
     * @see com.ibm.xml.xpointer.SpanTerm
     * @see com.ibm.xml.xpointer.OtherTerm
     * @see #getOtherTermsVector
     * @see #insertOtherTerm
     * @see #removeLastOtherTerm
     */
    public void appendOtherTerm(OtherTerm otherTerm) {
        if (null == this.otherTerms)
            this.otherTerms = new Vector();
        this.otherTerms.addElement(otherTerm);
        this.string = null;
    }

    /**
     * Inserts the specified <var>otherTerm</var> to the beginning of the other terms
     * for this XPointer.
     * @param   otherTerm   The RelTerm, AttrTerm, StringTerm, or SpanTerm to be inserted.
     * @see com.ibm.xml.xpointer.RelTerm
     * @see com.ibm.xml.xpointer.AttrTerm
     * @see com.ibm.xml.xpointer.StringTerm
     * @see com.ibm.xml.xpointer.SpanTerm
     * @see com.ibm.xml.xpointer.OtherTerm
     * @see #getOtherTermsVector
     * @see #appendOtherTerm
     * @see #removeLastOtherTerm
     */
    public void insertOtherTerm(OtherTerm otherTerm) {
        if (null == this.otherTerms)
            this.otherTerms = new Vector();
        this.otherTerms.insertElementAt(otherTerm, 0);
        this.string = null;
    }

    /**
     * Returns and removes the <var>OtherTerm</var> at the end of this XPointer.
     * @return              The RelTerm, AttrTerm, StringTerm, or SpanTerm that was at the end
     *                      of this XPointer's other term Vector, or <var>null</var> if there
     *                      are no existing other terms.
     * @see com.ibm.xml.xpointer.RelTerm
     * @see com.ibm.xml.xpointer.AttrTerm
     * @see com.ibm.xml.xpointer.StringTerm
     * @see com.ibm.xml.xpointer.SpanTerm
     * @see com.ibm.xml.xpointer.OtherTerm
     * @see #getOtherTermsVector
     * @see #appendOtherTerm
     * @see #insertOtherTerm
     */
    public OtherTerm removeLastOtherTerm() {
        OtherTerm ret = null;
        synchronized (this.otherTerms) {
            int len = this.otherTerms.size();
            if (len > 0) {
                ret = (OtherTerm)this.otherTerms.elementAt(len-1);
                this.otherTerms.removeElementAt(len-1);
            }
        }
        this.string = null;
        return ret;
    }

    /**
     * Returns this XPointer in the form of either:
     * <ul>
     * <li><code><var>absoluteTerm</var></code>
     * <li><code><var>absoluteTerm</var>.<var>otherTerm</var></code>
     * <li><code><var>absoluteTerm</var>.<var>otherTerm</var>.<var>otherTerm</var>...</code>
     * <li><code><var>otherTerm</var></code>
     * <li><code><var>otherTerm</var>.<var>otherTerm</var>...</code>
     * </ul>
     * @return          A string represention of this XPointer.
     * @see com.ibm.xml.xpointer.AbsTerm#toString
     * @see com.ibm.xml.xpointer.RelTerm#toString
     * @see com.ibm.xml.xpointer.AttrTerm#toString
     * @see com.ibm.xml.xpointer.StringTerm#toString
     * @see com.ibm.xml.xpointer.SpanTerm#toString
     */
    public String toString() {
        if (null != this.string) {
            return this.string;
        }
        StringBuffer sb = new StringBuffer(100);
        if (null != this.absTerm) {
            sb.append(this.absTerm.toString());
            if (0 < this.otherTerms.size())
                sb.append(".");
        }
        int i = 0;
        for (Enumeration en = this.otherTerms.elements();  en.hasMoreElements();  ) {
            if (i > 0)  sb.append(".");
            i ++;
            sb.append(en.nextElement().toString());
        }
        return this.string = sb.toString();
    }

    /**
     * Returns the locations in the specified <var>document</var> that are satisfied by this XPointer.
     * <DL>
     *   <DT>The following terms of this XPointer are not currently supported:</DT>
     *   <DD><code>origin()</code></DD>
     *   <DD><code>html()</code></DD>
     *   <DD><code>span()</code></DD>
     *   <DD><code>string()</code></DD>
     * </DL>
     * @param   document    TXDocument instance to be examined.
     * @return              Matched locations (should never be <var>null</var>).
     * @see com.ibm.xml.xpointer.Pointed
     * @see org.w3c.dom.Document
     */
    public synchronized Pointed point(Document document) {
        Node origin = document.getDocumentElement();
        if (this.absTerm == null || this.absTerm.getType() == XPointer.ST_ROOT) {
        } else if (this.absTerm.getType() == XPointer.ST_ORIGIN
                   || this.absTerm.getType() == XPointer.ST_HTML) {
            return new Pointed();               // Not supported.
        } else {                                // ID
            DTD dtd = ((TXDocument)document).getDTD();
            if (dtd == null)  return new Pointed(); // No ID information.
            origin = dtd.checkID(this.absTerm.getParameter());
            if (origin == null)  return new Pointed(); // No such an ID.
        }
        return point(origin);
    }

    /**
     * Returns the locations in the specified <var>originNode</var> that are satisfied by this XPointer.
     * <DL>
     *   <DT>The following terms of this XPointer are not currently supported:</DT>
     *   <DD><code>span()</code></DD>
     *   <DD><code>string()</code></DD>
     * </DL>
     * @param   originNode  Node instance to be examined.
     * @return              Matched locations (should never be <var>null</var>).
     * @see com.ibm.xml.xpointer.Pointed
     * @see com.ibm.xml.parser.Child
     */
    public synchronized Pointed point(Node originNode) {
        Pointed pointed = new Pointed();
        pointed.add(originNode);
        this.previous = XPointer.ST_NONE;
        Enumeration en = this.otherTerms.elements();
        while (en.hasMoreElements()) {
            Object obj = en.nextElement();
            Pointed result = new Pointed();
            Enumeration en2 = pointed.elements();
            while (en2.hasMoreElements()) {
                Pointed.Item it = (Pointed.Item)en2.nextElement();
                if (it.node != null)
                    point(result, obj, (Child)it.node);
            }
            pointed = result;
        }
        return pointed;
    }

    /**
     * Returns an XPointer instance representing the specified <var>targetNode</var>.
     * @param   targetNode  Node to be represented as an XPointer.
     * @return  XPointer instance, or <VAR>null</VAR> if <var>targetNode</var> can not be
     *          represented by an XPointer.
     * @see com.ibm.xml.parser.Child#makeXPointer
     */
    public static XPointer makeXPointer(Child targetNode) {
        if (targetNode.getNodeType() == Node.DOCUMENT_NODE)
            return null;                        // XPointer can't point the whole of a document.
        if (targetNode.getNodeType() == Node.ENTITY_REFERENCE_NODE)
            return null;                        // XPointer can't point entity references.

                                                // Search for root Document
        Child ch = targetNode;
        while (ch.getNodeType() != Node.DOCUMENT_NODE) {
            ch = (Child)ch.getParentNode();
            if (ch == null)
                return null;                    // No document
        }
        TXDocument tdoc = (TXDocument)ch;
        if (targetNode.getParentNode() == tdoc) {   // targetNode is the root element
            return new XPointer(new AbsTerm(XPointer.ST_ROOT, null), null);
        }

        DTD dtd = tdoc.getDTD();
        if (dtd != null) {                      // Examine ID
            Child parent = targetNode.getNodeType() != Node.ELEMENT_NODE
                ? (Child)((Child)targetNode).getParentWithoutReference() : (Child)targetNode;
            do {
                TXElement el = (TXElement)parent;
                Enumeration attrs = el.attributeElements();
                while (attrs.hasMoreElements()) {
                    Attr a = (Attr)attrs.nextElement();
                    AttDef ad = dtd.getAttributeDeclaration(el.getNodeName(), a.getName());
                    if (ad != null && ad.getDeclaredType() == AttDef.ID) {
                        XPointer xp = new XPointer(new AbsTerm(XPointer.ST_ID, a.getValue()), null);
                        if (el != targetNode) {
                            ch = targetNode;
                            RelTerm lastrelterm = null;
                            do {
                                xp.insertOtherTerm(lastrelterm = makeRelTerm(ch));
                                ch = (Child)ch.getParentWithoutReference();
                            } while (ch != el);
                            lastrelterm.setType(XPointer.ST_CHILD);
                        }
                        return xp;
                    }
                }
                parent = (Child)parent.getParentWithoutReference();
            } while (!(parent instanceof TXDocument));
                                                // No ID attribute in parent elements
        }

        TXElement root = (TXElement)tdoc.getDocumentElement();
        XPointer xp = new XPointer(new AbsTerm(XPointer.ST_ROOT, null), null);
        ch = targetNode;
        RelTerm lastrelterm = null;
        do {
            xp.insertOtherTerm(lastrelterm = makeRelTerm(ch));
            ch = (Child)ch.getParentWithoutReference();
        } while (ch != root);
        lastrelterm.setType(XPointer.ST_CHILD);

        return xp;
    }

    private void point(Pointed result, Object obj, Child origin) {
        if (obj instanceof RelTerm) {
            RelTerm rel = (RelTerm)obj;

            int type = rel.getType();
            if (type == XPointer.ST_NONE) {
                if (this.previous == XPointer.ST_NONE) // This XPointer has an error.
                    return;
                type = this.previous;
            }

            int instance = rel.getInstance();
            if (instance == 0)  return;         // Invalid XPointer.
            Vector v;
            Child n;
            switch (type) {
              case XPointer.ST_CHILD:
                if (rel.isAll()) {
                    for (n = (Child)origin.getFirstWithoutReference();
                         n != null;  n = (Child)n.getNextWithoutReference()) {
                        if (rel.match(n))
                            result.add(n);
                    }
                } else if (instance > 0) {
                    for (n = (Child)origin.getFirstWithoutReference();
                         n != null;  n = (Child)n.getNextWithoutReference()) {
                        if (rel.match(n) && --instance == 0) {
                            result.add(n);
                            break;
                        }
                    }
                } else {
                    for (n = (Child)origin.getLastWithoutReference();
                         n != null;  n = (Child)n.getPreviousWithoutReference()) {
                        if (rel.match(n) && ++instance == 0) {
                            result.add(n);
                            break;
                        }
                    }
                }
                break;

              case XPointer.ST_DESCENDANT:
                if (!(origin instanceof Element))
                    return;
                v = new Vector();
                addMatchingNodes(v, (Element)origin, rel);
                if (rel.isAll()) {
                    for (int i = 0;  i < v.size();  i ++)
                        result.add((Node)v.elementAt(i));
                } else if (instance > 0) {
                    if (instance <= v.size())
                        result.add((Node)v.elementAt(instance-1));
                } else {
                    if (instance+v.size() >= 0)
                        result.add((Node)v.elementAt(instance+v.size()));
                }
                break;

              case XPointer.ST_ANCESTOR:
                v = new Vector();
                while (origin.getParentWithoutReference() instanceof Element) {
                    origin = (Child)origin.getParentWithoutReference();
                    if (rel.match(origin))
                        v.addElement(origin);
                }
                if (rel.isAll()) {
                    for (int i = 0;  i < v.size();  i ++)
                        result.add((Node)v.elementAt(i));
                } else if (instance > 0) {
                    if (instance <= v.size())
                        result.add((Node)v.elementAt(instance-1));
                } else {
                    if (instance+v.size() >= 0)
                        result.add((Node)v.elementAt(instance+v.size()));
                }
                break;

              case XPointer.ST_PRECEDING:
                v = new Vector();
                while (origin.getPreviousWithoutReference() != null
                       || origin.getParentWithoutReference() instanceof Element) {
                    if (origin.getPreviousWithoutReference() == null)
                        origin = (Child)origin.getParentWithoutReference();
                    else
                        origin = (Child)origin.getPreviousWithoutReference();
                    if (rel.match(origin))
                        v.addElement(origin);
                }
                if (rel.isAll()) {
                    for (int i = 0;  i < v.size();  i ++)
                        result.add((Node)v.elementAt(i));
                } else if (instance > 0) {
                    if (instance <= v.size())
                        result.add((Node)v.elementAt(instance-1));
                } else {
                    if (instance+v.size() >= 0)
                        result.add((Node)v.elementAt(instance+v.size()));
                }
                break;

              case XPointer.ST_FOLLOWING:
                v = new Vector();
                while (origin.getNextWithoutReference() != null
                       || origin.getParentWithoutReference() instanceof Element) {
                    if (origin.getNextWithoutReference() == null)
                        origin = (Child)origin.getParentWithoutReference();
                    else
                        origin = (Child)origin.getNextWithoutReference();
                    if (rel.match(origin))
                        v.addElement(origin);
                }
                if (rel.isAll()) {
                    for (int i = 0;  i < v.size();  i ++)
                        result.add((Node)v.elementAt(i));
                } else if (instance > 0) {
                    if (instance <= v.size())
                        result.add((Node)v.elementAt(instance-1));
                } else {
                    if (instance+v.size() >= 0)
                        result.add((Node)v.elementAt(instance+v.size()));
                }
                break;

              case XPointer.ST_PSIBLING:
                if (rel.isAll()) {
                    while ((origin = (Child)origin.getPreviousWithoutReference()) != null) {
                        if (rel.match(origin))
                            result.add(origin);
                    }
                } else if (instance > 0) {
                    while ((origin = (Child)origin.getPreviousWithoutReference()) != null) {
                        if (rel.match(origin) && --instance == 0) {
                            result.add(origin);
                            break;
                        }
                    }
                } else {
                    instance = Math.abs(instance);
                    for (n = (Child)((Child)origin.getParentWithoutReference()).getFirstWithoutReference();
                         n != null;
                         n = (Child)n.getNextWithoutReference()) {
                        if (rel.match(n) && ++instance == 0) {
                            result.add(n);
                            break;
                        }
                    }
                }
                break;

              case XPointer.ST_FSIBLING:
                if (rel.isAll()) {
                    while ((origin = (Child)origin.getNextWithoutReference()) != null) {
                        if (rel.match(origin))
                            result.add(origin);
                    }
                } else if (instance > 0) {
                    while ((origin = (Child)origin.getNextWithoutReference()) != null) {
                        if (rel.match(origin) && --instance == 0) {
                            result.add(origin);
                            break;
                        }
                    }
                } else {
                    instance = Math.abs(instance);
                    for (n = (Child)((Child)origin.getParentWithoutReference()).getLastWithoutReference();
                         n != null;
                         n = (Child)n.getPreviousWithoutReference()) {
                        if (rel.match(n) && ++instance == 0) {
                            result.add(n);
                            break;
                        }
                    }
                }
                break;
            }

            this.previous = type;

        } else if (obj instanceof StringTerm) {
            this.previous = XPointer.ST_NONE;
                                                // Not supported yet.

        } else if (obj instanceof SpanTerm) {
            this.previous = XPointer.ST_NONE;
                                                // Not supported.
        } else if (obj instanceof AttrTerm) {
            this.previous = XPointer.ST_NONE;
            if (origin instanceof Element) {
                Attr attr = ((Element)origin).getAttributeNode(((AttrTerm)obj).getName());
                if (attr != null) {
                    result.add(attr.getValue());
                }
            }

        } else {
            throw new LibraryException("com.ibm.xml.xpointer.XPointer#point(): Internal Error: unknown OtherTerm.");
        }
    }

    static private void addMatchingNodes(Vector v, Element el, RelTerm rel) {
        for (Node n = ((Child)el).getFirstWithoutReference();
             n != null;
             n = ((Child)n).getNextWithoutReference()) {
            if (rel.match(n))
                v.addElement(n);
            if (n instanceof Element)
                addMatchingNodes(v, (Element)n, rel);
        }
    }

    /**
     *
     * @param ch A child of el; TXElement or TXText or TXCDATASection or TXComment or TXPI
     * @return A type of this RelTem is always ST_NONE.
     */
    private static RelTerm makeRelTerm(Child ch)
        throws LibraryException, IllegalArgumentException {
        Parent parent = (Parent)ch.getParentWithoutReference();
        int type = ch.getNodeType();
        int count = 1;
        switch (type) {
          case Node.ELEMENT_NODE:
            String name = ch.getNodeName();
            for (Node child = parent.getFirstWithoutReference();
                 child != null;  child = ((Child)child).getNextWithoutReference()) {
                if (child.getNodeType() == type && child.getNodeName().equals(name)) {
                    if (child == ch) {
                        return new RelTerm(XPointer.ST_NONE, count, name);
                    } else
                        count ++;
                }
            }
            break;

          case Node.CDATA_SECTION_NODE:
            for (Node child = parent.getFirstWithoutReference();
                 child != null;  child = ((Child)child).getNextWithoutReference()) {
                if (child.getNodeType() == type) {
                    if (child == ch) {
                        return new RelTerm(XPointer.ST_NONE, count, XPointer.NT_CDATA);
                    } else
                        count ++;
                }
            }
            break;

          case Node.TEXT_NODE:
            for (Node child = parent.getFirstWithoutReference();
                 child != null;  child = ((Child)child).getNextWithoutReference()) {
                if (child.getNodeType() == type) {
                    if (child == ch) {
                        return new RelTerm(XPointer.ST_NONE, count, XPointer.NT_TEXT);
                    } else
                        count ++;
                }
            }
            break;

          case Node.COMMENT_NODE:
          case Node.PROCESSING_INSTRUCTION_NODE:
            for (Node child = parent.getFirstWithoutReference();
                 child != null;  child = ((Child)child).getNextWithoutReference()) {
                if (child.getNodeType() == type) {
                    if (child == ch) {
                        return new RelTerm(XPointer.ST_NONE, count,
                                           type == Node.COMMENT_NODE
                                           ? XPointer.NT_COMMENT
                                           : XPointer.NT_PI);
                    } else
                        count ++;
                }
            }
            break;

          default:
            throw new IllegalArgumentException("com.ibm.xml.xpointer.XPointer#makeRelTerm():");
        }
        throw new LibraryException("com.ibm.xml.xpointer.XPointer#makeRelTerm(): Specified child isn't a child of specified element.");
    }
}

