/*
   rsrc.c - res to rsrc converter

   Name:       RSRC - res to rsrc converter

   Version:    1.02

   Desc:       add/update ".rsrc" section in Win32 executable or library
               from a Win32 resource file (.res)

   Make:       for 32 bit compilers
               - GNU C (emx, djgpp, linux)
               - Watcom C/C++ 10.x

   Copyright (c) 1995-1999 Rainer Schnitker

   This file is part of RSXNT.

   RSXNT is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   RSXNT is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with RSXNT; see the file COPYING.  If not, write to
   the Free Software Foundation, 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <malloc.h>
#include <string.h>
#include <fcntl.h>
#include "port.h"
#include "ntbind.h"

#define INVAL_WCHAR ((wchar_t)0xffff)       /* invalid uni char */
#define INVAL_TYPE  0x7fff                  /* invalid rc type */

#define ROUNDUP(x, size) (((x)+(size-1))&~(size-1))

#pragma pack(1)

/* windows res structures */

struct ResourceHead
{
    unsigned long  DataSize;
    unsigned long  HeaderSize;
    unsigned short FirstWord[1];
};

struct ResourceTail
{
    unsigned long  DataVersion;
    unsigned short MemoryFlags;
    unsigned short LanguageId;
    unsigned long  Version;
    unsigned long  Characteristics;
};

struct ResourceDir
{
    unsigned long  Characteristics;
    unsigned long  TimeDateStamp;
    unsigned long  Version;
    unsigned short NumberOfNamedEntries;
    unsigned short NumberOfIdEntries;
};
#define SIZE_RESDIR (sizeof(struct ResourceDir))

struct ResourceDirEntry
{
    unsigned long  Name;
    unsigned long  OffsetToData;
};
#define SIZE_ENTRY (sizeof(struct ResourceDirEntry))

struct ResourceDataEntry
{
    unsigned long  OffsetToData;
    unsigned long  Size;
    unsigned long  CodePage;
    unsigned long  Reserved;
};
#define SIZE_RESDATA (sizeof(struct ResourceDataEntry))

#pragma pack()

/* internal structures */

struct MyResourceEntry
{
    wchar_t                 *EntryName;
    unsigned short          EntryId;
    unsigned short          LanguageId;
    unsigned long           Characteristics;
    void                    *data;
    unsigned long           size;
};

struct MyResourceType
{
    wchar_t                 *TypeName;
    unsigned short          TypeId;
    unsigned short          NumberOfEntries;
    struct MyResourceEntry  *MyResourceEntry;
};

/* --Variables---------------------------------------------------------- */

static struct MyResourceType *res_types;
static int no_res_types;            /* number of diffrent types */
static int no_res_entries;          /* number of resource entries */
static unsigned long string_space;  /* space for named strings */
static unsigned long data_space;    /* space for data */

/* --Functions---------------------------------------------------------- */

static void out_of_mem(void)
{
    write(1 , "out of memory", 13);
    exit(1);
}

static void *xmalloc (size_t n)
{
    long *q;
    q = malloc (n);
    if (q == NULL)
        out_of_mem ();
    return q;
}

static void *xrealloc (void *p, size_t n)
{
    long *q;
    q = realloc (p, n);
    if (q == NULL)
        out_of_mem ();
    return q;
}

static int u_strlen(wchar_t *u_string)
{
    int len = 0;
    while (u_string[len])
        len++;
    return len;
}

static int u_strcmp(const wchar_t *string1, const wchar_t *string2)
{
    int d;

    for (;;) {
        d = *string1 - *string2;
        if (d != 0 || *string1 == 0 || *string2 == 0)
            return d;
        ++string1;
        ++string2;
    }
}

static int wchar2multi(char *ansi, const wchar_t *uni, size_t n)
{
    int len = wcstombs(ansi, uni, n);

    if (len != -1)
        ansi[len] = '\0';

    return len;
}

static int add_resource(unsigned short TypeId, wchar_t *TypeName,
                unsigned short EntryId, wchar_t *EntryName,
                void * data, unsigned long size,
                struct ResourceTail *ResTail)
{
    int t, e;
    struct MyResourceEntry *res_entry;

    ++no_res_entries;
    /* FK: leave room for alignment */
    data_space += (size +3) & ~3;

    for (t = 0; t < no_res_types; t++)
        if (TypeId == res_types[t].TypeId)
            break;

    /* alloc a new resource type (if not found) */
    if (t == no_res_types) {
        if (no_res_types == 0)
            res_types = (struct MyResourceType *)
                xmalloc(sizeof(struct MyResourceType));
        else
            res_types = (struct MyResourceType *)
                xrealloc(res_types,
                (no_res_types + 1) * sizeof(struct MyResourceType));
        res_types[t].TypeName = TypeName;
        res_types[t].TypeId = TypeId;
        res_types[t].NumberOfEntries = 0;
        res_types[t].MyResourceEntry = (struct MyResourceEntry *)
                xmalloc(sizeof(struct MyResourceEntry));
        no_res_types++;
    }

    /* add new entry */
    e = res_types[t].NumberOfEntries++;
    res_types[t].MyResourceEntry = (struct MyResourceEntry *)
        xrealloc(res_types[t].MyResourceEntry,
                sizeof(struct MyResourceEntry) * (e + 1));

    res_entry = & res_types[t].MyResourceEntry[e];
    res_entry->EntryName = EntryName;
    res_entry->EntryId = EntryId;
    res_entry->LanguageId = ResTail->LanguageId;
    res_entry->Characteristics = ResTail->Characteristics;
    res_entry->data = data;
    res_entry->size = size;
    return 1;
}

static int build_tree(char *resfile)
{
    int handle;
    char *resources;
    int i, size, len;
    struct ResourceHead *ResHead;
    struct ResourceTail *ResTail;
    wchar_t *TypeName;
    wchar_t *EntryName;
    unsigned short TypeId, EntryId;
    void *pData;

    if ((handle = open(resfile, O_RDONLY | O_BINARY)) == -1) {
        printf("Could not open '%s'", resfile);
        return -1;
    }
    else {      /* test bad watcom wrc.exe */
        char watcomrc[8] = {0xd7, 0xc1, 0xd4, 0xc3, 0xcf, 0xcd, 0xd2, 0xc3};
        char magic[8];

        read(handle, magic, 8);
        if (strncmp(watcomrc, magic, 8) == 0) {
            puts("Watcom Resources are not supported\n");
            close(handle);
            return -1;
        }
        else    /* are there other bad rc ?? */
            lseek(handle, 0, SEEK_SET);
    }

    size = filelength(handle);

    if ((resources = (char *) malloc(size)) == 0) {
        puts("out of memory");
        return -1;
    }

    if (read(handle, resources, size) != size) {
        free(resources);
        close(handle);
        printf("Error reading from '%s'", resfile);
        return -1;
    } else
        close(handle);

    for (i = 0; i < size; ) {
        ResHead = (struct ResourceHead *) &resources[i];
        ResTail = (struct ResourceTail *)
            &resources[i + ResHead->HeaderSize - sizeof(*ResTail)];

        TypeName = (wchar_t *) (&ResHead->FirstWord[0]);

        if (*TypeName == INVAL_WCHAR) {     /* ordinal */
            TypeId = TypeName[1];
            EntryName = (wchar_t *) (&ResHead->FirstWord[2]);
        }
        else {                              /* uni-string */
            TypeId = INVAL_TYPE;
            len = u_strlen(TypeName) + 1;
            EntryName = (wchar_t *) & TypeName[len];
            string_space += len * sizeof(wchar_t);
        }
        if (*EntryName == INVAL_WCHAR) {
            EntryId = EntryName[1];
        }
        else {
            EntryId = INVAL_TYPE;
            len = u_strlen(EntryName) + 1;
            string_space += len * sizeof(wchar_t);
        }

        if (ResHead->DataSize > 0) {
            pData = & resources[i + ResHead->HeaderSize];
            if (!add_resource(TypeId, TypeName, EntryId, EntryName,
                    pData, ResHead->DataSize, ResTail)) {
                puts("out of memory error");
                free(resources);
                return -1;
            }
        }

        i += (ResHead->HeaderSize + ResHead->DataSize);

        /* align 4 */
        while(i % 4)
            i++;
    }

    return 0;
}

static int fcmp_types(const void *ele1, const void *ele2)
{
    struct MyResourceType *p1 = (struct MyResourceType *) ele1;
    struct MyResourceType *p2 = (struct MyResourceType *) ele2;

    if (p1->TypeId == INVAL_TYPE && p2->TypeId != INVAL_TYPE)
        return -1;
    else if (p1->TypeId != INVAL_TYPE && p2->TypeId == INVAL_TYPE)
        return +1;
    else if (p1->TypeId != INVAL_TYPE && p2->TypeId != INVAL_TYPE)
        return (p1->TypeId < p2->TypeId) ? -1 : +1;
    else
        return u_strcmp(p1->TypeName, p2->TypeName);
}

static int fcmp_entry(const void *ele1, const void *ele2)
{
    struct MyResourceEntry *p1 = (struct MyResourceEntry *) ele1;
    struct MyResourceEntry *p2 = (struct MyResourceEntry *) ele2;

    if (p1->EntryId == INVAL_TYPE && p2->EntryId != INVAL_TYPE)
        return -1;
    else if (p1->EntryId != INVAL_TYPE && p2->EntryId == INVAL_TYPE)
        return +1;
    else if (p1->EntryId != INVAL_TYPE && p2->EntryId != INVAL_TYPE)
        return (p1->EntryId < p2->EntryId) ? -1 : +1;
    else
        return u_strcmp(p1->EntryName, p2->EntryName);
}

static void sort_tree()
{
    int t;
    qsort(res_types, no_res_types, sizeof(struct MyResourceType), fcmp_types);

    for (t = 0; t < no_res_types; t++) {
        if (res_types[t].NumberOfEntries > 1)
            qsort(res_types[t].MyResourceEntry, res_types[t].NumberOfEntries,
                    sizeof(struct MyResourceEntry), fcmp_entry);
    }
}

static int print_resource()
{
    int t, e;
    struct MyResourceEntry *res_entry;
    char name[260];

    for (t = 0; t < no_res_types; t++) {
        puts("---------------------------------------------");
        if (res_types[t].TypeId != INVAL_TYPE) {
            printf("type id %X\n", res_types[t].TypeId);
        }
        else {
            wchar2multi(name, res_types->TypeName, 260);
            printf("Type name '%s'\n", name);
        }

        for (e = 0; e < res_types[t].NumberOfEntries; e++) {
            res_entry = & res_types[t].MyResourceEntry[e];
            if (res_entry->EntryId != INVAL_TYPE) {
                printf("Entry id %u\n", res_entry->EntryId);
            }
            else {
                wchar2multi(name, res_entry->EntryName, 260);
                printf("Entry name '%s'\n", name);
            }
            printf("sizeof Data = %lX\n", res_entry->size);
            printf("lang id %X\n", res_entry->LanguageId);
            printf("charact %lX\n", res_entry->Characteristics);
        }
        fflush(stdout);
    }
    return 0;
}

static int skip_exe_hdr(int filehandle, unsigned long *headoff)
{
    struct exe_hdr exehdr;

    read(filehandle, &exehdr, sizeof(struct exe_hdr));

    if (exehdr.signatur == 0x5a4d) {    /* falls exe-kopf */
        unsigned long new_off;

        lseek(filehandle, 0x3C, SEEK_SET);
        read(filehandle, &new_off, sizeof(long));

        if (new_off) {
            DWORD pe_magic;
            lseek(filehandle, new_off, SEEK_SET);
            read(filehandle, &pe_magic, sizeof(pe_magic));
            if (pe_magic == 0x4550) {
                *headoff = new_off + 4;
                return 0;
            }
        }
    }   /* exe files */
    *headoff = 0;
    return -1;
}

static int write_resdata(int handle, unsigned long offset, unsigned long size, unsigned long vaddr)
{
    struct MyResourceEntry *MyResEntry;
    struct ResourceDir ResDir;
    struct ResourceDirEntry ResEntry;
    struct ResourceDataEntry ResDataEntry;
    char *mem, *p;
    unsigned long stroff, nextoff;
    wchar_t *wc;
    int e, t, len;

    mem = p = xmalloc(size);
    stroff = offset;

    /**** write level 1 directory (type ids) ****/

    ResDir.Characteristics = 0;
    ResDir.TimeDateStamp = 0;
    ResDir.Version = 0x00000004;
    ResDir.NumberOfNamedEntries = 0;
    ResDir.NumberOfIdEntries = 0;

    for (t = 0; t < no_res_types; t++) {
        if (res_types[t].TypeId == INVAL_TYPE)
            ++ResDir.NumberOfNamedEntries;
        else
            ++ResDir.NumberOfIdEntries;
    }

    memcpy(p, &ResDir, SIZE_RESDIR);
    p += SIZE_RESDIR;

    /* write level 1 entries */

    nextoff = SIZE_RESDIR + no_res_types * SIZE_ENTRY;

    for (t = 0; t < no_res_types; t++) {
        if (res_types[t].TypeId == INVAL_TYPE) {
            ResEntry.Name = 0x80000000 | stroff;
            len = u_strlen(res_types[t].TypeName);
            wc = (wchar_t *) (mem + stroff);
            wc[0] = len;
            memcpy(wc + 1, res_types[t].TypeName, len * 2);
            stroff += sizeof(wchar_t) * (len + 1);
        } else
            ResEntry.Name = res_types[t].TypeId;
        ResEntry.OffsetToData = 0x80000000 | nextoff;
        nextoff += SIZE_RESDIR + (res_types[t].NumberOfEntries * SIZE_ENTRY);

        memcpy(p, &ResEntry, SIZE_ENTRY);
        p += SIZE_ENTRY;
    }

    /* write level 2  */

    for (t = 0; t < no_res_types; t++) {
        ResDir.Characteristics = 0;
        ResDir.TimeDateStamp = 0;
        ResDir.Version = 0x00000004;
        ResDir.NumberOfNamedEntries = 0;
        ResDir.NumberOfIdEntries = 0;


        for (e = 0; e < res_types[t].NumberOfEntries; e++) {
            MyResEntry = & res_types[t].MyResourceEntry[e];

            if (MyResEntry->EntryId == INVAL_TYPE)
                ++ResDir.NumberOfNamedEntries;
            else
                ++ResDir.NumberOfIdEntries;
        }
        memcpy(p, &ResDir, SIZE_RESDIR);
        p += SIZE_RESDIR;

        for (e = 0; e < res_types[t].NumberOfEntries; e++) {
            MyResEntry = & res_types[t].MyResourceEntry[e];

            if (MyResEntry->EntryId == INVAL_TYPE) {
                ResEntry.Name = 0x80000000 | stroff;
                len = u_strlen(MyResEntry->EntryName);
                wc = (wchar_t *) (mem + stroff);
                wc[0] = len;
                memcpy(wc + 1, MyResEntry->EntryName, len * 2);
                stroff += sizeof(wchar_t) * (len + 1);
            } else
                ResEntry.Name = MyResEntry->EntryId;
            ResEntry.OffsetToData = 0x80000000 | nextoff;
            nextoff += SIZE_RESDIR + SIZE_ENTRY;

            memcpy(p, &ResEntry, SIZE_ENTRY);
            p += SIZE_ENTRY;
        }
    }


    /* write level 3 directories (language dir) */

    for (t = 0; t < no_res_types; t++) {
        for (e = 0; e < res_types[t].NumberOfEntries; e++) {
            MyResEntry = & res_types[t].MyResourceEntry[e];

            ResDir.Characteristics = MyResEntry->Characteristics;
            ResDir.TimeDateStamp = 0;
            ResDir.Version = 0x00000004;
            ResDir.NumberOfNamedEntries = 0;
            ResDir.NumberOfIdEntries = 1;

            memcpy(p, &ResDir, SIZE_RESDIR);
            p += SIZE_RESDIR;

            ResEntry.Name = MyResEntry->LanguageId;
            ResEntry.OffsetToData = nextoff;
            nextoff += SIZE_RESDATA;

            memcpy(p, &ResEntry, SIZE_ENTRY);
            p += SIZE_ENTRY;
        }
    }


    /* write leaf nodes */

    nextoff = offset + string_space;

    for (t = 0; t < no_res_types; t++) {
        for (e = 0; e < res_types[t].NumberOfEntries; e++) {
            MyResEntry = & res_types[t].MyResourceEntry[e];

            ResDataEntry.OffsetToData = vaddr + nextoff;
            ResDataEntry.Size = MyResEntry->size;
            ResDataEntry.CodePage = 0;
            ResDataEntry.Reserved = 0;

            /* FK: leave room for aligment */   
            nextoff += (MyResEntry->size + 3) & ~3;
            memcpy(p, &ResDataEntry, SIZE_RESDATA);
            p += SIZE_RESDATA;
        }
    }

    /* write leaf data */

    nextoff = offset + string_space;

    for (t = 0; t < no_res_types; t++) {
        for (e = 0; e < res_types[t].NumberOfEntries; e++) {
            MyResEntry = & res_types[t].MyResourceEntry[e];

            memcpy(mem + nextoff, MyResEntry->data, MyResEntry->size);
            /* FK: leave room for aligment */   
            nextoff += (MyResEntry->size + 3) & ~3;
        }
    }


    /* ok, lets go */
    write(handle, mem, size);
    free(mem);
    return 0;
}

static int write_resource(char *exefile)
{
    int             handle, bNew, i;
    FILEHDR         file_hdr;
    NTOPTHDR        ntopt_hdr;
    SCNHDR          secres;
    unsigned long   headoff, res_hdr_off;
    unsigned long   offset, vaddr;

    offset = SIZE_RESDIR + no_res_types * SIZE_ENTRY;       /* Level2 */

    offset += no_res_types * SIZE_RESDIR;
    offset += no_res_entries * SIZE_ENTRY;                  /* Level3 */

    offset += no_res_entries * (SIZE_RESDIR + SIZE_ENTRY);  /* Leaf nodes */

    offset += no_res_entries * SIZE_RESDATA;                /* first free */
    string_space = (string_space + 3) & ~3;

    handle = open(exefile, O_RDWR | O_BINARY);
    if (handle == -1) {
        printf("cannot open '%s'\n", exefile);
        close(handle);
        return -1;
    }

    if (skip_exe_hdr(handle, &headoff) < 0) {
        puts("no a valid Win32 file");
        return (-1);
    }

    lseek(handle, headoff, SEEK_SET);
    read(handle, &file_hdr, sizeof(FILEHDR));

    if (file_hdr.f_magic != 0x14C) {
        puts("no COFF header found");
        close(handle);
        return (-1);
    }

    read(handle, &ntopt_hdr, sizeof(NTOPTHDR));

    if (ntopt_hdr.Magic != 0x10b) {
        puts("no optional header found");
        close(handle);
        return (-1);
    }

    if (ntopt_hdr.CheckSum) {
        puts("can't handle checksum field != 0");
        close(handle);
        return (-1);
    }

    /* test if last section is .rsrc */

    res_hdr_off = headoff + sizeof(FILEHDR) + sizeof(NTOPTHDR);
    res_hdr_off += (file_hdr.f_nscns - 1) * sizeof(SCNHDR);

    for (i = 0; i < file_hdr.f_nscns - 1; i++) {
        read(handle, &secres, sizeof(SCNHDR));
        if (strncmp(secres.s_name, ".rsrc", 5) == 0) {
            puts(".rsrc section is not the last entry!\nGive up!");
            close(handle);
            return -1;
        }
    }
    read(handle, &secres, sizeof(SCNHDR));

    bNew = 1;
    if (strncmp(secres.s_name, ".rsrc", 5) == 0) {
        bNew = 0;

        chsize(handle, secres.s_scnptr);
        close(handle);
        handle = open(exefile, O_RDWR | O_BINARY);

        vaddr = secres.s_vaddr;
        puts("update .rsrc section");
    }
    else {
        puts("add .rsrc section");
        vaddr = secres.s_vaddr + ROUNDUP(secres.s_size, ntopt_hdr.SectionAlignment);
        res_hdr_off += sizeof(SCNHDR);
    }

    memcpy(secres.s_name, ".rsrc\0\0\0", 8);
    secres.s_paddr = offset + string_space + data_space;
    secres.s_vaddr = vaddr;
    secres.s_size = ROUNDUP(secres.s_paddr, ntopt_hdr.FileAlignment);
    secres.s_scnptr = ROUNDUP(filelength(handle), ntopt_hdr.FileAlignment);
    secres.s_relptr = 0;
    secres.s_lnnoptr = 0;
    secres.s_nreloc = 0;
    secres.s_nlnno = 0;
    secres.s_flags = 0xC0000040L;   /* read, write; data */

    if (bNew)
        ntopt_hdr.SizeOfImage += ROUNDUP(secres.s_size, ntopt_hdr.SectionAlignment);
    ntopt_hdr.DataDirectory[DIR_RESOURCE].VirtualAddress = secres.s_vaddr;
    ntopt_hdr.DataDirectory[DIR_RESOURCE].Size = secres.s_paddr;

    if (bNew) {
        file_hdr.f_nscns++;
        lseek(handle, headoff, SEEK_SET);
        write(handle, &file_hdr, sizeof(FILEHDR));
    }
    else {
        lseek(handle, headoff + sizeof(FILEHDR), SEEK_SET);
    }
    write(handle, &ntopt_hdr, sizeof(NTOPTHDR));
    lseek(handle, res_hdr_off, SEEK_SET);
    write(handle, &secres, sizeof(SCNHDR));

    lseek(handle, secres.s_scnptr, SEEK_SET);
    write_resdata(handle, offset, secres.s_size, secres.s_vaddr);

    close(handle);
    return 0;
}

int main(int argc, char **argv)
{
    char *resfile = NULL;
    char *exefile = NULL;
    int i;
    int opt_dumpres = 0;
    int opt_touch = 0;

    for (i = 1; i < argc; i++) {
        char *p = strchr(argv[i], '.');

        if (p) {
            if (stricmp(p, ".res") == 0)
                resfile = argv[i];
            else if (stricmp(p, ".exe") == 0 || stricmp(p, ".dll") == 0)
                exefile = argv[i];
        }
        else if (stricmp(argv[i], "--dump") == 0)   /* dump resource */
            opt_dumpres = 1;
        else if (stricmp(argv[i], "--touch") == 0)  /* dump resource */
            opt_touch = 1;
    }

    if (resfile == NULL || (exefile == NULL && !opt_dumpres)) {
        puts("rsrc version 1.50 (c) 1996-1999 Rainer Schnitker");
        puts("usage:");
        puts("\trsrc [options] resfile exefile");
        puts("\t\toptions:");
        puts("\t\t--dump: dumps resfile (does not touch exefile)");
        puts("\t\t--touch: creates .rce file (for makefiles)");
        return -1;
    }

    if (access(resfile, 0) < 0) {
        puts("cannot found resource file");
        return 1;
    }

    if (exefile != NULL && access(exefile, 0) < 0) {
        puts("cannot found exe file");
        return 1;
    }

    if (build_tree(resfile) != -1) {
        sort_tree();
        if (opt_dumpres) {
            print_resource();
            return 0;
        }
        write_resource(exefile);

        if (opt_touch) {
            int len = strlen(resfile);
            while (len > 0 && resfile[len] != '.')
                --len;
            strcpy(resfile + len, ".rce");
            close (open (resfile,
                O_WRONLY | O_CREAT | O_TRUNC | O_BINARY, 0666));
        }
    }
    return 0;
}
