/*
** Module   :MMF.H
** Abstract :Memory Mapped Files Interface
**
** Copyright (C) Sergey I. Yevtushenko
**
** Log: Wed  03/09/97       Created
*/

#include <mmf.h>

#define INCL_DOSPROCESS
#define INCL_DOSEXCEPTIONS
#define INCL_ERRORS
#include <os2.h>
#include <stdio.h>

/* Internal structures and constants */

#define PAG_SIZE    4096
#define PAG_MASK    0xFFFFF000

typedef struct
{
    ULONG ulFlags;
    HFILE hFile;
    PVOID pData;
    ULONG ulSize;
} MMFENTRY;

typedef MMFENTRY* PMMF;

/* Static data */

MMFENTRY mmfTable[MMF_MAX];

/* Local prototypes */

ULONG APIENTRY PageFaultHandler(PEXCEPTIONREPORTRECORD p1,
                                PEXCEPTIONREGISTRATIONRECORD p2,
                                PCONTEXTRECORD p3,
                                PVOID pv);

static PMMF Locate(void *addr);
static PMMF LocateFree(void);

/* Local functions implementation */

PMMF Locate(void *addr)
{
    int i;
    for(i = 0; i < MMF_MAX; i++)
    {
        if(mmfTable[i].ulFlags & MMF_USEDENTRY)
        {
            if(   (ULONG)mmfTable[i].pData <= (ULONG)addr
               && ((ULONG)mmfTable[i].pData+mmfTable[i].ulSize) >= (ULONG)addr)
            {
                return &mmfTable[i];
            }
        }
    }
    return 0;
}

PMMF LocateFree(void)
{
    int i;
    for(i = 0; i < MMF_MAX; i++)
    {
        if(!(mmfTable[i].ulFlags & MMF_USEDENTRY))
        {
            return &mmfTable[i];
        }
    }
    return 0;
}


/* API Implementation */

void DosInitMMF(void *excrec)
{
    EXCEPTIONREGISTRATIONRECORD * RegRec;
    APIRET  rc;

    RegRec = (EXCEPTIONREGISTRATIONRECORD *)excrec;

    RegRec->ExceptionHandler = (ERR) PageFaultHandler;
    rc = DosSetExceptionHandler(RegRec);
}

int DosAllocMMF(char *file, void **ppdata, int opt)
{
    FILESTATUS3 fsts3ConfigInfo = {{0}};
    ULONG  ulBufSize = sizeof(FILESTATUS3);
    APIRET rc    = NO_ERROR;
    ULONG  ulAct = 0;

    HFILE  hFile = 0;
    PVOID  pData = 0;
    PMMF   pMMF  = 0;

/* Check parameters */

    if(!file || opt > MMF_READWRITE || !ppdata)
        return ERROR_INVALID_PARAMETER;

/* Locate free entry in table */

    pMMF = LocateFree();

    if(!pMMF)
        return ERROR_MMF_TOO_MANY_MMF_OPEN;

/* Open file */

    rc = DosOpen((PCSZ)file,
                 &hFile,
                 &ulAct,
                 0L,
                 FILE_ARCHIVED | FILE_NORMAL,
                 OPEN_ACTION_FAIL_IF_NEW | OPEN_ACTION_OPEN_IF_EXISTS,
                 OPEN_FLAGS_NOINHERIT |
                 OPEN_SHARE_DENYNONE  |
                 OPEN_ACCESS_READWRITE,
                 0L);

    if(rc)
        return rc;

/* Query file size */

    rc = DosQueryFileInfo( hFile,
                           FIL_STANDARD,
                           &fsts3ConfigInfo,
                           ulBufSize);

    if(rc || fsts3ConfigInfo.cbFile == 0)
    {
        DosClose(hFile);
        return (rc) ? rc: ERROR_MMF_ZERO_FILE;
    }

/* Allocate memory */

    rc = DosAllocMem(&pData, fsts3ConfigInfo.cbFile, PAG_READ | PAG_WRITE);

    if(rc)
    {
        DosClose(hFile);
        return rc;
    }

/* Everything seems ok, fill data structure and return pointer to memory */

    pMMF->ulFlags   = opt | MMF_USEDENTRY;
    pMMF->hFile     = hFile;
    pMMF->pData     = pData;
    pMMF->ulSize    = fsts3ConfigInfo.cbFile;

    *ppdata = pData;

    return NO_ERROR;
}

int DosFreeMMF(void *pdata)
{
    PMMF pMMF = 0;

    pMMF = Locate(pdata);

    if(!pMMF)
        return ERROR_MMF_INVALID_REGION;

/* No automatic update when area freed

    if(pMMF->ulFlags & MMF_READWRITE)
        DosUpdateMMF(pdata);

*/
    DosClose(pMMF->hFile);
    DosFreeMem(pMMF->pData);

    pMMF->ulFlags   = 0;
    pMMF->hFile     = 0;
    pMMF->pData     = 0;
    pMMF->ulSize    = 0;

    return NO_ERROR;
}

int DosUpdateMMF(void *pdata)
{
    PMMF   pMMF   = 0;
    PBYTE  pArea  = 0;
    ULONG  ulPos  = 0;
    ULONG  ulFlag = 0;
    ULONG  ulSize = PAG_SIZE;
    APIRET rc     = NO_ERROR;

    pMMF = Locate(pdata);

    if(!pMMF)
        return ERROR_MMF_INVALID_REGION;

/* locate all regions which needs update, and actually update them */

    for(pArea = (PBYTE)pMMF->pData; ulPos < pMMF->ulSize; )
    {
        rc = DosQueryMem(pArea, &ulSize, &ulFlag);
        if(rc)
            return rc;

        if(ulFlag & PAG_WRITE)
        {
            ULONG ulTemp = 0;
            /* set pointer */
            rc = DosSetFilePtr(pMMF->hFile,
                               (ULONG)pArea - (ULONG)pMMF->pData,
                               FILE_BEGIN,
                               &ulTemp);
            if(!rc)
            {
                ULONG  ulWrote = 0;

                if(ulPos - pMMF->ulSize >= PAG_SIZE)
                    rc = DosWrite(pMMF->hFile,
                                  pArea,
                                  PAG_SIZE,
                                  &ulWrote);
                else
                    rc = DosWrite(pMMF->hFile,
                                  pArea,
                                  ulPos - pMMF->ulSize,
                                  &ulWrote);
                if(rc)
                    return rc;
            }
        }

        ulPos += PAG_SIZE;
        pArea += PAG_SIZE;
    }

    return NO_ERROR;
}

/* Main worker :) */


#define TRACE {printf("%s(%d) - %d\n", __FILE__,__LINE__,rc);}


ULONG APIENTRY PageFaultHandler(PEXCEPTIONREPORTRECORD p1,
							PEXCEPTIONREGISTRATIONRECORD p2,
							PCONTEXTRECORD p3,
							PVOID pv)
{
    if( p1->ExceptionNum == XCPT_ACCESS_VIOLATION
      && ( p1->ExceptionInfo[0] == XCPT_WRITE_ACCESS
        || p1->ExceptionInfo[0] == XCPT_READ_ACCESS))
    {
        PMMF   pMMF   = 0;
        PVOID  pPage  = 0;
        APIRET rc     = NO_ERROR;
        ULONG  ulFlag = 0;
        ULONG  ulSize = PAG_SIZE;

        pMMF = Locate((void *)p1->ExceptionInfo[1]);

        if(!pMMF)
            return XCPT_CONTINUE_SEARCH;

        pPage = (PVOID)(p1->ExceptionInfo[1] & PAG_MASK);

/* Query affected page flags */

        rc = DosQueryMem(pPage, &ulSize, &ulFlag);

        if(rc)
            return XCPT_CONTINUE_SEARCH;

/*
** There can be three cases:
**
**  1. We trying to read page              - always OK, commit it
**  2. We trying to write committed page   - OK if READ/WRITE mode
**  3. We trying to write uncommitted page - OK if READ/WRITE mode
**                                           but we need to commit it.
*/

/* filter out case 2 */

        if(p1->ExceptionInfo[0] == XCPT_WRITE_ACCESS
          && !(pMMF->ulFlags & MMF_READWRITE))
        {
            return XCPT_CONTINUE_SEARCH;
        }

/* if page not committed, commit it and mark as readonly */

        if(!(ulFlag & PAG_COMMIT))
        {
            ULONG ulTemp = 0;

            rc = DosSetMem(pPage, PAG_SIZE, PAG_COMMIT | PAG_READ | PAG_WRITE);

            if(rc)
                return XCPT_CONTINUE_SEARCH;

            /* set position */

            rc = DosSetFilePtr(pMMF->hFile,
                               (ULONG)pPage - (ULONG)pMMF->pData,
                               FILE_BEGIN,
                               &ulTemp);

            /* read page from disk */

            if(!rc)  /* Actually ignore errors here */
            {
                rc = DosRead(pMMF->hFile,
                             pPage,
                             PAG_SIZE,
                             &ulTemp);
            }

            rc = DosSetMem(pPage, PAG_SIZE, PAG_READ);

            if(rc)
                return XCPT_CONTINUE_SEARCH;

        }

/* if page already committed, and accessed for writing - mark them writable */

        if(p1->ExceptionInfo[0] == XCPT_WRITE_ACCESS)
        {
            rc = DosSetMem(pPage, PAG_SIZE, PAG_READ | PAG_WRITE);
        }
        return XCPT_CONTINUE_EXECUTION;
    }
    return XCPT_CONTINUE_SEARCH;
}

