#include <time.h>
#include <sys/time.h>
#include <sys/timext.h>
#include <assert.h>

#define TIME_T_MAX 0xffffffffUL

/* Return true iff adding A to T does not overflow time_t. */
#define ADD_OK(t,a) \
    ((a) < 0 ? (t) >= (time_t)(-(a)) \
   : (a) > 0 ? (t) <= TIME_T_MAX - (a) \
   : 1)

struct _dstswitch {
  time_t time;                  /* UTC */
  int shift;
};

static struct _tzinfo {
  char tzname[4];
  char dstzname[4];
  int tz, dst, shift;
  int sm, sw, sd, st;
  int em, ew, ed, et;
  } 
_tzi;

static struct _dstswitch _dstsw[2*(2106 - 1970 +1)+2] = 
				{{0,0}, {TIME_T_MAX, 0}};
static int _dstsw_count = 2;

	static struct _dstswitch *
find_switch(time_t t)
{
  int lo, hi, i;

  if (t == TIME_T_MAX)
    {
      i = _dstsw_count - 2;
      assert (_dstsw[i].time <= t);
      assert (t <= _dstsw[i+1].time);
    }
  else
    {
      lo = 0; hi = _dstsw_count - 2;
      for (;;)
        {
          i = (lo + hi) / 2;
          if (_dstsw[i].time > t)
            hi = i - 1;
          else if (_dstsw[i+1].time <= t)
            lo = i + 1;
          else
            break;
        }
      assert (_dstsw[i].time <= t);
      assert (t < _dstsw[i+1].time);
    }
  return &_dstsw[i];
}

/* Note that this function returns -1 on error (overflow). */
	extern int 
_localtime2gmt(time_t *p, int is_dst)
{
  time_t x;
  struct _dstswitch *sw;
  int count;

  if (is_dst > 0) {
      /* Our caller says that *P is specified as DST.  Compute UTC
         from *P, assuming that daylight saving applies. */

      if (!ADD_OK (*p, _tzi.tz - _tzi.shift))
        return -1;
      x = *p + _tzi.tz - _tzi.shift;
      sw = find_switch (x);
      *p = x;
      return sw->shift != 0;
  } else 
    if (is_dst == 0) {
      /* Our caller says that *P is specified as standard time.
         Compute UTC from *P, assuming that daylight saving does not
         apply. */

      if (!ADD_OK (*p, _tzi.tz))
        return -1;
      x = *p + _tzi.tz;
      sw = find_switch (x);
      *p = x;
      return sw->shift != 0;
  } else {
   /* Our caller does not know whether *P is specified as DST or
         standard time.  Try to find out.  First try DST. */
      count = 0;
      if (ADD_OK (*p, _tzi.tz - _tzi.shift)) {
          x = *p + _tzi.tz - _tzi.shift;
          sw = find_switch (x);
          if (sw->shift != 0) {
              *p = x;
              return 1;         /* DST */
          }
          ++count;
      }

      if (ADD_OK (*p, _tzi.tz))
        {
          x = *p + _tzi.tz;
          sw = find_switch (x);
          if (sw->shift == 0)
            {
              *p = x;
              return 0;         /* Not DST */
            }
          ++count;
        }

      if (count != 2)
        return -1;              /* Overflow */

      /* Assume that DST does not apply in the gap.  This makes moving
         into the gap from below work, but breaks moving into the gap
         from above.  The advantage of this choice is that ftime()
         works correctly in the gap if the clock is not adjusted. */

      *p += _tzi.tz;
      return 0;                 /* Not DST */
    }
}

