/* popen.c -- popen/pclose for OS/2, originally by Glenn Gribble. */
/* $Id: popen.c,v 1.1 2000/04/03 18:17:44 veit Exp $ */

#define INCL_DOS
#include <os2.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/param.h>
#include <unistd.h>
#include <process.h>
#include <io.h>
#include <stdio.h>
#include <stdiox.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/errnox.h>
#include <sys/debugx.h>

/* First, a singly linked-list to keep track of pipes */
	/* Map Handles onto termination codes. */
struct _hfPipe_llist {
  HFILE hfPipe;   /* key */
  APIRET rc;    /* val */
  struct _hfPipe_llist *prev; 
};
	/* The __popen_LL*() functions use it. */
static
struct _hfPipe_llist * hfPipeLL_p = (struct _hfPipe_llist*)NULL;


	static __inline__ void
__popen_LLinsert(HFILE hfPipe, APIRET rc)
{
	struct _hfPipe_llist *
  new = (struct _hfPipe_llist*)malloc(sizeof(struct _hfPipe_llist));

  new->hfPipe  = hfPipe;
  new->rc   = rc;
  new->prev = hfPipeLL_p; /* we point to the previously first entry */

  hfPipeLL_p = new; /* Now we are the first entry in the linked list */
}
	static __inline__ void
__popen_LLdelete(HFILE hfPipe)
{
 struct _hfPipe_llist *this = hfPipeLL_p; /* initialize to first entry */
 struct _hfPipe_llist *last = (struct _hfPipe_llist*)NULL;

  while (this) {
    if (this->hfPipe != hfPipe) {
      /* try the next one. */
      last = this;
      this = this->prev;
      continue;
    }
    /* Delete this node and leave. */
    if (last)
      last->prev = this->prev;
    else /* deleted first entry */
      hfPipeLL_p = this->prev;
    free(this);

    return;
  }
}
	static __inline__ ULONG
__popen_LLlookup(HFILE key)
{
 struct _hfPipe_llist * this = hfPipeLL_p;

  while (this) {
    if (this->hfPipe == key)
      return(this->rc);

    this = this->prev;
  }
  /* Zero is special in this context anyway. */
  return 0; /* not found */
}

#ifdef DEBUG
	static ULONG
__popen_LLlength(void)
{
  struct _hfPipe_llist *this = hfPipeLL_p;
  size_t len;

  for (len = 0; this; ++len)
    this = this->prev;

  return len;
}

	extern ULONG
__popen_LLprint(void)
{
  struct _hfPipe_llist *this = hfPipeLL_p;
  size_t i;

  for (i = 0; this; ++i) {
      printf ("hfPipe_ll[%d] == (%5d --> %5d)\n",
			  i, this->hfPipe, this->rc);
      this = this->prev;
  }
  if (i == 0)
    printf ("No entries.\n");

  return i;
}
#endif

/*
 *  Routine: popen
 *  Returns: FILE pointer to pipe.
 *  Action : Exec program connected via pipe, connect a FILE * to the
 *           pipe and return it.
 *  Params : Command - Program to run
 *           Mode    - Mode to open pipe.  "r" implies pipe is connected
 *                     to the programs stdout, "w" connects to stdin.
 */
	FILE *
_popen(const char *Command, const char *Mode)
{
 HFILE End1, End2, Old,Tmp, StdIO;
 APIRET rc =
	DosCreatePipe(&End1, &End2, PIPE_BUF);
		if (rc) {errno = _rc2Errno(rc); return (FILE*)NULL;}
 
	StdIO = (*Mode == 'w') ? STDIN_FILENO : 
 		 ((*Mode == 'r') ? STDOUT_FILENO : (HFILE)-1);

	if (StdIO == (HFILE)-1) {
		errno = EINVAL;
		return (FILE*)NULL;
	}
	if (StdIO == STDIN_FILENO) {
		Tmp = End1; End1 = End2; End2 = Tmp;
	}

	Old = (HFILE)-1; /* save stdin or stdout to Old */
	rc = DosDupHandle(StdIO, &Old);
		if (rc) {errno = _rc2Errno(rc); return (FILE*)NULL;}
	rc = DosSetFHState(Old, OPEN_FLAGS_NOINHERIT);
		if (rc) {errno = _rc2Errno(rc); return (FILE*)NULL;}
	
	Tmp = StdIO; /* redirect stdin or stdout to End2 */
	rc = DosDupHandle(End2, &Tmp);
		if (rc) {errno = _rc2Errno(rc); return (FILE*)NULL;}
	rc = DosClose(End2);
		if (rc) {errno = _rc2Errno(rc); return (FILE*)NULL;}
	rc = DosSetFHState(End1, OPEN_FLAGS_NOINHERIT);
		if (rc) {errno = _rc2Errno(rc); return (FILE*)NULL;}
	
 {char CmdLine[MAXPATHLEN], *CmdLine_p;
 RESULTCODES Result;
 FILE* File;
 const char *CmdExe = getenv("EMXSHELL");

	if (CmdExe == NULL)
		if ((CmdExe = getenv("COMSPEC")) == NULL )
				CmdExe = "cmd.exe";
	if (! strcpy(CmdLine, CmdExe) )
		return (FILE*)NULL;
	CmdLine_p = strdup(CmdLine); strlwr(CmdLine_p);
	if (!CmdLine_p)
		return(FILE*)NULL;
	if (strstr(CmdLine_p, "cmd.exe") || strstr(CmdLine_p, "4os2.exe"))
		strcat(CmdLine, " /c ");
	else
		strcat(CmdLine, " -c ");
	free(CmdLine_p);
	strcat(CmdLine, Command);
	_dPrintf("libext: popen(): Executing %s... \n", CmdLine);

	CmdLine[strlen(CmdLine) + 1] = '\0'; /* two zeroes */
	rc = DosExecPgm(CmdLine, MAXPATHLEN, EXEC_ASYNCRESULT, 
				CmdLine, 0, &Result, CmdExe);
	Tmp = StdIO; /* restore stdin or stdout */
	DosDupHandle(Old, &Tmp);
	DosClose(Old);

	if (rc) {
		DosClose(End1);
		_dPrintf("libext: popen(): DosExecPgm(): rc = %lu... \n", rc);
		return (FILE*)NULL;
	}
	__popen_LLinsert((PID) End1, (ULONG) Result.codeTerminate);
	/* Emx does not allow mixing operating system handles and
	 * C library handles, so we have to convert.
	 */
	File = fdopen(_imphandle((int)End1), Mode);


	return(File);
 }
}

/*
 *  Routine: popenRW
 *  Returns: PID of child process
 *  Action : Exec program connected via pipe, connect int fd's to 
 *           both the stdin and stdout of the process.
 *  Params : Command - Program to run
 *           Pipes   - Array of 2 ints to store the pipe descriptors
 *                     Pipe[0] writes to child's stdin,
 *                     Pipe[1] reads from child's stdout/stderr
 */
	extern int
_popenRW(const char *const *argv, int *pipes)
{
 HFILE Out1, Out2, In1, In2;
 HFILE Old0 = (HFILE)-1, Old1 = (HFILE)-1, Old2 = (HFILE)-1, Tmp;
 APIRET rc;
 int pid;

    if ( (rc = DosCreatePipe(&Out2, &Out1, 4096)) )
        {errno = _rc2Errno(rc); return(-1);}

    if ( (rc = DosCreatePipe (&In1, &In2, 4096)) )    {
        DosClose (Out1);
        DosClose (Out2);
        {errno = _rc2Errno(rc); return(-1);}
    }

    /* Save std{in,out,err} */
    DosDupHandle (STDIN_FILENO, &Old0);
    DosSetFHState (Old1, OPEN_FLAGS_NOINHERIT);
    DosDupHandle (STDOUT_FILENO, &Old1);
    DosSetFHState (Old2, OPEN_FLAGS_NOINHERIT);
    /* DosDupHandle (STDERR_FILENO, &Old2);
    DosSetFHState (Old2, OPEN_FLAGS_NOINHERIT); */

    /* Redirect std{in,out,err} */
    Tmp = STDIN_FILENO;
    DosDupHandle(In1, &Tmp);
    Tmp = STDOUT_FILENO;
    DosDupHandle(Out1, &Tmp);
    /* Tmp = STDERR_FILENO;
    DosDupHandle(Out1, &Tmp); */

    /* Close file handles not needed in child */
    
    DosClose(In1);
    DosClose(Out1);
    DosSetFHState (In2, OPEN_FLAGS_NOINHERIT);
    DosSetFHState (Out2, OPEN_FLAGS_NOINHERIT);

    /* Spawn we now our hoary brood. */
    pid = spawnvp(P_NOWAIT, argv[0], argv);

    /* Restore std{in,out} */
    Tmp = STDIN_FILENO;
    DosDupHandle (Old0, &Tmp);
    DosClose (Old0);
    Tmp = STDOUT_FILENO;
    DosDupHandle (Old1, &Tmp);
    DosClose (Old1);

    if (pid == (-1)) {
        DosClose (In2);
        DosClose (Out2);
        return(-1);
    }
    
    pipes[0] = In2;
    _setmode(pipes[0], O_BINARY);
    pipes[1] = Out2;
    _setmode(pipes[1], O_BINARY);

    /* Save ID of write-to-child pipe for pclose() */
    __popen_LLinsert(In2, (APIRET)pid);
    
    return(pid);
}


/*
 *  Routine: pclose
 *  Returns: TRUE on success
 *  Action : Close a pipe opened with popen();
 *  Params : Pipe - pipe to close
 */
	extern int
_pclose(FILE *Pipe) 
{
    RESULTCODES rc;
    PID hfPipe, hfPipe1;
    int Handle = fileno (Pipe);

    fclose(Pipe);

    rc.codeTerminate = (APIRET)-1;

    hfPipe1 = (PID) __popen_LLlookup ((PID) Handle);
    /* if hfPipe1 is zero, something's seriously wrong */
    if (hfPipe1 != 0) {
        DosWaitChild (DCWA_PROCESSTREE, DCWW_WAIT, &rc, &hfPipe, hfPipe1);
        __popen_LLdelete((PID) Handle);
    }
    return(rc.codeTerminate == 0 ? (int)rc.codeResult : -1);
}


#if DIAGNOSTIC
	void
main(void)
{
  FILE *fp1, *fp2, *fp3;
  int c;

  __popen_LLprint();
  fp1 = popen ("gcc --version", "r");
  __popen_LLprint();
  fp2 = popen ("link386 /?", "r");
  __popen_LLprint();
  fp3 = popen ("dir", "r");
  __popen_LLprint();

  while ((c = getc (fp1)) != EOF)
    printf ("%c", c);

  while ((c = getc (fp2)) != EOF)
    printf ("%c", c);

  while ((c = getc (fp3)) != EOF)
    printf ("%c", c);

  pclose (fp1);
  __popen_LLprint();
  pclose (fp2);
  __popen_LLprint();
  pclose (fp3);
  __popen_LLprint();

  return;
}

#endif /* DIAGNOSTIC */

	extern FILE * 
popen(const char*Command, const char*Mode) { return _popen(Command, Mode);}

	extern int
pclose(FILE *Pipe) { return _pclose(Pipe);}
